<?php
/**
 * Security Utilities
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Utilities;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Security utilities for input validation and output escaping.
 */
class QCNC_Security {

    /**
     * Verify nonce.
     *
     * @param string $nonce_action Nonce action.
     * @param string $nonce_field  Nonce field name.
     * @return bool Verification status.
     */
    public function verify_nonce( $nonce_action, $nonce_field = 'qcnc_nonce' ) {
        $nonce = isset( $_REQUEST[ $nonce_field ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $nonce_field ] ) ) : '';
        
        if ( ! wp_verify_nonce( $nonce, $nonce_action ) ) {
            return false;
        }

        return true;
    }

    /**
     * Check user capability.
     *
     * @param string $capability Required capability.
     * @return bool Has capability.
     */
    public function check_capability( $capability = 'manage_woocommerce' ) {
        return current_user_can( $capability );
    }

    /**
     * Verify nonce and capability.
     *
     * @param string $nonce_action Nonce action.
     * @param string $capability   Required capability.
     * @param string $nonce_field  Nonce field name.
     * @return bool Verification status.
     */
    public function verify_request( $nonce_action, $capability = 'manage_woocommerce', $nonce_field = 'qcnc_nonce' ) {
        if ( ! $this->verify_nonce( $nonce_action, $nonce_field ) ) {
            return false;
        }

        if ( ! $this->check_capability( $capability ) ) {
            return false;
        }

        return true;
    }

    /**
     * Sanitize product cost data.
     *
     * @param array $data Raw data.
     * @return array Sanitized data.
     */
    public function sanitize_cost_data( $data ) {
        return [
            'product_id'      => isset( $data['product_id'] ) ? absint( $data['product_id'] ) : 0,
            'variation_id'    => isset( $data['variation_id'] ) ? absint( $data['variation_id'] ) : null,
            'cost'            => isset( $data['cost'] ) ? floatval( $data['cost'] ) : 0.0,
            'packaging_cost'  => isset( $data['packaging_cost'] ) ? floatval( $data['packaging_cost'] ) : 0.0,
            'handling_cost'   => isset( $data['handling_cost'] ) ? floatval( $data['handling_cost'] ) : 0.0,
            'currency'        => isset( $data['currency'] ) ? sanitize_text_field( $data['currency'] ) : get_woocommerce_currency(),
        ];
    }

    /**
     * Sanitize report filters.
     *
     * @param array $filters Raw filters.
     * @return array Sanitized filters.
     */
    public function sanitize_report_filters( $filters ) {
        $sanitized = [];

        if ( isset( $filters['date_from'] ) ) {
            $sanitized['date_from'] = sanitize_text_field( $filters['date_from'] );
        }

        if ( isset( $filters['date_to'] ) ) {
            $sanitized['date_to'] = sanitize_text_field( $filters['date_to'] );
        }

        if ( isset( $filters['limit'] ) ) {
            $sanitized['limit'] = absint( $filters['limit'] );
        }

        if ( isset( $filters['product_id'] ) ) {
            $sanitized['product_id'] = absint( $filters['product_id'] );
        }

        if ( isset( $filters['category_id'] ) ) {
            $sanitized['category_id'] = absint( $filters['category_id'] );
        }

        return $sanitized;
    }

    /**
     * Escape data for CSV export.
     *
     * @param mixed $value Value to escape.
     * @return string Escaped value.
     */
    public function escape_csv( $value ) {
        if ( is_array( $value ) ) {
            $value = implode( ', ', $value );
        }

        $value = str_replace( '"', '""', $value );
        
        if ( strpos( $value, ',' ) !== false || strpos( $value, '"' ) !== false || strpos( $value, "\n" ) !== false ) {
            $value = '"' . $value . '"';
        }

        return $value;
    }

    /**
     * Sanitize array of integers.
     *
     * @param array $array Raw array.
     * @return array Sanitized array.
     */
    public function sanitize_int_array( $array ) {
        if ( ! is_array( $array ) ) {
            return [];
        }

        return array_map( 'absint', $array );
    }

    /**
     * Validate date format.
     *
     * @param string $date   Date string.
     * @param string $format Expected format.
     * @return bool Valid status.
     */
    public function validate_date( $date, $format = 'Y-m-d' ) {
        $d = \DateTime::createFromFormat( $format, $date );
        return $d && $d->format( $format ) === $date;
    }

    /**
     * Send JSON error response and die.
     *
     * @param string $message Error message.
     * @param int    $code    HTTP status code.
     */
    public function json_error( $message, $code = 400 ) {
        wp_send_json_error( [ 'message' => $message ], $code );
    }

    /**
     * Send JSON success response and die.
     *
     * @param mixed $data Response data.
     * @param int   $code HTTP status code.
     */
    public function json_success( $data, $code = 200 ) {
        wp_send_json_success( $data, $code );
    }
}
