<?php
/**
 * Logger
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Utilities;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Logging utility for debugging and error tracking.
 */
class QCNC_Logger {

    /**
     * Log file path.
     *
     * @var string
     */
    private $log_file;

    /**
     * Debug mode enabled.
     *
     * @var bool
     */
    private $debug_enabled;

    /**
     * Constructor.
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->log_file = $upload_dir['basedir'] . '/qcnc-logs/debug.log';
        $this->debug_enabled = get_option( 'qcnc_enable_debug_logging', false );

        // Create log directory if it doesn't exist.
        $log_dir = dirname( $this->log_file );
        if ( ! file_exists( $log_dir ) ) {
            wp_mkdir_p( $log_dir );
        }
    }

    /**
     * Log a message.
     *
     * @param string $message Message to log.
     * @param string $level   Log level (info, warning, error).
     */
    public function log( $message, $level = 'info' ) {
        if ( ! $this->debug_enabled && 'error' !== $level ) {
            return;
        }

        $timestamp = current_time( 'Y-m-d H:i:s' );
        $formatted_message = sprintf(
            "[%s] [%s] %s\n",
            $timestamp,
            strtoupper( $level ),
            $message
        );

        // Write to file.
        // error_log( $formatted_message, 3, $this->log_file );

        // Also log errors to PHP error log.
        if ( 'error' === $level ) {
            // error_log( 'QCNC Error: ' . $message );
        }
    }

    /**
     * Log an error.
     *
     * @param string $message Error message.
     */
    public function error( $message ) {
        $this->log( $message, 'error' );
    }

    /**
     * Log a warning.
     *
     * @param string $message Warning message.
     */
    public function warning( $message ) {
        $this->log( $message, 'warning' );
    }

    /**
     * Log an info message.
     *
     * @param string $message Info message.
     */
    public function info( $message ) {
        $this->log( $message, 'info' );
    }

    /**
     * Get log file path.
     *
     * @return string Log file path.
     */
    public function get_log_file() {
        return $this->log_file;
    }

    /**
     * Clear log file.
     *
     * @return bool Success status.
     */
    public function clear_log() {
        if ( file_exists( $this->log_file ) ) {
            return wp_delete_file( $this->log_file );
        }
        return true;
    }

    /**
     * Get recent log entries.
     *
     * @param int $lines Number of lines to retrieve.
     * @return array Log entries.
     */
    public function get_recent_logs( $lines = 100 ) {
        if ( ! file_exists( $this->log_file ) ) {
            return [];
        }

        $file = file( $this->log_file );
        if ( false === $file ) {
            return [];
        }

        return array_slice( $file, -$lines );
    }

    /**
     * Cleanup old logs.
     *
     * @param int $days Retention period in days.
     * @return int Number of deleted logs (or 0 if not implemented/applicable).
     */
    public function cleanup_old_logs( $days = 30 ) {
        if ( ! file_exists( $this->log_file ) ) {
            return 0;
        }

        // Simple implementation: if file stays too large, maybe truncate it? 
        // For now, since it's a single file, we can't easily delete "old" rows without parsing.
        // If the file is modified longer than $days ago, delete it?
        
        $mtime = filemtime( $this->log_file );
        if ( $mtime && ( time() - $mtime ) > ( $days * DAY_IN_SECONDS ) ) {
            $this->clear_log();
            return 1; // 1 file deleted
        }

        return 0;
    }
}
