<?php
/**
 * CSV Handler
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Utilities;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Handles CSV import and export operations.
 */
class QCNC_CSV_Handler {

    /**
     * Parse CSV file.
     *
     * @param string $file_path File path.
     * @return array|false Parsed data or false on failure.
     */
    public function parse_csv( $file_path ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $data = [];
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        $handle = fopen( $file_path, 'r' );

        if ( false === $handle ) {
            return false;
        }

        // Get headers.
        $headers = fgetcsv( $handle );
        if ( false === $headers ) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
            fclose( $handle );
            return false;
        }

        // Parse rows.
        while ( ( $row = fgetcsv( $handle ) ) !== false ) {
            if ( count( $row ) === count( $headers ) ) {
                $data[] = array_combine( $headers, $row );
            }
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose( $handle );
        return $data;
    }

    /**
     * Validate CSV data for cost import.
     *
     * @param array $data CSV data.
     * @return array Validation result with errors.
     */
    public function validate_cost_import( $data ) {
        $errors = [];
        $valid_rows = [];

        foreach ( $data as $index => $row ) {
            $row_errors = [];
            $row_num = $index + 2; // Account for header row.

            // Validate product_id.
            if ( empty( $row['product_id'] ) || ! is_numeric( $row['product_id'] ) ) {
                $row_errors[] = "Row {$row_num}: Invalid or missing product_id";
            } else {
                $product = wc_get_product( absint( $row['product_id'] ) );
                if ( ! $product ) {
                    $row_errors[] = "Row {$row_num}: Product #{$row['product_id']} not found";
                }
            }

            // Validate cost.
            if ( ! isset( $row['cost'] ) || ! is_numeric( $row['cost'] ) ) {
                $row_errors[] = "Row {$row_num}: Invalid or missing cost";
            }

            // Validate variation_id if present.
            if ( ! empty( $row['variation_id'] ) && ! is_numeric( $row['variation_id'] ) ) {
                $row_errors[] = "Row {$row_num}: Invalid variation_id";
            }

            if ( empty( $row_errors ) ) {
                $valid_rows[] = $row;
            } else {
                $errors = array_merge( $errors, $row_errors );
            }
        }

        return [
            'valid'       => empty( $errors ),
            'errors'      => $errors,
            'valid_rows'  => $valid_rows,
            'total_rows'  => count( $data ),
            'valid_count' => count( $valid_rows ),
            'error_count' => count( $data ) - count( $valid_rows ),
        ];
    }

    /**
     * Generate CSV content from array.
     *
     * @param array  $data    Data to export.
     * @param array  $headers Headers (optional).
     * @return string CSV content.
     */
    public function generate_csv( $data, $headers = null ) {
        if ( empty( $data ) ) {
            return '';
        }

        ob_start();
        $output = fopen( 'php://output', 'w' );

        // Add BOM for UTF-8.
        fprintf( $output, chr(0xEF).chr(0xBB).chr(0xBF) );

        // Add headers.
        if ( null === $headers ) {
            $headers = array_keys( $data[0] );
        }
        fputcsv( $output, $headers );

        // Add data rows.
        foreach ( $data as $row ) {
            fputcsv( $output, $row );
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose( $output );
        return ob_get_clean();
    }

    /**
     * Export costs to CSV.
     *
     * @return array CSV data.
     */
    public function export_costs_template() {
        return [
            [
                'product_id'     => '',
                'variation_id'   => '',
                'cost'           => '',
                'packaging_cost' => '',
                'handling_cost'  => '',
                'currency'       => get_woocommerce_currency(),
            ],
        ];
    }

    /**
     * Download CSV file.
     *
     * @param string $content  CSV content.
     * @param string $filename Filename.
     */
    public function download_csv( $content, $filename ) {
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );
        header( 'Pragma: no-cache' );
        header( 'Expires: 0' );
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- This is a raw CSV download, escaping would corrupt the data format.
        echo $content;
        exit;
    }
}
