<?php
/**
 * Cache Manager
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Utilities;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Manages caching for performance optimization.
 */
class QCNC_Cache_Manager {

    /**
     * Cache group prefix.
     *
     * @var string
     */
    private $cache_group = 'qcnc';

    /**
     * Default cache duration.
     *
     * @var int
     */
    private $default_duration;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->default_duration = absint( get_option( 'qcnc_cache_duration', 43200 ) );
    }

    /**
     * Get cached value.
     *
     * @param string $key Cache key.
     * @return mixed|false Cached value or false.
     */
    public function get( $key ) {
        $full_key = $this->get_full_key( $key );
        
        // Try object cache first.
        $value = wp_cache_get( $full_key, $this->cache_group );
        
        if ( false !== $value ) {
            return $value;
        }

        // Try transient cache.
        return get_transient( $full_key );
    }

    /**
     * Set cached value.
     *
     * @param string $key      Cache key.
     * @param mixed  $value    Value to cache.
     * @param int    $duration Cache duration in seconds.
     * @return bool Success status.
     */
    public function set( $key, $value, $duration = null ) {
        if ( null === $duration ) {
            $duration = $this->default_duration;
        }

        $full_key = $this->get_full_key( $key );

        // Set in object cache.
        wp_cache_set( $full_key, $value, $this->cache_group, $duration );

        // Set in transient cache.
        return set_transient( $full_key, $value, $duration );
    }

    /**
     * Delete cached value.
     *
     * @param string $key Cache key.
     * @return bool Success status.
     */
    public function delete( $key ) {
        $full_key = $this->get_full_key( $key );

        // Delete from object cache.
        wp_cache_delete( $full_key, $this->cache_group );

        // Delete from transient cache.
        return delete_transient( $full_key );
    }

    /**
     * Delete all caches matching pattern.
     *
     * @param string $pattern Key pattern.
     * @return bool Success status.
     */
    public function delete_pattern( $pattern ) {
        global $wpdb;

        $pattern = $this->get_full_key( $pattern );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} 
                 WHERE option_name LIKE %s 
                 OR option_name LIKE %s",
                '_transient_' . $pattern . '%',
                '_transient_timeout_' . $pattern . '%'
            )
        );

        return true;
    }

    /**
     * Delete product-related caches.
     *
     * @param int $product_id Product ID.
     * @return bool Success status.
     */
    public function delete_product_cache( $product_id ) {
        $this->delete( "product_cost_{$product_id}_0" );
        $this->delete_pattern( "product_cost_{$product_id}_" );
        $this->delete_pattern( 'report_product_' );
        $this->delete_pattern( 'dashboard_summary_' );
        
        return true;
    }

    /**
     * Delete order-related caches.
     *
     * @param int $order_id Order ID.
     * @return bool Success status.
     */
    public function delete_order_cache( $order_id ) {
        $this->delete( "order_profit_{$order_id}" );
        $this->delete_pattern( 'report_customer_' );
        $this->delete_pattern( 'dashboard_summary_' );
        
        return true;
    }

    /**
     * Clear all plugin caches.
     *
     * @return bool Success status.
     */
    public function clear_all() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_qcnc_%' 
             OR option_name LIKE '_transient_timeout_qcnc_%'"
        );

        // Clear object cache.
        wp_cache_flush();

        return true;
    }

    /**
     * Get full cache key with prefix.
     *
     * @param string $key Base key.
     * @return string Full key.
     */
    private function get_full_key( $key ) {
        return 'qcnc_' . $key;
    }

    /**
     * Store report cache in database.
     *
     * @param string $report_type Report type.
     * @param array  $filters     Filters.
     * @param array  $data        Report data.
     * @param int    $duration    Cache duration.
     * @return bool Success status.
     */
    public function cache_report( $report_type, $filters, $data, $duration = 86400 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_report_cache';

        $cache_key = $this->generate_report_cache_key( $report_type, $filters );
        $expires_at = wp_date( 'Y-m-d H:i:s', time() + $duration );

        // Delete existing cache.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->delete( $table, [ 'cache_key' => $cache_key ] );

        // Insert new cache.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        return $wpdb->insert(
            $table,
            [
                'cache_key'   => $cache_key,
                'report_type' => $report_type,
                'date_from'   => $filters['date_from'] ?? null,
                'date_to'     => $filters['date_to'] ?? null,
                'filters'     => wp_json_encode( $filters ),
                'report_data' => wp_json_encode( $data ),
                'expires_at'  => $expires_at,
                'created_at'  => current_time( 'mysql' ),
            ]
        );
    }

    /**
     * Get cached report from database.
     *
     * @param string $report_type Report type.
     * @param array  $filters     Filters.
     * @return array|null Cached report or null.
     */
    public function get_cached_report( $report_type, $filters ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_report_cache';

        $cache_key = $this->generate_report_cache_key( $report_type, $filters );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $cached = $wpdb->get_row(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE cache_key = %s AND expires_at > NOW()",
                $cache_key
            ),
            ARRAY_A
        );

        if ( $cached ) {
            return json_decode( $cached['report_data'], true );
        }

        return null;
    }

    /**
     * Generate cache key for report.
     *
     * @param string $report_type Report type.
     * @param array  $filters     Filters.
     * @return string Cache key.
     */
    private function generate_report_cache_key( $report_type, $filters ) {
        return md5( $report_type . '_' . wp_json_encode( $filters ) );
    }

    /**
     * Clean expired report caches.
     *
     * @return int Number of deleted records.
     */
    public function cleanup_expired_reports() {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_report_cache';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->query( "DELETE FROM {$table} WHERE expires_at < NOW()" );
    }
}
