<?php
/**
 * Payment Gateway Integration
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Integrations;

use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Manages payment gateway fee calculations.
 */
class QCNC_Payment_Gateway_Integration {

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Logger $logger Logger.
     */
    public function __construct( QCNC_Logger $logger ) {
        $this->logger = $logger;
        $this->init_hooks();
    }

    /**
     * Initialize hooks.
     */
    private function init_hooks() {
        add_action( 'woocommerce_payment_gateways', [ $this, 'register_gateway_fees' ] );
    }

    /**
     * Register gateway fees when gateways are loaded.
     *
     * @param array $gateways Available gateways.
     * @return array Gateways.
     */
    public function register_gateway_fees( $gateways ) {
        // Hook to auto-detect and register new gateways.
        return $gateways;
    }

    /**
     * Get payment gateway fee configuration.
     *
     * @param string $gateway_id Gateway ID.
     * @param string $currency   Currency.
     * @return array|null Fee config or null.
     */
    public function get_gateway_fee_config( $gateway_id, $currency ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_row(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE gateway_id = %s AND currency = %s AND is_active = 1",
                $gateway_id,
                $currency
            ),
            ARRAY_A
        );
    }

    /**
     * Update gateway fee configuration.
     *
     * @param string $gateway_id      Gateway ID.
     * @param string $currency        Currency.
     * @param array  $fee_config      Fee configuration.
     * @return bool Success status.
     */
    public function update_gateway_fee( $gateway_id, $currency, $fee_config ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        $existing = $this->get_gateway_fee_config( $gateway_id, $currency );

        $data = [
            'gateway_name'   => $fee_config['gateway_name'],
            'fee_type'       => $fee_config['fee_type'],
            'percentage_fee' => floatval( $fee_config['percentage_fee'] ),
            'fixed_fee'      => floatval( $fee_config['fixed_fee'] ),
            'is_active'      => isset( $fee_config['is_active'] ) ? absint( $fee_config['is_active'] ) : 1,
            'updated_at'     => current_time( 'mysql' ),
        ];

        if ( $existing ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->update(
                $table,
                $data,
                [
                    'gateway_id' => $gateway_id,
                    'currency'   => $currency,
                ]
            );
        } else {
            $data['gateway_id'] = $gateway_id;
            $data['currency']   = $currency;
            $data['created_at'] = current_time( 'mysql' );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->insert( $table, $data );
        }

        if ( false !== $result ) {
            $this->logger->log( "Gateway fee updated: {$gateway_id} ({$currency})" );
            return true;
        }

        return false;
    }

    /**
     * Get all active gateway fees.
     *
     * @return array Gateway fees.
     */
    public function get_all_gateway_fees() {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM {$table} WHERE is_active = 1 ORDER BY gateway_name ASC",
            ARRAY_A
        );
    }

    /**
     * Delete gateway fee configuration.
     *
     * @param int $fee_id Fee ID.
     * @return bool Success status.
     */
    public function delete_gateway_fee( $fee_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->delete( $table, [ 'id' => absint( $fee_id ) ], [ '%d' ] );

        return false !== $result;
    }
}
