<?php
/**
 * Query Builder
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Database;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Builds optimized SQL queries with prepared statements.
 */
class QCNC_Query_Builder {

    /**
     * Get orders with profit data.
     *
     * @param array $args Query arguments.
     * @return array Orders with profit.
     */
    public function get_orders_with_profit( $args = [] ) {
        global $wpdb;

        $defaults = [
            'date_from'  => '',
            'date_to'    => '',
            'status'     => [],
            'limit'      => 20,
            'offset'     => 0,
            'orderby'    => 'calculated_at',
            'order'      => 'DESC',
        ];

        $args = wp_parse_args( $args, $defaults );

        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $where_clauses = [ '1=1' ];
        $prepare_values = [];

        // Date filter.
        if ( ! empty( $args['date_from'] ) && ! empty( $args['date_to'] ) ) {
            $where_clauses[] = 'calculated_at BETWEEN %s AND %s';
            $prepare_values[] = $args['date_from'] . ' 00:00:00';
            $prepare_values[] = $args['date_to'] . ' 23:59:59';
        }

        $where_sql = implode( ' AND ', $where_clauses );

        $orderby = sanitize_sql_orderby( $args['orderby'] . ' ' . $args['order'] );
        if ( ! $orderby ) {
            $orderby = 'calculated_at DESC';
        }

        $query = "SELECT * FROM {$profit_table} WHERE {$where_sql} ORDER BY {$orderby} LIMIT %d OFFSET %d";
        $prepare_values[] = absint( $args['limit'] );
        $prepare_values[] = absint( $args['offset'] );

        if ( ! empty( $prepare_values ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            $query = $wpdb->prepare( $query, $prepare_values );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results( $query, ARRAY_A );
    }

    /**
     * Get product costs with filters.
     *
     * @param array $args Query arguments.
     * @return array Product costs.
     */
    public function get_product_costs( $args = [] ) {
        global $wpdb;

        $defaults = [
            'product_ids' => [],
            'is_current'  => true,
            'limit'       => 100,
            'offset'      => 0,
        ];

        $args = wp_parse_args( $args, $defaults );

        $table = $wpdb->prefix . 'qcnc_product_costs';
        $where_clauses = [];
        $prepare_values = [];

        if ( ! empty( $args['product_ids'] ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $args['product_ids'] ), '%d' ) );
            $where_clauses[] = "product_id IN ({$placeholders})";
            $prepare_values = array_merge( $prepare_values, $args['product_ids'] );
        }

        if ( $args['is_current'] ) {
            $where_clauses[] = 'is_current = 1';
        }

        $where_sql = ! empty( $where_clauses ) ? 'WHERE ' . implode( ' AND ', $where_clauses ) : '';

        $query = "SELECT * FROM {$table} {$where_sql} ORDER BY product_id ASC LIMIT %d OFFSET %d";
        $prepare_values[] = absint( $args['limit'] );
        $prepare_values[] = absint( $args['offset'] );

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results( $wpdb->prepare( $query, $prepare_values ), ARRAY_A );
    }

    /**
     * Get profit summary statistics.
     *
     * @param string $date_from Start date.
     * @param string $date_to   End date.
     * @return array Summary stats.
     */
    public function get_profit_summary( $date_from = '', $date_to = '' ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        $where_clauses = [ '1=1' ];
        $prepare_values = [];

        if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
            $where_clauses[] = 'calculated_at BETWEEN %s AND %s';
            $prepare_values[] = $date_from . ' 00:00:00';
            $prepare_values[] = $date_to . ' 23:59:59';
        }

        $where_sql = implode( ' AND ', $where_clauses );

        $query = "
            SELECT 
                COUNT(order_id) as total_orders,
                SUM(revenue) as total_revenue,
                SUM(cogs_total) as total_cogs,
                SUM(net_profit) as total_profit,
                AVG(margin_pct) as avg_margin
            FROM {$table}
            WHERE {$where_sql}
        ";

        if ( ! empty( $prepare_values ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $query = $wpdb->prepare( $query, $prepare_values );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_row( $query, ARRAY_A );
    }
}
