<?php
/**
 * Database Schema Manager
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Database;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Handles database schema creation and migrations.
 */
class QCNC_Database_Schema {

    /**
     * Create all plugin tables.
     */
    public static function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        $prefix          = $wpdb->prefix;

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $tables = [
            self::get_product_costs_schema( $prefix, $charset_collate ),
            self::get_order_profit_schema( $prefix, $charset_collate ),
            self::get_order_items_profit_schema( $prefix, $charset_collate ),
            self::get_cost_rules_schema( $prefix, $charset_collate ),
            self::get_margin_alerts_schema( $prefix, $charset_collate ),
            self::get_payment_gateway_fees_schema( $prefix, $charset_collate ),
            self::get_report_cache_schema( $prefix, $charset_collate ),
            self::get_product_profits_schema( $prefix, $charset_collate ),
            self::get_customer_clv_schema( $prefix, $charset_collate ),
            self::get_customer_acquisition_costs_schema( $prefix, $charset_collate ),
            self::get_customer_segments_schema( $prefix, $charset_collate ),
        ];

        foreach ( $tables as $table_sql ) {
            dbDelta( $table_sql );
        }

        // Update database version.
        update_option( 'qcnc_db_version', QCNC_VERSION );
    }

    /**
     * Get product costs table schema.
     */
    /*
    private static function get_product_costs_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_product_costs (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id BIGINT(20) UNSIGNED NOT NULL,
            variation_id BIGINT(20) UNSIGNED DEFAULT NULL,
            cost DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            packaging_cost DECIMAL(19,4) DEFAULT '0.0000',
            handling_cost DECIMAL(19,4) DEFAULT '0.0000',
            effective_from DATETIME NOT NULL,
            effective_to DATETIME DEFAULT NULL,
            is_current TINYINT(1) NOT NULL DEFAULT 1,
            created_by BIGINT(20) UNSIGNED DEFAULT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_product_id (product_id),
            KEY idx_variation_id (variation_id),
            KEY idx_is_current (is_current),
            KEY idx_effective_dates (effective_from, effective_to),
            KEY idx_product_current (product_id, is_current)
        ) $charset_collate;";
    }
    */

    /**
     * Get product costs table schema.
     */
    private static function get_product_costs_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_product_costs (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id BIGINT(20) UNSIGNED NOT NULL,
            variation_id BIGINT(20) UNSIGNED DEFAULT NULL,
            cost DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            packaging_cost DECIMAL(19,4) DEFAULT '0.0000',
            handling_cost DECIMAL(19,4) DEFAULT '0.0000',
            warehousing_cost DECIMAL(19,4) DEFAULT '0.0000',
            labor_cost DECIMAL(19,4) DEFAULT '0.0000',
            shipping_material_cost DECIMAL(19,4) DEFAULT '0.0000',
            custom_cost_1 DECIMAL(19,4) DEFAULT '0.0000',
            custom_cost_1_label VARCHAR(100) DEFAULT '',
            custom_cost_2 DECIMAL(19,4) DEFAULT '0.0000',
            custom_cost_2_label VARCHAR(100) DEFAULT '',
            custom_cost_3 DECIMAL(19,4) DEFAULT '0.0000',
            custom_cost_3_label VARCHAR(100) DEFAULT '',
            effective_from DATETIME NOT NULL,
            effective_to DATETIME DEFAULT NULL,
            is_current TINYINT(1) NOT NULL DEFAULT 1,
            created_by BIGINT(20) UNSIGNED DEFAULT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME DEFAULT NULL,
            notes TEXT DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_product_id (product_id),
            KEY idx_variation_id (variation_id),
            KEY idx_is_current (is_current),
            KEY idx_effective_dates (effective_from, effective_to),
            KEY idx_product_current (product_id, is_current)
        ) $charset_collate;";
    }


    /**
     * Get order profit table schema.
     */
    private static function get_order_profit_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_order_profit (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            order_id BIGINT(20) UNSIGNED NOT NULL,
            revenue DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            cogs_total DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            shipping_cost DECIMAL(19,4) DEFAULT '0.0000',
            payment_fee DECIMAL(19,4) DEFAULT '0.0000',
            tax_amount DECIMAL(19,4) DEFAULT '0.0000',
            discount_amount DECIMAL(19,4) DEFAULT '0.0000',
            refund_amount DECIMAL(19,4) DEFAULT '0.0000',
            gross_profit DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            net_profit DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            margin_pct DECIMAL(10,4) DEFAULT '0.0000',
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            calculated_at DATETIME NOT NULL,
            needs_recalc TINYINT(1) DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY idx_order_id (order_id),
            KEY idx_net_profit (net_profit),
            KEY idx_margin_pct (margin_pct),
            KEY idx_calculated_at (calculated_at),
            KEY idx_needs_recalc (needs_recalc)
        ) $charset_collate;";
    }

    /**
     * Get order items profit table schema.
     */
    private static function get_order_items_profit_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_order_items_profit (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            order_id BIGINT(20) UNSIGNED NOT NULL,
            order_item_id BIGINT(20) UNSIGNED NOT NULL,
            product_id BIGINT(20) UNSIGNED NOT NULL,
            variation_id BIGINT(20) UNSIGNED DEFAULT NULL,
            quantity INT(11) NOT NULL DEFAULT 1,
            item_revenue DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            item_cogs DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            item_profit DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            cost_per_unit DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY idx_order_id (order_id),
            KEY idx_order_item_id (order_item_id),
            KEY idx_product_id (product_id),
            KEY idx_variation_id (variation_id)
        ) $charset_collate;";
    }

    /**
     * Get cost rules table schema.
     */
    private static function get_cost_rules_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_cost_rules (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            rule_name VARCHAR(255) NOT NULL,
            rule_type VARCHAR(50) NOT NULL COMMENT 'category|tag|global|supplier',
            target_ids TEXT DEFAULT NULL COMMENT 'JSON array of category/tag IDs',
            adjustment_type VARCHAR(20) NOT NULL COMMENT 'percentage|fixed',
            adjustment_value DECIMAL(19,4) NOT NULL,
            cost_component VARCHAR(50) NOT NULL COMMENT 'base|packaging|handling',
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            priority INT(11) DEFAULT 10,
            created_at DATETIME NOT NULL,
            updated_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_is_active (is_active),
            KEY idx_priority (priority)
        ) $charset_collate;";
    }

    /**
     * Get margin alerts table schema.
     */
    private static function get_margin_alerts_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_margin_alerts (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            alert_type VARCHAR(50) NOT NULL COMMENT 'low_margin|loss_order|monthly_summary',
            entity_type VARCHAR(50) NOT NULL COMMENT 'product|order|global',
            entity_id BIGINT(20) UNSIGNED DEFAULT NULL,
            alert_data TEXT DEFAULT NULL COMMENT 'JSON data',
            threshold_value DECIMAL(19,4) DEFAULT NULL,
            actual_value DECIMAL(19,4) DEFAULT NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'pending' COMMENT 'pending|sent|failed',
            sent_at DATETIME DEFAULT NULL,
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY idx_alert_type (alert_type),
            KEY idx_entity (entity_type, entity_id),
            KEY idx_status (status),
            KEY idx_created_at (created_at)
        ) $charset_collate;";
    }

    /**
     * Get payment gateway fees table schema.
     */
    private static function get_payment_gateway_fees_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_payment_gateway_fees (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            gateway_id VARCHAR(100) NOT NULL,
            gateway_name VARCHAR(255) NOT NULL,
            fee_type VARCHAR(20) NOT NULL COMMENT 'percentage|fixed|mixed',
            percentage_fee DECIMAL(10,4) DEFAULT '0.0000',
            fixed_fee DECIMAL(19,4) DEFAULT '0.0000',
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            created_at DATETIME NOT NULL,
            updated_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY idx_gateway_currency (gateway_id, currency),
            KEY idx_is_active (is_active)
        ) $charset_collate;";
    }

    /**
     * Get report cache table schema.
     */
    private static function get_report_cache_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_report_cache (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            cache_key VARCHAR(255) NOT NULL,
            report_type VARCHAR(50) NOT NULL,
            date_from DATE DEFAULT NULL,
            date_to DATE DEFAULT NULL,
            filters TEXT DEFAULT NULL COMMENT 'JSON filters',
            report_data LONGTEXT NOT NULL COMMENT 'JSON report data',
            expires_at DATETIME NOT NULL,
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY idx_cache_key (cache_key),
            KEY idx_report_type (report_type),
            KEY idx_expires_at (expires_at),
            KEY idx_date_range (date_from, date_to)
        ) $charset_collate;";
    }

    /**
     * Migrate database to add new cost fields.
     */
    public static function migrate_add_custom_costs() {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';
        
        $columns_to_add = [
            "warehousing_cost DECIMAL(19,4) DEFAULT '0.0000'",
            "labor_cost DECIMAL(19,4) DEFAULT '0.0000'",
            "shipping_material_cost DECIMAL(19,4) DEFAULT '0.0000'",
            "custom_cost_1 DECIMAL(19,4) DEFAULT '0.0000'",
            "custom_cost_1_label VARCHAR(100) DEFAULT ''",
            "custom_cost_2 DECIMAL(19,4) DEFAULT '0.0000'",
            "custom_cost_2_label VARCHAR(100) DEFAULT ''",
            "custom_cost_3 DECIMAL(19,4) DEFAULT '0.0000'",
            "custom_cost_3_label VARCHAR(100) DEFAULT ''",
            "notes TEXT DEFAULT NULL",
        ];
        
        foreach ( $columns_to_add as $column_def ) {
            $column_name = explode( ' ', $column_def )[0];
            
            // Check if column exists
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $column_exists = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SHOW COLUMNS FROM {$table} LIKE %s",
                    $column_name
                )
            );
            
            // Add column if it doesn't exist
            if ( empty( $column_exists ) ) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.SchemaChange
                $wpdb->query( "ALTER TABLE {$table} ADD COLUMN {$column_def}" );
            }
        }
        
        update_option( 'qcnc_custom_costs_migrated', '1' );
    }

    /**
     * Get product profits table schema (for advanced dashboard).
     */
    private static function get_product_profits_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_product_profits (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id BIGINT(20) UNSIGNED NOT NULL,
            date DATE NOT NULL,
            profit DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            margin DECIMAL(10,4) NOT NULL DEFAULT '0.0000',
            revenue DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            costs DECIMAL(19,4) NOT NULL DEFAULT '0.0000',
            quantity INT(11) NOT NULL DEFAULT 0,
            order_count INT(11) NOT NULL DEFAULT 0,
            created_at DATETIME NOT NULL,
            updated_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY idx_product_date (product_id, date),
            KEY idx_product_id (product_id),
            KEY idx_date (date),
            KEY idx_profit (profit),
            KEY idx_margin (margin)
        ) $charset_collate;";
    }

    /**
     * Create customer lifetime value table
     */
    private static function get_customer_clv_schema( $prefix, $charset_collate ) {

        return "CREATE TABLE {$prefix}qcnc_customer_clv (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            customer_email varchar(100) NOT NULL,
            total_orders int(11) DEFAULT 0,
            total_revenue decimal(20,4) DEFAULT 0.0000,
            total_profit decimal(20,4) DEFAULT 0.0000,
            avg_order_value decimal(20,4) DEFAULT 0.0000,
            avg_profit_per_order decimal(20,4) DEFAULT 0.0000,
            first_order_date datetime DEFAULT NULL,
            last_order_date datetime DEFAULT NULL,
            customer_age_days int(11) DEFAULT 0,
            predicted_clv decimal(20,4) DEFAULT 0.0000,
            predicted_lifetime_profit decimal(20,4) DEFAULT 0.0000,
            acquisition_cost decimal(20,4) DEFAULT 0.0000,
            cac_to_clv_ratio decimal(10,4) DEFAULT 0.0000,
            customer_tier varchar(20) DEFAULT 'bronze',
            churn_risk_score decimal(5,2) DEFAULT 0.00,
            churn_risk_level varchar(20) DEFAULT 'low',
            days_since_last_order int(11) DEFAULT 0,
            purchase_frequency decimal(10,2) DEFAULT 0.00,
            currency varchar(10) DEFAULT 'USD',
            last_calculated datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY customer_id (customer_id),
            KEY customer_email (customer_email),
            KEY customer_tier (customer_tier),
            KEY churn_risk_level (churn_risk_level),
            KEY predicted_clv (predicted_clv),
            KEY last_order_date (last_order_date)
        ) $charset_collate;";
    }

    /**
     * Create customer acquisition costs table
     */
    private static function get_customer_acquisition_costs_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_customer_acquisition_costs (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            source varchar(100) DEFAULT NULL,
            medium varchar(100) DEFAULT NULL,
            campaign varchar(100) DEFAULT NULL,
            acquisition_cost decimal(20,4) DEFAULT 0.0000,
            acquisition_date datetime DEFAULT NULL,
            currency varchar(10) DEFAULT 'USD',
            notes text DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY customer_id (customer_id),
            KEY source (source),
            KEY campaign (campaign),
            KEY acquisition_date (acquisition_date)
        ) $charset_collate;";
    }

    /**
     * Create customer segments table
     */
    private static function get_customer_segments_schema( $prefix, $charset_collate ) {
        return "CREATE TABLE {$prefix}qcnc_customer_segments (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            segment_name varchar(100) NOT NULL,
            segment_type varchar(50) NOT NULL,
            conditions text NOT NULL,
            customer_count int(11) DEFAULT 0,
            avg_clv decimal(20,4) DEFAULT 0.0000,
            total_revenue decimal(20,4) DEFAULT 0.0000,
            is_active tinyint(1) DEFAULT 1,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY segment_type (segment_type),
            KEY is_active (is_active)
        ) $charset_collate;";
    }




}
