<?php
/**
 * Enhanced Cron Jobs Handler
 *
 * Schedules and executes automated tasks including:
 * - Daily profit summaries
 * - Anomaly detection
 * - Alert cleanup
 * - Cache cleanup
 *
 * @package QuarkcodeNeuralCommerce
 * @subpackage Cron
 */

namespace QuarkcodeNeuralCommerce\Cron;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class QCNC_Cron_Jobs {

    /**
     * Initialize cron jobs
     */
    public function __construct() {
        // Register custom cron schedules
        add_filter( 'cron_schedules', array( $this, 'add_custom_cron_schedules' ) );

        // Schedule cron jobs on activation
        add_action( 'qcnc_plugin_activated', array( $this, 'schedule_cron_jobs' ) );

        // Unschedule on deactivation
        add_action( 'qcnc_plugin_deactivated', array( $this, 'unschedule_cron_jobs' ) );

        // Register cron job handlers
        add_action( 'qcnc_daily_profit_summary', array( $this, 'send_daily_profit_summary' ) );
        add_action( 'qcnc_anomaly_detection', array( $this, 'run_anomaly_detection' ) );
        add_action( 'qcnc_hourly_profit_check', array( $this, 'check_recent_orders_profit' ) );
        add_action( 'qcnc_weekly_cleanup', array( $this, 'cleanup_old_data' ) );
        add_action( 'qcnc_cache_cleanup', array( $this, 'cleanup_cache' ) );

        // Add settings page hooks
        add_action( 'admin_init', array( $this, 'maybe_reschedule_jobs' ) );
    }

    /**
     * Add custom cron schedules
     *
     * @param array $schedules Existing schedules
     * @return array Modified schedules
     */
    public function add_custom_cron_schedules( $schedules ) {
        // Every 6 hours
        $schedules['qcnc_six_hours'] = array(
            'interval' => 6 * HOUR_IN_SECONDS,
            'display'  => __( 'Every 6 Hours', 'quarkcode-neuralcommerce-lite' ),
        );

        // Twice daily
        $schedules['qcnc_twice_daily'] = array(
            'interval' => 12 * HOUR_IN_SECONDS,
            'display'  => __( 'Twice Daily', 'quarkcode-neuralcommerce-lite' ),
        );

        return $schedules;
    }

    /**
     * Schedule all cron jobs
     */
    public function schedule_cron_jobs() {
        // Daily profit summary (sent at configured time, default 8 AM)
        $summary_time = get_option( 'qcnc_daily_summary_time', '08:00' );
        $summary_timestamp = $this->get_next_scheduled_time( $summary_time );

        if ( ! wp_next_scheduled( 'qcnc_daily_profit_summary' ) ) {
            wp_schedule_event( $summary_timestamp, 'daily', 'qcnc_daily_profit_summary' );
        }

        // Anomaly detection (runs twice daily at 6 AM and 6 PM)
        if ( ! wp_next_scheduled( 'qcnc_anomaly_detection' ) ) {
            wp_schedule_event( strtotime( 'today 6:00' ), 'qcnc_twice_daily', 'qcnc_anomaly_detection' );
        }

        // Hourly profit check for recent orders
        if ( ! wp_next_scheduled( 'qcnc_hourly_profit_check' ) ) {
            wp_schedule_event( time(), 'hourly', 'qcnc_hourly_profit_check' );
        }

        // Weekly cleanup (every Sunday at 2 AM)
        if ( ! wp_next_scheduled( 'qcnc_weekly_cleanup' ) ) {
            wp_schedule_event( strtotime( 'next Sunday 2:00' ), 'weekly', 'qcnc_weekly_cleanup' );
        }

        // Cache cleanup (every 6 hours)
        if ( ! wp_next_scheduled( 'qcnc_cache_cleanup' ) ) {
            wp_schedule_event( time(), 'qcnc_six_hours', 'qcnc_cache_cleanup' );
        }
    }

    /**
     * Unschedule all cron jobs
     */
    public function unschedule_cron_jobs() {
        $cron_jobs = array(
            'qcnc_daily_profit_summary',
            'qcnc_anomaly_detection',
            'qcnc_hourly_profit_check',
            'qcnc_weekly_cleanup',
            'qcnc_cache_cleanup',
        );

        foreach ( $cron_jobs as $job ) {
            $timestamp = wp_next_scheduled( $job );
            if ( $timestamp ) {
                wp_unschedule_event( $timestamp, $job );
            }
            wp_clear_scheduled_hook( $job );
        }
    }

    /**
     * Reschedule jobs if settings changed
     */
    public function maybe_reschedule_jobs() {
        $last_scheduled = get_option( 'qcnc_cron_last_scheduled', 0 );
        $settings_modified = get_option( 'qcnc_settings_last_modified', 0 );

        // Reschedule if settings were modified after last schedule
        if ( $settings_modified > $last_scheduled ) {
            $this->unschedule_cron_jobs();
            $this->schedule_cron_jobs();
            update_option( 'qcnc_cron_last_scheduled', time() );
        }
    }

    /**
     * Send daily profit summary
     */
    public function send_daily_profit_summary() {
        // Check if daily summary is enabled
        $enabled_alerts = get_option( 'qcnc_enabled_alerts', array() );
        if ( ! in_array( 'daily_summary', $enabled_alerts ) ) {
            return;
        }

        // Get the alert system instance
        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();
        $alert_system->send_daily_summary();

        // Log execution
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Daily profit summary sent', array(
            'time' => current_time( 'mysql' ),
        ) );
    }

    /**
     * Run anomaly detection
     */
    public function run_anomaly_detection() {
        // Check if anomaly detection is enabled
        $enabled_alerts = get_option( 'qcnc_enabled_alerts', array() );
        if ( ! in_array( 'anomaly_detected', $enabled_alerts ) ) {
            return;
        }

        // Get the alert system instance
        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();
        $alert_system->detect_anomalies();

        // Log execution
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Anomaly detection completed', array(
            'time' => current_time( 'mysql' ),
        ) );
    }

    /**
     * Check recent orders for profitability issues
     */
    public function check_recent_orders_profit() {
        global $wpdb;

        // Get orders from the last hour that haven't been checked
        $orders = wc_get_orders( array(
            'limit'        => 50,
            'date_created' => '>' . ( time() - HOUR_IN_SECONDS ),
            'status'       => array( 'completed', 'processing' ),
        ) );

        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();

        foreach ( $orders as $order ) {
            // Check if already analyzed
            $analyzed = get_post_meta( $order->get_id(), '_qcnc_profit_analyzed', true );
            if ( $analyzed ) {
                continue;
            }

            // Check profitability
            $alert_system->check_order_profitability( $order->get_id() );

            // Mark as analyzed
            update_post_meta( $order->get_id(), '_qcnc_profit_analyzed', current_time( 'mysql' ) );
        }

        if ( count( $orders ) > 0 ) {
            $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
            $logger->log( 'Hourly profit check completed', array(
                'orders_checked' => count( $orders ),
            ) );
        }
    }

    /**
     * Cleanup old data
     */
    public function cleanup_old_data() {
        global $wpdb;

        // Get cleanup settings
        $alert_retention = get_option( 'qcnc_alert_retention_days', 90 );
        $log_retention = get_option( 'qcnc_log_retention_days', 30 );

        // Cleanup old alerts
        $alerts_table = $wpdb->prefix . 'qcnc_margin_alerts';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $deleted_alerts = $wpdb->query( $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "DELETE FROM {$alerts_table} 
             WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)
             AND status = 'read'",
            $alert_retention
        ) );

        // Cleanup old logs
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logs_deleted = $logger->cleanup_old_logs( $log_retention );

        // Cleanup old cost history (keep 1 year)
        $history_table = $wpdb->prefix . 'qcnc_cost_history';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $deleted_history = $wpdb->query(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "DELETE FROM {$history_table} 
             WHERE created_at < DATE_SUB(NOW(), INTERVAL 365 DAY)"
        );

        $logger->log( 'Weekly cleanup completed', array(
            'alerts_deleted' => $deleted_alerts,
            'logs_deleted' => $logs_deleted,
            'history_deleted' => $deleted_history,
        ) );
    }

    /**
     * Cleanup cache
     */
    public function cleanup_cache() {
        // Clean up transients
        delete_expired_transients();

        // Clean up plugin-specific cache
        $cache_manager = new \QuarkcodeNeuralCommerce\Utilities\Cache_Manager();
        $cache_manager->clear_expired_cache();

        // Cleanup report cache
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_qcnc_%' 
             OR option_name LIKE '_transient_timeout_qcnc_%'"
        );

        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Cache cleanup completed' );
    }

    /**
     * Get next scheduled time for a specific time of day
     *
     * @param string $time Time in HH:MM format
     * @return int Unix timestamp
     */
    private function get_next_scheduled_time( $time ) {
        list( $hour, $minute ) = explode( ':', $time );

        $now = current_time( 'timestamp' );
        $scheduled = strtotime( "today {$hour}:{$minute}", $now );

        // If the time has passed today, schedule for tomorrow
        if ( $scheduled < $now ) {
            $scheduled = strtotime( "tomorrow {$hour}:{$minute}", $now );
        }

        return $scheduled;
    }

    /**
     * Get cron job status (for admin display)
     *
     * @return array Cron job statuses
     */
    public static function get_cron_status() {
        $jobs = array(
            'qcnc_daily_profit_summary' => 'Daily Profit Summary',
            'qcnc_anomaly_detection' => 'Anomaly Detection',
            'qcnc_hourly_profit_check' => 'Hourly Profit Check',
            'qcnc_weekly_cleanup' => 'Weekly Cleanup',
            'qcnc_cache_cleanup' => 'Cache Cleanup',
        );

        $status = array();

        foreach ( $jobs as $hook => $label ) {
            $next_run = wp_next_scheduled( $hook );
            $status[] = array(
                'hook' => $hook,
                'label' => $label,
                'scheduled' => $next_run ? true : false,
                'next_run' => $next_run ? wp_date( 'Y-m-d H:i:s', $next_run ) : 'Not scheduled',
                'next_run_relative' => $next_run ? human_time_diff( time(), $next_run ) : 'N/A',
            );
        }

        return $status;
    }

    /**
     * Manually trigger a cron job (for testing)
     *
     * @param string $job_name Job name
     * @return bool Success
     */
    public static function trigger_manual( $job_name ) {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return false;
        }

        switch ( $job_name ) {
            case 'daily_summary':
                do_action( 'qcnc_daily_profit_summary' );
                break;
            case 'anomaly_detection':
                do_action( 'qcnc_anomaly_detection' );
                break;
            case 'hourly_check':
                do_action( 'qcnc_hourly_profit_check' );
                break;
            case 'cleanup':
                do_action( 'qcnc_weekly_cleanup' );
                break;
            case 'cache':
                do_action( 'qcnc_cache_cleanup' );
                break;
            default:
                return false;
        }

        return true;
    }

    /**
     * Register CLV cron schedules
     */
    public function register_clv_schedules() {
        // Daily CLV recalculation
        if ( ! wp_next_scheduled( 'qcnc_daily_clv_recalculation' ) ) {
            wp_schedule_event( strtotime( '02:00:00' ), 'daily', 'qcnc_daily_clv_recalculation' );
        }

        // Weekly churn risk check
        if ( ! wp_next_scheduled( 'qcnc_weekly_churn_check' ) ) {
            wp_schedule_event( strtotime( 'next Monday 03:00' ), 'weekly', 'qcnc_weekly_churn_check' );
        }

        // Monthly CLV summary
        if ( ! wp_next_scheduled( 'qcnc_monthly_clv_summary' ) ) {
            wp_schedule_event( strtotime( 'first day of next month 04:00' ), 'monthly', 'qcnc_monthly_clv_summary' );
        }

        // Hourly check for high-risk customers (optional)
        if ( ! wp_next_scheduled( 'qcnc_hourly_high_risk_check' ) ) {
            wp_schedule_event( time(), 'hourly', 'qcnc_hourly_high_risk_check' );
        }

        // Hook the actions
        add_action( 'qcnc_daily_clv_recalculation', array( $this, 'run_daily_clv_recalculation' ) );
        add_action( 'qcnc_weekly_churn_check', array( $this, 'run_weekly_churn_check' ) );
        add_action( 'qcnc_monthly_clv_summary', array( $this, 'run_monthly_clv_summary' ) );
        add_action( 'qcnc_hourly_high_risk_check', array( $this, 'run_hourly_high_risk_check' ) );
        add_action( 'qcnc_calculate_new_customer_clv', array( $this, 'calculate_new_customer_clv' ), 10, 1 );
    }

    /**
     * Run daily CLV recalculation for all customers
     */
    public function run_daily_clv_recalculation() {
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Starting daily CLV recalculation', 'info' );

        $clv_calculator = new \QuarkcodeNeuralCommerce\Core\QCNC_CLV_Calculator();

        // Get all customers with orders in last 30 days (active customers)
        $active_customers = $this->get_active_customers( 30 );

        $processed = 0;
        $success = 0;
        $failed = 0;

        foreach ( $active_customers as $customer_id ) {
            $result = $clv_calculator->calculate_customer_clv( $customer_id );
            $processed++;

            if ( $result !== false ) {
                $success++;
            } else {
                $failed++;
            }

            // Prevent timeout - add small delay
            if ( $processed % 50 === 0 ) {
                sleep( 1 );
            }
        }

        $logger->log( sprintf( 
            'Daily CLV recalculation complete. Processed: %d, Success: %d, Failed: %d',
            $processed, $success, $failed
        ), 'info' );

        // Store the last run time
        update_option( 'qcnc_last_daily_clv_run', current_time( 'mysql' ) );
    }

    /**
     * Get active customers (with orders in last X days)
     * 
     * @param int $days
     * @return array Customer IDs
     */
    private function get_active_customers( $days = 30 ) {
        global $wpdb;

        $date_from = wp_date( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );

        // Get customers who placed orders in last X days
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $customer_ids = $wpdb->get_col( $wpdb->prepare(
            "SELECT DISTINCT meta.meta_value as customer_id
            FROM {$wpdb->posts} as posts
            INNER JOIN {$wpdb->postmeta} as meta ON posts.ID = meta.post_id
            WHERE posts.post_type = 'shop_order'
            AND posts.post_status IN ('wc-completed', 'wc-processing')
            AND posts.post_date >= %s
            AND meta.meta_key = '_customer_user'
            AND meta.meta_value > 0",
            $date_from
        ) );

        return array_map( 'intval', $customer_ids );
    }

    /**
     * Run weekly churn risk check and send notifications
     */
    public function run_weekly_churn_check() {
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Starting weekly churn risk check', 'info' );

        $clv_calculator = new \QuarkcodeNeuralCommerce\Core\QCNC_CLV_Calculator();

        // Get high-risk customers
        $high_risk_customers = $clv_calculator->get_churn_risk_customers( 'high' );

        if ( empty( $high_risk_customers ) ) {
            $logger->log( 'No high-risk customers found', 'info' );
            return;
        }

        // Send churn risk email notification
        $alert_system = new \QuarkcodeNeuralCommerce\Core\Margin_Alert_System();

        foreach ( $high_risk_customers as $customer_data ) {
            $this->send_churn_risk_internal_alert( $customer_data );
        }

        $logger->log( sprintf( 'Churn risk check complete. Found %d high-risk customers', count( $high_risk_customers ) ), 'info' );

        update_option( 'qcnc_last_churn_check', current_time( 'mysql' ) );
    }

    /**
     * Send internal churn risk alert to admin
     * 
     * @param array $customer_data
     */
    private function send_churn_risk_internal_alert( $customer_data ) {
        $admin_email = get_option( 'admin_email' );
        $site_name = get_bloginfo( 'name' );

        $customer = get_userdata( $customer_data['customer_id'] );
        if ( ! $customer ) {
            return;
        }

        $subject = sprintf( 
            '[%s] Churn Risk Alert: %s', 
            $site_name, 
            $customer->display_name 
        );

        $message = sprintf(
            "High churn risk detected for customer:

    " .
            "Customer: %s
    " .
            "Email: %s
    " .
            "Predicted Lifetime Profit: %s
    " .
            "Churn Risk Score: %s%%
    " .
            "Days Since Last Order: %d
    " .
            "Total Orders: %d

    " .
            "Recommended Actions:
    " .
            "1. Send a personalized re-engagement email
    " .
            "2. Offer a special discount or incentive
    " .
            "3. Reach out via phone/SMS if possible

    " .
            "View customer details: %s",
            $customer->display_name,
            $customer->user_email,
            wc_price( $customer_data['predicted_lifetime_profit'] ),
            round( $customer_data['churn_risk_score'] ),
            $customer_data['days_since_last_order'],
            $customer_data['total_orders'],
            admin_url( 'user-edit.php?user_id=' . $customer_data['customer_id'] )
        );

        wp_mail( $admin_email, $subject, $message );
    }

    /**
     * Run monthly CLV summary report
     */
    public function run_monthly_clv_summary() {
        $logger = new \QuarkcodeNeuralCommerce\Utilities\QCNC_Logger();
        $logger->log( 'Generating monthly CLV summary', 'info' );

        $clv_calculator = new \QuarkcodeNeuralCommerce\Core\QCNC_CLV_Calculator();
        $summary_stats = $clv_calculator->get_clv_summary_stats();

        // Send summary email to admin
        $this->send_monthly_clv_summary_email( $summary_stats );

        $logger->log( 'Monthly CLV summary sent', 'info' );
        update_option( 'qcnc_last_monthly_summary', current_time( 'mysql' ) );
    }

    /**
     * Send monthly CLV summary email
     * 
     * @param array $summary_stats
     */
    private function send_monthly_clv_summary_email( $summary_stats ) {
        $admin_email = get_option( 'admin_email' );
        $site_name = get_bloginfo( 'name' );
        $currency = get_woocommerce_currency_symbol();

        $overview = $summary_stats['overview'];

        $subject = sprintf( '[%s] Monthly CLV Summary Report', $site_name );

        $message = sprintf(
            "Monthly Customer Lifetime Value Report
    " .
            "======================================

    " .
            "Total Customers: %s
    " .
            "Average CLV: %s
    " .
            "Total Predicted CLV: %s
    " .
            "Average Predicted Lifetime Profit: %s
    " .
            "Total Predicted Profit: %s
    " .
            "Average Acquisition Cost: %s
    " .
            "Average CAC:CLV Ratio: %s:1

    " .
            "Customer Tiers:
    ",
            number_format( $overview['total_customers'] ?? 0 ),
            wc_price( $overview['avg_clv'] ?? 0 ),
            wc_price( $overview['total_predicted_clv'] ?? 0 ),
            wc_price( $overview['avg_lifetime_profit'] ?? 0 ),
            wc_price( $overview['total_predicted_profit'] ?? 0 ),
            wc_price( $overview['avg_acquisition_cost'] ?? 0 ),
            number_format( $overview['avg_cac_clv_ratio'] ?? 0, 2 )
        );

        // Add tier breakdown
        foreach ( $summary_stats['tiers'] as $tier_data ) {
            $message .= sprintf(
                "  - %s: %d customers
    ",
                strtoupper( $tier_data['customer_tier'] ),
                $tier_data['count']
            );
        }

        $message .= "
    Churn Risk Distribution:
    ";
        foreach ( $summary_stats['churn_risk'] as $churn_data ) {
            $message .= sprintf(
                "  - %s Risk: %d customers
    ",
                ucfirst( $churn_data['churn_risk_level'] ),
                $churn_data['count']
            );
        }

        $message .= sprintf(
            "

    View full CLV dashboard: %s",
            admin_url( 'admin.php?page=qcnc-customer-clv' )
        );

        wp_mail( $admin_email, $subject, $message );
    }

    /**
     * Hourly check for newly high-risk customers
     */
    public function run_hourly_high_risk_check() {
        // Check if enabled
        $enabled = get_option( 'qcnc_hourly_risk_check_enabled', 'no' );
        if ( $enabled !== 'yes' ) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_customer_clv';

        // Get customers who became high-risk in last hour
        $one_hour_ago = wp_date( 'Y-m-d H:i:s', strtotime( '-1 hour' ) );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $new_high_risk = $wpdb->get_results( $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM $table 
            WHERE churn_risk_level = 'high' 
            AND updated_at >= %s",
            $one_hour_ago
        ), ARRAY_A );

        foreach ( $new_high_risk as $customer_data ) {
            $this->send_churn_risk_internal_alert( $customer_data );
        }
    }

    /**
     * Calculate CLV for new customer (delayed)
     * 
     * @param int $customer_id
     */
    public function calculate_new_customer_clv( $customer_id ) {
        $clv_calculator = new \QuarkcodeNeuralCommerce\Core\QCNC_CLV_Calculator();
        $clv_calculator->calculate_customer_clv( $customer_id );
    }

    /**
     * Clear all CLV cron schedules (for deactivation)
     */
    public function clear_clv_schedules() {
        wp_clear_scheduled_hook( 'qcnc_daily_clv_recalculation' );
        wp_clear_scheduled_hook( 'qcnc_weekly_churn_check' );
        wp_clear_scheduled_hook( 'qcnc_monthly_clv_summary' );
        wp_clear_scheduled_hook( 'qcnc_hourly_high_risk_check' );
        wp_clear_scheduled_hook( 'qcnc_calculate_new_customer_clv' );
    }

}
