<?php
/**
 * Report Generator
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Core;

use QuarkcodeNeuralCommerce\Utilities\QCNC_Cache_Manager;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Generates profit reports and analytics.
 */
class QCNC_Report_Generator {

    /**
     * Cache manager instance.
     *
     * @var QCNC_Cache_Manager
     */
    private $cache_manager;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Cache_Manager $cache_manager Cache manager.
     * @param QCNC_Logger        $logger        Logger.
     */
    public function __construct( QCNC_Cache_Manager $cache_manager, QCNC_Logger $logger ) {
        $this->cache_manager = $cache_manager;
        $this->logger        = $logger;
    }

    /**
     * Generate profit by product report.
     *
     * @param string $date_from Start date (Y-m-d).
     * @param string $date_to   End date (Y-m-d).
     * @param int    $limit     Limit results.
     * @return array Report data.
     */
    public function get_profit_by_product( $date_from = '', $date_to = '', $limit = 100 ) {
        $cache_key = "report_product_{$date_from}_{$date_to}_{$limit}";
        
        // Disable cache temporarily
        $cached    = $this->cache_manager->get( $cache_key );
        // $cached = false;

        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $items_table = $wpdb->prefix . 'qcnc_order_items_profit';
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        $where_clause = '1=1';
        $params = [];

        if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
            // Use order date instead of calculated_at
            $where_clause .= ' AND o.date_created_gmt BETWEEN %s AND %s';
            $params[] = $date_from . ' 00:00:00';
            $params[] = $date_to . ' 23:59:59';
        }

        $query = "
            SELECT 
                oip.product_id,
                oip.variation_id,
                COUNT(DISTINCT oip.order_id) as order_count,
                SUM(oip.quantity) as total_quantity,
                SUM(oip.item_revenue) as total_revenue,
                SUM(oip.item_cogs) as total_cogs,
                SUM(oip.item_profit) as total_profit,
                AVG((oip.item_profit / NULLIF(oip.item_revenue, 0)) * 100) as avg_margin_pct
            FROM {$items_table} oip
            INNER JOIN {$profit_table} op ON oip.order_id = op.order_id
            INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE {$where_clause}
            GROUP BY oip.product_id, oip.variation_id
            ORDER BY total_profit DESC
            LIMIT %d
        ";

        $params[] = $limit;
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $prepared_query = $wpdb->prepare( $query, $params );
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results( $prepared_query, ARRAY_A );

        // Enrich with product data
        foreach ( $results as &$row ) {
            $product_id = $row['variation_id'] ? $row['variation_id'] : $row['product_id'];
            $product = wc_get_product( $product_id );

            if ( $product ) {
                $row['product_name'] = $product->get_name();
                $row['sku'] = $product->get_sku();
                $row['product_link'] = admin_url( 'post.php?post=' . $row['product_id'] . '&action=edit' );
            }
        }

        $this->cache_manager->set( $cache_key, $results, 6 * HOUR_IN_SECONDS );

        return $results;
    }


    /**
     * Generate profit by category report.
     *
     * @param string $date_from Start date (Y-m-d).
     * @param string $date_to   End date (Y-m-d).
     * @return array Report data.
     */
    public function get_profit_by_category( $date_from = '', $date_to = '' ) {
        $cache_key = "report_category_{$date_from}_{$date_to}";
        $cached    = $this->cache_manager->get( $cache_key );
        
        // Disable cache temporarily
        // $cached = false;


        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $items_table = $wpdb->prefix . 'qcnc_order_items_profit';
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        $where_clause = '1=1';
        $params = [];

        if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
            // Use order date instead of calculated_at
            $where_clause .= ' AND o.date_created_gmt BETWEEN %s AND %s';
            $params[] = $date_from . ' 00:00:00';
            $params[] = $date_to . ' 23:59:59';
        }

        $query = "
            SELECT 
                tt.term_id,
                t.name as category_name,
                COUNT(DISTINCT oip.order_id) as order_count,
                SUM(oip.quantity) as total_quantity,
                SUM(oip.item_revenue) as total_revenue,
                SUM(oip.item_cogs) as total_cogs,
                SUM(oip.item_profit) as total_profit,
                AVG((oip.item_profit / NULLIF(oip.item_revenue, 0)) * 100) as avg_margin_pct
            FROM {$items_table} oip
            INNER JOIN {$profit_table} op ON oip.order_id = op.order_id
            INNER JOIN {$orders_table} o ON op.order_id = o.id
            INNER JOIN {$wpdb->term_relationships} tr ON oip.product_id = tr.object_id
            INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id AND tt.taxonomy = 'product_cat'
            INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
            WHERE {$where_clause}
            GROUP BY tt.term_id, t.name
            ORDER BY total_profit DESC
        ";

        if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            $prepared_query = $wpdb->prepare( $query, $params );
        } else {
            $prepared_query = $query;
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results( $prepared_query, ARRAY_A );

        $this->cache_manager->set( $cache_key, $results, 6 * HOUR_IN_SECONDS );

        return $results;
    }


    /**
     * Generate profit by customer report.
     *
     * @param string $date_from Start date (Y-m-d).
     * @param string $date_to   End date (Y-m-d).
     * @param int    $limit     Limit results.
     * @return array Report data.
     */
    public function get_profit_by_customer( $date_from = '', $date_to = '', $limit = 100 ) {
        $cache_key = "report_customer_{$date_from}_{$date_to}_{$limit}";
        $cached    = $this->cache_manager->get( $cache_key );
        
        // Disable cache temporarily
        // $cached = false;
        

        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        $where_clause = '1=1';
        $params = [];

        if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
            // Use order date instead of calculated_at
            $where_clause .= ' AND o.date_created_gmt BETWEEN %s AND %s';
            $params[] = $date_from . ' 00:00:00';
            $params[] = $date_to . ' 23:59:59';
        }

        // Check if HPOS is enabled
        if ( class_exists( '\\Automattic\\WooCommerce\\Utilities\\OrderUtil' ) &&
            \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled() ) {
            
            $query = "
                SELECT 
                    o.customer_id,
                    om.meta_value as customer_email,
                    COUNT(op.order_id) as order_count,
                    SUM(op.revenue) as total_revenue,
                    SUM(op.net_profit) as total_profit,
                    AVG(op.margin_pct) as avg_margin_pct
                FROM {$profit_table} op
                INNER JOIN {$orders_table} o ON op.order_id = o.id
                LEFT JOIN {$wpdb->prefix}wc_orders_meta om ON o.id = om.order_id AND om.meta_key = '_billing_email'
                WHERE {$where_clause}
                GROUP BY o.customer_id, om.meta_value
                HAVING o.customer_id > 0
                ORDER BY total_profit DESC
                LIMIT %d
            ";
        } else {
            $query = "
                SELECT 
                    pm.meta_value as customer_id,
                    pm2.meta_value as customer_email,
                    COUNT(op.order_id) as order_count,
                    SUM(op.revenue) as total_revenue,
                    SUM(op.net_profit) as total_profit,
                    AVG(op.margin_pct) as avg_margin_pct
                FROM {$profit_table} op
                INNER JOIN {$orders_table} o ON op.order_id = o.id
                INNER JOIN {$wpdb->postmeta} pm ON op.order_id = pm.post_id AND pm.meta_key = '_customer_user'
                LEFT JOIN {$wpdb->postmeta} pm2 ON op.order_id = pm2.post_id AND pm2.meta_key = '_billing_email'
                WHERE {$where_clause}
                GROUP BY pm.meta_value, pm2.meta_value
                HAVING pm.meta_value > 0
                ORDER BY total_profit DESC
                LIMIT %d
            ";
        }

        $params[] = $limit;
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $prepared_query = $wpdb->prepare( $query, $params );
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results( $prepared_query, ARRAY_A );

        // Enrich with customer data
        foreach ( $results as &$row ) {
            if ( $row['customer_id'] > 0 ) {
                $customer = get_userdata( $row['customer_id'] );
                if ( $customer ) {
                    $row['customer_name'] = $customer->display_name;
                }
            } else {
                $row['customer_name'] = __( 'Guest', 'quarkcode-neuralcommerce-lite' );
            }
        }

        $this->cache_manager->set( $cache_key, $results, 6 * HOUR_IN_SECONDS );

        return $results;
    }


    /**
     * Generate profit by date range report.
     *
     * @param string $date_from Start date (Y-m-d).
     * @param string $date_to   End date (Y-m-d).
     * @param string $interval  Interval (day, week, month).
     * @return array Report data.
     */
    public function get_profit_by_date( $date_from, $date_to, $interval = 'day' ) {
        global $wpdb;
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        $date_format = $this->get_mysql_date_format( $interval );

        $query = $wpdb->prepare(
            //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT DATE_FORMAT(o.date_created_gmt, %s) as period, COUNT(op.order_id) as order_count, SUM(op.revenue) as total_revenue, SUM(op.cogs_total) as total_cogs, SUM(op.net_profit) as total_profit, AVG(op.margin_pct) as avg_margin_pct FROM {$profit_table} op INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE o.date_created_gmt BETWEEN %s AND %s
            GROUP BY period
            ORDER BY period ASC",
            $date_format,
            $date_from . ' 00:00:00',
            $date_to . ' 23:59:59'
        );

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results( $query, ARRAY_A );
    }


    /**
     * Get MySQL date format based on interval.
     *
     * @param string $interval Interval.
     * @return string Date format.
     */
    private function get_mysql_date_format( $interval ) {
        switch ( $interval ) {
            case 'week':
                return '%Y-W%u';
            case 'month':
                return '%Y-%m';
            case 'year':
                return '%Y';
            default:
                return '%Y-%m-%d';
        }
    }

    /**
     * Get dashboard summary data.
     *
     * @param string $date_from Start date (Y-m-d).
     * @param string $date_to   End date (Y-m-d).
     * @return array Summary data.
     */
    public function get_dashboard_summary( $date_from = '', $date_to = '' ) {
        $cache_key = "dashboard_summary_{$date_from}_{$date_to}";
        
        $cached = $this->cache_manager->get( $cache_key );

        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        if ( empty( $date_from ) || empty( $date_to ) ) {
            $date_from = wp_date( 'Y-m-01' );
            $date_to = wp_date( 'Y-m-d' );
        }

        // Use ORDER date, not calculation date
        $query = "
            SELECT 
                COUNT(op.order_id) as total_orders,
                SUM(op.revenue) as total_revenue,
                SUM(op.cogs_total) as total_cogs,
                SUM(op.cogs_total + COALESCE(op.shipping_cost, 0) + COALESCE(op.payment_fee, 0)) as total_costs,
                SUM(op.gross_profit) as total_gross_profit,
                SUM(op.net_profit) as total_net_profit,
                AVG(op.margin_pct) as avg_margin,
                SUM(op.shipping_cost) as total_shipping_cost,
                SUM(op.payment_fee) as total_payment_fees,
                COUNT(CASE WHEN op.net_profit < 0 THEN 1 END) as loss_orders_count,
                COUNT(CASE WHEN op.margin_pct < 15 THEN 1 END) as low_margin_orders_count
            FROM {$profit_table} op
            INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE o.date_created_gmt BETWEEN %s AND %s
        ";

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $prepared_query = $wpdb->prepare( $query, $date_from . ' 00:00:00', $date_to . ' 23:59:59' );
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $summary = $wpdb->get_row( $prepared_query, ARRAY_A );

        // ✅ Ensure all values are properly typed and handle nulls
        $summary = array(
            'total_orders'              => intval( $summary['total_orders'] ?? 0 ),
            'total_revenue'             => floatval( $summary['total_revenue'] ?? 0 ),
            'total_cogs'                => floatval( $summary['total_cogs'] ?? 0 ),
            'total_costs'               => floatval( $summary['total_costs'] ?? 0 ), 
            'total_gross_profit'        => floatval( $summary['total_gross_profit'] ?? 0 ),
            'total_net_profit'          => floatval( $summary['total_net_profit'] ?? 0 ),
            'avg_margin'                => floatval( $summary['avg_margin'] ?? 0 ),
            'total_shipping_cost'       => floatval( $summary['total_shipping_cost'] ?? 0 ),
            'total_payment_fees'        => floatval( $summary['total_payment_fees'] ?? 0 ),
            'loss_orders_count'         => intval( $summary['loss_orders_count'] ?? 0 ),
            'low_margin_orders_count'   => intval( $summary['low_margin_orders_count'] ?? 0 ),
        );

        // Get top products
        $summary['top_products'] = $this->get_profit_by_product( $date_from, $date_to, 10 );

        // Get worst products
        $summary['worst_products'] = $this->get_worst_products( $date_from, $date_to, 10 );

        $this->cache_manager->set( $cache_key, $summary, 2 * HOUR_IN_SECONDS );

        return $summary;
    }



    /**
     * Get worst performing products.
     *
     * @param string $date_from Start date.
     * @param string $date_to   End date.
     * @param int    $limit     Limit.
     * @return array Products.
     */
    public function get_worst_products( $date_from, $date_to, $limit = 10 ) {
        global $wpdb;
        $items_table = $wpdb->prefix . 'qcnc_order_items_profit';
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        $query = $wpdb->prepare(
            //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT oip.product_id, oip.variation_id, SUM(oip.item_profit) as total_profit, SUM(oip.item_revenue) as total_revenue FROM {$items_table} oip INNER JOIN {$profit_table} op ON oip.order_id = op.order_id INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE o.date_created_gmt BETWEEN %s AND %s
            GROUP BY oip.product_id, oip.variation_id
            ORDER BY total_profit ASC
            LIMIT %d",
            $date_from . ' 00:00:00',
            $date_to . ' 23:59:59',
            $limit
        );

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results( $query, ARRAY_A );

        foreach ( $results as &$row ) {
            $product_id = $row['variation_id'] ? $row['variation_id'] : $row['product_id'];
            $product = wc_get_product( $product_id );

            if ( $product ) {
                $row['product_name'] = $product->get_name();
                $row['sku'] = $product->get_sku();
            }
        }

        return $results;
    }


    /**
     * Export report to CSV.
     *
     * @param array  $data        Report data.
     * @param string $report_type Report type.
     * @return string CSV content.
     */
    public function export_to_csv( $data, $report_type ) {
        if ( empty( $data ) ) {
            return '';
        }

        ob_start();
        $output = fopen( 'php://output', 'w' );

        // Add headers.
        $headers = array_keys( $data[0] );
        fputcsv( $output, $headers );

        // Add data rows.
        foreach ( $data as $row ) {
            fputcsv( $output, $row );
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose( $output );
        return ob_get_clean();
    }

    /**
     * Get dashboard metrics for date range
     *
     * @param string $date_from Start date
     * @param string $date_to End date
     * @return array Dashboard metrics
     */
    public function get_dashboard_metrics( $date_from, $date_to ) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_profit';
        
        // Get aggregated metrics
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_row( $wpdb->prepare(
            //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT SUM(op.revenue) as total_revenue, SUM(op.cogs_total + COALESCE(op.shipping_cost, 0) + COALESCE(op.payment_fee, 0)) as total_costs, SUM(op.net_profit) as net_profit, COUNT(op.order_id) as order_count FROM {$profit_table} op INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE o.date_created_gmt BETWEEN %s AND %s",
            $date_from . ' 00:00:00',
            $date_to . ' 23:59:59'
        ) );
        
        // Calculate derived metrics
        $total_revenue = floatval( $results->total_revenue ?? 0 );
        $total_costs = floatval( $results->total_costs ?? 0 );
        $net_profit = floatval( $results->net_profit ?? 0 );
        $order_count = intval( $results->order_count ?? 0 );

        // ✅ CALCULATE MARGIN FROM TOTALS (gives 72.50%)
        $profit_margin = $total_revenue > 0 
            ? (($total_revenue - $total_costs) / $total_revenue) * 100 
            : 0;
        
        return array(
            'total_revenue' => $total_revenue,
            'total_costs' => $total_costs,
            'net_profit' => $net_profit,
            'profit_margin' => $profit_margin,
            'avg_order_value' => $order_count > 0 ? $total_revenue / $order_count : 0,
            'profit_per_order' => $order_count > 0 ? $net_profit / $order_count : 0,
            'order_count' => $order_count
        );
    }

    /**
     * Get comparison metrics vs previous period
     *
     * @param string $date_from Start date
     * @param string $date_to End date
     * @param string $comparison_type Type of comparison (previous_period, previous_year, budget)
     * @return array Comparison data with percentage changes
     */
    public function get_comparison_metrics( $date_from, $date_to, $comparison_type = 'previous_period' ) {
        // Get current period metrics
        $current = $this->get_dashboard_metrics( $date_from, $date_to );
        
        // Calculate comparison period dates
        $start = new \DateTime( $date_from );
        $end = new \DateTime( $date_to );
        $diff = $start->diff( $end )->days;
        
        if ( $comparison_type === 'previous_year' ) {
            $comp_start = $start->modify( '-1 year' )->format( 'Y-m-d' );
            $comp_end = $end->modify( '-1 year' )->format( 'Y-m-d' );
        } else {
            // Previous period (default)
            $comp_end = $start->modify( '-1 day' )->format( 'Y-m-d' );
            $comp_start = $start->modify( '-' . $diff . ' days' )->format( 'Y-m-d' );
        }
        
        // Get comparison period metrics
        $previous = $this->get_dashboard_metrics( $comp_start, $comp_end );
        
        // Calculate percentage changes
        return array(
            'revenue_change' => $this->calculate_percentage_change( $previous['total_revenue'], $current['total_revenue'] ),
            'cost_change' => $this->calculate_percentage_change( $previous['total_costs'], $current['total_costs'] ),
            'profit_change' => $this->calculate_percentage_change( $previous['net_profit'], $current['net_profit'] ),
            'margin_change' => $current['profit_margin'] - $previous['profit_margin'],
            'aov_change' => $this->calculate_percentage_change( $previous['avg_order_value'], $current['avg_order_value'] ),
            'ppo_change' => $this->calculate_percentage_change( $previous['profit_per_order'], $current['profit_per_order'] )
        );
    }

    /**
     * Calculate percentage change between two values
     *
     * @param float $old Old value
     * @param float $new New value
     * @return float Percentage change
     */
    private function calculate_percentage_change( $old, $new ) {
        if ( $old == 0 ) {
            return $new > 0 ? 100 : 0;
        }
        return ( ( $new - $old ) / abs( $old ) ) * 100;
    }

}
