<?php
/**
 * Product Cost Manager
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Core;

use QuarkcodeNeuralCommerce\Utilities\QCNC_Cache_Manager;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


/**
 * Manages product COGS with historical versioning.
 */
class QCNC_Product_Cost_Manager {

    /**
     * Cache manager instance.
     *
     * @var QCNC_Cache_Manager
     */
    private $cache_manager;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Cache_Manager $cache_manager Cache manager.
     * @param QCNC_Logger        $logger        Logger.
     */
    public function __construct( QCNC_Cache_Manager $cache_manager, QCNC_Logger $logger ) {
        $this->cache_manager = $cache_manager;
        $this->logger        = $logger;
    }

    /**
     * Set product cost with versioning.
     *
     * @param int    $product_id      Product ID.
     * @param float  $cost            Product cost.
     * @param int    $variation_id    Variation ID (optional).
     * @param float  $packaging_cost  Packaging cost (optional).
     * @param float  $handling_cost   Handling cost (optional).
     * @param string $currency        Currency code.
     * @return int|false Cost ID on success, false on failure.
     */
    /*
    public function set_product_cost( $product_id, $cost, $variation_id = null, $packaging_cost = 0.0, $handling_cost = 0.0, $currency = '', $additional_costs = [] ) {
        global $wpdb;

        if ( empty( $currency ) ) {
            $currency = get_woocommerce_currency();
        }

        $product_id   = absint( $product_id );
        $variation_id = $variation_id ? absint( $variation_id ) : null;

        // Validate product exists
        $product = wc_get_product( $variation_id ? $variation_id : $product_id );
        if ( ! $product ) {
            $this->logger->log( "Product #{$product_id} not found", 'error' );
            return false;
        }

        $table       = $wpdb->prefix . 'qcnc_product_costs';
        $current_time = current_time( 'mysql' );
        $user_id     = get_current_user_id();

        $target_product_id = $variation_id ? $variation_id : $product_id;
        $old_cost_meta = get_post_meta( $target_product_id, '_qcnc_product_cost', true );
        $old_total_cost_meta = get_post_meta( $target_product_id, '_qcnc_total_cost', true );

        // Start transaction
        $wpdb->query( 'START TRANSACTION' );

        try {
            // Get the existing current cost (if any)
            $existing_cost = $wpdb->get_row(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
                    "SELECT * FROM {$table} WHERE product_id = %d AND " . ( $variation_id ? "variation_id = %d" : "variation_id IS NULL" ) . "
                    AND is_current = 1",
                    $variation_id ? [ $product_id, $variation_id ] : [ $product_id ]
                ),
                ARRAY_A
            );

            // Check if the new cost is actually different from existing
            if ( $existing_cost ) {
                $same_cost = 
                    floatval( $existing_cost['cost'] ) === floatval( $cost ) &&
                    floatval( $existing_cost['packaging_cost'] ) === floatval( $packaging_cost ) &&
                    floatval( $existing_cost['handling_cost'] ) === floatval( $handling_cost ) &&
                    floatval( $existing_cost['warehousing_cost'] ?? 0 ) === floatval( $additional_costs['warehousing_cost'] ?? 0 ) &&
                    floatval( $existing_cost['labor_cost'] ?? 0 ) === floatval( $additional_costs['labor_cost'] ?? 0 ) &&
                    floatval( $existing_cost['shipping_material_cost'] ?? 0 ) === floatval( $additional_costs['shipping_material_cost'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_1'] ?? 0 ) === floatval( $additional_costs['custom_cost_1'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_2'] ?? 0 ) === floatval( $additional_costs['custom_cost_2'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_3'] ?? 0 ) === floatval( $additional_costs['custom_cost_3'] ?? 0 );

                if ( $same_cost ) {
                    // No change in cost, don't create a new record
                    $wpdb->query( 'COMMIT' );
                    $this->logger->log( "No cost change for product #{$product_id}, skipping update", 'info' );
                    return $existing_cost['id'];
                }
            }

            // Mark existing current cost as historical
            if ( $existing_cost ) {
                $wpdb->update(
                    $table,
                    [
                        'is_current'   => 0,
                        'effective_to' => $current_time,
                        'updated_at'   => $current_time,
                    ],
                    [ 'id' => $existing_cost['id'] ],
                    [ '%d', '%s', '%s' ],
                    [ '%d' ]
                );
            }

            // Prepare data for insertion
            $data = [
                'product_id'             => $product_id,
                'variation_id'           => $variation_id,
                'cost'                   => $cost,
                'currency'               => $currency,
                'packaging_cost'         => $packaging_cost,
                'handling_cost'          => $handling_cost,
                'warehousing_cost'       => isset( $additional_costs['warehousing_cost'] ) ? floatval( $additional_costs['warehousing_cost'] ) : 0.0,
                'labor_cost'             => isset( $additional_costs['labor_cost'] ) ? floatval( $additional_costs['labor_cost'] ) : 0.0,
                'shipping_material_cost' => isset( $additional_costs['shipping_material_cost'] ) ? floatval( $additional_costs['shipping_material_cost'] ) : 0.0,
                'custom_cost_1'          => isset( $additional_costs['custom_cost_1'] ) ? floatval( $additional_costs['custom_cost_1'] ) : 0.0,
                'custom_cost_1_label'    => isset( $additional_costs['custom_cost_1_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_1_label'] ) : '',
                'custom_cost_2'          => isset( $additional_costs['custom_cost_2'] ) ? floatval( $additional_costs['custom_cost_2'] ) : 0.0,
                'custom_cost_2_label'    => isset( $additional_costs['custom_cost_2_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_2_label'] ) : '',
                'custom_cost_3'          => isset( $additional_costs['custom_cost_3'] ) ? floatval( $additional_costs['custom_cost_3'] ) : 0.0,
                'custom_cost_3_label'    => isset( $additional_costs['custom_cost_3_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_3_label'] ) : '',
                'notes'                  => isset( $additional_costs['notes'] ) ? sanitize_textarea_field( $additional_costs['notes'] ) : '',
                'effective_from'         => $current_time,
                'effective_to'           => null,
                'is_current'             => 1,
                'created_by'             => $user_id,
                'created_at'             => $current_time,
            ];

            $format = [ '%d', '%d', '%f', '%s', '%f', '%f', '%f', '%f', '%f', '%f', '%s', '%f', '%s', '%f', '%s', '%s', '%s', '%s', '%d', '%d', '%s' ];

            // Insert new cost record
            $result = $wpdb->insert( $table, $data, $format );

            if ( false === $result ) {
                throw new \Exception( 'Failed to insert cost record' );
            }

            $cost_id = $wpdb->insert_id;

            // Commit transaction
            $wpdb->query( 'COMMIT' );

            // Clear cache
            $this->cache_manager->delete_product_cache( $product_id );

            // Calculate total cost
            $total_cost = $cost + $packaging_cost + $handling_cost + 
                        ( $data['warehousing_cost'] ?? 0 ) + 
                        ( $data['labor_cost'] ?? 0 ) + 
                        ( $data['shipping_material_cost'] ?? 0 ) + 
                        ( $data['custom_cost_1'] ?? 0 ) + 
                        ( $data['custom_cost_2'] ?? 0 ) + 
                        ( $data['custom_cost_3'] ?? 0 );

            // Store cost in product meta for quick access
            update_post_meta( $target_product_id, '_qcnc_product_cost', $cost );
            update_post_meta( $target_product_id, '_qcnc_current_cost', $cost );
            update_post_meta( $target_product_id, '_qcnc_total_cost', $total_cost );

            // 👉 FIRE ALERT HOOK IF COST CHANGED
            if ( $old_cost_meta && floatval( $old_cost_meta ) != floatval( $cost ) ) {
                do_action( 
                    'qcnc_product_cost_updated', 
                    $target_product_id,
                    floatval( $old_cost_meta ), 
                    floatval( $cost ),
                    floatval( $old_total_cost_meta ? $old_total_cost_meta : $old_cost_meta ),
                    floatval( $total_cost )
                );
                
                $this->logger->log( 
                    "Cost alert triggered for product #{$target_product_id}: " . 
                    "{$old_cost_meta} → {$cost} (Total: {$old_total_cost_meta} → {$total_cost})"
                );
            }

            $this->logger->log( "Cost set for product #{$product_id}: {$cost} {$currency}, Total: {$total_cost}" );

            return $cost_id;

        } catch ( \Exception $e ) {
            $wpdb->query( 'ROLLBACK' );
            $this->logger->log( "Error setting product cost: " . $e->getMessage(), 'error' );
            return false;
        }
    }
    */

    /**
     * Set product cost with versioning.
     *
     * @param int    $product_id      Product ID.
     * @param float  $cost            Product cost.
     * @param int    $variation_id    Variation ID (optional).
     * @param float  $packaging_cost  Packaging cost (optional).
     * @param float  $handling_cost   Handling cost (optional).
     * @param string $currency        Currency code.
     * @return int|false Cost ID on success, false on failure.
     */
    public function set_product_cost( $product_id, $cost, $variation_id = null, $packaging_cost = 0.0, $handling_cost = 0.0, $currency = '', $additional_costs = array() ) {
        global $wpdb;

        if ( empty( $currency ) ) {
            $currency = get_woocommerce_currency();
        }

        $product_id   = absint( $product_id );
        $variation_id = $variation_id ? absint( $variation_id ) : null;

        // Validate product exists
        $product = wc_get_product( $variation_id ? $variation_id : $product_id );
        if ( ! $product ) {
            $this->logger->log( "Product #{$product_id} not found", 'error' );
            return false;
        }

        $table        = $wpdb->prefix . 'qcnc_product_costs';
        $current_time = current_time( 'mysql' );
        $user_id      = get_current_user_id();

        $target_product_id   = $variation_id ? $variation_id : $product_id;
        $old_cost_meta       = get_post_meta( $target_product_id, '_qcnc_product_cost', true );
        $old_total_cost_meta = get_post_meta( $target_product_id, '_qcnc_total_cost', true );

        // Start transaction
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query( 'START TRANSACTION' );

        try {
            // ✅ FIXED: Prepare query differently for variation vs simple product
            if ( $variation_id ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
                $existing_cost = $wpdb->get_row(
                    $wpdb->prepare(
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        "SELECT * FROM {$table} WHERE product_id = %d AND variation_id = %d AND is_current = 1",
                        $product_id,
                        $variation_id
                    ),
                    ARRAY_A
                );
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
                $existing_cost = $wpdb->get_row(
                    $wpdb->prepare(
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        "SELECT * FROM {$table} WHERE product_id = %d AND variation_id IS NULL AND is_current = 1",
                        $product_id
                    ),
                    ARRAY_A
                );
            }

            // Check if the new cost is actually different from existing
            if ( $existing_cost ) {
                $same_cost = 
                    floatval( $existing_cost['cost'] ) === floatval( $cost ) &&
                    floatval( $existing_cost['packaging_cost'] ) === floatval( $packaging_cost ) &&
                    floatval( $existing_cost['handling_cost'] ) === floatval( $handling_cost ) &&
                    floatval( $existing_cost['warehousing_cost'] ?? 0 ) === floatval( $additional_costs['warehousing_cost'] ?? 0 ) &&
                    floatval( $existing_cost['labor_cost'] ?? 0 ) === floatval( $additional_costs['labor_cost'] ?? 0 ) &&
                    floatval( $existing_cost['shipping_material_cost'] ?? 0 ) === floatval( $additional_costs['shipping_material_cost'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_1'] ?? 0 ) === floatval( $additional_costs['custom_cost_1'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_2'] ?? 0 ) === floatval( $additional_costs['custom_cost_2'] ?? 0 ) &&
                    floatval( $existing_cost['custom_cost_3'] ?? 0 ) === floatval( $additional_costs['custom_cost_3'] ?? 0 );

                if ( $same_cost ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                    $wpdb->query( 'COMMIT' );
                    $this->logger->log( "No cost change for product #{$product_id}, skipping update", 'info' );
                    return $existing_cost['id'];
                }
            }

            // Mark existing current cost as historical
            if ( $existing_cost ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $wpdb->update(
                    $table,
                    array(
                        'is_current'   => 0,
                        'effective_to' => $current_time,
                        'updated_at'   => $current_time,
                    ),
                    array( 'id' => $existing_cost['id'] ),
                    array( '%d', '%s', '%s' ),
                    array( '%d' )
                );
            }

            // Prepare data for insertion
            $data = array(
                'product_id'             => $product_id,
                'variation_id'           => $variation_id,
                'cost'                   => $cost,
                'currency'               => $currency,
                'packaging_cost'         => $packaging_cost,
                'handling_cost'          => $handling_cost,
                'warehousing_cost'       => isset( $additional_costs['warehousing_cost'] ) ? floatval( $additional_costs['warehousing_cost'] ) : 0.0,
                'labor_cost'             => isset( $additional_costs['labor_cost'] ) ? floatval( $additional_costs['labor_cost'] ) : 0.0,
                'shipping_material_cost' => isset( $additional_costs['shipping_material_cost'] ) ? floatval( $additional_costs['shipping_material_cost'] ) : 0.0,
                'custom_cost_1'          => isset( $additional_costs['custom_cost_1'] ) ? floatval( $additional_costs['custom_cost_1'] ) : 0.0,
                'custom_cost_1_label'    => isset( $additional_costs['custom_cost_1_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_1_label'] ) : '',
                'custom_cost_2'          => isset( $additional_costs['custom_cost_2'] ) ? floatval( $additional_costs['custom_cost_2'] ) : 0.0,
                'custom_cost_2_label'    => isset( $additional_costs['custom_cost_2_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_2_label'] ) : '',
                'custom_cost_3'          => isset( $additional_costs['custom_cost_3'] ) ? floatval( $additional_costs['custom_cost_3'] ) : 0.0,
                'custom_cost_3_label'    => isset( $additional_costs['custom_cost_3_label'] ) ? sanitize_text_field( $additional_costs['custom_cost_3_label'] ) : '',
                'notes'                  => isset( $additional_costs['notes'] ) ? sanitize_textarea_field( $additional_costs['notes'] ) : '',
                'effective_from'         => $current_time,
                'effective_to'           => null,
                'is_current'             => 1,
                'created_by'             => $user_id,
                'created_at'             => $current_time,
            );

            $format = array( '%d', '%d', '%f', '%s', '%f', '%f', '%f', '%f', '%f', '%f', '%s', '%f', '%s', '%f', '%s', '%s', '%s', '%s', '%d', '%d', '%s' );

            // Insert new cost record
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->insert( $table, $data, $format );

            if ( false === $result ) {
                throw new \Exception( 'Failed to insert cost record' );
            }

            $cost_id = $wpdb->insert_id;

            // Commit transaction
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->query( 'COMMIT' );

            // Clear cache
            $this->cache_manager->delete_product_cache( $product_id );

            // Calculate total cost
            $total_cost = $cost + $packaging_cost + $handling_cost + 
                        ( $data['warehousing_cost'] ?? 0 ) + 
                        ( $data['labor_cost'] ?? 0 ) + 
                        ( $data['shipping_material_cost'] ?? 0 ) + 
                        ( $data['custom_cost_1'] ?? 0 ) + 
                        ( $data['custom_cost_2'] ?? 0 ) + 
                        ( $data['custom_cost_3'] ?? 0 );

            // Store cost in product meta for quick access
            update_post_meta( $target_product_id, '_qcnc_product_cost', $cost );
            update_post_meta( $target_product_id, '_qcnc_current_cost', $cost );
            update_post_meta( $target_product_id, '_qcnc_total_cost', $total_cost );

            // Fire alert hook if cost changed
            if ( $old_cost_meta && floatval( $old_cost_meta ) !== floatval( $cost ) ) {
                do_action( 
                    'qcnc_product_cost_updated', 
                    $target_product_id,
                    floatval( $old_cost_meta ), 
                    floatval( $cost ),
                    floatval( $old_total_cost_meta ? $old_total_cost_meta : $old_cost_meta ),
                    floatval( $total_cost )
                );
                
                $this->logger->log( 
                    "Cost alert triggered for product #{$target_product_id}: " . 
                    "{$old_cost_meta} → {$cost} (Total: {$old_total_cost_meta} → {$total_cost})"
                );
            }

            $this->logger->log( "Cost set for product #{$product_id}: {$cost} {$currency}, Total: {$total_cost}" );

            return $cost_id;

        } catch ( \Exception $e ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->query( 'ROLLBACK' );
            $this->logger->log( "Error setting product cost: " . $e->getMessage(), 'error' );
            return false;
        }
    }

    /**
     * Get current product cost.
     *
     * @param int $product_id   Product ID.
     * @param int $variation_id Variation ID (optional).
     * @return array|null Cost data or null if not found.
     */
    public function get_product_cost( $product_id, $variation_id = null ) {
        $cache_key = "product_cost_{$product_id}_" . ( $variation_id ? $variation_id : '0' );
        $cached    = $this->cache_manager->get( $cache_key );

        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        $query = $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM {$table} 
            WHERE product_id = %d 
            AND is_current = 1",
            $product_id
        );

        if ( $variation_id ) {
            $query .= $wpdb->prepare( " AND variation_id = %d", $variation_id );
        } else {
            $query .= " AND variation_id IS NULL";
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $cost = $wpdb->get_row( $query, ARRAY_A );

        if ( $cost ) {
            // Calculate total cost with ALL components
            $cost['total_cost'] = floatval( $cost['cost'] ?? 0 ) + 
                                floatval( $cost['packaging_cost'] ?? 0 ) + 
                                floatval( $cost['handling_cost'] ?? 0 ) +
                                floatval( $cost['warehousing_cost'] ?? 0 ) +
                                floatval( $cost['labor_cost'] ?? 0 ) +
                                floatval( $cost['shipping_material_cost'] ?? 0 ) +
                                floatval( $cost['custom_cost_1'] ?? 0 ) +
                                floatval( $cost['custom_cost_2'] ?? 0 ) +
                                floatval( $cost['custom_cost_3'] ?? 0 );
            
            $this->cache_manager->set( $cache_key, $cost, 12 * HOUR_IN_SECONDS );
        }

        return $cost;
    }


    /**
     * Get product cost at specific date (historical).
     *
     * @param int    $product_id   Product ID.
     * @param string $date         Date in Y-m-d H:i:s format.
     * @param int    $variation_id Variation ID (optional).
     * @return array|null Cost data or null if not found.
     */
    /*
    public function get_product_cost_at_date( $product_id, $date, $variation_id = null ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        $query = $wpdb->prepare(
            "SELECT * FROM {$table} 
             WHERE product_id = %d 
             AND effective_from <= %s 
             AND (effective_to IS NULL OR effective_to > %s)
             ORDER BY effective_from DESC 
             LIMIT 1",
            $product_id,
            $date,
            $date
        );

        if ( $variation_id ) {
            $variation_condition = $wpdb->prepare( " AND variation_id = %d", $variation_id );
            $query = str_replace( 'ORDER BY', $variation_condition . ' ORDER BY', $query );
        }

        $cost = $wpdb->get_row( $query, ARRAY_A );

        if ( $cost ) {
            $cost['total_cost'] = floatval( $cost['cost'] ) + floatval( $cost['packaging_cost'] ) + floatval( $cost['handling_cost'] );
        }

        return $cost;
    }
    */

    /**
     * Get product cost at specific date (historical).
     *
     * @param int    $product_id   Product ID.
     * @param string $date         Date in Y-m-d H:i:s format.
     * @param int    $variation_id Variation ID (optional).
     * @return array|null Cost data or null if not found.
     */
    public function get_product_cost_at_date( $product_id, $date, $variation_id = null ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // Build query based on whether we have a variation
        if ( $variation_id ) {
            $query = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} 
                WHERE product_id = %d 
                AND variation_id = %d
                AND effective_from <= %s 
                AND (effective_to IS NULL OR effective_to > %s)
                ORDER BY effective_from DESC 
                LIMIT 1",
                $product_id,
                $variation_id,
                $date,
                $date
            );
        } else {
            $query = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} 
                WHERE product_id = %d 
                AND variation_id IS NULL
                AND effective_from <= %s 
                AND (effective_to IS NULL OR effective_to > %s)
                ORDER BY effective_from DESC 
                LIMIT 1",
                $product_id,
                $date,
                $date
            );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $cost = $wpdb->get_row( $query, ARRAY_A );

        // FALLBACK: If no historical cost found, use current cost
        if ( ! $cost ) {
            $this->logger->log( 
                "No historical cost found for product #{$product_id}" . 
                ( $variation_id ? " (variation #{$variation_id})" : "" ) . 
                " at {$date}, falling back to current cost", 
                'info' 
            );
            
            $cost = $this->get_product_cost( $product_id, $variation_id );
        }

        if ( $cost ) {
            // Calculate total cost with ALL components
            $cost['total_cost'] = floatval( $cost['cost'] ?? 0 ) + 
                                floatval( $cost['packaging_cost'] ?? 0 ) + 
                                floatval( $cost['handling_cost'] ?? 0 ) +
                                floatval( $cost['warehousing_cost'] ?? 0 ) +
                                floatval( $cost['labor_cost'] ?? 0 ) +
                                floatval( $cost['shipping_material_cost'] ?? 0 ) +
                                floatval( $cost['custom_cost_1'] ?? 0 ) +
                                floatval( $cost['custom_cost_2'] ?? 0 ) +
                                floatval( $cost['custom_cost_3'] ?? 0 );
        }

        return $cost;
    }



    /**
     * Get cost history for a product.
     *
     * @param int $product_id   Product ID.
     * @param int $variation_id Variation ID (optional).
     * @param int $limit        Number of records to retrieve.
     * @return array Cost history.
     */
    public function get_cost_history( $product_id, $variation_id = null, $limit = 50 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        $query = $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM {$table} 
            WHERE product_id = %d",
            $product_id
        );

        if ( $variation_id ) {
            $query .= $wpdb->prepare( " AND variation_id = %d", $variation_id );
        } else {
            $query .= " AND variation_id IS NULL";
        }

        // UPDATED: Order by is_current first, then by date
        $query .= " ORDER BY is_current DESC, effective_from DESC";

        if ( $limit > 0 ) {
            $query .= $wpdb->prepare( " LIMIT %d", $limit );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results( $query, ARRAY_A );
    }


    /**
     * Bulk update costs from array.
     *
     * @param array $cost_data Array of cost data.
     * @return array Results with success/error counts.
     */
    public function bulk_update_costs( $cost_data ) {
        $results = [
            'success' => 0,
            'failed'  => 0,
            'errors'  => [],
        ];

        foreach ( $cost_data as $index => $item ) {
            if ( empty( $item['product_id'] ) || ! isset( $item['cost'] ) ) {
                $results['failed']++;
                $results['errors'][] = "Row {$index}: Missing product_id or cost";
                continue;
            }

            $product_id    = absint( $item['product_id'] );
            $variation_id  = ! empty( $item['variation_id'] ) ? absint( $item['variation_id'] ) : null;
            $cost          = floatval( $item['cost'] );
            $packaging     = isset( $item['packaging_cost'] ) ? floatval( $item['packaging_cost'] ) : 0.0;
            $handling      = isset( $item['handling_cost'] ) ? floatval( $item['handling_cost'] ) : 0.0;
            $currency      = ! empty( $item['currency'] ) ? sanitize_text_field( $item['currency'] ) : '';

            $result = $this->set_product_cost( $product_id, $cost, $variation_id, $packaging, $handling, $currency );

            if ( $result ) {
                $results['success']++;
            } else {
                $results['failed']++;
                $results['errors'][] = "Row {$index}: Failed to update product #{$product_id}";
            }
        }

        $this->logger->log( "Bulk update completed: {$results['success']} success, {$results['failed']} failed" );

        return $results;
    }

    /**
     * Delete product cost (mark as inactive rather than delete).
     *
     * @param int $cost_id Cost ID.
     * @return bool Success status.
     */
    public function delete_cost( $cost_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->update(
            $table,
            [
                'is_current'   => 0,
                'effective_to' => current_time( 'mysql' ),
            ],
            [ 'id' => absint( $cost_id ) ],
            [ '%d', '%s' ],
            [ '%d' ]
        );

        if ( $result ) {
            $this->logger->log( "Cost #{$cost_id} marked as inactive" );
            return true;
        }

        return false;
    }

    /**
     * Get all products without costs.
     *
     * @param int $limit Limit number of results.
     * @return array Product IDs without costs.
     */
    public function get_products_without_costs( $limit = 100 ) {
        global $wpdb;

        $query = $wpdb->prepare(
            "SELECT DISTINCT p.ID 
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->prefix}qcnc_product_costs c ON p.ID = c.product_id AND c.is_current = 1
             WHERE p.post_type IN ('product', 'product_variation')
             AND p.post_status = 'publish'
             AND c.id IS NULL
             LIMIT %d",
            $limit
        );

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        return $wpdb->get_col( $query );
    }

    /**
     * Get total cost including all components.
     *
     * @param int $product_id   Product ID.
     * @param int $variation_id Variation ID (optional).
     * @return float Total cost.
     */
    public function get_total_cost( $product_id, $variation_id = null ) {
        $cost_data = $this->get_product_cost( $product_id, $variation_id );

        if ( ! $cost_data ) {
            return 0.0;
        }

        return floatval( $cost_data['cost'] ) + 
            floatval( $cost_data['packaging_cost'] ) + 
            floatval( $cost_data['handling_cost'] ) +
            floatval( $cost_data['warehousing_cost'] ?? 0 ) +
            floatval( $cost_data['labor_cost'] ?? 0 ) +
            floatval( $cost_data['shipping_material_cost'] ?? 0 ) +
            floatval( $cost_data['custom_cost_1'] ?? 0 ) +
            floatval( $cost_data['custom_cost_2'] ?? 0 ) +
            floatval( $cost_data['custom_cost_3'] ?? 0 );
    }


    /**
     * Search products by cost range.
     *
     * @param float $min_cost Minimum cost.
     * @param float $max_cost Maximum cost.
     * @return array Product IDs.
     */
    public function search_by_cost_range( $min_cost, $max_cost ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        $query = $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT DISTINCT product_id FROM {$table}
             WHERE is_current = 1
             AND (cost + packaging_cost + handling_cost) BETWEEN %f AND %f
             ORDER BY product_id ASC",
            $min_cost,
            $max_cost
        );

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_col( $query );
    }

    /**
     * Clean up duplicate current cost records.
     * Ensures only one record per product has is_current = 1.
     *
     * @param int $product_id Product ID.
     * @param int $variation_id Variation ID (optional).
     */
    public function cleanup_duplicate_current_costs( $product_id, $variation_id = null ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // Get all current costs for this product
        if ( $variation_id ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $current_costs = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT * FROM {$table} 
                    WHERE product_id = %d 
                    AND variation_id = %d
                    AND is_current = 1
                    ORDER BY effective_from DESC",
                    $product_id,
                    $variation_id
                ),
                ARRAY_A
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $current_costs = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT * FROM {$table} 
                    WHERE product_id = %d 
                    AND variation_id IS NULL
                    AND is_current = 1
                    ORDER BY effective_from DESC",
                    $product_id
                ),
                ARRAY_A
            );
        }

        // If more than one current cost exists, keep only the newest
        if ( count( $current_costs ) > 1 ) {
            $this->logger->log( "Found " . count( $current_costs ) . " current costs for product #{$product_id}, cleaning up...", 'warning' );
            
            // Keep the first one (newest), mark others as historical
            for ( $i = 1; $i < count( $current_costs ); $i++ ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $wpdb->update(
                    $table,
                    [
                        'is_current'   => 0,
                        'effective_to' => $current_costs[$i]['effective_from'],
                    ],
                    [ 'id' => $current_costs[$i]['id'] ],
                    [ '%d', '%s' ],
                    [ '%d' ]
                );
            }
            
            $this->logger->log( "Cleanup complete for product #{$product_id}", 'info' );
        }
    }


}
