<?php
/**
 * Order Profit Calculator
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Core;

use QuarkcodeNeuralCommerce\Utilities\QCNC_Cache_Manager;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Calculates order profitability.
 */
class QCNC_Order_Profit_Calculator {

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    private $cost_manager;

    /**
     * Cache manager instance.
     *
     * @var QCNC_Cache_Manager
     */
    private $cache_manager;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Product_Cost_Manager $cost_manager  Cost manager.
     * @param QCNC_Cache_Manager        $cache_manager Cache manager.
     * @param QCNC_Logger               $logger        Logger.
     */
    public function __construct( QCNC_Product_Cost_Manager $cost_manager, QCNC_Cache_Manager $cache_manager, QCNC_Logger $logger ) {
        $this->cost_manager  = $cost_manager;
        $this->cache_manager = $cache_manager;
        $this->logger        = $logger;
    }

    /**
     * Calculate profit for an order.
     *
     * @param int $order_id Order ID.
     * @return array|false Profit data or false on failure.
     */
    public function calculate_order_profit( $order_id ) {
        $order = wc_get_order( $order_id );

        if ( ! $order ) {
            $this->logger->log( "Order #{$order_id} not found", 'error' );
            return false;
        }

        // ADD THIS CHECK - Skip if it's a refund
        if ( $order instanceof \WC_Order_Refund ) {
            $this->logger->log( "Skipping refund #{$order_id}", 'info' );
            return false;
        }

        global $wpdb;
        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $items_table  = $wpdb->prefix . 'qcnc_order_items_profit';

        // Start transaction.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query( 'START TRANSACTION' );

        try {
            $order_date = $order->get_date_created() ? $order->get_date_created()->date( 'Y-m-d H:i:s' ) : current_time( 'mysql' );

            // Calculate revenue.
            $revenue = floatval( $order->get_total() ) - floatval( $order->get_total_tax() );

            // Calculate COGS.
            $cogs_total = 0.0;
            $items_profit_data = [];

            foreach ( $order->get_items() as $item_id => $item ) {
                $product_id   = $item->get_product_id();
                $variation_id = $item->get_variation_id();
                $quantity     = $item->get_quantity();
                $item_total   = floatval( $item->get_total() );

                // Get cost at order date.
                $cost_data = $this->cost_manager->get_product_cost_at_date(
                    $product_id,
                    $order_date,
                    $variation_id ? $variation_id : null
                );

                $cost_per_unit = 0.0;
                if ( $cost_data ) {
                    $cost_per_unit = floatval( $cost_data['total_cost'] );
                }

                $item_cogs   = $cost_per_unit * $quantity;
                $item_profit = $item_total - $item_cogs;
                $cogs_total += $item_cogs;

                $items_profit_data[] = [
                    'order_id'       => $order_id,
                    'order_item_id'  => $item_id,
                    'product_id'     => $product_id,
                    'variation_id'   => $variation_id ? $variation_id : null,
                    'quantity'       => $quantity,
                    'item_revenue'   => $item_total,
                    'item_cogs'      => $item_cogs,
                    'item_profit'    => $item_profit,
                    'cost_per_unit'  => $cost_per_unit,
                    'created_at'     => current_time( 'mysql' ),
                ];
            }

            // Calculate shipping cost.
            $shipping_cost = floatval( $order->get_shipping_total() );

            // Calculate payment gateway fee.
            $payment_fee = $this->calculate_payment_fee( $order );

            // Get tax and discount.
            $tax_amount      = floatval( $order->get_total_tax() );
            $discount_amount = floatval( $order->get_total_discount() );

            // Get refund amount.
            $refund_amount = floatval( $order->get_total_refunded() );

            // Calculate profits.
            $gross_profit = $revenue - $cogs_total;
            $net_profit   = $gross_profit - $shipping_cost - $payment_fee;
            $margin_pct   = $revenue > 0 ? ( $net_profit / $revenue ) * 100 : 0;

            // Prepare profit data.
            $profit_data = [
                'order_id'        => $order_id,
                'revenue'         => $revenue,
                'cogs_total'      => $cogs_total,
                'shipping_cost'   => $shipping_cost,
                'payment_fee'     => $payment_fee,
                'tax_amount'      => $tax_amount,
                'discount_amount' => $discount_amount,
                'refund_amount'   => $refund_amount,
                'gross_profit'    => $gross_profit,
                'net_profit'      => $net_profit,
                'margin_pct'      => $margin_pct,
                'currency'        => $order->get_currency(),
                //'calculated_at'   => current_time( 'mysql' ),
                'calculated_at' => $order_date,
                'needs_recalc'    => 0,
            ];

            // Delete existing records.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->delete( $profit_table, [ 'order_id' => $order_id ], [ '%d' ] );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->delete( $items_table, [ 'order_id' => $order_id ], [ '%d' ] );

            // Insert order profit.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->insert( $profit_table, $profit_data );

            if ( false === $result ) {
                throw new \Exception( 'Failed to insert order profit' );
            }

            // Insert items profit.
            foreach ( $items_profit_data as $item_data ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $wpdb->insert( $items_table, $item_data );
            }

            // Commit transaction.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->query( 'COMMIT' );

            // Update order meta.
            $order->update_meta_data( '_qcnc_net_profit', $net_profit );
            $order->update_meta_data( '_qcnc_margin_pct', $margin_pct );
            $order->save_meta_data();

            // Clear cache.
            $this->cache_manager->delete_order_cache( $order_id );

            $this->logger->log( "Profit calculated for order #{$order_id}: Net Profit = {$net_profit}, Margin = {$margin_pct}%" );

            // 👉 TRIGGER PROFIT CALCULATED ACTION FOR ALERTS
            do_action( 'qcnc_order_profit_calculated', $order_id, array(
                'profit' => $net_profit,
                'margin_percentage' => $margin_pct,
                'revenue' => $revenue,
                'cost' => $cogs_total,
                'gross_profit' => $gross_profit,
                'shipping_cost' => $shipping_cost,
                'payment_fee' => $payment_fee,
            ) );

            return $profit_data;


        } catch ( \Exception $e ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->query( 'ROLLBACK' );
            $this->logger->log( "Error calculating order profit: " . $e->getMessage(), 'error' );
            return false;
        }
    }

    /**
     * Calculate payment gateway fee for order.
     *
     * @param \WC_Order $order Order object.
     * @return float Payment fee.
     */
    private function calculate_payment_fee( $order ) {
        // ADD THIS CHECK
        if ( ! $order || ! method_exists( $order, 'get_payment_method' ) ) {
            return 0.0;
        }
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        $payment_method = $order->get_payment_method();
        $order_total    = floatval( $order->get_total() );
        $currency       = $order->get_currency();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $fee_config = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} 
                 WHERE gateway_id = %s 
                 AND currency = %s 
                 AND is_active = 1 
                 LIMIT 1",
                $payment_method,
                $currency
            ),
            ARRAY_A
        );

        if ( ! $fee_config ) {
            return 0.0;
        }

        $fee = 0.0;

        switch ( $fee_config['fee_type'] ) {
            case 'percentage':
                $fee = ( $order_total * floatval( $fee_config['percentage_fee'] ) ) / 100;
                break;

            case 'fixed':
                $fee = floatval( $fee_config['fixed_fee'] );
                break;

            case 'mixed':
                $fee = ( ( $order_total * floatval( $fee_config['percentage_fee'] ) ) / 100 ) + floatval( $fee_config['fixed_fee'] );
                break;
        }

        return $fee;
    }

    /**
     * Recalculate order profit (for refunds, updates).
     *
     * @param int $order_id Order ID.
     * @return array|false Updated profit data.
     */
    public function recalculate_order_profit( $order_id ) {
        $this->logger->log( "Recalculating profit for order #{$order_id}" );
        return $this->calculate_order_profit( $order_id );
    }

    /**
     * Get order profit data.
     *
     * @param int $order_id Order ID.
     * @return array|null Profit data or null.
     */
    public function get_order_profit( $order_id ) {
        $cache_key = "order_profit_{$order_id}";
        $cached    = $this->cache_manager->get( $cache_key );

        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $profit = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->prepare( "SELECT * FROM {$table} WHERE order_id = %d", $order_id ),
            ARRAY_A
        );

        if ( $profit ) {
            $this->cache_manager->set( $cache_key, $profit, 6 * HOUR_IN_SECONDS );
        }

        return $profit;
    }

    /**
     * Get order items profit breakdown.
     *
     * @param int $order_id Order ID.
     * @return array Items profit data.
     */
    public function get_order_items_profit( $order_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_items_profit';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->prepare( "SELECT * FROM {$table} WHERE order_id = %d", $order_id ),
            ARRAY_A
        );
    }

    /**
     * Bulk recalculate profits for multiple orders.
     *
     * @param array $order_ids Order IDs.
     * @return array Results.
     */
    public function bulk_recalculate( $order_ids ) {
        $results = [
            'success' => 0,
            'failed'  => 0,
        ];

        foreach ( $order_ids as $order_id ) {
            if ( $this->calculate_order_profit( $order_id ) ) {
                $results['success']++;
            } else {
                $results['failed']++;
            }
        }

        return $results;
    }

    /**
     * Mark orders for recalculation.
     *
     * @param array $order_ids Order IDs.
     * @return bool Success status.
     */
    /*
    public function mark_for_recalc( $order_ids ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        $placeholders = implode( ',', array_fill( 0, count( $order_ids ), '%d' ) );

        $result = $wpdb->query(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "UPDATE {$table} SET needs_recalc = 1 WHERE order_id IN ({$placeholders})",
                $order_ids
            )
        );

        return false !== $result;
    }
        */

    /**
     * Mark orders for recalculation.
     *
     * @param array $order_ids Order IDs.
     * @return bool Success status.
     */
    public function mark_for_recalc( $order_ids ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        // Ensure $order_ids is an array
        if ( ! is_array( $order_ids ) ) {
            $order_ids = array( $order_ids );
        }

        // Sanitize order IDs
        $order_ids = array_map( 'absint', $order_ids );
        $order_ids = array_filter( $order_ids ); // Remove any zeros

        if ( empty( $order_ids ) ) {
            return false;
        }

        // WordPress's $wpdb->prepare() doesn't work well with dynamic IN clauses
        $order_ids_string = implode( ',', $order_ids );

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $result = $wpdb->query(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            "UPDATE {$table} SET needs_recalc = 1 WHERE order_id IN ({$order_ids_string})"
        );

        return false !== $result;
    }

    /**
     * Get orders that need recalculation.
     *
     * @param int $limit Limit.
     * @return array Order IDs.
     */
    public function get_orders_needing_recalc( $limit = 100 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_col(
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT order_id FROM {$table} WHERE needs_recalc = 1 LIMIT %d",
                $limit
            )
        );
    }

    /**
     * Get real-time stats for dashboard
     *
     * @return array Real-time statistics
     */
    public function get_real_time_stats() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_profit';
        
        // Get stats for last hour
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $stats = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            "SELECT COUNT(*) as order_count, SUM(net_profit) as total_profit, AVG(margin_pct) as avg_margin FROM {$table_name}
            WHERE calculated_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)"
        );
        
        return array(
            'hourly_orders' => intval( $stats->order_count ?? 0 ),
            'hourly_profit' => floatval( $stats->total_profit ?? 0 ),
            'hourly_margin' => floatval( $stats->avg_margin ?? 0 )
        );
    }

}
