<?php
/**
 * Enhanced Margin Alert System
 *
 * Comprehensive automated profit alerts with:
 * - Email/SMS when margins drop below threshold
 * - Cost increase alerts
 * - Unprofitable order flags
 * - Daily profit summaries
 * - Anomaly detection
 *
 * @package QuarkcodeNeuralCommerce
 * @subpackage Core
 */

namespace QuarkcodeNeuralCommerce\Core;

use QuarkcodeNeuralCommerce\Utilities\Logger;
use QuarkcodeNeuralCommerce\Utilities\Cache_Manager;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class QCNC_Margin_Alert_System {

    /**
     * Alert types
     */
    const ALERT_MARGIN_DROP = 'margin_drop';
    const ALERT_COST_INCREASE = 'cost_increase';
    const ALERT_UNPROFITABLE = 'unprofitable_order';
    const ALERT_ANOMALY = 'anomaly_detected';
    const ALERT_DAILY_SUMMARY = 'daily_summary';

    /**
     * Alert rate limit flag
     */
    private static $bulk_operation_in_progress = false;

    /**
     * Alert queue for batching
     */
    private static $alert_queue = array();

    /**
     * Initialize the alert system
     */
    public function __construct() {
        static $hooks_registered = false;
        if ( $hooks_registered ) {
            // error_log( '⚠️ QCNC: Duplicate instance prevented!' );
            return;
        }
        $hooks_registered = true;
        // error_log( '✅ QCNC: Registering hooks (first time)' );

        // Hook into order completion
        add_action( 'woocommerce_order_status_completed', array( $this, 'check_order_profitability' ), 10, 1 );

        // Hook into product cost updates
        add_action( 'qcnc_product_cost_updated', array( $this, 'check_cost_increase' ), 10, 3 );

        // Hook into profit calculations
        add_action( 'qcnc_order_profit_calculated', array( $this, 'check_margin_drop' ), 10, 2 );

        // Hook to detect bulk operations
        add_action( 'qcnc_before_bulk_profit_calculation', array( __CLASS__, 'set_bulk_operation_mode' ), 10, 0 );
        add_action( 'qcnc_after_bulk_profit_calculation', function() {
            self::set_bulk_operation_mode( false );
        }, 10, 0 );

        // Register daily summary cron
        add_action( 'qcnc_daily_profit_summary', array( $this, 'send_daily_summary' ) );

        // Register anomaly detection cron
        add_action( 'qcnc_anomaly_detection', array( $this, 'detect_anomalies' ) );

        // Add settings
        add_filter( 'qcnc_settings_tabs', array( $this, 'add_alerts_settings_tab' ) );
        add_action( 'qcnc_settings_alerts', array( $this, 'render_alerts_settings' ) );
        // Register digest cron jobs
        add_action( 'qcnc_hourly_digest', array( $this, 'send_digest_email' ) );
        add_action( 'qcnc_daily_digest', array( $this, 'send_digest_email' ) );

        // Schedule cron if not already scheduled
        if ( ! wp_next_scheduled( 'qcnc_hourly_digest' ) ) {
            wp_schedule_event( time(), 'hourly', 'qcnc_hourly_digest' );
        }

        if ( ! wp_next_scheduled( 'qcnc_daily_digest' ) ) {
            wp_schedule_event( strtotime( 'tomorrow 9:00 AM' ), 'daily', 'qcnc_daily_digest' );
        }
        
    }

    /**
     * Check if an order is unprofitable
     *
     * @param int $order_id Order ID
     */
    public function check_order_profitability( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        // Get order profit data
        $profit_data = $this->calculate_order_profit( $order_id );

        if ( $profit_data['margin_pct'] < 0 || $profit_data['net_profit'] < 0 ) {
            $this->trigger_alert( self::ALERT_UNPROFITABLE, array(
                'order_id' => $order_id,
                'order_number' => $order->get_order_number(),
                'profit' => $profit_data['net_profit'],
                'margin' => $profit_data['margin_pct'],
                'revenue' => $profit_data['revenue'],
                'cost' => $profit_data['cogs_total'],
            ) );
        }

        // Check if margin is below threshold
        $margin_threshold = get_option( 'qcnc_margin_threshold', 20 );
        if ( $profit_data['margin_pct'] < $margin_threshold ) {
            $this->trigger_alert( self::ALERT_MARGIN_DROP, array(
                'order_id' => $order_id,
                'order_number' => $order->get_order_number(),
                'margin' => $profit_data['margin_pct'],
                'threshold' => $margin_threshold,
                'profit' => $profit_data['net_profit'],
            ) );
        }
    }

    /**
     * Check for cost increases
     *
     * @param int $product_id Product ID
     * @param float $old_cost Old cost
     * @param float $new_cost New cost
     */
    public function check_cost_increase( $product_id, $old_cost, $new_cost ) {
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return;
        }

        // Calculate percentage increase
        $percentage_increase = ( ( $new_cost - $old_cost ) / $old_cost ) * 100;

        // Get threshold from settings
        $cost_increase_threshold = get_option( 'qcnc_cost_increase_threshold', 10 );

        if ( $percentage_increase > $cost_increase_threshold ) {
            $this->trigger_alert( self::ALERT_COST_INCREASE, array(
                'product_id' => $product_id,
                'product_name' => $product->get_name(),
                'sku' => $product->get_sku(),
                'old_cost' => $old_cost,
                'new_cost' => $new_cost,
                'increase_percentage' => round( $percentage_increase, 2 ),
                'price' => $product->get_price(),
            ) );
        }
    }

    /**
     * Check for margin drops
     *
     * @param int $order_id Order ID
     * @param array $profit_data Profit data
     */
    public function check_margin_drop( $order_id, $profit_data ) {
        global $wpdb;

        // Get average margin from last 30 days
        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $avg_margin = $wpdb->get_var( "SELECT AVG(margin_pct) FROM {$table_name} 
             WHERE calculated_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
             AND margin_pct > 0" );

        if ( ! $avg_margin ) {
            // error_log( 'QCNC: No historical margin data found for 30-day average.' );
            return;
        }

        // Check if current margin is significantly lower
        $current_margin = isset( $profit_data['margin_pct'] ) ? $profit_data['margin_pct'] : ( isset( $profit_data['margin_percentage'] ) ? $profit_data['margin_percentage'] : 0 );
        // error_log( "QCNC: Checking margin drop. Avg: {$avg_margin}, Current: {$current_margin}" );
        $margin_drop_threshold = get_option( 'qcnc_margin_drop_threshold', 30 ); // 30% drop
        $drop_percentage = ( ( $avg_margin - $current_margin ) / $avg_margin ) * 100;

         if ( $drop_percentage > $margin_drop_threshold ) {
            $order = wc_get_order( $order_id );

            // ✅ FIXED: Include full order details for email
            $this->trigger_alert( self::ALERT_MARGIN_DROP, array(
                'order_id' => $order_id,
                'order_number' => $order ? $order->get_order_number() : $order_id,
                'current_margin' => $current_margin,
                'average_margin' => round( $avg_margin, 2 ),
                'drop_percentage' => round( $drop_percentage, 2 ),

                // ✅ ADD THESE FIELDS FOR EMAIL DETAILS:
                'margin' => $current_margin,  // For template compatibility
                'revenue' => isset($profit_data['revenue']) ? $profit_data['revenue'] : 0,
                'profit' => isset($profit_data['net_profit']) ? $profit_data['net_profit'] : 0,
                'cost' => isset($profit_data['cogs_total']) ? $profit_data['cogs_total'] : 0,
                'threshold' => round( $avg_margin, 2 ),
            ) );
        }
    }

    /**
     * Detect profit anomalies using statistical analysis
     */
    public function detect_anomalies() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        // Get profit data from last 90 days
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $profit_data = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT order_id, net_profit, margin_pct, calculated_at FROM {$table_name}
             WHERE calculated_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)
             ORDER BY calculated_at ASC",
            ARRAY_A
        );

        if ( count( $profit_data ) < 30 ) {
            return; // Not enough data
        }

        // Calculate statistical metrics
        $profits = array_column( $profit_data, 'net_profit' );
        $margins = array_column( $profit_data, 'margin_pct' );

        $profit_mean = array_sum( $profits ) / count( $profits );
        $margin_mean = array_sum( $margins ) / count( $margins );

        // Calculate standard deviation
        $profit_std = $this->calculate_std_deviation( $profits, $profit_mean );
        $margin_std = $this->calculate_std_deviation( $margins, $margin_mean );

        // Check last 7 days for anomalies
        $recent_data = array_slice( $profit_data, -7 );
        $anomalies = array();

        foreach ( $recent_data as $data ) {
            $profit_z_score = ( $data['net_profit'] - $profit_mean ) / $profit_std;
            $margin_z_score = ( $data['margin_pct'] - $margin_mean ) / $margin_std;

            // If z-score is beyond 2 standard deviations, it's an anomaly
            if ( abs( $profit_z_score ) > 2 || abs( $margin_z_score ) > 2 ) {
                $anomalies[] = array(
                    'order_id' => $data['order_id'],
                    'profit' => $data['net_profit'],
                    'margin' => $data['margin_pct'],
                    'profit_z_score' => round( $profit_z_score, 2 ),
                    'margin_z_score' => round( $margin_z_score, 2 ),
                    'date' => $data['calculated_at'],
                );
            }
        }

        if ( ! empty( $anomalies ) ) {
            $this->trigger_alert( self::ALERT_ANOMALY, array(
                'anomalies' => $anomalies,
                'profit_mean' => round( $profit_mean, 2 ),
                'margin_mean' => round( $margin_mean, 2 ),
                'detection_period' => '90 days',
            ) );
        }
    }

    /**
     * Send daily profit summary
     */
    public function send_daily_summary() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        // Get yesterday's data
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $yesterday_data = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT COUNT(*) as order_count, SUM(revenue) as total_revenue, SUM(cogs_total) as total_cost, SUM(net_profit) as total_profit, AVG(margin_pct) as avg_margin FROM {$table_name}
             WHERE DATE(calculated_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)",
            ARRAY_A
        );

        // Get last 7 days comparison
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $last_week_data = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT AVG(net_profit) as avg_profit, AVG(margin_pct) as avg_margin FROM {$table_name}
             WHERE calculated_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)",
            ARRAY_A
        );

        // Get top profitable products
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $top_products = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT op.product_id, SUM(op.net_profit) as total_profit, AVG(op.margin_pct) as avg_margin, COUNT(*) as order_count FROM {$table_name} op
             WHERE DATE(op.calculated_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
             GROUP BY op.product_id
             ORDER BY total_profit DESC
             LIMIT 5",
            ARRAY_A
        );

        // Get products with low margins
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $low_margin_products = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT op.product_id, AVG(op.margin_pct) as avg_margin, COUNT(*) as order_count FROM {$table_name} op
             WHERE DATE(op.calculated_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
             GROUP BY op.product_id
             HAVING avg_margin < 15
             ORDER BY avg_margin ASC
             LIMIT 5",
            ARRAY_A
        );

        if ( $yesterday_data && $yesterday_data['order_count'] > 0 ) {
            $summary_data = array(
                'date' => wp_date( 'Y-m-d', strtotime( '-1 day' ) ),
                'yesterday' => $yesterday_data,
                'last_week' => $last_week_data,
                'top_products' => $top_products,
                'low_margin_products' => $low_margin_products,
            );

            $this->send_email_alert( self::ALERT_DAILY_SUMMARY, $summary_data );
        }
    }

    /**
     * Trigger an alert
     *
     * @param string $alert_type Alert type
     * @param array $data Alert data
     */
    /*
    private function trigger_alert( $alert_type, $data ) {
        // Check if alert type is enabled
        $enabled_alerts = get_option( 'qcnc_enabled_alerts', array(
            self::ALERT_MARGIN_DROP,
            self::ALERT_COST_INCREASE,
            self::ALERT_UNPROFITABLE,
            self::ALERT_ANOMALY,
            self::ALERT_DAILY_SUMMARY,
        ) );

        if ( ! in_array( $alert_type, $enabled_alerts ) ) {
            return;
        }

        // Log alert
        // error_log( "=== QCNC ALERT TRIGGERED ===" );
        // error_log( "Alert Type: " . $alert_type );
        // error_log( "Data: " . print_r( $data, true ) );

        // Save to database
        $this->save_alert( $alert_type, $data );

        // Send email notification
        $this->send_email_alert( $alert_type, $data );

        // Send SMS if enabled
        if ( get_option( 'qcnc_sms_alerts_enabled', false ) ) {
            $this->send_sms_alert( $alert_type, $data );
        }

        // Fire action hook for external integrations
        do_action( 'qcnc_alert_triggered', $alert_type, $data );
    }
        */

    /**
     * Set bulk operation flag
     * Call this before bulk profit calculations
     */
    public static function set_bulk_operation_mode( $enabled = true ) {
        self::$bulk_operation_in_progress = $enabled;

        if ( $enabled ) {
            // error_log( 'QCNC: Bulk operation mode ENABLED - alerts will be queued' );
        } else {
            // error_log( 'QCNC: Bulk operation mode DISABLED - sending queued alerts' );

            // Send digest of queued alerts
            if ( ! empty( self::$alert_queue ) ) {
                self::send_queued_alerts_digest();
            }
        }
    }

    /**
     * Send digest of queued alerts
     */
    /*
    private static function send_queued_alerts_digest() {
        if ( empty( self::$alert_queue ) ) {
            return;
        }

        $alert_counts = array();
        foreach ( self::$alert_queue as $alert ) {
            $type = $alert['type'];
            if ( ! isset( $alert_counts[ $type ] ) ) {
                $alert_counts[ $type ] = 0;
            }
            $alert_counts[ $type ]++;
        }

        $to = get_option( 'qcnc_alert_email', get_option( 'admin_email' ) );
        $subject = '[' . get_bloginfo( 'name' ) . '] 📊 Bulk Profit Calculation Summary';

        $message = '<h2>Profit Calculation Summary</h2>';
        $message .= '<p>The following alerts were triggered during bulk profit calculation:</p>';
        $message .= '<ul>';

        foreach ( $alert_counts as $type => $count ) {
            $message .= "<li><strong>{$count}</strong> {$type} alert(s)</li>";
        }

        $message .= '</ul>';
        $message .= '<p><strong>Total alerts:</strong> ' . count( self::$alert_queue ) . '</p>';
        $message .= '<p><small>Individual alert emails were suppressed to prevent spam. ';
        $message .= 'Check your dashboard for details.</small></p>';

        $headers = array( 'Content-Type: text/html; charset=UTF-8' );
        wp_mail( $to, $subject, $message, $headers );

        error_log( 'QCNC: Sent digest with ' . count( self::$alert_queue ) . ' queued alerts' );

        // Clear queue
        self::$alert_queue = array();
    }
    */

    /**
     * Build detailed digest email HTML with ALL alert details
     * 
     * @param array $alerts_by_type Alerts grouped by type
     * @return string HTML email content
     */
    private static function build_detailed_digest_email($alerts_by_type) {
        $site_name = get_bloginfo('name');
        $total_alerts = 0;
        foreach ($alerts_by_type as $alerts) {
            $total_alerts += count($alerts);
        }
        
        $current_date = wp_date('F j, Y g:i A');
        
        // ✅ FIX: Correct dashboard link
        $dashboard_url = admin_url('admin.php?page=qcnc-dashboard');
        $settings_url = admin_url('admin.php?page=quarkcode-neuralcommerce-settings&tab=alerts');
        
        ob_start();
        ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <?php
        // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedStylesheet -- Inline CSS required for HTML emails.
        // Email clients do not support external stylesheets, so inline styles are necessary for email rendering.
        ?>
        <style>
                body { 
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
                    line-height: 1.6; 
                    color: #333; 
                    margin: 0; 
                    padding: 0; 
                    background: #f5f5f5;
                }
                .email-wrapper { 
                    background: #f5f5f5; 
                    padding: 20px; 
                }
                .container { 
                    max-width: 600px; 
                    margin: 0 auto; 
                    background: white;
                    border-radius: 8px;
                    overflow: hidden;
                    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                }
                .header { 
                    background: linear-gradient(135deg, #2271b1 0%, #135e96 100%); 
                    color: white; 
                    padding: 30px 20px; 
                    text-align: center;
                }
                .header h1 { 
                    margin: 0; 
                    font-size: 24px; 
                    font-weight: 600;
                }
                .header p { 
                    margin: 10px 0 0 0; 
                    opacity: 0.9; 
                    font-size: 14px;
                }
                .content { 
                    padding: 30px 20px; 
                }
                .summary-box { 
                    background: #f9fafb; 
                    padding: 20px; 
                    margin: 0 0 25px 0; 
                    border-radius: 6px;
                    border-left: 4px solid #2271b1; 
                }
                .summary-box h2 { 
                    margin: 0 0 15px 0; 
                    color: #2271b1; 
                    font-size: 18px;
                    font-weight: 600;
                }
                .stat { 
                    font-size: 36px; 
                    font-weight: 700; 
                    color: #2271b1; 
                    margin: 0;
                }
                .stat-label { 
                    color: #666; 
                    font-size: 14px; 
                    margin: 5px 0 0 0;
                }
                .alert-section { 
                    margin: 30px 0; 
                }
                .alert-section-header { 
                    background: #f0f0f0; 
                    padding: 12px 15px; 
                    border-radius: 6px 6px 0 0;
                    border-left: 4px solid #d63638;
                    margin: 0 0 15px 0;
                }
                .alert-section-header.warning { border-left-color: #f0b849; }
                .alert-section-header.critical { border-left-color: #d63638; }
                .alert-section-header.info { border-left-color: #2271b1; }
                .alert-section-header h3 { 
                    margin: 0; 
                    font-size: 16px; 
                    color: #333;
                    font-weight: 600;
                }
                .alert-item { 
                    background: #fafafa; 
                    padding: 15px; 
                    margin: 0 0 12px 0; 
                    border-radius: 6px;
                    border-left: 3px solid #d63638;
                }
                .alert-item.warning { border-left-color: #f0b849; background: #fffbf0; }
                .alert-item.critical { border-left-color: #d63638; background: #fcf0f1; }
                .alert-item.info { border-left-color: #2271b1; background: #f0f6fc; }
                .alert-item-title { 
                    font-weight: 600; 
                    font-size: 15px; 
                    color: #1d2327; 
                    margin: 0 0 8px 0;
                }
                .alert-item-detail { 
                    display: flex; 
                    justify-content: space-between; 
                    padding: 5px 0;
                    font-size: 14px;
                }
                .alert-item-label { 
                    color: #666; 
                    font-weight: 500;
                }
                .alert-item-value { 
                    font-weight: 600; 
                    color: #1d2327;
                }
                .alert-item-value.negative { color: #d63638; }
                .alert-item-value.warning { color: #f0b849; }
                .alert-item-value.positive { color: #00a32a; }
                .divider { 
                    height: 1px; 
                    background: #e0e0e0; 
                    margin: 15px 0;
                }
                .actions-box { 
                    background: #f0f6fc; 
                    padding: 20px; 
                    border-radius: 6px; 
                    margin: 25px 0;
                    border-left: 4px solid #2271b1;
                }
                .actions-box h3 { 
                    margin: 0 0 15px 0; 
                    color: #2271b1; 
                    font-size: 16px;
                }
                .actions-box ul { 
                    margin: 0; 
                    padding: 0 0 0 20px;
                }
                .actions-box li { 
                    margin: 8px 0;
                    color: #333;
                }
                .btn { 
                    display: inline-block; 
                    padding: 12px 24px; 
                    background: #2271b1; 
                    color: white !important; 
                    text-decoration: none; 
                    border-radius: 5px;
                    font-weight: 600;
                    margin: 10px 5px 0 0;
                }
                .btn-secondary { 
                    background: #f0f0f0; 
                    color: #333 !important;
                }
                .footer { 
                    background: #fafafa; 
                    padding: 20px; 
                    text-align: center; 
                    font-size: 13px; 
                    color: #666;
                    border-top: 1px solid #e0e0e0;
                }
                .footer a { 
                    color: #2271b1; 
                    text-decoration: none;
                }
                @media only screen and (max-width: 600px) {
                    .email-wrapper { padding: 10px; }
                    .content { padding: 20px 15px; }
                    .alert-item-detail { flex-direction: column; }
                }
            </style>
    </head>
    <body>
    <div class="container">
        <div class="header">
            <h1>📊 Profit Alert Digest</h1>
            <p><?php echo esc_html($current_date); ?></p>
        </div>

        <div class="content">
            <!-- SUMMARY -->
            <div class="summary-box">
                <h2>Summary</h2>
                <p class="stat"><?php echo esc_html($total_alerts); ?></p>
                <p>Total alerts triggered</p>
                
                <ul>
                    <?php foreach ($alerts_by_type as $type => $alerts) : ?>
                        <li>
                            <strong><?php echo count($alerts); ?></strong>
                            <?php echo esc_html(self::get_alert_type_label($type)); ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>

            <?php
            // ═════════════════════════════════════════════════════════
            // MARGIN DROP / LOW MARGIN ALERTS
            // ═════════════════════════════════════════════════════════
            // Handle 'margin_drop', 'margindrop' and 'lowmargin' types
            $margin_alerts = array();
            if (isset($alerts_by_type['margin_drop'])) {
                $margin_alerts = array_merge($margin_alerts, $alerts_by_type['margin_drop']);
            }
            if (isset($alerts_by_type['margindrop'])) {
                $margin_alerts = array_merge($margin_alerts, $alerts_by_type['margindrop']);
            }
            if (isset($alerts_by_type['low_margin'])) {
                $margin_alerts = array_merge($margin_alerts, $alerts_by_type['low_margin']);
            }
            
            if (!empty($margin_alerts)) :
                $avg_margin = 0;
                $margin_count = 0;
                foreach ($margin_alerts as $alert) {
                    if (isset($alert['margin'])) {
                        $avg_margin += $alert['margin'];
                        $margin_count++;
                    }
                }
                if ($margin_count > 0) {
                    $avg_margin = $avg_margin / $margin_count;
                }
            ?>
            <div class="alert-section">
                <div class="alert-section-header warning">
                    <h3>⚠️ Low Margin Alerts (<?php echo count($margin_alerts); ?>)</h3>
                </div>
                
                <div style="background: #fffbf0; padding: 12px; margin-bottom: 15px; border-radius: 4px;">
                    <p style="margin: 0;">
                        Average margin: 
                        <strong style="color: #f0b849;"><?php echo number_format($avg_margin, 2); ?>%</strong>
                    </p>
                </div>
                
                <?php foreach ($margin_alerts as $alert) : 
                    $order = wc_get_order($alert['order_id'] ?? 0);
                    if (!$order) continue;
                ?>
                <div class="alert-item warning">
                    <div class="alert-item-title">
                        Order #<?php echo esc_html($alert['order_number'] ?? $order->get_order_number()); ?>
                    </div>
                    <?php if (isset($alert['revenue'])) : ?>
                    <div class="alert-item-detail">
                        <span>Revenue:</span>
                        <span><?php echo wp_kses_post(wc_price($alert['revenue'])); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($alert['profit'])) : ?>
                    <div class="alert-item-detail">
                        <span>Profit:</span>
                        <span><?php echo wp_kses_post(wc_price($alert['profit'])); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="alert-item-detail">
                        <span>Margin:</span>
                        <span style="color: #f0b849; font-weight: bold;">
                            <?php echo number_format($alert['margin'] ?? $alert['current_margin'] ?? 0, 2); ?>%
                        </span>
                    </div>
                    <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;">
                        <a href="<?php echo esc_url($order->get_edit_order_url()); ?>" style="color: #2271b1;">
                            View Order →
                        </a>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <?php
            // ═════════════════════════════════════════════════════════
            // UNPROFITABLE ORDER ALERTS
            // ═════════════════════════════════════════════════════════
            if ( isset( $alerts_by_type['unprofitable_order'] ) ) :
                $unprofitable_alerts = $alerts_by_type['unprofitable_order'];
            ?>
            <div class="alert-section">
                <div class="alert-section-header critical">
                    <h3>🚨 Unprofitable Orders (<?php echo count( $unprofitable_alerts ); ?>)</h3>
                </div>

                <?php foreach ( $unprofitable_alerts as $alert ) : 
                    $order = wc_get_order($alert['order_id'] ?? 0);
                    if (!$order) continue;
                ?>
                <div class="alert-item critical">
                    <div class="alert-item-title">
                        Order #<?php echo esc_html($alert['order_number'] ?? $order->get_order_number()); ?>
                    </div>
                    <?php if (isset($alert['revenue'])) : ?>
                    <div class="alert-item-detail">
                        <span>Revenue:</span>
                        <span><?php echo wp_kses_post(wc_price($alert['revenue'])); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($alert['profit'])) : ?>
                    <div class="alert-item-detail">
                        <span>Profit:</span>
                        <span style="color: #d63638; font-weight: bold;"><?php echo wp_kses_post(wc_price($alert['profit'])); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="alert-item-detail">
                        <span>Margin:</span>
                        <span style="color: #d63638; font-weight: bold;">
                            <?php echo number_format($alert['margin'] ?? 0, 2); ?>%
                        </span>
                    </div>
                    <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0; border-top-color: rgba(214, 54, 56, 0.2);">
                        <a href="<?php echo esc_url($order->get_edit_order_url()); ?>" style="color: #d63638;">
                            View Order →
                        </a>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <?php
            // ═════════════════════════════════════════════════════════
            // COST INCREASE ALERTS
            // ═════════════════════════════════════════════════════════
            if ( isset( $alerts_by_type['cost_increase'] ) ) :
                $alerts = $alerts_by_type['cost_increase'];
            ?>
            <div class="alert-section">
                <div class="alert-section-header warning">
                    <h3>📈 Cost Increase Alerts (<?php echo count( $alerts ); ?>)</h3>
                </div>

                <?php foreach ( $alerts as $alert ) : 
                    $product = wc_get_product( $alert['product_id'] );
                    if ( ! $product ) continue;
                ?>
                <div class="alert-item warning">
                    <div class="alert-item-title">
                        <?php echo esc_html( $alert['product_name'] ); ?>
                    </div>
                    <?php if ( ! empty( $alert['sku'] ) ) : ?>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">SKU:</span>
                        <span class="alert-item-value"><?php echo esc_html( $alert['sku'] ); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Old Cost:</span>
                        <span class="alert-item-value"><?php echo wp_kses_post(wc_price( $alert['old_cost'] )); ?></span>
                    </div>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">New Cost:</span>
                        <span class="alert-item-value warning">
                            <?php echo wp_kses_post(wc_price( $alert['new_cost'] )); ?>
                        </span>
                    </div>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Increase:</span>
                        <span class="alert-item-value warning">
                            +<?php echo number_format( $alert['increase_percentage'], 2 ); ?>%
                        </span>
                    </div>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Current Price:</span>
                        <span class="alert-item-value"><?php echo wp_kses_post(wc_price( $alert['price'] )); ?></span>
                    </div>
                    <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;">
                        <a href="<?php echo esc_url( get_edit_post_link( $alert['product_id'] ) ); ?>" class="btn btn-secondary" style="font-size: 13px; padding: 8px 15px;">
                            Edit Product →
                        </a>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <?php
            // ═════════════════════════════════════════════════════════
            // ANOMALY DETECTION ALERTS
            // ═════════════════════════════════════════════════════════
            if ( isset( $alerts_by_type['anomaly_detected'] ) ) :
                $alerts = $alerts_by_type['anomaly_detected'];
            ?>
            <div class="alert-section">
                <div class="alert-section-header info">
                    <h3>🔍 Anomaly Detection (<?php echo count( $alerts ); ?>)</h3>
                </div>

                <?php foreach ( $alerts as $alert ) : ?>
                <div class="alert-item info">
                    <p style="margin: 0 0 10px 0; font-weight: 600; color: #2271b1;">
                        Detected <?php echo count( $alert['anomalies'] ); ?> unusual profit patterns
                    </p>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Detection Period:</span>
                        <span class="alert-item-value"><?php echo esc_html( $alert['detection_period'] ); ?></span>
                    </div>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Average Profit:</span>
                        <span class="alert-item-value"><?php echo wp_kses_post(wc_price( $alert['profit_mean'] )); ?></span>
                    </div>
                    <div class="alert-item-detail">
                        <span class="alert-item-label">Average Margin:</span>
                        <span class="alert-item-value"><?php echo number_format( $alert['margin_mean'], 2 ); ?>%</span>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- ACTIONS -->
            <div class="actions-box">
                <h3>💡 Recommended Actions</h3>
                <ul>
                    <li>Investigate low-margin products for cost optimization</li>
                    <li>Consider adjusting prices to maintain target margins</li>
                    <li>
                        <a href="<?php echo esc_url($dashboard_url); ?>">
                            View detailed reports in dashboard →
                        </a>
                    </li>
                </ul>
            </div>
        </div>

        <div class="footer">
            <p>
                This is an automated digest from <strong><?php echo esc_html($site_name); ?></strong>
                NeuralCommerce Profit Analytics
            </p>
        </div>
    </div>
    </body>
    </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Get human-readable label for alert type
     */
    private static function get_alert_type_label($type) {
        $labels = array(
            'low_margin' => 'Low Margin Alerts',
            'margin_drop' => 'Margin Drop Alerts',
            'margindrop' => 'Margin Drop Alerts',
            'unprofitable_order' => 'Unprofitable Orders',
            'cost_increase' => 'Cost Increase Alerts',
            'anomaly_detected' => 'Anomaly Detections',
            'daily_summary' => 'Daily Summaries',
        );
        
        return isset($labels[$type]) ? $labels[$type] : ucwords(str_replace('_', ' ', $type));
    }


    /**
     * Send digest of queued alerts with FULL DETAILS
     */
    private static function send_queued_alerts_digest() {
        if ( empty( self::$alert_queue ) ) {
            return;
        }

        // Group alerts by type
        $alerts_by_type = array();
        foreach ( self::$alert_queue as $alert ) {
            $type = $alert['type'];
            if ( ! isset( $alerts_by_type[ $type ] ) ) {
                $alerts_by_type[ $type ] = array();
            }
            $alerts_by_type[ $type ][] = $alert['data'];
        }

        // Build detailed HTML email
        $to = get_option( 'qcnc_alert_email', get_option( 'admin_email' ) );

        // Add additional emails
        $additional_emails = get_option( 'qcnc_additional_alert_emails', '' );
        if ( ! is_array( $to ) ) {
            $to = array( $to );
        }
        if ( ! empty( $additional_emails ) ) {
            $to = array_merge( $to, array_map( 'trim', explode( ',', $additional_emails ) ) );
        }
        $to = array_unique( array_filter( $to ) );

        $subject = '[' . get_bloginfo( 'name' ) . '] 📊 Profit Alert Digest - ' . count( self::$alert_queue ) . ' alerts';

        $message = self::build_detailed_digest_email( $alerts_by_type );

        $headers = array( 'Content-Type: text/html; charset=UTF-8' );
        $sent = wp_mail( $to, $subject, $message, $headers );

        if ( $sent ) {
            // error_log( 'QCNC: Sent digest email with ' . count( self::$alert_queue ) . ' alerts to: ' . implode( ', ', $to ) );
        } else {
            // error_log( 'QCNC: Failed to send digest email' );
        }

        // Clear queue
        self::$alert_queue = array();
    }

    /**
     * Build detailed digest email HTML with ALL alert details
     * 
     * @param array $alerts_by_type Alerts grouped by type
     * @return string HTML email content
     */
    /*
    private static function build_detailed_digest_email( $alerts_by_type ) {
        $site_name = get_bloginfo( 'name' );
        $total_alerts = 0;
        foreach ( $alerts_by_type as $alerts ) {
            $total_alerts += count( $alerts );
        }

        $current_date = wp_date( 'F j, Y g:i A' );

        // ✅ FIX: Correct dashboard link
        $dashboard_url = admin_url( 'admin.php?page=qcnc-dashboard' );
        $settings_url = admin_url( 'admin.php?page=quarkcode-neuralcommerce-settings&tab=alerts' );

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body { 
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
                    line-height: 1.6; 
                    color: #333; 
                    margin: 0; 
                    padding: 0; 
                    background: #f5f5f5;
                }
                .email-wrapper { 
                    background: #f5f5f5; 
                    padding: 20px; 
                }
                .container { 
                    max-width: 600px; 
                    margin: 0 auto; 
                    background: white;
                    border-radius: 8px;
                    overflow: hidden;
                    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                }
                .header { 
                    background: linear-gradient(135deg, #2271b1 0%, #135e96 100%); 
                    color: white; 
                    padding: 30px 20px; 
                    text-align: center;
                }
                .header h1 { 
                    margin: 0; 
                    font-size: 24px; 
                    font-weight: 600;
                }
                .header p { 
                    margin: 10px 0 0 0; 
                    opacity: 0.9; 
                    font-size: 14px;
                }
                .content { 
                    padding: 30px 20px; 
                }
                .summary-box { 
                    background: #f9fafb; 
                    padding: 20px; 
                    margin: 0 0 25px 0; 
                    border-radius: 6px;
                    border-left: 4px solid #2271b1; 
                }
                .summary-box h2 { 
                    margin: 0 0 15px 0; 
                    color: #2271b1; 
                    font-size: 18px;
                    font-weight: 600;
                }
                .stat { 
                    font-size: 36px; 
                    font-weight: 700; 
                    color: #2271b1; 
                    margin: 0;
                }
                .stat-label { 
                    color: #666; 
                    font-size: 14px; 
                    margin: 5px 0 0 0;
                }
                .alert-section { 
                    margin: 30px 0; 
                }
                .alert-section-header { 
                    background: #f0f0f0; 
                    padding: 12px 15px; 
                    border-radius: 6px 6px 0 0;
                    border-left: 4px solid #d63638;
                    margin: 0 0 15px 0;
                }
                .alert-section-header.warning { border-left-color: #f0b849; }
                .alert-section-header.critical { border-left-color: #d63638; }
                .alert-section-header.info { border-left-color: #2271b1; }
                .alert-section-header h3 { 
                    margin: 0; 
                    font-size: 16px; 
                    color: #333;
                    font-weight: 600;
                }
                .alert-item { 
                    background: #fafafa; 
                    padding: 15px; 
                    margin: 0 0 12px 0; 
                    border-radius: 6px;
                    border-left: 3px solid #d63638;
                }
                .alert-item.warning { border-left-color: #f0b849; background: #fffbf0; }
                .alert-item.critical { border-left-color: #d63638; background: #fcf0f1; }
                .alert-item.info { border-left-color: #2271b1; background: #f0f6fc; }
                .alert-item-title { 
                    font-weight: 600; 
                    font-size: 15px; 
                    color: #1d2327; 
                    margin: 0 0 8px 0;
                }
                .alert-item-detail { 
                    display: flex; 
                    justify-content: space-between; 
                    padding: 5px 0;
                    font-size: 14px;
                }
                .alert-item-label { 
                    color: #666; 
                    font-weight: 500;
                }
                .alert-item-value { 
                    font-weight: 600; 
                    color: #1d2327;
                }
                .alert-item-value.negative { color: #d63638; }
                .alert-item-value.warning { color: #f0b849; }
                .alert-item-value.positive { color: #00a32a; }
                .divider { 
                    height: 1px; 
                    background: #e0e0e0; 
                    margin: 15px 0;
                }
                .actions-box { 
                    background: #f0f6fc; 
                    padding: 20px; 
                    border-radius: 6px; 
                    margin: 25px 0;
                    border-left: 4px solid #2271b1;
                }
                .actions-box h3 { 
                    margin: 0 0 15px 0; 
                    color: #2271b1; 
                    font-size: 16px;
                }
                .actions-box ul { 
                    margin: 0; 
                    padding: 0 0 0 20px;
                }
                .actions-box li { 
                    margin: 8px 0;
                    color: #333;
                }
                .btn { 
                    display: inline-block; 
                    padding: 12px 24px; 
                    background: #2271b1; 
                    color: white !important; 
                    text-decoration: none; 
                    border-radius: 5px;
                    font-weight: 600;
                    margin: 10px 5px 0 0;
                }
                .btn-secondary { 
                    background: #f0f0f0; 
                    color: #333 !important;
                }
                .footer { 
                    background: #fafafa; 
                    padding: 20px; 
                    text-align: center; 
                    font-size: 13px; 
                    color: #666;
                    border-top: 1px solid #e0e0e0;
                }
                .footer a { 
                    color: #2271b1; 
                    text-decoration: none;
                }
                @media only screen and (max-width: 600px) {
                    .email-wrapper { padding: 10px; }
                    .content { padding: 20px 15px; }
                    .alert-item-detail { flex-direction: column; }
                }
            </style>
            </head>
            <body>
            <div class="email-wrapper">
                <div class="container">
                    <div class="header">
                        <h1>📊 Profit Alert Digest</h1>
                        <p><?php echo esc_html( $current_date ); ?></p>
                    </div>

                    <div class="content">
                        <!-- SUMMARY -->
                        <div class="summary-box">
                            <h2>Summary</h2>
                            <p class="stat"><?php echo $total_alerts; ?></p>
                            <p class="stat-label">Total alerts triggered</p>

                            <div class="divider"></div>

                            <ul style="margin: 15px 0 0 0; padding-left: 20px;">
                                <?php foreach ( $alerts_by_type as $type => $alerts ) : ?>
                                    <li>
                                        <strong><?php echo count( $alerts ); ?></strong>
                                        <?php echo esc_html( self::get_alert_type_label( $type ) ); ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>

                        <?php
                        // ═════════════════════════════════════════════════════════
                        // UNPROFITABLE ORDERS
                        // ═════════════════════════════════════════════════════════
                        if ( isset( $alerts_by_type['unprofitable_order'] ) ) :
                            $alerts = $alerts_by_type['unprofitable_order'];
                        ?>
                        <div class="alert-section">
                            <div class="alert-section-header critical">
                                <h3>🚨 Unprofitable Orders (<?php echo count( $alerts ); ?>)</h3>
                            </div>

                            <?php foreach ( $alerts as $alert ) : 
                                $order = wc_get_order( $alert['order_id'] );
                                if ( ! $order ) continue;
                            ?>
                            <div class="alert-item critical">
                                <div class="alert-item-title">
                                    Order #<?php echo esc_html( $alert['order_number'] ); ?>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Revenue:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['revenue'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Cost:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['cost'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Net Profit:</span>
                                    <span class="alert-item-value negative">
                                        <?php echo wc_price( $alert['profit'] ); ?>
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Margin:</span>
                                    <span class="alert-item-value negative">
                                        <?php echo number_format( $alert['margin'], 2 ); ?>%
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Customer:</span>
                                    <span class="alert-item-value">
                                        <?php echo esc_html( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() ); ?>
                                    </span>
                                </div>
                                <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;">
                                    <a href="<?php echo esc_url( $order->get_edit_order_url() ); ?>" class="btn btn-secondary" style="font-size: 13px; padding: 8px 15px;">
                                        View Order →
                                    </a>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <?php
                        // ═════════════════════════════════════════════════════════
                        // LOW MARGIN ALERTS
                        // ═════════════════════════════════════════════════════════
                        if ( isset( $alerts_by_type['low_margin'] ) ) :
                            $alerts = $alerts_by_type['low_margin'];
                            $avg_margin = array_sum( array_column( $alerts, 'margin' ) ) / count( $alerts );
                        ?>
                        <div class="alert-section">
                            <div class="alert-section-header warning">
                                <h3>⚠️ Low Margin Alerts (<?php echo count( $alerts ); ?>)</h3>
                            </div>

                            <div style="background: #fffbf0; padding: 12px; margin-bottom: 15px; border-radius: 4px;">
                                <p style="margin: 0; color: #666; font-size: 14px;">
                                    Average margin across these orders: 
                                    <strong style="color: #f0b849; font-size: 16px;">
                                        <?php echo number_format( $avg_margin, 2 ); ?>%
                                    </strong>
                                </p>
                            </div>

                            <?php foreach ( $alerts as $alert ) : 
                                $order = wc_get_order( $alert['order_id'] );
                                if ( ! $order ) continue;
                            ?>
                            <div class="alert-item warning">
                                <div class="alert-item-title">
                                    Order #<?php echo esc_html( $alert['order_number'] ); ?>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Revenue:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['revenue'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Profit:</span>
                                    <span class="alert-item-value <?php echo $alert['profit'] < 0 ? 'negative' : ''; ?>">
                                        <?php echo wc_price( $alert['profit'] ); ?>
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Margin:</span>
                                    <span class="alert-item-value warning">
                                        <?php echo number_format( $alert['margin'], 2 ); ?>%
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Threshold:</span>
                                    <span class="alert-item-value">
                                        <?php echo number_format( $alert['threshold'], 2 ); ?>%
                                    </span>
                                </div>
                                <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;">
                                    <a href="<?php echo esc_url( $order->get_edit_order_url() ); ?>" class="btn btn-secondary" style="font-size: 13px; padding: 8px 15px;">
                                        View Order →
                                    </a>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <?php
                        // ═════════════════════════════════════════════════════════
                        // COST INCREASE ALERTS
                        // ═════════════════════════════════════════════════════════
                        if ( isset( $alerts_by_type['cost_increase'] ) ) :
                            $alerts = $alerts_by_type['cost_increase'];
                        ?>
                        <div class="alert-section">
                            <div class="alert-section-header warning">
                                <h3>📈 Cost Increase Alerts (<?php echo count( $alerts ); ?>)</h3>
                            </div>

                            <?php foreach ( $alerts as $alert ) : 
                                $product = wc_get_product( $alert['product_id'] );
                                if ( ! $product ) continue;
                            ?>
                            <div class="alert-item warning">
                                <div class="alert-item-title">
                                    <?php echo esc_html( $alert['product_name'] ); ?>
                                </div>
                                <?php if ( ! empty( $alert['sku'] ) ) : ?>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">SKU:</span>
                                    <span class="alert-item-value"><?php echo esc_html( $alert['sku'] ); ?></span>
                                </div>
                                <?php endif; ?>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Old Cost:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['old_cost'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">New Cost:</span>
                                    <span class="alert-item-value warning">
                                        <?php echo wc_price( $alert['new_cost'] ); ?>
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Increase:</span>
                                    <span class="alert-item-value warning">
                                        +<?php echo number_format( $alert['increase_percentage'], 2 ); ?>%
                                    </span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Current Price:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['price'] ); ?></span>
                                </div>
                                <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;">
                                    <a href="<?php echo esc_url( get_edit_post_link( $alert['product_id'] ) ); ?>" class="btn btn-secondary" style="font-size: 13px; padding: 8px 15px;">
                                        Edit Product →
                                    </a>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <?php
                        // ═════════════════════════════════════════════════════════
                        // ANOMALY DETECTION ALERTS
                        // ═════════════════════════════════════════════════════════
                        if ( isset( $alerts_by_type['anomaly_detected'] ) ) :
                            $alerts = $alerts_by_type['anomaly_detected'];
                        ?>
                        <div class="alert-section">
                            <div class="alert-section-header info">
                                <h3>🔍 Anomaly Detection (<?php echo count( $alerts ); ?>)</h3>
                            </div>

                            <?php foreach ( $alerts as $alert ) : ?>
                            <div class="alert-item info">
                                <p style="margin: 0 0 10px 0; font-weight: 600; color: #2271b1;">
                                    Detected <?php echo count( $alert['anomalies'] ); ?> unusual profit patterns
                                </p>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Detection Period:</span>
                                    <span class="alert-item-value"><?php echo esc_html( $alert['detection_period'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Average Profit:</span>
                                    <span class="alert-item-value"><?php echo wc_price( $alert['profit_mean'] ); ?></span>
                                </div>
                                <div class="alert-item-detail">
                                    <span class="alert-item-label">Average Margin:</span>
                                    <span class="alert-item-value"><?php echo number_format( $alert['margin_mean'], 2 ); ?>%</span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <!-- RECOMMENDED ACTIONS -->
                        <div class="actions-box">
                            <h3>💡 Recommended Actions</h3>
                            <ul>
                                <?php if ( isset( $alerts_by_type['unprofitable_order'] ) ) : ?>
                                    <li>Review unprofitable orders and adjust pricing strategy</li>
                                    <li>Check if shipping costs or fees are correctly calculated</li>
                                <?php endif; ?>
                                <?php if ( isset( $alerts_by_type['low_margin'] ) ) : ?>
                                    <li>Investigate low-margin products for cost optimization</li>
                                    <li>Consider adjusting prices to maintain target margins</li>
                                <?php endif; ?>
                                <?php if ( isset( $alerts_by_type['cost_increase'] ) ) : ?>
                                    <li>Update product prices to reflect new costs</li>
                                    <li>Review supplier agreements for better pricing</li>
                                <?php endif; ?>
                                <li>
                                    <a href="<?php echo esc_url( $dashboard_url ); ?>">
                                        View detailed reports in dashboard →
                                    </a>
                                </li>
                            </ul>
                        </div>

                        <!-- ACTION BUTTONS -->
                        <div style="text-align: center; margin: 25px 0;">
                            <a href="<?php echo esc_url( $dashboard_url ); ?>" class="btn">
                                View Dashboard
                            </a>
                            <a href="<?php echo esc_url( $settings_url ); ?>" class="btn btn-secondary">
                                Manage Alerts
                            </a>
                        </div>
                    </div>

                    <div class="footer">
                        <p style="margin: 0 0 10px 0;">
                            This is an automated digest from <strong><?php echo esc_html( $site_name ); ?></strong> 
                            NeuralCommerce Profit Analytics
                        </p>
                        <p style="margin: 0;">
                            <a href="<?php echo esc_url( $settings_url ); ?>">
                                Manage alert settings
                            </a>
                        </p>
                    </div>
                </div>
            </div>
            </body>
            </html>
        <?php
        return ob_get_clean();
    }
        */

    /**
     * Get human-readable label for alert type
     * 
     * @param string $type Alert type constant
     * @return string Label
     */
    /*
    private static function get_alert_type_label( $type ) {
        $labels = array(
            'low_margin' => 'Low Margin Alerts',
            'unprofitable_order' => 'Unprofitable Orders',
            'cost_increase' => 'Cost Increase Alerts',
            'anomaly_detected' => 'Anomaly Detections',
            'margin_drop' => 'Margin Drop Alerts',
            'daily_summary' => 'Daily Summaries',
        );

        return isset( $labels[ $type ] ) ? $labels[ $type ] : ucwords( str_replace( '_', ' ', $type ) );
    }
    */

    /**
     * Trigger an alert with intelligent batching
     *
     * @param string $alert_type Alert type
     * @param array $data Alert data
     */
    private function trigger_alert( $alert_type, $data ) {
        // Check if alert type is enabled
        $enabled_alerts = get_option( 'qcnc_enabled_alerts', array(
            self::ALERT_MARGIN_DROP,
            self::ALERT_COST_INCREASE,
            self::ALERT_UNPROFITABLE,
            self::ALERT_ANOMALY,
            self::ALERT_DAILY_SUMMARY,
        ) );

        if ( ! in_array( $alert_type, $enabled_alerts ) ) {
            return;
        }

        // Save to database (always save for records)
        $this->save_alert( $alert_type, $data );

        // ✅ NEW: Get digest mode setting
        $digest_mode = get_option( 'qcnc_alert_digest_mode', 'hourly' );

        // ✅ NEW: Always queue alerts for digest (except daily summary)
        if ( $alert_type !== self::ALERT_DAILY_SUMMARY && $digest_mode !== 'immediate' ) {
            // error_log( "QCNC: Alert queued for {$digest_mode} digest: {$alert_type}" );
            $this->queue_alert_for_digest( $alert_type, $data );
            return;
        }

        // Only send immediately if digest mode is "immediate"
        if ( $digest_mode === 'immediate' ) {
            // Check rate limit
            $rate_limit_key = 'qcnc_alert_rate_limit';
            $sent_count = get_transient( $rate_limit_key );
            $max_per_hour = get_option( 'qcnc_max_alerts_per_hour', 10 );

            if ( $sent_count && $sent_count >= $max_per_hour ) {
                // error_log( "QCNC: Alert skipped - rate limit reached ({$sent_count}/{$max_per_hour} per hour)" );
                $this->queue_alert_for_digest( $alert_type, $data );
                return;
            }

            // Send email notification
            $this->send_email_alert( $alert_type, $data );

            // Increment rate limit counter
            if ( ! $sent_count ) {
                set_transient( $rate_limit_key, 1, HOUR_IN_SECONDS );
            } else {
                set_transient( $rate_limit_key, $sent_count + 1, HOUR_IN_SECONDS );
            }

            // error_log( "QCNC: Alert sent. Count: " . ( $sent_count + 1 ) . "/{$max_per_hour}" );
        }

        // Fire action hook for external integrations
        do_action( 'qcnc_alert_triggered', $alert_type, $data );
    }

    /**
     * Queue alert for digest email
     */
    private function queue_alert_for_digest( $alert_type, $data ) {
        $queue = get_option( 'qcnc_alert_digest_queue', array() );

        $queue[] = array(
            'type' => $alert_type,
            'data' => $data,
            'timestamp' => current_time( 'timestamp' ),
        );

        update_option( 'qcnc_alert_digest_queue', $queue, false );
    }

    /**
     * Send digest email with all queued alerts
     */
    public function send_digest_email() {
        $queue = get_option('qcnc_alert_digest_queue', array());
        
        if (empty($queue)) {
            // error_log('QCNC: No alerts in queue for digest');
            return false;
        }

        // Group alerts by type
        $alerts_by_type = array();
        foreach ($queue as $alert) {
            $type = $alert['type'];
            if (!isset($alerts_by_type[$type])) {
                $alerts_by_type[$type] = array();
            }
            $alerts_by_type[$type][] = $alert['data'];
        }

        $to = get_option('qcnc_alert_email', get_option('admin_email'));
        
        // Add additional emails
        $additional_emails = get_option('qcnc_additional_alert_emails', '');
        if (!is_array($to)) {
            $to = array($to);
        }
        if (!empty($additional_emails)) {
            $to = array_merge($to, array_map('trim', explode(',', $additional_emails)));
        }
        $to = array_unique(array_filter($to));
        
        $subject = '[' . get_bloginfo('name') . '] 📊 Profit Alert Digest - ' . count($queue) . ' alerts';
        
        // ✅ USE ENHANCED VERSION WITH DETAILS
        $message = self::build_detailed_digest_email($alerts_by_type);
        
        $headers = array('Content-Type: text/html; charset=UTF-8');
        $sent = wp_mail($to, $subject, $message, $headers);
        
        if ($sent) {
            // error_log('QCNC: Digest email sent with ' . count($queue) . ' alerts to: ' . implode(', ', $to));
            // Clear the queue after successful send
            delete_option('qcnc_alert_digest_queue');
        } else {
            // error_log('QCNC: Failed to send digest email');
        }
        
        return $sent;
    }


    /**
     * Build digest email HTML
     */
    /*
    private function build_digest_email( $queue ) {
        $alert_counts = array();
        $critical_alerts = array();
        $margin_alerts = array();
        $cost_alerts = array();

        // Categorize alerts
        foreach ( $queue as $alert ) {
            $type = $alert['type'];

            if ( ! isset( $alert_counts[ $type ] ) ) {
                $alert_counts[ $type ] = 0;
            }
            $alert_counts[ $type ]++;

            // Collect critical alerts for detail
            if ( $type === self::ALERT_UNPROFITABLE ) {
                $critical_alerts[] = $alert;
            } elseif ( $type === self::ALERT_MARGIN_DROP ) {
                $margin_alerts[] = $alert;
            } elseif ( $type === self::ALERT_COST_INCREASE ) {
                $cost_alerts[] = $alert;
            }
        }

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #2271b1; color: white; padding: 20px; border-radius: 5px 5px 0 0; }
                .content { background: #f9f9f9; padding: 20px; }
                .summary-box { background: white; padding: 15px; margin: 15px 0; border-left: 4px solid #2271b1; }
                .critical-box { border-left-color: #dc3232; }
                .warning-box { border-left-color: #f0b849; }
                .alert-item { background: white; padding: 10px; margin: 10px 0; border-radius: 3px; }
                .footer { background: #f0f0f0; padding: 15px; text-align: center; font-size: 12px; color: #666; }
                h2 { margin-top: 0; color: #2271b1; }
                .stat { font-size: 24px; font-weight: bold; color: #2271b1; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1 style="margin: 0;">📊 Profit Alert Digest</h1>
                    <p style="margin: 5px 0 0 0;">
                        <?php echo wp_date( 'F j, Y g:i A' ); ?>
                    </p>
                </div>

                <div class="content">
                    <div class="summary-box">
                        <h2>Summary</h2>
                        <p class="stat"><?php echo count( $queue ); ?></p>
                        <p>Total alerts triggered</p>

                        <ul style="margin: 15px 0; padding-left: 20px;">
                            <?php foreach ( $alert_counts as $type => $count ) : ?>
                                <li>
                                    <strong><?php echo $count; ?></strong> 
                                    <?php echo $this->get_alert_type_label( $type ); ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>

                    <?php if ( ! empty( $critical_alerts ) ) : ?>
                        <div class="summary-box critical-box">
                            <h2>🚨 Critical: Unprofitable Orders</h2>
                            <?php foreach ( array_slice( $critical_alerts, 0, 5 ) as $alert ) : ?>
                                <div class="alert-item">
                                    <strong>Order #<?php echo esc_html( $alert['data']['order_number'] ?? $alert['data']['order_id'] ); ?></strong><br>
                                    Margin: <span style="color: #dc3232;"><?php echo number_format( $alert['data']['margin'] ?? 0, 2 ); ?>%</span><br>
                                    Profit: <?php echo wc_price( $alert['data']['profit'] ?? 0 ); ?>
                                </div>
                            <?php endforeach; ?>

                            <?php if ( count( $critical_alerts ) > 5 ) : ?>
                                <p><em>+ <?php echo count( $critical_alerts ) - 5; ?> more unprofitable orders</em></p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <?php if ( ! empty( $margin_alerts ) ) : ?>
                        <div class="summary-box warning-box">
                            <h2>⚠️ Low Margin Alerts</h2>
                            <p><?php echo count( $margin_alerts ); ?> orders with margins below threshold</p>

                            <?php 
                            $avg_margin = 0;
                            foreach ( $margin_alerts as $alert ) {
                                $avg_margin += $alert['data']['margin'] ?? 0;
                            }
                            $avg_margin = $avg_margin / count( $margin_alerts );
                            ?>
                            <p>Average margin: <strong><?php echo number_format( $avg_margin, 2 ); ?>%</strong></p>
                        </div>
                    <?php endif; ?>

                    <?php if ( ! empty( $cost_alerts ) ) : ?>
                        <div class="summary-box warning-box">
                            <h2>📈 Cost Increase Alerts</h2>
                            <?php foreach ( array_slice( $cost_alerts, 0, 5 ) as $alert ) : ?>
                                <div class="alert-item">
                                    <strong><?php echo esc_html( $alert['data']['product_name'] ?? 'Product' ); ?></strong><br>
                                    Cost increased: <span style="color: #f0b849;">
                                        <?php echo number_format( $alert['data']['increase_percentage'] ?? 0, 1 ); ?>%
                                    </span><br>
                                    New cost: <?php echo wc_price( $alert['data']['new_cost'] ?? 0 ); ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <div class="summary-box">
                        <h2>💡 Recommended Actions</h2>
                        <ul>
                            <?php if ( ! empty( $critical_alerts ) ) : ?>
                                <li>Review unprofitable orders and adjust pricing or costs</li>
                            <?php endif; ?>
                            <?php if ( ! empty( $margin_alerts ) ) : ?>
                                <li>Investigate low-margin products for cost optimization</li>
                            <?php endif; ?>
                            <?php if ( ! empty( $cost_alerts ) ) : ?>
                                <li>Update product prices to maintain target margins</li>
                            <?php endif; ?>
                            <li>
                                <a href="<?php echo admin_url( 'admin.php?page=quarkcode-neuralcommerce-lite' ); ?>">
                                    View detailed reports in dashboard
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>

                <div class="footer">
                    <p>
                        This is an automated digest from NeuralCommerce Profit Analytics<br>
                        <a href="<?php echo admin_url( 'admin.php?page=quarkcode-neuralcommerce-settings&tab=alerts' ); ?>">
                            Manage alert settings
                        </a>
                    </p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
        */

    /**
     * Get human-readable label for alert type
     */
    /*
    private function get_alert_type_label( $type ) {
        $labels = array(
            self::ALERT_MARGIN_DROP => 'Low Margin Alerts',
            self::ALERT_COST_INCREASE => 'Cost Increase Alerts',
            self::ALERT_UNPROFITABLE => 'Unprofitable Orders',
            self::ALERT_ANOMALY => 'Anomaly Detections',s
            self::ALERT_DAILY_SUMMARY => 'Daily Summaries',
        );

        return $labels[ $type ] ?? $type;
    }
    */


    /**
     * Save alert to database
     *
     * @param string $alert_type Alert type
     * @param array $data Alert data
     */
    private function save_alert( $alert_type, $data ) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'qcnc_margin_alerts';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->insert(
            $table_name,
            array(
                'alert_type' => $alert_type,
                'alert_data' => wp_json_encode( $data ),
                'status' => 'unread',
                'created_at' => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%s', '%s' )
        );
    }

    /**
     * Send email alert
     *
     * @param string $alert_type Alert type
     * @param array $data Alert data
     */
    private function send_email_alert( $alert_type, $data ) {
        // error_log( "=== SENDING EMAIL ALERT ===" );
        // error_log( "Alert Type: " . $alert_type );
        $to = get_option( 'qcnc_alert_email', get_option( 'admin_email' ) );
        $additional_emails = get_option( 'qcnc_additional_alert_emails', '' );

        if ( ! is_array( $to ) ) {
            $to = array( $to );
        }

        if ( ! empty( $additional_emails ) ) {
            $to = array_merge( 
                $to, 
                array_map( 'trim', explode( ',', $additional_emails ) ) 
            );
        }
        $to = array_unique( array_filter( $to ) );

        $subject = $this->get_alert_subject( $alert_type, $data );
        $message = $this->get_alert_message( $alert_type, $data );

        // Use HTML email
        $set_html_content_type = function() { return 'text/html'; };
        add_filter( 'wp_mail_content_type', $set_html_content_type );

        $headers = array(
            'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
        );

        $sent = wp_mail( $to, $subject, $message, $headers );
        // error_log( "Email sent result: " . ( $sent ? 'TRUE' : 'FALSE' ) );
        if ( ! $sent ) {
           // error_log( "Email failed for: " . print_r( $to, true ) );
        }

        // Reset content type
        remove_filter( 'wp_mail_content_type', $set_html_content_type );
    }

    /**
     * Get alert subject
     */
    private function get_alert_subject( $alert_type, $data ) {
        $site_name = get_bloginfo( 'name' );

        switch ( $alert_type ) {
            case self::ALERT_MARGIN_DROP:
                return "[{$site_name}] ⚠️ Low Profit Margin Alert";

            case self::ALERT_COST_INCREASE:
                return "[{$site_name}] 📈 Product Cost Increase Alert";

            case self::ALERT_UNPROFITABLE:
                return "[{$site_name}] 🚨 Unprofitable Order Alert";

            case self::ALERT_ANOMALY:
                return "[{$site_name}] 🔍 Profit Anomaly Detected";

            case self::ALERT_DAILY_SUMMARY:
                return "[{$site_name}] 📊 Daily Profit Summary - " . wp_date( 'M d, Y', strtotime( '-1 day' ) );

            default:
                return "[{$site_name}] Profit Alert";
        }
    }

    /**
     * Get alert message
     */
    private function get_alert_message( $alert_type, $data ) {
        ob_start();

        $template_path = QCNC_TEMPLATES_DIR . "email/alert-{$alert_type}.php";

        // Use specific template if exists, otherwise use generic
        if ( file_exists( $template_path ) ) {
            include $template_path;
        } else {
            include QCNC_TEMPLATES_DIR . 'email/generic-alert.php';
        }

        return ob_get_clean();
    }

    /**
     * Get SMS message (short version)
     */
    private function get_sms_message( $alert_type, $data ) {
        switch ( $alert_type ) {
            case self::ALERT_MARGIN_DROP:
                return "Low margin alert: Order #{$data['order_number']} has {$data['margin']}% margin (threshold: {$data['threshold']}%)";

            case self::ALERT_COST_INCREASE:
                return "Cost increased: {$data['product_name']} cost rose {$data['increase_percentage']}% to " . wc_price( $data['new_cost'] );

            case self::ALERT_UNPROFITABLE:
                return "Unprofitable order: #{$data['order_number']} margin: {$data['margin']}%";

            case self::ALERT_ANOMALY:
                return "Anomaly detected: " . count( $data['anomalies'] ) . " unusual profit patterns in last 7 days";

            default:
                return "Profit alert triggered. Check your email for details.";
        }
    }

    /**
     * Calculate standard deviation
     */
    private function calculate_std_deviation( $values, $mean ) {
        $variance = 0;
        foreach ( $values as $value ) {
            $variance += pow( $value - $mean, 2 );
        }
        $variance /= count( $values );
        return sqrt( $variance );
    }

    /**
     * Calculate order profit
     */
    private function calculate_order_profit( $order_id ) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $profit_data = $wpdb->get_row( $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM {$table_name} WHERE order_id = %d",
            $order_id
        ), ARRAY_A );

        if ( $profit_data ) {
            return $profit_data;
        }

        // Calculate if not exists
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return array(
                'net_profit' => 0,
                'margin_pct' => 0,
                'revenue' => 0,
                'cogs_total' => 0,
            );
        }

        $revenue = $order->get_total();
        $cost = 0;

        foreach ( $order->get_items() as $item ) {
            $product_id = $item->get_product_id();
            $quantity = $item->get_quantity();
            $product_cost = get_post_meta( $product_id, '_qcnc_product_cost', true );

            if ( $product_cost ) {
                $cost += floatval( $product_cost ) * $quantity;
            }
        }

        $profit = $revenue - $cost;
        $margin_pct = $revenue > 0 ? ( $profit / $revenue ) * 100 : 0;

        return array(
            'revenue' => $revenue,
            'cogs_total' => $cost,
            'net_profit' => $profit,
            'margin_pct' => $margin_pct,
        );
    }

    /**
     * Add alerts settings tab
     */
    public function add_alerts_settings_tab( $tabs ) {
        $tabs['alerts'] = __( 'Profit Alerts', 'quarkcode-neuralcommerce-lite' );
        return $tabs;
    }

    /**
     * Render alerts settings
     */
    public function render_alerts_settings() {
        ?>
        <h2><?php esc_html_e( 'Automated Profit Alerts Settings', 'quarkcode-neuralcommerce-lite' ); ?></h2>
        <p><?php esc_html_e( 'Configure automated alerts to catch profit issues before they cost thousands.', 'quarkcode-neuralcommerce-lite' ); ?></p>

        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e( 'Alert Email Recipients', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <input type="email" name="qcnc_alert_email" 
                           value="<?php echo esc_attr( get_option( 'qcnc_alert_email', get_option( 'admin_email' ) ) ); ?>" 
                           class="regular-text" />
                    <p class="description"><?php esc_html_e( 'Primary email for alerts', 'quarkcode-neuralcommerce-lite' ); ?></p>

                    <textarea name="qcnc_additional_alert_emails" 
                              rows="3" 
                              class="large-text"
                              placeholder="email1@example.com, email2@example.com"><?php 
                        echo esc_textarea( get_option( 'qcnc_additional_alert_emails', '' ) ); 
                    ?></textarea>
                    <p class="description"><?php esc_html_e( 'Additional emails (comma-separated)', 'quarkcode-neuralcommerce-lite' ); ?></p>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Margin Threshold', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <input type="number" name="qcnc_margin_threshold" 
                           value="<?php echo esc_attr( get_option( 'qcnc_margin_threshold', 20 ) ); ?>" 
                           min="0" max="100" step="0.1" />
                    <span>%</span>
                    <p class="description"><?php esc_html_e( 'Alert when order margin falls below this percentage', 'quarkcode-neuralcommerce-lite' ); ?></p>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Cost Increase Threshold', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <input type="number" name="qcnc_cost_increase_threshold" 
                           value="<?php echo esc_attr( get_option( 'qcnc_cost_increase_threshold', 10 ) ); ?>" 
                           min="0" max="100" step="0.1" />
                    <span>%</span>
                    <p class="description"><?php esc_html_e( 'Alert when product cost increases by this percentage', 'quarkcode-neuralcommerce-lite' ); ?></p>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Margin Drop Threshold', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <input type="number" name="qcnc_margin_drop_threshold" 
                           value="<?php echo esc_attr( get_option( 'qcnc_margin_drop_threshold', 30 ) ); ?>" 
                           min="0" max="100" step="0.1" />
                    <span>%</span>
                    <p class="description"><?php esc_html_e( 'Alert when margin drops by this percentage compared to 30-day average', 'quarkcode-neuralcommerce-lite' ); ?></p>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Enabled Alerts', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <?php
                    $enabled_alerts = get_option( 'qcnc_enabled_alerts', array(
                        'margin_drop', 'cost_increase', 'unprofitable_order', 'anomaly_detected', 'daily_summary'
                    ) );
                    ?>
                    <label>
                        <input type="checkbox" name="qcnc_enabled_alerts[]" value="margin_drop" 
                               <?php checked( in_array( 'margin_drop', $enabled_alerts ) ); ?> />
                        <?php esc_html_e( 'Low Margin Alerts', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label><br/>

                    <label>
                        <input type="checkbox" name="qcnc_enabled_alerts[]" value="cost_increase" 
                               <?php checked( in_array( 'cost_increase', $enabled_alerts ) ); ?> />
                        <?php esc_html_e( 'Cost Increase Alerts', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label><br/>

                    <label>
                        <input type="checkbox" name="qcnc_enabled_alerts[]" value="unprofitable_order" 
                               <?php checked( in_array( 'unprofitable_order', $enabled_alerts ) ); ?> />
                        <?php esc_html_e( 'Unprofitable Order Alerts', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label><br/>

                    <label>
                        <input type="checkbox" name="qcnc_enabled_alerts[]" value="anomaly_detected" 
                               <?php checked( in_array( 'anomaly_detected', $enabled_alerts ) ); ?> />
                        <?php esc_html_e( 'Anomaly Detection Alerts', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label><br/>

                    <label>
                        <input type="checkbox" name="qcnc_enabled_alerts[]" value="daily_summary" 
                               <?php checked( in_array( 'daily_summary', $enabled_alerts ) ); ?> />
                        <?php esc_html_e( 'Daily Profit Summary', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Rate Limiting', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <input type="number" name="qcnc_max_alerts_per_hour" 
                           value="<?php echo esc_attr( get_option( 'qcnc_max_alerts_per_hour', 10 ) ); ?>" 
                           min="1" max="100" step="1" />
                    <span><?php esc_html_e( 'emails per hour', 'quarkcode-neuralcommerce-lite' ); ?></span>
                    <p class="description">
                        <?php esc_html_e( 'Maximum alert emails per hour to prevent spam', 'quarkcode-neuralcommerce-lite' ); ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th scope="row"><?php esc_html_e( 'Alert Delivery Mode', 'quarkcode-neuralcommerce-lite' ); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="qcnc_alert_digest_mode" value="1" 
                               <?php checked( get_option( 'qcnc_alert_digest_mode', false ) ); ?> />
                        <?php esc_html_e( 'Send daily digest instead of immediate alerts', 'quarkcode-neuralcommerce-lite' ); ?>
                    </label>
                    <p class="description">
                        <?php esc_html_e( 'Batch all alerts into a single daily email (recommended to prevent spam)', 'quarkcode-neuralcommerce-lite' ); ?>
                    </p>
                </td>
            </tr>

        </table>
        <?php
    }
}