<?php
/**
 * Cost Rules Engine
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Core;

use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Manages dynamic cost rules and automation.
 */
class QCNC_Cost_Rules_Engine {

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    private $cost_manager;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Product_Cost_Manager $cost_manager Cost manager.
     * @param QCNC_Logger               $logger       Logger.
     */
    public function __construct( QCNC_Product_Cost_Manager $cost_manager, QCNC_Logger $logger ) {
        $this->cost_manager = $cost_manager;
        $this->logger       = $logger;
    }

    /**
     * Create a new cost rule.
     *
     * @param array $rule_data Rule data.
     * @return int|false Rule ID or false on failure.
     */
    public function create_rule( $rule_data ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_cost_rules';

        $defaults = [
            'rule_name'        => '',
            'rule_type'        => 'category',
            'target_ids'       => null,
            'adjustment_type'  => 'percentage',
            'adjustment_value' => 0.0,
            'cost_component'   => 'base',
            'is_active'        => 1,
            'priority'         => 10,
            'created_at'       => current_time( 'mysql' ),
        ];

        $rule_data = wp_parse_args( $rule_data, $defaults );

        // Encode target_ids as JSON.
        if ( is_array( $rule_data['target_ids'] ) ) {
            $rule_data['target_ids'] = wp_json_encode( $rule_data['target_ids'] );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert( $table, $rule_data );

        if ( $result ) {
            $rule_id = $wpdb->insert_id;
            $this->logger->log( "Cost rule created: {$rule_data['rule_name']} (ID: {$rule_id})" );
            return $rule_id;
        }

        return false;
    }

    /**
     * Update a cost rule.
     *
     * @param int   $rule_id   Rule ID.
     * @param array $rule_data Updated rule data.
     * @return bool Success status.
     */
    public function update_rule( $rule_id, $rule_data ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_cost_rules';

        $rule_data['updated_at'] = current_time( 'mysql' );

        // Encode target_ids as JSON.
        if ( isset( $rule_data['target_ids'] ) && is_array( $rule_data['target_ids'] ) ) {
            $rule_data['target_ids'] = wp_json_encode( $rule_data['target_ids'] );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->update(
            $table,
            $rule_data,
            [ 'id' => absint( $rule_id ) ],
            null,
            [ '%d' ]
        );

        return false !== $result;
    }

    /**
     * Delete a cost rule.
     *
     * @param int $rule_id Rule ID.
     * @return bool Success status.
     */
    public function delete_rule( $rule_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_cost_rules';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->delete( $table, [ 'id' => absint( $rule_id ) ], [ '%d' ] );

        if ( $result ) {
            $this->logger->log( "Cost rule deleted: ID {$rule_id}" );
            return true;
        }

        return false;
    }

    /**
     * Get all active rules.
     *
     * @return array Active rules.
     */
    public function get_active_rules() {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_cost_rules';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $rules = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT * FROM {$table} WHERE is_active = 1 ORDER BY priority ASC",
            ARRAY_A
        );

        // Decode JSON target_ids.
        foreach ( $rules as &$rule ) {
            if ( ! empty( $rule['target_ids'] ) ) {
                $rule['target_ids'] = json_decode( $rule['target_ids'], true );
            }
        }

        return $rules;
    }

    /**
     * Apply rules to a product.
     *
     * @param int $product_id Product ID.
     * @return bool Success status.
     */
    public function apply_rules_to_product( $product_id ) {
        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            return false;
        }

        $rules = $this->get_active_rules();
        $applicable_rules = [];

        foreach ( $rules as $rule ) {
            if ( $this->is_rule_applicable_to_product( $rule, $product ) ) {
                $applicable_rules[] = $rule;
            }
        }

        if ( empty( $applicable_rules ) ) {
            return false;
        }

        // Get current cost.
        $current_cost_data = $this->cost_manager->get_product_cost( $product_id );

        if ( ! $current_cost_data ) {
            $this->logger->log( "No current cost found for product #{$product_id}, skipping rule application" );
            return false;
        }

        $base_cost      = floatval( $current_cost_data['cost'] );
        $packaging_cost = floatval( $current_cost_data['packaging_cost'] );
        $handling_cost  = floatval( $current_cost_data['handling_cost'] );

        // Apply rules.
        foreach ( $applicable_rules as $rule ) {
            switch ( $rule['cost_component'] ) {
                case 'base':
                    $base_cost = $this->apply_adjustment( $base_cost, $rule );
                    break;
                case 'packaging':
                    $packaging_cost = $this->apply_adjustment( $packaging_cost, $rule );
                    break;
                case 'handling':
                    $handling_cost = $this->apply_adjustment( $handling_cost, $rule );
                    break;
            }
        }

        // Update cost if changed.
        if ( $base_cost != $current_cost_data['cost'] || 
             $packaging_cost != $current_cost_data['packaging_cost'] || 
             $handling_cost != $current_cost_data['handling_cost'] ) {
            
            $this->cost_manager->set_product_cost(
                $product_id,
                $base_cost,
                null,
                $packaging_cost,
                $handling_cost,
                $current_cost_data['currency']
            );

            $this->logger->log( "Rules applied to product #{$product_id}" );
            return true;
        }

        return false;
    }

    /**
     * Check if rule is applicable to product.
     *
     * @param array       $rule    Rule data.
     * @param \WC_Product $product Product object.
     * @return bool Applicable status.
     */
    private function is_rule_applicable_to_product( $rule, $product ) {
        switch ( $rule['rule_type'] ) {
            case 'global':
                return true;

            case 'category':
                if ( empty( $rule['target_ids'] ) ) {
                    return false;
                }
                $product_categories = $product->get_category_ids();
                return ! empty( array_intersect( $rule['target_ids'], $product_categories ) );

            case 'tag':
                if ( empty( $rule['target_ids'] ) ) {
                    return false;
                }
                $product_tags = $product->get_tag_ids();
                return ! empty( array_intersect( $rule['target_ids'], $product_tags ) );

            default:
                return false;
        }
    }

    /**
     * Apply cost adjustment based on rule.
     *
     * @param float $current_value Current cost value.
     * @param array $rule          Rule data.
     * @return float Adjusted value.
     */
    private function apply_adjustment( $current_value, $rule ) {
        $adjustment = floatval( $rule['adjustment_value'] );

        switch ( $rule['adjustment_type'] ) {
            case 'percentage':
                return $current_value * ( 1 + ( $adjustment / 100 ) );

            case 'fixed':
                return $current_value + $adjustment;

            default:
                return $current_value;
        }
    }

    /**
     * Apply rules to all products in bulk.
     *
     * @param int $batch_size Batch size.
     * @return array Results.
     */
    public function apply_rules_to_all_products( $batch_size = 50 ) {
        $args = [
            'post_type'      => 'product',
            'post_status'    => 'publish',
            'posts_per_page' => $batch_size,
            'fields'         => 'ids',
        ];

        $product_ids = get_posts( $args );
        $results = [
            'processed' => 0,
            'updated'   => 0,
        ];

        foreach ( $product_ids as $product_id ) {
            $results['processed']++;
            if ( $this->apply_rules_to_product( $product_id ) ) {
                $results['updated']++;
            }
        }

        return $results;
    }
}
