<?php
/**
 * Core Plugin Class
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce;

use QuarkcodeNeuralCommerce\Admin\QCNC_Admin as Admin;
use QuarkcodeNeuralCommerce\Admin\QCNC_Admin_Ajax;
use QuarkcodeNeuralCommerce\Core\QCNC_Product_Cost_Manager;
use QuarkcodeNeuralCommerce\Core\QCNC_Order_Profit_Calculator;
use QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System;
use QuarkcodeNeuralCommerce\Core\QCNC_Cost_Rules_Engine;
use QuarkcodeNeuralCommerce\Core\QCNC_Report_Generator;
use QuarkcodeNeuralCommerce\Integrations\QCNC_Payment_Gateway_Integration;
use QuarkcodeNeuralCommerce\Integrations\QCNC_Shipping_Integration;
use QuarkcodeNeuralCommerce\API\QCNC_Products_Api;
use QuarkcodeNeuralCommerce\API\QCNC_Orders_Api;
use QuarkcodeNeuralCommerce\API\QCNC_Reports_Api;
use QuarkcodeNeuralCommerce\Database\QCNC_Database_Schema;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Cache_Manager;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;
use QuarkcodeNeuralCommerce\Cron\QCNC_Cron_Jobs;
use QuarkcodeNeuralCommerce\Admin\QCNC_Alert_Test_Handler; 

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Main plugin core class (Singleton).
 */
class QCNC_Core {

    /**
     * Single instance of the class.
     *
     * @var QCNC_Core
     */
    private static $instance = null;

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    public $product_cost_manager;

    /**
     * Order profit calculator instance.
     *
     * @var QCNC_Order_Profit_Calculator
     */
    public $order_profit_calculator;

    /**
     * Margin alert system instance.
     *
     * @var QCNC_Margin_Alert_System
     */
    public $margin_alert_system;

    /**
     * Cost rules engine instance.
     *
     * @var QCNC_Cost_Rules_Engine
     */
    public $cost_rules_engine;

    /**
     * Report generator instance.
     *
     * @var QCNC_Report_Generator
     */
    public $report_generator;

    /**
     * Cache manager instance.
     *
     * @var QCNC_Cache_Manager
     */
    public $cache_manager;

    /**
     * Security utility instance.
     *
     * @var QCNC_Security
     */
    public $security;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    public $logger;

    /**
     * Alert test handler instance.
     *
     * @var QCNC_Alert_Test_Handler
     */
    public $alert_test_handler;

    /**
     * Get single instance.
     *
     * @return QCNC_Core
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        $this->init_utilities();
        $this->init_core_components();
        $this->init_integrations();
        $this->init_admin();
        $this->init_api();
        $this->init_cron();
        $this->register_hooks();
        $this->init_alert_test_handler();
    }

    /**
     * Initialize utility classes.
     */
    private function init_utilities() {
        $this->cache_manager = new QCNC_Cache_Manager();
        $this->security      = new QCNC_Security();
        $this->logger        = new QCNC_Logger();
    }

    /**
     * Initialize core components.
     */
    private function init_core_components() {
        $this->product_cost_manager    = new QCNC_Product_Cost_Manager( $this->cache_manager, $this->logger );
        $this->cost_rules_engine       = new QCNC_Cost_Rules_Engine( $this->product_cost_manager, $this->logger );
        $this->order_profit_calculator = new QCNC_Order_Profit_Calculator(
            $this->product_cost_manager,
            $this->cache_manager,
            $this->logger
        );
        $this->margin_alert_system     = new QCNC_Margin_Alert_System;
        $this->report_generator        = new QCNC_Report_Generator( $this->cache_manager, $this->logger );
    }

    /**
     * Initialize integrations.
     */
    private function init_integrations() {
        new QCNC_Payment_Gateway_Integration( $this->logger );
        new QCNC_Shipping_Integration( $this->logger );
    }


    /**
     * Initialize admin area.
     */
    private function init_admin() {
        if ( is_admin() ) {
            new Admin(
                $this->product_cost_manager,
                $this->order_profit_calculator,
                $this->report_generator,
                $this->security
            );
            new QCNC_Admin_Ajax(
                $this->product_cost_manager,
                $this->order_profit_calculator,
                $this->security,
                $this->logger
            );
        }
    }

    /**
     * Initialize REST API endpoints.
     */
    private function init_api() {
        add_action( 'rest_api_init', [ $this, 'register_api_routes' ] );
    }

    /**
     * Register REST API routes.
     */
    /*
    public function register_api_routes() {
        $products_api = new QCNC_Products_Api( $this->product_cost_manager, $this->security );
        $products_api->init();
        
        $orders_api = new QCNC_Orders_Api( $this->order_profit_calculator, $this->security );
        $orders_api->init();
        
        $reports_api = new QCNC_Reports_Api( $this->report_generator, $this->security );
        $reports_api->init();
    }
        */

    /**
     * Register REST API routes.
     */
    public function register_api_routes() {
        $products_api = new QCNC_Products_Api( $this->product_cost_manager, $this->security );
        $products_api->init();
        
        $orders_api = new QCNC_Orders_Api( $this->order_profit_calculator, $this->security );
        $orders_api->init();
        
        $reports_api = new QCNC_Reports_Api( $this->report_generator, $this->security );
        $reports_api->init();
        
        // ✅ DEBUG: Log that API is registered (remove after testing)
        // error_log( 'QCNC: Advanced Reports API registered' );
    }



    /**
     * Initialize cron jobs.
     */
    private function init_cron() {
        new QCNC_Cron_Jobs(
            $this->order_profit_calculator,
            $this->margin_alert_system,
            $this->cache_manager,
            $this->logger
        );
    }


    /**
     * Register WordPress hooks.
     */
    private function register_hooks() {
        // WooCommerce order hooks.
        add_action( 'woocommerce_order_status_changed', [ $this, 'on_order_status_changed' ], 10, 4 );
        add_action( 'woocommerce_payment_complete', [ $this, 'on_payment_complete' ], 10, 1 );
        add_action( 'woocommerce_order_refunded', [ $this, 'on_order_refunded' ], 10, 2 );

        // WooCommerce product hooks.
        add_action( 'woocommerce_update_product', [ $this, 'on_product_saved' ], 10, 1 );
        add_action( 'woocommerce_new_product', [ $this, 'on_product_saved' ], 10, 1 );

        // Plugin action links.
        add_filter( 'plugin_action_links_' . QCNC_PLUGIN_BASENAME, [ $this, 'plugin_action_links' ] );
    }

    /**
     * Handle order status changed event.
     *
     * @param int    $order_id   Order ID.
     * @param string $old_status Old status.
     * @param string $new_status New status.
     * @param object $order      Order object.
     */
    public function on_order_status_changed( $order_id, $old_status, $new_status, $order ) {
        $this->logger->log( "Order #{$order_id} status changed from {$old_status} to {$new_status}" );
        
        if ( in_array( $new_status, [ 'completed', 'processing' ], true ) ) {
            $this->order_profit_calculator->calculate_order_profit( $order_id );
            $this->margin_alert_system->check_order_profitability( $order_id );
        }
    }

    /**
     * Handle payment complete event.
     *
     * @param int $order_id Order ID.
     */
    public function on_payment_complete( $order_id ) {
        $this->logger->log( "Payment completed for order #{$order_id}" );
        $this->order_profit_calculator->calculate_order_profit( $order_id );
    }

    /**
     * Handle order refund event.
     *
     * @param int $order_id  Order ID.
     * @param int $refund_id Refund ID.
     */
    public function on_order_refunded( $order_id, $refund_id ) {
        $this->logger->log( "Order #{$order_id} refunded (Refund ID: {$refund_id})" );
        $this->order_profit_calculator->recalculate_order_profit( $order_id );
        $this->cache_manager->delete_order_cache( $order_id );
    }

    /**
     * Handle product save event.
     *
     * @param int $product_id Product ID.
     */
    public function on_product_saved( $product_id ) {
        $this->logger->log( "Product #{$product_id} saved" );
        $this->cost_rules_engine->apply_rules_to_product( $product_id );
        $this->cache_manager->delete_product_cache( $product_id );
    }

    /**
     * Add plugin action links.
     *
     * @param array $links Existing links.
     * @return array Modified links.
     */
    public function plugin_action_links( $links ) {
        $plugin_links = [
            '<a href="' . esc_url( admin_url( 'admin.php?page=qcnc-dashboard' ) ) . '">' . esc_html__( 'Dashboard', 'quarkcode-neuralcommerce-lite' ) . '</a>',
            '<a href="' . esc_url( admin_url( 'admin.php?page=qcnc-settings' ) ) . '">' . esc_html__( 'Settings', 'quarkcode-neuralcommerce-lite' ) . '</a>',
        ];

        return array_merge( $plugin_links, $links );
    }

    /**
     * Initialize alert system
     */
    /*
    private function init_alert_system() {
        if ( class_exists( 'QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System' ) ) {
            new Core\QCNC_Margin_Alert_System();
        }
    }
        */

    /**
     * Initialize cron system
     */
    private function init_cron_system() {
        if ( class_exists( 'QuarkcodeNeuralCommerce\Cron\QCNC_Cron_Jobs' ) ) {
            new Cron\QCNC_Cron_Jobs();
        }
    }

    /**
     * Initialize alert test handler
     */
    private function init_alert_test_handler() {
        if ( is_admin() ) {
            $this->alert_test_handler = new QCNC_Alert_Test_Handler();
        }
    }





}
