<?php
/**
 * Plugin Activator
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce;

use QuarkcodeNeuralCommerce\Database\QCNC_Database_Schema;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Plugin activation handler.
 */
class QCNC_Activator {

    /**
     * Run activation tasks.
     */
    public static function activate() {
        // Create database tables.
        self::create_tables();

        // Set default options.
        self::set_default_options();

        // Schedule cron jobs.
        self::schedule_cron_jobs();

        // Insert default payment gateway fees.
        self::insert_default_gateway_fees();

        // Set activation flag.
        update_option( 'qcnc_activated', true );
        update_option( 'qcnc_version', QCNC_VERSION );
        update_option( 'qcnc_activation_time', current_time( 'timestamp' ) );

        // Flush rewrite rules.
        flush_rewrite_rules();
    }

    /**
     * Schedule cron jobs on activation
     */
    public static function schedule_cron_jobs() {
        // Trigger the cron job scheduling
        do_action( 'qcnc_plugin_activated' );
    }


    /**
     * Create database tables.
     */
    private static function create_tables() {
        require_once QCNC_INCLUDES_DIR . 'database/class-qcnc-database-schema.php';
        QCNC_Database_Schema::create_tables();
    }

    /**
     * Set default plugin options.
     */
    private static function set_default_options() {
        $defaults = [
            'qcnc_low_margin_threshold'    => 15.0,  // 15% margin alert.
            'qcnc_loss_order_alert'        => true,
            'qcnc_monthly_summary_email'   => true,
            'qcnc_alert_email'             => get_option( 'admin_email' ),
            'qcnc_default_currency'        => get_woocommerce_currency(),
            'qcnc_enable_slack_alerts'     => false,
            'qcnc_slack_webhook_url'       => '',
            'qcnc_calculation_on_save'     => true,
            'qcnc_cache_duration'          => 43200, // 12 hours.
            'qcnc_enable_debug_logging'    => false,
        ];

        foreach ( $defaults as $option_name => $option_value ) {
            if ( false === get_option( $option_name ) ) {
                add_option( $option_name, $option_value );
            }
        }
    }

    /**
     * Schedule cron jobs.
     */
    /*
    private static function schedule_cron_jobs() {
        // Daily profit recalculation.
        if ( ! wp_next_scheduled( 'qcnc_daily_profit_recalc' ) ) {
            wp_schedule_event( strtotime( 'tomorrow 2:00 AM' ), 'daily', 'qcnc_daily_profit_recalc' );
        }

        // Weekly report generation.
        if ( ! wp_next_scheduled( 'qcnc_weekly_report' ) ) {
            wp_schedule_event( strtotime( 'next monday 3:00 AM' ), 'weekly', 'qcnc_weekly_report' );
        }

        // Monthly summary email.
        if ( ! wp_next_scheduled( 'qcnc_monthly_summary' ) ) {
            wp_schedule_event( strtotime( 'first day of next month 9:00 AM' ), 'monthly', 'qcnc_monthly_summary' );
        }

        // Hourly cache cleanup.
        if ( ! wp_next_scheduled( 'qcnc_cache_cleanup' ) ) {
            wp_schedule_event( time(), 'hourly', 'qcnc_cache_cleanup' );
        }
    }
        */

    /**
     * Insert default payment gateway fee configurations.
     */
    private static function insert_default_gateway_fees() {
        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';
        $currency = get_woocommerce_currency();

        $default_gateways = [
            [
                'gateway_id'     => 'stripe',
                'gateway_name'   => 'Stripe',
                'fee_type'       => 'mixed',
                'percentage_fee' => 2.9,
                'fixed_fee'      => 0.30,
                'currency'       => $currency,
            ],
            [
                'gateway_id'     => 'paypal',
                'gateway_name'   => 'PayPal',
                'fee_type'       => 'mixed',
                'percentage_fee' => 3.49,
                'fixed_fee'      => 0.49,
                'currency'       => $currency,
            ],
            [
                'gateway_id'     => 'razorpay',
                'gateway_name'   => 'Razorpay',
                'fee_type'       => 'percentage',
                'percentage_fee' => 2.0,
                'fixed_fee'      => 0.00,
                'currency'       => $currency,
            ],
            [
                'gateway_id'     => 'cod',
                'gateway_name'   => 'Cash on Delivery',
                'fee_type'       => 'fixed',
                'percentage_fee' => 0.0,
                'fixed_fee'      => 0.00,
                'currency'       => $currency,
            ],
        ];

        foreach ( $default_gateways as $gateway ) {
            $gateway['created_at'] = current_time( 'mysql' );
            $gateway['is_active']  = 1;

            // Check if gateway already exists.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $exists = $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
                    "SELECT id FROM {$table} WHERE gateway_id = %s AND currency = %s",
                    $gateway['gateway_id'],
                    $gateway['currency']
                )
            );

            if ( ! $exists ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $wpdb->insert( $table, $gateway );
            }
        }
    }
}