<?php
/**
 * Reports REST API
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\API;

use QuarkcodeNeuralCommerce\Core\QCNC_Report_Generator;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Reports API endpoints.
 */
class QCNC_Reports_Api extends QCNC_Rest_Controller {

    /**
     * Report generator instance.
     *
     * @var QCNC_Report_Generator
     */
    private $report_generator;

    /**
     * Constructor.
     *
     * @param QCNC_Report_Generator $report_generator Report generator.
     * @param QCNC_Security         $security         Security.
     */
    public function __construct( QCNC_Report_Generator $report_generator, QCNC_Security $security ) {
        parent::__construct( $security );
        $this->report_generator = $report_generator;
        // Routes will be registered via rest_api_init hook
    }

    /**
     * Initialize the API.
     */
    public function init() {
        $this->register_routes();
    }


    /**
     * Register API routes.
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/reports/dashboard',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_dashboard_report' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/reports/product',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_product_report' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/reports/category',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_category_report' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/reports/customer',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_customer_report' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );
    }

    /**
     * Get dashboard report.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_dashboard_report( $request ) {
        $date_from = sanitize_text_field( $request->get_param( 'date_from' ) ) ?: wp_date( 'Y-m-01' );
        $date_to = sanitize_text_field( $request->get_param( 'date_to' ) ) ?: wp_date( 'Y-m-d' );

        $data = $this->report_generator->get_dashboard_summary( $date_from, $date_to );

        return $this->prepare_response( $data );
    }

    /**
     * Get product profit report.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_product_report( $request ) {
        $date_from = sanitize_text_field( $request->get_param( 'date_from' ) ) ?: '';
        $date_to = sanitize_text_field( $request->get_param( 'date_to' ) ) ?: '';
        $limit = absint( $request->get_param( 'limit' ) ) ?: 100;

        $data = $this->report_generator->get_profit_by_product( $date_from, $date_to, $limit );

        return $this->prepare_response( $data );
    }

    /**
     * Get category profit report.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_category_report( $request ) {
        $date_from = sanitize_text_field( $request->get_param( 'date_from' ) ) ?: '';
        $date_to = sanitize_text_field( $request->get_param( 'date_to' ) ) ?: '';

        $data = $this->report_generator->get_profit_by_category( $date_from, $date_to );

        return $this->prepare_response( $data );
    }

    /**
     * Get customer profit report.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_customer_report( $request ) {
        $date_from = sanitize_text_field( $request->get_param( 'date_from' ) ) ?: '';
        $date_to = sanitize_text_field( $request->get_param( 'date_to' ) ) ?: '';
        $limit = absint( $request->get_param( 'limit' ) ) ?: 100;

        $data = $this->report_generator->get_profit_by_customer( $date_from, $date_to, $limit );

        return $this->prepare_response( $data );
    }
}
