<?php
/**
 * Products REST API
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\API;

use QuarkcodeNeuralCommerce\Core\QCNC_Product_Cost_Manager;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Products API endpoints.
 */
class QCNC_Products_Api extends QCNC_Rest_Controller {

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    private $cost_manager;

    /**
     * Constructor.
     *
     * @param QCNC_Product_Cost_Manager $cost_manager Cost manager.
     * @param QCNC_Security             $security     Security.
     */
    public function __construct( QCNC_Product_Cost_Manager $cost_manager, QCNC_Security $security ) {
        parent::__construct( $security );
        $this->cost_manager = $cost_manager;
        // Routes will be registered via rest_api_init hook
    }

    /**
     * Initialize the API.
     */
    public function init() {
        $this->register_routes();
    }


    /**
     * Register API routes.
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/products/(?P<id>\d+)/cost',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [ $this, 'get_product_cost' ],
                    'permission_callback' => [ $this, 'check_permission' ],
                    'args'                => [
                        'id' => [
                            'required'          => true,
                            'validate_callback' => function( $param ) {
                                return is_numeric( $param );
                            },
                        ],
                    ],
                ],
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [ $this, 'update_product_cost' ],
                    'permission_callback' => [ $this, 'check_permission' ],
                    'args'                => $this->get_cost_schema(),
                ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/products/costs',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_products_costs' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );
    }

    /**
     * Get product cost.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error Response.
     */
    public function get_product_cost( $request ) {
        $product_id = absint( $request['id'] );
        $cost_data = $this->cost_manager->get_product_cost( $product_id );

        if ( ! $cost_data ) {
            return $this->prepare_error( __( 'Cost not found', 'quarkcode-neuralcommerce-lite' ), 404 );
        }

        return $this->prepare_response( $cost_data );
    }

    /**
     * Update product cost.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error Response.
     */
    public function update_product_cost( $request ) {
        $product_id = absint( $request['id'] );
        $cost = floatval( $request['cost'] );
        $packaging_cost = isset( $request['packaging_cost'] ) ? floatval( $request['packaging_cost'] ) : 0.0;
        $handling_cost = isset( $request['handling_cost'] ) ? floatval( $request['handling_cost'] ) : 0.0;

        $result = $this->cost_manager->set_product_cost( $product_id, $cost, null, $packaging_cost, $handling_cost );

        if ( $result ) {
            return $this->prepare_response( [
                'success' => true,
                'cost_id' => $result,
                'message' => __( 'Cost updated successfully', 'quarkcode-neuralcommerce-lite' ),
            ] );
        }

        return $this->prepare_error( __( 'Failed to update cost', 'quarkcode-neuralcommerce-lite' ) );
    }

    /**
     * Get multiple products costs.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_products_costs( $request ) {
        $page = absint( $request->get_param( 'page' ) ) ?: 1;
        $per_page = absint( $request->get_param( 'per_page' ) ) ?: 20;

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        $offset = ( $page - 1 ) * $per_page;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE is_current = 1" );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $costs = $wpdb->get_results(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE is_current = 1 LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        return $this->prepare_response( [
            'costs'       => $costs,
            'total'       => $total,
            'page'        => $page,
            'per_page'    => $per_page,
            'total_pages' => ceil( $total / $per_page ),
        ] );
    }

    /**
     * Get cost schema for validation.
     *
     * @return array Schema.
     */
    private function get_cost_schema() {
        return [
            'cost' => [
                'required'          => true,
                'type'              => 'number',
                'validate_callback' => function( $param ) {
                    return is_numeric( $param ) && $param >= 0;
                },
            ],
            'packaging_cost' => [
                'type'              => 'number',
                'validate_callback' => function( $param ) {
                    return is_numeric( $param ) && $param >= 0;
                },
            ],
            'handling_cost' => [
                'type'              => 'number',
                'validate_callback' => function( $param ) {
                    return is_numeric( $param ) && $param >= 0;
                },
            ],
        ];
    }
}
