<?php
/**
 * Orders REST API
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\API;

use QuarkcodeNeuralCommerce\Core\QCNC_Order_Profit_Calculator;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Orders API endpoints.
 */
class QCNC_Orders_Api extends QCNC_Rest_Controller {

    /**
     * Order profit calculator instance.
     *
     * @var QCNC_Order_Profit_Calculator
     */
    private $calculator;

    /**
     * Constructor.
     *
     * @param QCNC_Order_Profit_Calculator $calculator Calculator.
     * @param QCNC_Security                $security   Security.
     */
    public function __construct( QCNC_Order_Profit_Calculator $calculator, QCNC_Security $security ) {
        parent::__construct( $security );
        $this->calculator = $calculator;
        // Routes will be registered via rest_api_init hook
    }

    /**
     * Initialize the API.
     */
    public function init() {
        $this->register_routes();
    }

    /**
     * Register API routes.
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/orders/(?P<id>\d+)/profit',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [ $this, 'get_order_profit' ],
                    'permission_callback' => [ $this, 'check_permission' ],
                ],
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [ $this, 'recalculate_order_profit' ],
                    'permission_callback' => [ $this, 'check_permission' ],
                ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/orders/profit',
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_orders_profit' ],
                'permission_callback' => [ $this, 'check_permission' ],
            ]
        );
    }

    /**
     * Get order profit.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error Response.
     */
    public function get_order_profit( $request ) {
        $order_id = absint( $request['id'] );
        $profit_data = $this->calculator->get_order_profit( $order_id );

        if ( ! $profit_data ) {
            return $this->prepare_error( __( 'Profit data not found', 'quarkcode-neuralcommerce-lite' ), 404 );
        }

        // Get items breakdown.
        $items = $this->calculator->get_order_items_profit( $order_id );
        $profit_data['items'] = $items;

        return $this->prepare_response( $profit_data );
    }

    /**
     * Recalculate order profit.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error Response.
     */
    public function recalculate_order_profit( $request ) {
        $order_id = absint( $request['id'] );
        $result = $this->calculator->recalculate_order_profit( $order_id );

        if ( $result ) {
            return $this->prepare_response( [
                'success'     => true,
                'message'     => __( 'Profit recalculated successfully', 'quarkcode-neuralcommerce-lite' ),
                'profit_data' => $result,
            ] );
        }

        return $this->prepare_error( __( 'Failed to recalculate profit', 'quarkcode-neuralcommerce-lite' ) );
    }

    /**
     * Get multiple orders profit.
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response Response.
     */
    public function get_orders_profit( $request ) {
        $page = absint( $request->get_param( 'page' ) ) ?: 1;
        $per_page = absint( $request->get_param( 'per_page' ) ) ?: 20;

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        $offset = ( $page - 1 ) * $per_page;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $orders = $wpdb->get_results(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} ORDER BY calculated_at DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        return $this->prepare_response( [
            'orders'      => $orders,
            'total'       => $total,
            'page'        => $page,
            'per_page'    => $per_page,
            'total_pages' => ceil( $total / $per_page ),
        ] );
    }
}
