<?php
/**
 * Alert Testing AJAX Handler
 * 
 * Handles test email sending via AJAX without requiring functions.php code
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Admin;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class QCNC_Alert_Test_Handler {

    /**
     * Initialize the test handler
     */
    public function __construct() {
        // error_log( 'QCNC Alert Test Handler: Initializing...' );

        // Register AJAX handlers for both logged-in users
        add_action( 'wp_ajax_qcnc_test_alert', array( $this, 'handle_test_alert' ) );

        // error_log( 'QCNC Alert Test Handler: AJAX action registered!' );
    }

    /**
     * Handle AJAX test alert request
     */
    public function handle_test_alert() {
        // Security check
        check_ajax_referer( 'qcnc_test_alert', 'nonce' );

        // Permission check
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array(
                'message' => __( 'You do not have permission to test alerts.', 'quarkcode-neuralcommerce-lite' )
            ) );
        }

        // Get alert type
        $alert_type = isset( $_POST['alert_type'] ) ? sanitize_text_field( wp_unslash($_POST['alert_type']) ) : '';

        if ( empty( $alert_type ) ) {
            wp_send_json_error( array(
                'message' => __( 'Alert type is required.', 'quarkcode-neuralcommerce-lite' )
            ) );
        }

        // error_log( "=== QCNC Alert Test Started: {$alert_type} ===" );

        // Process based on alert type
        try {
            switch ( $alert_type ) {
                case 'low_margin':
                    $this->test_low_margin();
                    break;

                case 'cost_increase':
                    $this->test_cost_increase();
                    break;

                case 'unprofitable_order':
                    $this->test_unprofitable_order();
                    break;

                case 'anomaly_detected':
                    $this->test_anomaly_detected();
                    break;

                case 'daily_summary':
                    $this->test_daily_summary();
                    break;

                default:
                    wp_send_json_error( array(
                        'message' => __( 'Invalid alert type.', 'quarkcode-neuralcommerce-lite' )
                    ) );
            }

            $email = get_option( 'qcnc_alert_email', get_option( 'admin_email' ) );

            wp_send_json_success( array(
                'message' => sprintf(
                    /* translators: %s: Email address */
                    __( 'Test email sent successfully to %s! Check your inbox.', 'quarkcode-neuralcommerce-lite' ),
                    $email
                )
            ) );

        } catch ( Exception $e ) {
            // error_log( 'QCNC Alert Test Error: ' . $e->getMessage() );

            wp_send_json_error( array(
                'message' => sprintf(
                    /* translators: %s: Error message */
                    __( 'Error sending test: %s', 'quarkcode-neuralcommerce-lite' ),
                    $e->getMessage()
                )
            ) );
        }
    }

    /**
     * Test low margin alert
     */
    private function test_low_margin() {
        // error_log( 'Testing low_margin alert...' );

        // Trigger the action with test data
        do_action( 'qcnc_order_profit_calculated', 456, array(
            'net_profit' => 12.00,
            'margin_pct' => 12.0,
            'revenue' => 100.00,
            'cogs_total' => 88.00,
        ) );

        // error_log( 'Low margin alert triggered!' );
    }

    /**
     * Test cost increase alert
     */
    private function test_cost_increase() {
        // error_log( 'Testing cost_increase alert...' );

        // Get a real product or use test data
        $products = wc_get_products( array( 'limit' => 1 ) );
        $product_id = ! empty( $products ) ? $products[0]->get_id() : 123;

        // Trigger the action
        do_action( 'qcnc_product_cost_updated', $product_id, 10.00, 15.00 );

        // error_log( 'Cost increase alert triggered!' );
    }

    /**
     * Test unprofitable order alert
     */
    private function test_unprofitable_order() {
        // error_log( 'Testing unprofitable_order alert...' );

        if ( ! class_exists( 'QuarkcodeNeuralCommerce\\Core\\QCNC_Margin_Alert_System' ) ) {
            throw new \Exception( 'Alert system class not found.' );
        }

        // Create instance
        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();

        // Use reflection to access private method
        $reflection = new \ReflectionClass( $alert_system );
        $method = $reflection->getMethod( 'send_email_alert' );
        $method->setAccessible( true );

        // Call method with test data
        $method->invoke(
            $alert_system,
            'unprofitable_order',
            array(
                'order_id' => 789,
                'order_number' => '#789',
                'profit' => -25.00,
                'margin' => -12.5,
                'revenue' => 200.00,
                'cost' => 225.00,
            )
        );

        // error_log( 'Unprofitable order alert sent!' );
    }

    /**
     * Test anomaly detection alert
     */
    private function test_anomaly_detected() {
        // error_log( 'Testing anomaly_detected alert...' );

        if ( ! class_exists( 'QuarkcodeNeuralCommerce\\Core\\QCNC_Margin_Alert_System' ) ) {
            throw new \Exception( 'Alert system class not found.' );
        }

        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();

        $reflection = new \ReflectionClass( $alert_system );
        $method = $reflection->getMethod( 'send_email_alert' );
        $method->setAccessible( true );

        $test_data = array(
            'anomalies' => array(
                array(
                    'order_id' => 1001,
                    'profit' => -50.00,
                    'margin' => -15.0,
                    'profit_z_score' => -3.5,
                    'margin_z_score' => -2.8,
                    'date' => wp_date( 'Y-m-d H:i:s', strtotime( '-2 days' ) ),
                ),
                array(
                    'order_id' => 1002,
                    'profit' => 5.00,
                    'margin' => 3.5,
                    'profit_z_score' => -2.2,
                    'margin_z_score' => -2.1,
                    'date' => wp_date( 'Y-m-d H:i:s', strtotime( '-1 day' ) ),
                ),
                array(
                    'order_id' => 1003,
                    'profit' => 150.00,
                    'margin' => 60.0,
                    'profit_z_score' => 3.2,
                    'margin_z_score' => 2.9,
                    'date' => wp_date( 'Y-m-d H:i:s' ),
                ),
            ),
            'profit_mean' => 35.50,
            'margin_mean' => 28.5,
            'detection_period' => '90 days',
        );

        $method->invoke( $alert_system, 'anomaly_detected', $test_data );

        // error_log( 'Anomaly detection alert sent!' );
    }

    /**
     * Test daily summary alert
     */
    private function test_daily_summary() {
        // error_log( 'Testing daily_summary alert...' );

        if ( ! class_exists( 'QuarkcodeNeuralCommerce\\Core\\QCNC_Margin_Alert_System' ) ) {
            throw new \Exception( 'Alert system class not found.' );
        }

        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();

        $reflection = new \ReflectionClass( $alert_system );
        $method = $reflection->getMethod( 'send_email_alert' );
        $method->setAccessible( true );

        $test_data = array(
            'date' => wp_date( 'Y-m-d', strtotime( '-1 day' ) ),
            'yesterday' => array(
                'order_count' => 45,
                'total_revenue' => 5234.50,
                'total_cost' => 3356.25,
                'total_profit' => 1878.25,
                'avg_margin' => 35.89,
            ),
            'last_week' => array(
                'avg_profit' => 41.73,
                'avg_margin' => 32.15,
            ),
            'top_products' => array(
                array(
                    'product_id' => 101,
                    'total_profit' => 456.00,
                    'avg_margin' => 45.5,
                    'order_count' => 12,
                ),
                array(
                    'product_id' => 102,
                    'total_profit' => 345.00,
                    'avg_margin' => 38.2,
                    'order_count' => 8,
                ),
                array(
                    'product_id' => 103,
                    'total_profit' => 289.00,
                    'avg_margin' => 42.1,
                    'order_count' => 15,
                ),
            ),
            'low_margin_products' => array(
                array(
                    'product_id' => 201,
                    'avg_margin' => 8.5,
                    'order_count' => 5,
                ),
                array(
                    'product_id' => 202,
                    'avg_margin' => 12.3,
                    'order_count' => 3,
                ),
            ),
        );

        $method->invoke( $alert_system, 'daily_summary', $test_data );

        // error_log( 'Daily summary alert sent!' );
    }
}

// Initialize the handler
new QCNC_Alert_Test_Handler();
