<?php
/**
 * Admin Controller
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Admin;

use QuarkcodeNeuralCommerce\Core\QCNC_Product_Cost_Manager;
use QuarkcodeNeuralCommerce\Core\QCNC_Order_Profit_Calculator;
use QuarkcodeNeuralCommerce\Core\QCNC_Report_Generator;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Manages admin UI and pages.
 */
class QCNC_Admin {

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    private $cost_manager;

    /**
     * Order profit calculator instance.
     *
     * @var QCNC_Order_Profit_Calculator
     */
    private $calculator;

    /**
     * Report generator instance.
     *
     * @var QCNC_Report_Generator
     */
    private $report_generator;

    /**
     * Security instance.
     *
     * @var QCNC_Security
     */
    private $security;

    /**
     * Constructor.
     *
     * @param QCNC_Product_Cost_Manager    $cost_manager      Cost manager.
     * @param QCNC_Order_Profit_Calculator $calculator        Calculator.
     * @param QCNC_Report_Generator        $report_generator  Report generator.
     * @param QCNC_Security                $security          Security.
     */
    public function __construct( 
        QCNC_Product_Cost_Manager $cost_manager,
        QCNC_Order_Profit_Calculator $calculator,
        QCNC_Report_Generator $report_generator,
        QCNC_Security $security
    ) {
        $this->cost_manager     = $cost_manager;
        $this->calculator       = $calculator;
        $this->report_generator = $report_generator;
        $this->security         = $security;

        $this->init_hooks();
    }

    /**
     * Initialize hooks.
     */
    private function init_hooks() {
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_filter( 'manage_shop_order_posts_columns', [ $this, 'add_order_columns' ] );
        add_action( 'manage_shop_order_posts_custom_column', [ $this, 'render_order_column' ], 10, 2 );
        add_filter( 'manage_edit-product_columns', [ $this, 'add_product_columns' ] );
        add_action( 'manage_product_posts_custom_column', [ $this, 'render_product_column' ], 10, 2 );
        add_action( 'wp_ajax_qcnc_get_products_list', [ $this, 'get_products_list' ] );
        add_action( 'wp_ajax_qcnc_save_products_inline', [ $this, 'save_products_inline' ] );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_alert_test_scripts' ) );

    }

    /**
     * Add admin menu pages.
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Neural Commerce', 'quarkcode-neuralcommerce-lite'),
            __('Neural Commerce', 'quarkcode-neuralcommerce-lite'),
            'manage_woocommerce',
            'qcnc-dashboard',
            array($this, 'render_dashboard_page'), // UPDATED
            'dashicons-chart-area',
            56
        );

        // Advanced Dashboard submenu (first item, same as main menu)
        add_submenu_page(
            'qcnc-dashboard',
            __( 'Dashboard', 'quarkcode-neuralcommerce-lite' ),
            __( 'Dashboard', 'quarkcode-neuralcommerce-lite' ),
            'manage_woocommerce',
            'qcnc-dashboard', // Same slug as parent
            [ $this, 'render_dashboard_page' ]
        );

        add_submenu_page(
            'qcnc-dashboard',
            __( 'Product Costs', 'quarkcode-neuralcommerce-lite' ),
            __( 'Product Costs', 'quarkcode-neuralcommerce-lite' ),
            'manage_woocommerce',
            'qcnc-product-costs',
            [ $this, 'render_product_costs_page' ]
        );

        add_submenu_page(
            'qcnc-dashboard',
            __( 'Order Profitability', 'quarkcode-neuralcommerce-lite' ),
            __( 'Order Profitability', 'quarkcode-neuralcommerce-lite' ),
            'manage_woocommerce',
            'qcnc-order-profitability',
            [ $this, 'render_order_profitability_page' ]
        );

        add_submenu_page(
            'qcnc-dashboard',
            __( 'Reports', 'quarkcode-neuralcommerce-lite' ),
            __( 'Reports', 'quarkcode-neuralcommerce-lite' ),
            'manage_woocommerce',
            'qcnc-reports',
            [ $this, 'render_reports_page' ]
        );

        // ✅ ADD Pro version menu link
        add_submenu_page(
            'qcnc-dashboard',
            __('Get Pro Version', 'quarkcode-neuralcommerce-lite'),
            '<span style="color:#f18500;">⭐ Get Pro Version</span>',
            'manage_woocommerce',
            'qcnc-upgrade',
            array($this, 'render_upgrade_page')
        );

        add_submenu_page(
            'qcnc-dashboard',
            __( 'Settings', 'quarkcode-neuralcommerce-lite' ),
            __( 'Settings', 'quarkcode-neuralcommerce-lite' ),
            'manage_woocommerce',
            'qcnc-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    /**
     * Enqueue admin assets.
     *
     * @param string $hook Current admin page hook.
     */
    public function enqueue_assets( $hook ) {
        $screen = get_current_screen();
        $is_product_list = ( $screen && 'edit-product' === $screen->id );
        
        if ( strpos( $hook, 'qcnc-' ) === false && ! $is_product_list ) {
            return;
        }

        // Enqueue styles.
        wp_enqueue_style(
            'qcnc-admin',
            QCNC_ASSETS_URL . 'css/admin.css',
            [],
            QCNC_VERSION
        );

        wp_enqueue_style(
            'qcnc-dashboard',
            QCNC_ASSETS_URL . 'css/dashboard.css',
            [],
            QCNC_VERSION
        );

        // Enqueue consolidated admin view styles (extracted from inline)
        wp_enqueue_style(
            'qcnc-admin-views',
            QCNC_ASSETS_URL . 'css/qcnc-admin-views.css',
            [],
            QCNC_VERSION
        );

        // Enqueue scripts.
        wp_enqueue_script(
            'qcnc-admin',
            QCNC_ASSETS_URL . 'js/admin.js',
            [ 'jquery' ],
            QCNC_VERSION,
            true
        );

        // Enqueue consolidated admin view scripts (extracted from inline)
        wp_enqueue_script(
            'qcnc-admin-views',
            QCNC_ASSETS_URL . 'js/qcnc-admin-views.js',
            [ 'jquery', 'qcnc-admin' ],
            QCNC_VERSION,
            true
        );

        // Localize script.
        wp_localize_script(
            'qcnc-admin',
            'qcncAdmin',
            array(
                'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
                'nonce'         => wp_create_nonce( 'qcnc_admin_nonce' ),
                'upgradeUrl'    => admin_url( 'admin.php?page=qcnc-upgrade' ),
                'i18n'          => array(
                    'confirmDelete'  => __( 'Are you sure you want to delete this?', 'quarkcode-neuralcommerce-lite' ),
                    'confirmCleanup' => __( 'Clean up duplicate costs?', 'quarkcode-neuralcommerce-lite' ),
                    'addGatewayFee'  => __( 'Add Gateway Fee', 'quarkcode-neuralcommerce-lite' ),
                    'saving'         => __( 'Saving...', 'quarkcode-neuralcommerce-lite' ),
                    'saved'          => __( 'Saved', 'quarkcode-neuralcommerce-lite' ),
                    'error'          => __( 'Error occurred', 'quarkcode-neuralcommerce-lite' ),
                ),
            )
        );


        // Advanced dashboard assets (ONLY on dashboard page)
        if ( $hook === 'toplevel_page_qcnc-dashboard' ) {
            
            // Chart.js
            wp_enqueue_script(
                'chartjs',
                QCNC_ASSETS_URL . 'js/chart.min.js', 
                array(),
                '3.9.1',
                true
            );


            // Advanced dashboard CSS
            wp_enqueue_style(
                'qcnc-dashboard-advanced',
                QCNC_ASSETS_URL . 'css/dashboard-advanced.css',
                array(),
                QCNC_VERSION
            );

            // Advanced dashboard JS
            wp_enqueue_script(
                'qcnc-dashboard-advanced',
                QCNC_ASSETS_URL . 'js/dashboard-advanced.js',
                array( 'jquery', 'chartjs' ),
                QCNC_VERSION,
                true
            );

            // Localize advanced dashboard script
            wp_localize_script(
                'qcnc-dashboard-advanced',
                'qcnc_ajax',
                array(
                    'ajax_url' => admin_url( 'admin-ajax.php' ),
                    'nonce'    => wp_create_nonce( 'qcnc_ajax_nonce' )
                )
            );
        }

        // Enqueue AI product cost script on product pages
        if (get_post_type() === 'product') {
            wp_enqueue_script(
                'qcnc-product-cost-ai',
                QCNC_PLUGIN_URL . 'assets/js/product-cost-ai.js',
                ['jquery'],
                QCNC_VERSION,
                true
            );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'qcnc-dead-stock' ) {
            wp_enqueue_style(
                'qcnc-dead-stock',
                QCNC_ASSETS_URL . 'css/dead-stock.css',
                array(),
                QCNC_VERSION
            );
        }

        // Find where you localize the script (around line 280) and UPDATE it:
        wp_localize_script(
            'qcnc-dead-stock',
            'qcncAdminAjax',
            array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'adminUrl' => admin_url(), // ADD THIS LINE
                'nonce'   => wp_create_nonce( 'qcnc-admin-ajax' ),
            )
        );

        // Dead Stock JavaScript
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'qcnc-dead-stock' ) {
            wp_enqueue_script(
                'qcnc-dead-stock',
                QCNC_ASSETS_URL . 'js/dead-stock.js',
                array( 'jquery' ),
                QCNC_VERSION,
                true
            );
            
            // Localize script with AJAX data
            wp_localize_script(
                'qcnc-dead-stock',
                'qcncAdminAjax',
                array(
                    'ajaxurl' => admin_url( 'admin-ajax.php' ),
                    'adminUrl' => admin_url(), // Make sure this line is here
                    'nonce'   => wp_create_nonce( 'qcnc-admin-ajax' ),
                )
            );
        }

        // ✅ FIXED: Advanced Reports assets - Better screen detection
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'qcnc-advanced-reports' ) {
            
            // Chart.js (already bundled)
            wp_enqueue_script(
                'qcnc-chart-js',
                QCNC_ASSETS_URL . 'js/chart.min.js',
                array(),
                QCNC_VERSION,
                true
            );

            // Advanced Reports JS
            wp_enqueue_script(
                'qcnc-reports-advanced',
                QCNC_ASSETS_URL . 'js/reports-advanced.js',
                array( 'jquery', 'qcnc-chart-js' ),
                QCNC_VERSION,
                true
            );

            // Advanced Reports CSS
            wp_enqueue_style(
                'qcnc-reports-advanced',
                QCNC_ASSETS_URL . 'css/reports-advanced.css',
                array(),
                QCNC_VERSION
            );

            // Localized strings
            wp_localize_script(
                'qcnc-reports-advanced',
                'qcnc_i18n',
                array(
                    'dead_stock_empty' => __( 'No dead stock items in the selected range.', 'quarkcode-neuralcommerce-lite' ),
                )
            );
        }

        // Upgrade page assets
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'qcnc-upgrade' ) {
            wp_enqueue_style(
                'qcnc-upgrade-page',
                QCNC_ASSETS_URL . 'css/qcnc-upgrade-page.css',
                array(),
                QCNC_VERSION
            );

            wp_enqueue_script(
                'qcnc-upgrade-page',
                QCNC_ASSETS_URL . 'js/qcnc-upgrade-page.js',
                array(),
                QCNC_VERSION,
                true
            );
        }

        
    }

    /**
     * Render advanced dashboard
     */
    public function render_advanced_dashboard() {
        // Get core instance
        $core = \QuarkcodeNeuralCommerce\QCNC_Core::instance();
        
        // Pass instances to view
        $profit_calculator = $core->order_profit_calculator;
        $report_generator = $core->report_generator;
        $ai_insights = $core->ai_insights_engine;
        
        // Get date range
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $date_from = isset($_GET['date_from']) ? sanitize_text_field(wp_unslash($_GET['date_from'])) : wp_date('Y-m-d', strtotime('-30 days'));
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $date_to = isset($_GET['date_to']) ? sanitize_text_field(wp_unslash($_GET['date_to'])) : wp_date('Y-m-d');
        
        // ✅ Get summary statistics for stat boxes
        $summary = $report_generator->get_dashboard_summary($date_from, $date_to);
        
        // Include view
        include_once QCNC_PLUGIN_DIR . 'includes/admin/views/dashboard-advanced.php';
    }



    /**
     * Render dashboard page.
     */
    public function render_dashboard_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $date_from = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : wp_date( 'Y-m-d', strtotime( '-30 days' ) );
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $date_to = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : wp_date( 'Y-m-d' );

        $summary = $this->report_generator->get_dashboard_summary( $date_from, $date_to );

        include QCNC_INCLUDES_DIR . 'admin/views/dashboard.php';
    }

    /**
     * Render product costs page.
     */
    public function render_product_costs_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        include QCNC_INCLUDES_DIR . 'admin/views/product-costs.php';
    }

    /**
     * Render order profitability page.
     */
    public function render_order_profitability_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        include QCNC_INCLUDES_DIR . 'admin/views/order-profitability.php';
    }

    /**
     * Render reports page.
     */
    public function render_reports_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        include QCNC_INCLUDES_DIR . 'admin/views/reports.php';
    }

    /**
     * Render settings page.
     */
    /*
    public function render_settings_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Save settings if form submitted.
        if ( isset( $_POST['qcnc_save_settings'] ) ) {
            // Check nonce manually since form uses 'qcnc_nonce'
            if ( isset( $_POST['qcnc_nonce'] ) && 
                wp_verify_nonce( $_POST['qcnc_nonce'], 'qcnc_save_settings' ) ) {
                $this->save_settings();
            } else {
                add_settings_error(
                    'qcnc_settings',
                    'invalid_nonce',
                    __( 'Security check failed. Please try again.', 'quarkcode-neuralcommerce-lite' ),
                    'error'
                );
            }
        }

        include QCNC_INCLUDES_DIR . 'admin/views/settings.php';
    }
        */

    /**
     * Render settings page.
     */
    public function render_settings_page() {
        if ( ! $this->security->check_capability( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Save settings if form submitted.
        if ( isset( $_POST['qcnc_save_settings'] ) ) {
            // ✅ FIXED: Sanitize and unslash nonce before verification
            $nonce = isset( $_POST['qcnc_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['qcnc_nonce'] ) ) : '';
            
            if ( wp_verify_nonce( $nonce, 'qcnc_save_settings' ) ) {
                $this->save_settings();
            } else {
                add_settings_error(
                    'qcnc_settings',
                    'invalid_nonce',
                    __( 'Security check failed. Please try again.', 'quarkcode-neuralcommerce-lite' ),
                    'error'
                );
            }
        }

        include QCNC_INCLUDES_DIR . 'admin/views/settings.php';
    }

    /**
     * Render products editor page.
     */
    public function render_products_editor_page() {
        require_once plugin_dir_path( __FILE__ ) . 'views/products-inline-editor.php';
    }

    /**
     * Render bulk cost updates page
     */
    public function render_bulk_costs_page() {
        require_once plugin_dir_path( __FILE__ ) . 'views/bulk-cost-update.php';
    }

    /**
     * Save settings.
     */
    private function save_settings() {
        // error_log( '=== QCNC Save Settings Called ===' );
        // error_log( 'Alert Email: ' . ( sanitize_email(wp_unslash($_POST['qcnc_alert_email'] ?? 'NOT SET' ))) );
        $settings = [
            'qcnc_low_margin_threshold',
            'qcnc_loss_order_alert',
            'qcnc_monthly_summary_email',
            'qcnc_alert_email',
            'qcnc_enable_slack_alerts',
            'qcnc_slack_webhook_url',
            'qcnc_enable_debug_logging',
            'qcnc_delete_data_on_uninstall',
        ];

        foreach ( $settings as $setting ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if ( isset( $_POST[ $setting ] ) ) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                $value = sanitize_text_field( wp_unslash( $_POST[ $setting ] ) );
                update_option( $setting, $value );
            }
        }

        // Add to your save settings logic:
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if (isset($_POST['qcnc_delete_data_on_uninstall'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option('qcnc_delete_data_on_uninstall', 1);
        } else {
            delete_option('qcnc_delete_data_on_uninstall');
        }

        // Save AI settings
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if (isset($_POST['qcnc_ai_provider'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option('qcnc_ai_provider', sanitize_text_field(wp_unslash($_POST['qcnc_ai_provider'])));
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if (isset($_POST['qcnc_ai_api_key'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option('qcnc_ai_api_key', sanitize_text_field(wp_unslash($_POST['qcnc_ai_api_key'])));
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if (isset($_POST['qcnc_ai_model'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option('qcnc_ai_model', sanitize_text_field(wp_unslash($_POST['qcnc_ai_model'])));
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if (isset($_POST['qcnc_ai_rate_limit'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option('qcnc_ai_rate_limit', intval($_POST['qcnc_ai_rate_limit']));
        }

        // Save alert settings
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_alert_email'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option( 'qcnc_alert_email', sanitize_email( wp_unslash($_POST['qcnc_alert_email']) ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_additional_alert_emails'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option( 'qcnc_additional_alert_emails', sanitize_textarea_field( wp_unslash($_POST['qcnc_additional_alert_emails']) ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_margin_threshold'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option( 'qcnc_margin_threshold', floatval( $_POST['qcnc_margin_threshold'] ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_cost_increase_threshold'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option( 'qcnc_cost_increase_threshold', floatval( $_POST['qcnc_cost_increase_threshold'] ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_margin_drop_threshold'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            update_option( 'qcnc_margin_drop_threshold', floatval( $_POST['qcnc_margin_drop_threshold'] ) );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_enabled_alerts'] ) ) {
            // ✅ FIXED: Unslash before sanitization
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $enabled_alerts = isset( $_POST['qcnc_enabled_alerts'] ) && is_array( $_POST['qcnc_enabled_alerts'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['qcnc_enabled_alerts'] ) ) 
                : array();
            update_option( 'qcnc_enabled_alerts', $enabled_alerts );
        } else {
            update_option( 'qcnc_enabled_alerts', array() );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_alert_digest_mode'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $digest_mode = sanitize_text_field( wp_unslash( $_POST['qcnc_alert_digest_mode'] ) );
            
            // Validate mode value
            $valid_modes = array( 'immediate', 'hourly', 'daily' );
            if ( in_array( $digest_mode, $valid_modes, true ) ) {
                update_option( 'qcnc_alert_digest_mode', $digest_mode );
            }
        }
        
        // Save rate limiting
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['qcnc_max_alerts_per_hour'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $max_alerts = intval( $_POST['qcnc_max_alerts_per_hour'] );
            
            // Ensure it's between 1 and 100
            $max_alerts = max( 1, min( 100, $max_alerts ) );
            
            update_option( 'qcnc_max_alerts_per_hour', $max_alerts );
        }

        // Update settings modified timestamp
        update_option( 'qcnc_settings_last_modified', time() );


        add_settings_error(
            'qcnc_settings',
            'settings_updated',
            __( 'Settings saved successfully.', 'quarkcode-neuralcommerce-lite' ),
            'success'
        );
    }

    /**
     * Add profit columns to orders list.
     *
     * @param array $columns Existing columns.
     * @return array Modified columns.
     */
    public function add_order_columns( $columns ) {
        $new_columns = [];

        foreach ( $columns as $key => $value ) {
            $new_columns[ $key ] = $value;

            if ( 'order_total' === $key ) {
                $new_columns['qcnc_profit'] = __( 'Net Profit', 'quarkcode-neuralcommerce-lite' );
                $new_columns['qcnc_margin'] = __( 'Margin %', 'quarkcode-neuralcommerce-lite' );
            }
        }

        return $new_columns;
    }

    /**
     * Render order column content.
     *
     * @param string $column  Column name.
     * @param int    $post_id Post ID.
     */
    public function render_order_column( $column, $post_id ) {
        if ( 'qcnc_profit' === $column ) {
            $profit_data = $this->calculator->get_order_profit( $post_id );
            if ( $profit_data ) {
                $net_profit = floatval( $profit_data['net_profit'] );
                $color = $net_profit >= 0 ? 'green' : 'red';
                echo '<span style="color: ' . esc_attr( $color ) . '; font-weight: bold;">' . 
                     wp_kses_post( wc_price( $net_profit ) ) . '</span>';
            } else {
                echo '—';
            }
        }

        if ( 'qcnc_margin' === $column ) {
            $profit_data = $this->calculator->get_order_profit( $post_id );
            if ( $profit_data ) {
                $margin = floatval( $profit_data['margin_pct'] );
                $color = $margin >= 15 ? 'green' : ( $margin >= 0 ? 'orange' : 'red' );
                echo '<span style="color: ' . esc_attr( $color ) . ';">' . 
                     esc_html( number_format( $margin, 2 ) ) . '%</span>';
            } else {
                echo '—';
            }
        }
    }

    /**
     * Add cost columns to products list.
     *
     * @param array $columns Existing columns.
     * @return array Modified columns.
     */
    public function add_product_columns( $columns ) {
        $new_columns = [];

        foreach ( $columns as $key => $value ) {
            $new_columns[ $key ] = $value;

            if ( 'price' === $key ) {
                $new_columns['qcnc_cost'] = __( 'Cost', 'quarkcode-neuralcommerce-lite' );
            }
        }

        return $new_columns;
    }

    /**
     * Render product column content.
     *
     * @param string $column  Column name.
     * @param int    $post_id Post ID.
     */
    public function render_product_column( $column, $post_id ) {
        if ( 'qcnc_cost' === $column ) {
            $cost_data = $this->cost_manager->get_product_cost( $post_id );
            if ( $cost_data ) {
                echo wp_kses_post( wc_price( $cost_data['total_cost'] ) );
            } else {
                echo '<span style="color: #999;">—</span>';
            }
        }
    }

    /**
     * Get products list for dropdown selector via AJAX.
     */
    public function get_products_list() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $limit = isset( $_POST['limit'] ) ? absint( $_POST['limit'] ) : 500;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $search = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';

        $args = [
            'status'  => 'publish',
            'limit'   => $limit,
            'orderby' => 'name',
            'order'   => 'ASC',
            'return'  => 'ids',
        ];

        if ( ! empty( $search ) ) {
            $args['s'] = $search;
        }

        $product_ids = wc_get_products( $args );
        $products = [];

        foreach ( $product_ids as $product_id ) {
            $product = wc_get_product( $product_id );
            
            if ( $product ) {
                $products[] = [
                    'id'   => $product_id,
                    'name' => $product->get_name(),
                    'sku'  => $product->get_sku() ?: '',
                    'price' => $product->get_price(),
                ];
            }
        }

        $this->security->json_success( $products );
    }

    /**
     * AJAX handler to save products inline (cost and price)
     */
    public function save_products_inline() {
        check_ajax_referer( 'qcnc_save_products', 'qcnc_products_nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $costs = isset( $_POST['cost'] ) ? sanitize_text_field( wp_unslash( $_POST['cost'] ) ) : [];
        $prices = isset( $_POST['price'] ) ? sanitize_text_field( wp_unslash( $_POST['price'] ) ) : [];

        $updated_products = [];
        $errors = [];

        // Update costs
        if ( isset( $this->cost_manager ) ) {
            foreach ( $costs as $product_id => $cost ) {
                $product_id = absint( $product_id );
                $cost = floatval( $cost );
                
                $result = $this->cost_manager->set_product_cost(
                    $product_id,
                    $cost,
                    null, // variation_id
                    0.0,  // packaging_cost
                    0.0,  // handling_cost
                    '',   // currency
                    []    // additional_costs
                );
                
                if ( !$result ) {
                    $errors[] = "Failed to update cost for product #{$product_id}";
                } else {
                    $updated_products[$product_id] = true;
                }
            }
        }

        // Update prices
        foreach ( $prices as $product_id => $price ) {
            $product_id = absint( $product_id );
            $price = floatval( $price );
            
            $product = wc_get_product( $product_id );
            if ( $product ) {
                $product->set_regular_price( $price );
                $product->save();
                $updated_products[$product_id] = true;
            } else {
                $errors[] = "Product #{$product_id} not found";
            }
        }

        $updated = count( $updated_products );

        if ( ! empty( $errors ) ) {
            wp_send_json_error( [
                'message' => 'Some products failed to update',
                'errors' => $errors,
                'updated' => $updated
            ] );
        }

        wp_send_json_success( [
            'message' => 'Products updated successfully',
            'updated' => $updated
        ] );
    }

    /**
     * Add this method to class-qcnc-admin.php
     * Inside the QCNC_Admin class
     */

    /**
     * Enqueue alert test scripts
     */
    public function enqueue_alert_test_scripts( $hook ) {
        // Only load on plugin pages
        if ( strpos( $hook, 'qcnc-' ) === false ) {
            return;
        }

        // Enqueue JavaScript
        wp_enqueue_script(
            'qcnc-alert-test',
            QCNC_PLUGIN_URL . 'includes/admin/js/qcnc-alert-test.js',
            array( 'jquery' ),
            QCNC_VERSION,
            true
        );

        // Localize script with AJAX URL and nonce
        wp_localize_script(
            'qcnc-alert-test',
            'qcncAlertTest',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'qcnc_test_alert' )
            )
        );
    }

    /**
     * Render dead stock analysis page
     */
    public function render_dead_stock_page() {
        require_once QCNC_PLUGIN_DIR . 'includes/admin/views/dead-stock-analysis.php';
    }

    /**
     * Render Customer CLV page
     */
    public function render_clv_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Enqueue CLV specific assets
        wp_enqueue_style( 
            'qcnc-clv-dashboard', 
            QCNC_ASSETS_URL . 'css/clv-dashboard.css', 
            array(), 
            QCNC_VERSION 
        );

        wp_enqueue_script( 
            'qcnc-clv-dashboard', 
            QCNC_ASSETS_URL . 'js/clv-dashboard.js', 
            array( 'jquery' ), 
            QCNC_VERSION, 
            true 
        );

        // Localize script
        wp_localize_script( 'qcnc-clv-dashboard', 'qcncClvData', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'qcnc_ajax_nonce' ),
            'currency' => get_woocommerce_currency_symbol(),
            'currencyCode' => get_woocommerce_currency(),
            'tierThresholds' => get_option( 'qcnc_clv_tier_thresholds', array(
                'vip' => 10000,
                'gold' => 5000,
                'silver' => 1000,
                'bronze' => 0
            ) )
        ) );

        // Include the view template
        include_once QCNC_PLUGIN_DIR . 'includes/admin/views/customer-clv.php';
    }

    /**
     * ALSO ADD THESE HELPER METHODS TO THE Admin CLASS:
     */

    /**
     * Get tier badge HTML
     * 
     * @param string $tier
     * @return string
     */
    public static function get_tier_badge( $tier ) {
        $colors = array(
            'vip' => '#9333EA',
            'gold' => '#F59E0B',
            'silver' => '#6B7280',
            'bronze' => '#92400E'
        );

        $color = isset( $colors[ $tier ] ) ? $colors[ $tier ] : '#6B7280';
        $label = strtoupper( $tier );

        return sprintf( 
            '<span class="qcnc-tier-badge" style="background-color: %s; color: white; padding: 2px 8px; border-radius: 4px; font-size: 11px; font-weight: 600;">%s</span>',
            esc_attr( $color ),
            esc_html( $label )
        );
    }

    /**
     * Get churn risk badge HTML
     * 
     * @param string $level
     * @param float $score
     * @return string
     */
    public static function get_churn_risk_badge( $level, $score = 0 ) {
        $colors = array(
            'high' => '#DC2626',
            'medium' => '#F59E0B',
            'low' => '#10B981'
        );

        $color = isset( $colors[ $level ] ) ? $colors[ $level ] : '#6B7280';
        $label = ucfirst( $level );

        if ( $score > 0 ) {
            $label .= ' (' . round( $score ) . '%)';
        }

        return sprintf( 
            '<span class="qcnc-churn-badge" style="background-color: %s; color: white; padding: 2px 8px; border-radius: 4px; font-size: 11px; font-weight: 600;">%s</span>',
            esc_attr( $color ),
            esc_html( $label )
        );
    }

    /**
     * Format CLV value
     * 
     * @param float $value
     * @return string
     */
    public static function format_clv_value( $value ) {
        return wc_price( $value );
    }

    /**
     * Calculate CAC to CLV ratio color
     * 
     * @param float $ratio
     * @return string
     */
    public static function get_cac_clv_ratio_color( $ratio ) {
        if ( $ratio >= 3 ) {
            return '#10B981'; // Green - Excellent
        } elseif ( $ratio >= 2 ) {
            return '#F59E0B'; // Yellow - Good
        } else {
            return '#DC2626'; // Red - Poor
        }
    }

    /**
     * Render Advanced Reports (Enterprise) page.
     */
    public function render_advanced_reports_page() {
        // Make core services available to the view
        $core = \QuarkcodeNeuralCommerce\QCNC_Core::instance();

        $qcnc_report_generator = $core->report_generator;
        $qcnc_security         = $core->security;

        // ✅ FIXED: Correct path to reports-advanced.php
        $view_file = QCNC_INCLUDES_DIR . 'admin/views/reports-advanced.php';

        if ( file_exists( $view_file ) ) {
            include $view_file;
        } else {
            echo '<div class="notice notice-error"><p>';
            esc_html_e( 'Advanced Reports view file not found at: ', 'quarkcode-neuralcommerce-lite' );
            echo esc_html( $view_file );
            echo '</p></div>';
        }
    }

    /**
     * Render Upgrade to Pro page (LITE version) - Enhanced Version
     */
    public function render_upgrade_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Pro features list for upgrade page (defined inline since we removed the restrictions class)
        $features = array(
            array(
                'title' => __( 'Unlimited Products', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Track costs for unlimited products', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Advanced Dashboard', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Interactive charts, heatmaps, and real-time insights', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'AI-Powered Insights', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Get AI recommendations and profit forecasting', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Bulk Operations', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Bulk import/export costs via CSV', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Customer Lifetime Value', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Track CLV, customer segments, and churn risk', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Dead Stock Analysis', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Identify slow-moving inventory automatically', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Advanced Reports', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Detailed profit analysis with multiple dimensions', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'SMS Alerts', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Real-time margin alerts via SMS', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Custom Cost Rules', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Add warehousing, labor, and packaging costs', 'quarkcode-neuralcommerce-lite' ),
            ),
            array(
                'title' => __( 'Priority Support', 'quarkcode-neuralcommerce-lite' ),
                'desc'  => __( 'Get help when you need it', 'quarkcode-neuralcommerce-lite' ),
            ),
        );
        $upgrade_url = apply_filters( 'qcnc_lite_upgrade_url', 'https://quarkcodeai.com/product/quarkcode-neural-commerce/' );
        // Styles are enqueued via wp_enqueue_style in enqueue_assets() method
        ?>
        <div class="wrap qcnc-upgrade-page">
            
            <!-- Hero Section (Smaller) -->
            <div class="qcnc-upgrade-hero" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 40px 30px; border-radius: 16px; margin: 20px 0; text-align: center; box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);">
                <span style="background: rgba(255,255,255,0.2); padding: 6px 14px; border-radius: 20px; font-size: 12px; display: inline-block; margin-bottom: 15px;">
                    ⚡ TRUSTED BY 500+ WOOCOMMERCE STORES
                </span>
                <h1 style="color: white; font-size: 36px; margin: 15px 0; line-height: 1.2;">
                    <?php esc_html_e( 'Transform Your Store with', 'quarkcode-neuralcommerce-lite' ); ?><br/>
                    <span style="background: linear-gradient(to right, #ffd700, #ffed4e); -webkit-background-clip: text; -webkit-text-fill-color: transparent; background-clip: text;">
                        <?php esc_html_e( 'Neural Commerce Pro', 'quarkcode-neuralcommerce-lite' ); ?>
                    </span>
                </h1>
                <p style="font-size: 18px; margin: 0 0 30px 0; opacity: 0.95; max-width: 700px; margin-left: auto; margin-right: auto;">
                    <?php esc_html_e( 'Stop guessing. Start knowing. Make data-driven decisions that increase your profits by up to 40%', 'quarkcode-neuralcommerce-lite' ); ?>
                </p>
                <a href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" class="button button-hero" style="background: white; color: #667eea; font-size: 20px; padding: 15px 40px; height: auto; line-height: 1.5; border: none; box-shadow: 0 8px 16px rgba(0,0,0,0.2); border-radius: 30px; font-weight: 600; margin-right: 15px;">
                    <?php esc_html_e( '🚀 Get Pro Version', 'quarkcode-neuralcommerce-lite' ); ?>
                </a>
                <a href="<?php echo esc_url( $upgrade_url . '#pricing' ); ?>" target="_blank" class="button button-hero" style="background: transparent; color: white; font-size: 16px; padding: 14px 35px; height: auto; line-height: 1.5; border: 2px solid white; box-shadow: none; border-radius: 30px;">
                    <?php esc_html_e( 'View Pricing', 'quarkcode-neuralcommerce-lite' ); ?>
                </a>
            </div>

            <!-- Social Proof Stats (Fixed CSS) -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 40px 0;">
                <div class="qcnc-stats-box">
                    <span class="qcnc-stats-number">500+</span>
                    <span><?php esc_html_e( 'Active Users', 'quarkcode-neuralcommerce-lite' ); ?></span>
                </div>
                <div class="qcnc-stats-box">
                    <span class="qcnc-stats-number">$2M+</span>
                    <span><?php esc_html_e( 'Profits Tracked', 'quarkcode-neuralcommerce-lite' ); ?></span>
                </div>
                <div class="qcnc-stats-box">
                    <span class="qcnc-stats-number">40%</span>
                    <span><?php esc_html_e( 'Avg. Profit Increase', 'quarkcode-neuralcommerce-lite' ); ?></span>
                </div>
                <div class="qcnc-stats-box">
                    <span class="qcnc-stats-number">4.9★</span>
                    <span><?php esc_html_e( 'Customer Rating', 'quarkcode-neuralcommerce-lite' ); ?></span>
                </div>
            </div>

            <!-- Why Upgrade Section -->
            <div style="background: #f8f9fa; padding: 40px; border-radius: 12px; margin: 40px 0;">
                <h2 style="text-align: center; font-size: 36px; margin-bottom: 40px;">
                    <?php esc_html_e( '💡 What\'s in Neural Commerce Pro?', 'quarkcode-neuralcommerce-lite' ); ?>
                </h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 30px;">
                    
                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08); transition: transform 0.2s;">
                        <div class="qcnc-feature-icon">📊</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'AI-Powered Insights', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Get instant AI recommendations on pricing, inventory, and profit optimization based on your store data.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>

                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
                        <div class="qcnc-feature-icon">📈</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'Advanced Analytics', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Access unlimited historical data, customer lifetime value tracking, and profit forecasting tools.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>

                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
                        <div class="qcnc-feature-icon">⚡</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'Bulk Operations', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Import/export thousands of product costs in seconds. Save hours of manual data entry.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>

                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
                        <div class="qcnc-feature-icon">👥</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'Customer Insights', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Track customer lifetime value, identify your best customers, and predict churn risk.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>

                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
                        <div class="qcnc-feature-icon">🎯</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'Dead Stock Detection', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Automatically identify slow-moving inventory before it becomes a cash flow problem.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>

                    <div style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.08);">
                        <div class="qcnc-feature-icon">🏆</div>
                        <h3 style="color: #667eea; margin: 0 0 10px 0; font-size: 20px;"><?php esc_html_e( 'Priority Support', 'quarkcode-neuralcommerce-lite' ); ?></h3>
                        <p style="color: #666; margin: 0; line-height: 1.6;"><?php esc_html_e( 'Get help when you need it with priority email and chat support from our expert team.', 'quarkcode-neuralcommerce-lite' ); ?></p>
                    </div>
                </div>
            </div>

            <!-- Feature Comparison Table (Complete) -->
            <div class="qcnc-comparison-table" style="margin: 60px 0;">
                <h2 style="text-align: center; font-size: 36px; margin-bottom: 30px;">
                    <?php esc_html_e( '📋 Lite vs Pro Features', 'quarkcode-neuralcommerce-lite' ); ?>
                </h2>
                <table class="wp-list-table widefat fixed striped" style="margin-top: 20px; font-size: 15px;">
                    <thead>
                        <tr style="background: #667eea;">
                            <th style="width: 50%; color: white; padding: 15px;"><?php esc_html_e( 'Feature', 'quarkcode-neuralcommerce-lite' ); ?></th>
                            <th style="text-align: center; width: 25%; color: white;"><?php esc_html_e( 'Lite', 'quarkcode-neuralcommerce-lite' ); ?></th>
                            <th style="text-align: center; width: 25%; background: #5568d3; color: white;"><?php esc_html_e( 'Pro', 'quarkcode-neuralcommerce-lite' ); ?></th>
                        </tr>
                    </thead>
                    <tbody style="font-size: 14px;">
                        <tr>
                            <td><strong><?php esc_html_e( 'Products with Cost Tracking', 'quarkcode-neuralcommerce-lite' ); ?></strong></td>
                            <td style="text-align: center;">♾️ Unlimited</td>
                            <td style="text-align: center; background: #f0f9ff;"><strong>♾️ Unlimited</strong></td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e( 'Historical Data', 'quarkcode-neuralcommerce-lite' ); ?></strong></td>
                            <td style="text-align: center;">♾️ Unlimited</td>
                            <td style="text-align: center; background: #f0f9ff;"><strong style="color: #10b981;">♾️ Unlimited</strong></td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e( 'Order Profitability Display', 'quarkcode-neuralcommerce-lite' ); ?></strong></td>
                            <td style="text-align: center;">♾️ Unlimited</td>
                            <td style="text-align: center; background: #f0f9ff;"><strong>♾️ Unlimited</strong></td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e( 'Margin Alerts', 'quarkcode-neuralcommerce-lite' ); ?></strong></td>
                            <td style="text-align: center;">♾️ Unlimited</td>
                            <td style="text-align: center; background: #f0f9ff;"><strong>♾️ Unlimited</strong></td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e( 'Priority Support', 'quarkcode-neuralcommerce-lite' ); ?></strong></td>
                            <td style="text-align: center;">Community</td>
                            <td style="text-align: center; background: #f0f9ff;"><strong style="color: #10b981;">✅ Priority</strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <!-- Testimonials -->
            <div style="margin: 60px 0;">
                <h2 style="text-align: center; font-size: 36px; margin-bottom: 40px;">
                    <?php esc_html_e( '💬 What Our Customers Say', 'quarkcode-neuralcommerce-lite' ); ?>
                </h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                    <div class="qcnc-testimonial">
                        <p style="font-style: italic; color: #444; margin: 0 0 15px 0;">"Neural Commerce Pro helped us identify unprofitable products and increase our margins by 35% in just 2 months!"</p>
                        <strong style="color: #667eea;">- Sarah K.</strong>, Online Retailer
                    </div>
                    <div class="qcnc-testimonial">
                        <p style="font-style: italic; color: #444; margin: 0 0 15px 0;">"The AI insights are game-changing. It's like having a data analyst on staff 24/7."</p>
                        <strong style="color: #667eea;">- Mike T.</strong>, E-commerce Manager
                    </div>
                    <div class="qcnc-testimonial">
                        <p style="font-style: italic; color: #444; margin: 0 0 15px 0;">"Best investment we've made. ROI was positive within the first month!"</p>
                        <strong style="color: #667eea;">- Jessica M.</strong>, Store Owner
                    </div>
                </div>
            </div>

            <!-- Screenshot Gallery Section -->
            <?php
            // Get screenshots from the folder
            $screenshots_dir = QCNC_PLUGIN_DIR . 'assets/images/screenshots/';
            $screenshots_url = QCNC_ASSETS_URL . 'images/screenshots/';
            $screenshots = [];
            
            if ( is_dir( $screenshots_dir ) ) {
                $files = glob( $screenshots_dir . '*.{jpg,jpeg,png,gif,webp}', GLOB_BRACE );
                foreach ( $files as $file ) {
                    $filename = basename( $file );
                    // Create caption from filename (remove extension and replace - or _ with space)
                    $caption = str_replace( [ '-', '_' ], ' ', pathinfo( $filename, PATHINFO_FILENAME ) );
                    $caption = ucwords( $caption );
                    
                    $screenshots[] = [
                        'url' => $screenshots_url . $filename,
                        'caption' => $caption
                    ];
                }
            }
            
            if ( ! empty( $screenshots ) ) :
            ?>
            <div style="margin: 60px 0;">
                <h2 style="text-align: center; font-size: 36px; margin-bottom: 20px;">
                    <?php esc_html_e( '📸 See Neural Commerce Pro in Action', 'quarkcode-neuralcommerce-lite' ); ?>
                </h2>
                <p style="text-align: center; color: #666; font-size: 16px; margin-bottom: 30px;">
                    <?php esc_html_e( 'Click any screenshot to view full size', 'quarkcode-neuralcommerce-lite' ); ?>
                </p>
                
                <div class="qcnc-screenshot-grid">
                    <?php foreach ( $screenshots as $index => $screenshot ) : ?>
                        <div class="qcnc-screenshot-item" onclick="qcncOpenLightbox(<?php echo esc_attr( $index ); ?>)">
                            <img src="<?php echo esc_url( $screenshot['url'] ); ?>" alt="<?php echo esc_attr( $screenshot['caption'] ); ?>" loading="lazy" />
                            <div class="qcnc-screenshot-caption">
                                <?php echo esc_html( $screenshot['caption'] ); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Lightbox Modal -->
            <div id="qcnc-lightbox" class="qcnc-lightbox" onclick="qcncCloseLightbox()">
                <span class="qcnc-lightbox-close" onclick="qcncCloseLightbox()">&times;</span>
                <div class="qcnc-lightbox-content" onclick="event.stopPropagation()">
                    <img id="qcnc-lightbox-img" src="" alt="" />
                    <div id="qcnc-lightbox-caption" class="qcnc-lightbox-caption"></div>
                </div>
            </div>

            <?php
            // Pass screenshot data to the enqueued script using wp_add_inline_script
            wp_add_inline_script(
                'qcnc-upgrade-page',
                'var qcncScreenshots = ' . wp_json_encode( $screenshots ) . ';',
                'before'
            );
            endif;
            ?>

            <!-- CTA Section -->
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 60px 40px; border-radius: 16px; text-align: center; margin: 60px 0;">
                <h2 style="color: white; font-size: 40px; margin: 0 0 20px 0;">
                    <?php esc_html_e( '🎯 Ready to Maximize Your Profits?', 'quarkcode-neuralcommerce-lite' ); ?>
                </h2>
                <p style="font-size: 20px; margin: 0 0 30px 0; opacity: 0.9;">
                    <?php esc_html_e( 'Join 500+ successful WooCommerce stores using Neural Commerce Pro', 'quarkcode-neuralcommerce-lite' ); ?>
                </p>
                <a href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" class="button button-hero" style="background: white; color: #667eea; font-size: 22px; padding: 20px 50px; height: auto; line-height: 1.5; border: none; box-shadow: 0 8px 16px rgba(0,0,0,0.3); border-radius: 30px; font-weight: 600;">
                    <?php esc_html_e( '🚀 Get Pro Version Now', 'quarkcode-neuralcommerce-lite' ); ?>
                </a>
                <p style="margin: 20px 0 0 0; opacity: 0.8; font-size: 14px;">
                    <?php esc_html_e( '✓ 30-Day Money-Back Guarantee  ✓ Instant Access  ✓ Priority Support', 'quarkcode-neuralcommerce-lite' ); ?>
                </p>
            </div>
        </div>
        <?php
    }






}