<?php
/**
 * Admin AJAX Handler
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

namespace QuarkcodeNeuralCommerce\Admin;

use QuarkcodeNeuralCommerce\Core\QCNC_Report_Generator;
use QuarkcodeNeuralCommerce\Core\QCNC_AI_Insights_Engine;
use QuarkcodeNeuralCommerce\Core\QCNC_Product_Cost_Manager;
use QuarkcodeNeuralCommerce\Core\QCNC_Order_Profit_Calculator;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Security;
use QuarkcodeNeuralCommerce\Utilities\QCNC_Logger;
use QuarkcodeNeuralCommerce\Utilities\QCNC_CSV_Handler;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Handles AJAX requests for admin operations.
 */
class QCNC_Admin_Ajax {

    /**
     * Product cost manager instance.
     *
     * @var QCNC_Product_Cost_Manager
     */
    private $cost_manager;

    /**
     * Order profit calculator instance.
     *
     * @var QCNC_Order_Profit_Calculator
     */
    private $calculator;

    /**
     * Security instance.
     *
     * @var QCNC_Security
     */
    private $security;

    /**
     * Logger instance.
     *
     * @var QCNC_Logger
     */
    private $logger;

    /**
     * Constructor.
     *
     * @param QCNC_Product_Cost_Manager    $cost_manager Cost manager.
     * @param QCNC_Order_Profit_Calculator $calculator   Calculator.
     * @param QCNC_Security                $security     Security.
     * @param QCNC_Logger                  $logger       Logger.
     */
    public function __construct( 
        QCNC_Product_Cost_Manager $cost_manager,
        QCNC_Order_Profit_Calculator $calculator,
        QCNC_Security $security,
        QCNC_Logger $logger
    ) {
        $this->cost_manager = $cost_manager;
        $this->calculator   = $calculator;
        $this->security     = $security;
        $this->logger       = $logger;

        $this->register_ajax_actions();
    }

    /**
     * Register AJAX actions.
     */
    private function register_ajax_actions() {
        add_action( 'wp_ajax_qcnc_update_product_cost', [ $this, 'update_product_cost' ] );
        add_action( 'wp_ajax_qcnc_get_product_cost', [ $this, 'get_product_cost' ] );
        add_action( 'wp_ajax_qcnc_bulk_import_costs', [ $this, 'bulk_import_costs' ] );
        add_action( 'wp_ajax_qcnc_export_costs', [ $this, 'export_costs' ] );
        add_action( 'wp_ajax_qcnc_recalculate_order', [ $this, 'recalculate_order' ] );
        add_action( 'wp_ajax_qcnc_get_product_costs_list', [ $this, 'get_product_costs_list' ] );
        add_action( 'wp_ajax_qcnc_delete_cost', [ $this, 'delete_cost' ] );
        add_action( 'wp_ajax_qcnc_get_orders_profit', [ $this, 'get_orders_profit' ] );
        add_action( 'wp_ajax_qcnc_get_products_list', [ $this, 'get_products_list' ] );
        add_action( 'wp_ajax_qcnc_get_product_details', [ $this, 'get_product_details' ] );
        add_action( 'wp_ajax_qcnc_get_order_count', [ $this, 'get_order_count' ] );
        add_action( 'wp_ajax_qcnc_start_bulk_processing', [ $this, 'start_bulk_processing' ] );
        add_action( 'wp_ajax_qcnc_process_order_batch', [ $this, 'process_order_batch' ] );
        add_action( 'wp_ajax_qcnc_clear_dashboard_cache', [ $this, 'clear_dashboard_cache' ] );
        add_action( 'wp_ajax_qcnc_reset_profit_data', [ $this, 'reset_profit_data' ] );
        add_action( 'wp_ajax_qcnc_get_order_details', [ $this, 'get_order_details' ] );
        add_action( 'wp_ajax_qcnc_get_gateway_fee', [ $this, 'get_gateway_fee' ] );
        add_action( 'wp_ajax_qcnc_save_gateway_fee', [ $this, 'save_gateway_fee' ] );
        add_action( 'wp_ajax_qcnc_delete_gateway_fee', [ $this, 'delete_gateway_fee' ] );
        add_action( 'wp_ajax_qcnc_get_cost_history', [ $this, 'get_cost_history' ] );
        add_action( 'wp_ajax_qcnc_cleanup_duplicate_costs', [ $this, 'cleanup_duplicate_costs' ] );
        // Add new advanced dashboard actions
        add_action('wp_ajax_qcnc_get_advanced_dashboard_data', array($this, 'get_advanced_dashboard_data'));
        add_action('wp_ajax_qcnc_get_profit_forecast', array($this, 'get_profit_forecast'));
        add_action('wp_ajax_qcnc_get_realtime_stats', array($this, 'get_realtime_stats'));
        add_action('wp_ajax_qcnc_export_dashboard', array($this, 'export_dashboard'));
        
        add_action( 'wp_ajax_qcnc_save_products_inline', array( $this, 'save_products_inline' ) );
        
        // Calculate all profits
        add_action( 'wp_ajax_qcnc_calculate_all_profits', array( $this, 'calculate_all_profits' ) );
        add_action( 'wp_ajax_qcnc_send_digest_now', array( $this, 'send_digest_now' ) );


    }

    /**
     * Calculate profit data for all orders via AJAX.
     */
    public function calculate_all_profits() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        global $wpdb;

        // Get all completed and processing orders
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $order_ids = $wpdb->get_col(
            "SELECT id FROM {$wpdb->prefix}wc_orders 
            WHERE type = 'shop_order'
            AND status IN ('wc-completed', 'wc-processing')
            ORDER BY id DESC
            LIMIT 100"
        );

        // Fallback for legacy orders table
        if ( empty( $order_ids ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $order_ids = $wpdb->get_col(
                "SELECT ID FROM {$wpdb->posts} 
                WHERE post_type = 'shop_order' 
                AND post_status IN ('wc-completed', 'wc-processing')
                ORDER BY ID DESC
                LIMIT 100"
            );
        }

        if ( empty( $order_ids ) ) {
            $this->security->json_error( __( 'No orders found to process', 'quarkcode-neuralcommerce-lite' ) );
        }

        $processed = 0;
        $errors = 0;

        foreach ( $order_ids as $order_id ) {
            try {
                $order = wc_get_order( $order_id );
                if ( $order && ! ( $order instanceof \WC_Order_Refund ) ) {
                    $result = $this->calculator->calculate_order_profit( $order_id );
                    if ( $result ) {
                        $processed++;
                    } else {
                        $errors++;
                    }
                }
            } catch ( \Exception $e ) {
                $this->logger->log( 'Error calculating profit for order #' . $order_id . ': ' . $e->getMessage(), 'error' );
                $errors++;
            }
        }

        $this->security->json_success( array(
            'message' => sprintf(
                /* translators: 1: Number of orders processed, 2: Number of errors */
                __( 'Successfully calculated profit for %1$d orders (%2$d errors)', 'quarkcode-neuralcommerce-lite' ),
                $processed,
                $errors
            ),
            'processed' => $processed,
            'errors' => $errors,
            'total' => count( $order_ids ),
        ) );
    }

    /**
     * Update product cost via AJAX.
     */
    public function update_product_cost() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $cost = isset( $_POST['cost'] ) ? floatval( $_POST['cost'] ) : 0.0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $packaging_cost = isset( $_POST['packaging_cost'] ) ? floatval( $_POST['packaging_cost'] ) : 0.0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $handling_cost = isset( $_POST['handling_cost'] ) ? floatval( $_POST['handling_cost'] ) : 0.0;

        if ( ! $product_id ) {
            $this->security->json_error( __( 'Invalid product ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Collect additional costs
        $additional_costs = [
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'warehousing_cost'       => isset( $_POST['warehousing_cost'] ) ? floatval( $_POST['warehousing_cost'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'labor_cost'             => isset( $_POST['labor_cost'] ) ? floatval( $_POST['labor_cost'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'shipping_material_cost' => isset( $_POST['shipping_material_cost'] ) ? floatval( $_POST['shipping_material_cost'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_1'          => isset( $_POST['custom_cost_1'] ) ? floatval( $_POST['custom_cost_1'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_1_label'    => isset( $_POST['custom_cost_1_label'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_cost_1_label'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_2'          => isset( $_POST['custom_cost_2'] ) ? floatval( $_POST['custom_cost_2'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_2_label'    => isset( $_POST['custom_cost_2_label'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_cost_2_label'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_3'          => isset( $_POST['custom_cost_3'] ) ? floatval( $_POST['custom_cost_3'] ) : 0.0,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'custom_cost_3_label'    => isset( $_POST['custom_cost_3_label'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_cost_3_label'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'notes'                  => isset( $_POST['notes'] ) ? sanitize_textarea_field( wp_unslash( $_POST['notes'] ) ) : '',
        ];

        $result = $this->cost_manager->set_product_cost(
            $product_id,
            $cost,
            null,
            $packaging_cost,
            $handling_cost,
            '',
            $additional_costs
        );

        // Check for WP_Error from validation
        if ( is_wp_error( $result ) ) {
            $this->security->json_error( $result->get_error_message() );
            return;
        }

        if ( $result ) {
            $this->security->json_success(
                array(
                    'message' => __( 'Cost updated successfully', 'quarkcode-neuralcommerce-lite' ),
                    'cost_id' => $result,
                )
            );
        } else {
            $this->security->json_error( __( 'Failed to update cost', 'quarkcode-neuralcommerce-lite' ) );
        }

    }


    /**
     * Get product cost via AJAX.
     */
    public function get_product_cost() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;

        if ( ! $product_id ) {
            $this->security->json_error( __( 'Invalid product ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        $cost_data = $this->cost_manager->get_product_cost( $product_id );

        if ( $cost_data ) {
            $this->security->json_success( $cost_data );
        } else {
            $this->security->json_success( [
                'cost'           => 0,
                'packaging_cost' => 0,
                'handling_cost'  => 0,
                'total_cost'     => 0,
            ] );
        }
    }

    /**
     * Bulk import costs from CSV via AJAX.
     */
    public function bulk_import_costs() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // ✅ FIXED: Validate and sanitize file upload
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( ! isset( $_FILES['csv_file'] ) ) {
            $this->security->json_error( __( 'No file uploaded', 'quarkcode-neuralcommerce-lite' ) );
        }

        // ✅ FIXED: Sanitize file array elements
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $file = array(
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'name'     => isset( $_FILES['csv_file']['name'] ) ? sanitize_file_name( wp_unslash( $_FILES['csv_file']['name'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'type'     => isset( $_FILES['csv_file']['type'] ) ? sanitize_mime_type( wp_unslash( $_FILES['csv_file']['type'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'tmp_name' => isset( $_FILES['csv_file']['tmp_name'] ) ? sanitize_text_field( wp_unslash( $_FILES['csv_file']['tmp_name'] ) ) : '',
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'error'    => isset( $_FILES['csv_file']['error'] ) ? intval( $_FILES['csv_file']['error'] ) : UPLOAD_ERR_NO_FILE,
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'size'     => isset( $_FILES['csv_file']['size'] ) ? intval( $_FILES['csv_file']['size'] ) : 0,
        );

        if ( $file['error'] !== UPLOAD_ERR_OK ) {
            $this->security->json_error( __( 'File upload error', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Validate file exists and is readable
        if ( empty( $file['tmp_name'] ) || ! is_readable( $file['tmp_name'] ) ) {
            $this->security->json_error( __( 'Unable to read uploaded file', 'quarkcode-neuralcommerce-lite' ) );
        }

        $csv_handler = new QCNC_CSV_Handler();
        $file_path = $file['tmp_name'];

        // Parse CSV.
        $data = $csv_handler->parse_csv( $file_path );

        if ( false === $data ) {
            $this->security->json_error( __( 'Failed to parse CSV file', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Validate data.
        $validation = $csv_handler->validate_cost_import( $data );

        if ( ! $validation['valid'] ) {
            $this->security->json_error( array(
                'message' => __( 'CSV validation failed', 'quarkcode-neuralcommerce-lite' ),
                'errors'  => $validation['errors'],
            ) );
        }

        // Import costs.
        $results = $this->cost_manager->bulk_update_costs( $validation['valid_rows'] );

        $this->security->json_success( array(
            'message' => sprintf(
                /* translators: 1: Number of successful imports, 2: Number of failed imports */
                __( 'Import completed: %1$d succeeded, %2$d failed', 'quarkcode-neuralcommerce-lite' ),
                $results['success'],
                $results['failed']
            ),
            'success' => $results['success'],
            'failed'  => $results['failed'],
            'errors'  => $results['errors'],
        ) );
    }

    /**
     * Export costs to CSV via AJAX.
     */
    public function export_costs() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'Security check failed', 'quarkcode-neuralcommerce-lite' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $costs = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT product_id, variation_id, cost, packaging_cost, handling_cost, currency FROM {$table} 
             WHERE is_current = 1 
             ORDER BY product_id ASC",
            ARRAY_A
        );

        $csv_handler = new QCNC_CSV_Handler();
        $csv_content = $csv_handler->generate_csv( $costs );

        $filename = 'product-costs-' . wp_date( 'Y-m-d' ) . '.csv';
        $csv_handler->download_csv( $csv_content, $filename );
    }

    /**
     * Recalculate order profit via AJAX.
     */
    public function recalculate_order() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;

        if ( ! $order_id ) {
            $this->security->json_error( __( 'Invalid order ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        $result = $this->calculator->recalculate_order_profit( $order_id );

        if ( $result ) {
            $this->security->json_success( [
                'message'     => __( 'Order profit recalculated successfully', 'quarkcode-neuralcommerce-lite' ),
                'profit_data' => $result,
            ] );
        } else {
            $this->security->json_error( __( 'Failed to recalculate order profit', 'quarkcode-neuralcommerce-lite' ) );
        }
    }

    /**
     * Get product costs list with pagination via AJAX.
     */
    public function get_product_costs_list() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $page = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $per_page = isset( $_POST['per_page'] ) ? absint( $_POST['per_page'] ) : 20;
        $offset = ( $page - 1 ) * $per_page;

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE is_current = 1" );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $costs = $wpdb->get_results(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE is_current = 1 ORDER BY product_id ASC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        // Enrich with product data.
        foreach ( $costs as &$cost ) {
            $product_id = $cost['variation_id'] ? $cost['variation_id'] : $cost['product_id'];
            $product = wc_get_product( $product_id );

            if ( $product ) {
                $cost['product_name'] = $product->get_name();
                $cost['sku'] = $product->get_sku();
                $cost['price'] = $product->get_price();
            }
        }

        $this->security->json_success( [
            'costs'       => $costs,
            'total'       => $total,
            'page'        => $page,
            'per_page'    => $per_page,
            'total_pages' => ceil( $total / $per_page ),
        ] );
    }

    /**
     * Delete product cost via AJAX.
     */
    public function delete_cost() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $cost_id = isset( $_POST['cost_id'] ) ? absint( $_POST['cost_id'] ) : 0;

        if ( ! $cost_id ) {
            $this->security->json_error( __( 'Invalid cost ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        $result = $this->cost_manager->delete_cost( $cost_id );

        if ( $result ) {
            $this->security->json_success( [
                'message' => __( 'Cost deleted successfully', 'quarkcode-neuralcommerce-lite' ),
            ] );
        } else {
            $this->security->json_error( __( 'Failed to delete cost', 'quarkcode-neuralcommerce-lite' ) );
        }
    }

    /**
     * Get orders with profit data via AJAX.
     */
    public function get_orders_profit() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $page = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $per_page = isset( $_POST['per_page'] ) ? absint( $_POST['per_page'] ) : 20;
        $offset = ( $page - 1 ) * $per_page;

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $orders = $wpdb->get_results(
            $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} ORDER BY calculated_at DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        // Enrich with order data.
        foreach ( $orders as &$order_data ) {
            $order = wc_get_order( $order_data['order_id'] );

            if ( $order ) {
                $order_data['order_number'] = $order->get_order_number();
                $order_data['order_date'] = $order->get_date_created()->date( 'Y-m-d H:i:s' );
                $order_data['order_status'] = $order->get_status();
                $order_data['customer_name'] = $order->get_formatted_billing_full_name();
            }
        }

        $this->security->json_success( [
            'orders'      => $orders,
            'total'       => $total,
            'page'        => $page,
            'per_page'    => $per_page,
            'total_pages' => ceil( $total / $per_page ),
        ] );
    }

    /**
     * Get detailed product information via AJAX.
     */
    public function get_product_details() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;

        if ( ! $product_id ) {
            $this->security->json_error( __( 'Invalid product ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            $this->security->json_error( __( 'Product not found', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Get existing cost data
        $existing_cost = $this->cost_manager->get_product_cost( $product_id );

        // Get currency symbol
        $currency_symbol = get_woocommerce_currency_symbol();

        // Prepare product data
        $product_data = [
            'id'                     => $product_id,
            'name'                   => $product->get_name(),
            'sku'                    => $product->get_sku(),
            'type'                   => $product->get_type(),
            'regular_price'          => $product->get_regular_price(),
            'sale_price'             => $product->get_sale_price(),
            'price'                  => $product->get_price(),
            'regular_price_formatted' => $currency_symbol . number_format( (float) $product->get_regular_price(), 2 ),
            'sale_price_formatted'   => $product->get_sale_price() ? $currency_symbol . number_format( (float) $product->get_sale_price(), 2 ) : '',
            'price_formatted'        => $currency_symbol . number_format( (float) $product->get_price(), 2 ),
            'currency'               => get_woocommerce_currency(),
            'existing_cost'          => $existing_cost ?: null,
        ];

        $this->security->json_success( $product_data );
    }

    /**
     * Get order count statistics via AJAX.
     */
    public function get_order_count() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        global $wpdb;
        
        // Total orders (excluding refunds)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $total = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders 
            WHERE type = 'shop_order'
            AND status IN ('wc-completed', 'wc-processing')"
        );
        
        // If using legacy orders table
        if ( ! $total ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $total = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->posts} 
                WHERE post_type = 'shop_order' 
                AND post_status IN ('wc-completed', 'wc-processing')"
            );
        }

        // Processed orders
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $processed = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}qcnc_order_profit"
        );

        $this->security->json_success( [
            'total'     => (int) $total,
            'processed' => (int) $processed,
            'missing'   => (int) ( $total - $processed ),
        ] );
    }

    /**
     * Start bulk order processing via AJAX.
     */
    public function start_bulk_processing() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // Triggers bulk operation mode in alert system
        do_action( 'qcnc_before_bulk_profit_calculation' );

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $mode = isset( $_POST['mode'] ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : 'all';
        
        global $wpdb;
        
        // Get total orders to process
        if ( 'missing' === $mode ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $total = $wpdb->get_var(
                "SELECT COUNT(DISTINCT o.id) 
                FROM {$wpdb->prefix}wc_orders o
                LEFT JOIN {$wpdb->prefix}qcnc_order_profit p ON o.id = p.order_id
                WHERE o.type = 'shop_order'
                AND o.status IN ('wc-completed', 'wc-processing')
                AND p.order_id IS NULL"
            );
            
            // Fallback for legacy orders
            if ( ! $total ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $total = $wpdb->get_var(
                    "SELECT COUNT(DISTINCT p.ID) 
                    FROM {$wpdb->posts} p
                    LEFT JOIN {$wpdb->prefix}qcnc_order_profit op ON p.ID = op.order_id
                    WHERE p.post_type = 'shop_order'
                    AND p.post_status IN ('wc-completed', 'wc-processing')
                    AND op.order_id IS NULL"
                );
            }
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $total = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders 
                WHERE type = 'shop_order'
                AND status IN ('wc-completed', 'wc-processing')"
            );
            
            if ( ! $total ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $total = $wpdb->get_var(
                    "SELECT COUNT(*) FROM {$wpdb->posts} 
                    WHERE post_type = 'shop_order' 
                    AND post_status IN ('wc-completed', 'wc-processing')"
                );
            }
        }

        $this->security->json_success( [
            'total'      => (int) $total,
            'batch_size' => 10, // Process 10 orders at a time
            'mode'       => $mode,
        ] );
    }

    /**
     * Process batch of orders via AJAX.
     */
    public function process_order_batch() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $offset     = isset( $_POST['offset'] ) ? absint( $_POST['offset'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $batch_size = isset( $_POST['batch_size'] ) ? absint( $_POST['batch_size'] ) : 10;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $mode       = isset( $_POST['mode'] ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : 'all';

        global $wpdb;

        // Get order IDs to process (excluding refunds)
        if ( 'missing' === $mode ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $order_ids = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT DISTINCT o.id 
                    FROM {$wpdb->prefix}wc_orders o
                    LEFT JOIN {$wpdb->prefix}qcnc_order_profit p ON o.id = p.order_id
                    WHERE o.type = 'shop_order'
                    AND o.status IN ('wc-completed', 'wc-processing')
                    AND p.order_id IS NULL
                    LIMIT %d OFFSET %d",
                    $batch_size,
                    $offset
                )
            );
            
            // Fallback for legacy orders
            if ( empty( $order_ids ) ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
                $order_ids = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT DISTINCT p.ID 
                        FROM {$wpdb->posts} p
                        LEFT JOIN {$wpdb->prefix}qcnc_order_profit op ON p.ID = op.order_id
                        WHERE p.post_type = 'shop_order'
                        AND p.post_status IN ('wc-completed', 'wc-processing')
                        AND op.order_id IS NULL
                        LIMIT %d OFFSET %d",
                        $batch_size,
                        $offset
                    )
                );
            }
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $order_ids = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT id FROM {$wpdb->prefix}wc_orders 
                    WHERE type = 'shop_order'
                    AND status IN ('wc-completed', 'wc-processing')
                    LIMIT %d OFFSET %d",
                    $batch_size,
                    $offset
                )
            );
            
            if ( empty( $order_ids ) ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $order_ids = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT ID FROM {$wpdb->posts} 
                        WHERE post_type = 'shop_order' 
                        AND post_status IN ('wc-completed', 'wc-processing')
                        LIMIT %d OFFSET %d",
                        $batch_size,
                        $offset
                    )
                );
            }
        }

        $processed = 0;
        $errors    = 0;

        // Check if this is the first batch
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $offset = isset( $_POST['offset'] ) ? absint( $_POST['offset'] ) : 0;
        if ( $offset === 0 ) {
            // First batch - enable bulk mode
            do_action( 'qcnc_before_bulk_profit_calculation' );
        }

        foreach ( $order_ids as $order_id ) {
            try {
                // Double-check it's not a refund
                $order = wc_get_order( $order_id );
                if ( $order && ! ( $order instanceof \WC_Order_Refund ) ) {
                    $result = $this->calculator->calculate_order_profit( $order_id );
                    if ( $result ) {
                        $processed++;
                    } else {
                        $errors++;
                    }
                }
            } catch ( \Exception $e ) {
                $this->logger->log( 'Error processing order #' . $order_id . ': ' . $e->getMessage(), 'error' );
                $errors++;
            }
        }

         // Check if this is the last batch
         // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $total = isset( $_POST['total'] ) ? absint( $_POST['total'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $batch_size = isset( $_POST['batch_size'] ) ? absint( $_POST['batch_size'] ) : 10;
        $is_last_batch = ( $offset + $batch_size ) >= $total;

        if ( $is_last_batch ) {
            // Last batch - disable bulk mode and send digest
            do_action( 'qcnc_after_bulk_profit_calculation' );
        }

        $this->security->json_success( [
            'processed'       => $processed,
            'errors'          => $errors,
            'total_processed' => $offset + $processed,
        ] );
    }

    /**
     * Clear dashboard cache via AJAX.
     */
    public function clear_dashboard_cache() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}options 
            WHERE option_name LIKE '_transient_qcnc_%' 
            OR option_name LIKE '_transient_timeout_qcnc_%'"
        );

        $this->security->json_success( [
            'message' => __( 'Cache cleared successfully', 'quarkcode-neuralcommerce-lite' ),
        ] );
    }

    /**
     * Reset all profit data via AJAX.
     */
    public function reset_profit_data() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        global $wpdb;
        
        // Delete all profit data
        $wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}qcnc_order_profit" );
        $wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}qcnc_order_items_profit" );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}options 
            WHERE option_name LIKE '_transient_qcnc_%' 
            OR option_name LIKE '_transient_timeout_qcnc_%'"
        );

        $this->security->json_success( [
            'message' => __( 'All profit data has been deleted successfully', 'quarkcode-neuralcommerce-lite' ),
        ] );
    }

    /**
     * Get order profit details via AJAX.
     */
    public function get_order_details() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;

        if ( ! $order_id ) {
            $this->security->json_error( __( 'Invalid order ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        $order = wc_get_order( $order_id );
        
        if ( ! $order ) {
            $this->security->json_error( __( 'Order not found', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Get profit data
        $profit_data = $this->calculator->get_order_profit( $order_id );
        
        if ( ! $profit_data ) {
            $this->security->json_error( __( 'Profit data not found. Please process this order first.', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Get items breakdown
        $items_data = $this->calculator->get_order_items_profit( $order_id );

        // Format order details
        $details = [
            'order_info' => [
                'order_number'   => $order->get_order_number(),
                'order_date'     => $order->get_date_created()->date( 'F j, Y g:i a' ),
                'order_status'   => wc_get_order_status_name( $order->get_status() ),
                'customer_name'  => $order->get_formatted_billing_full_name(),
                'customer_email' => $order->get_billing_email(),
            ],
            'profit_summary' => [
                'revenue'         => floatval( $profit_data['revenue'] ),
                'cogs_total'      => floatval( $profit_data['cogs_total'] ),
                'shipping_cost'   => floatval( $profit_data['shipping_cost'] ),
                'payment_fee'     => floatval( $profit_data['payment_fee'] ),
                'tax_amount'      => floatval( $profit_data['tax_amount'] ),
                'discount_amount' => floatval( $profit_data['discount_amount'] ),
                'gross_profit'    => floatval( $profit_data['gross_profit'] ),
                'net_profit'      => floatval( $profit_data['net_profit'] ),
                'margin_pct'      => floatval( $profit_data['margin_pct'] ),
                'currency'        => $profit_data['currency'],
            ],
            'items' => [],
        ];

        // Format items
        foreach ( $items_data as $item ) {
            $product_id = $item['variation_id'] ? $item['variation_id'] : $item['product_id'];
            $product = wc_get_product( $product_id );
            
            $details['items'][] = [
                'product_name'   => $product ? $product->get_name() : __( 'Unknown Product', 'quarkcode-neuralcommerce-lite' ),
                'sku'            => $product ? $product->get_sku() : '',
                'quantity'       => intval( $item['quantity'] ),
                'revenue'        => floatval( $item['item_revenue'] ),
                'cogs'           => floatval( $item['item_cogs'] ),
                'profit'         => floatval( $item['item_profit'] ),
                'cost_per_unit'  => floatval( $item['cost_per_unit'] ),
                'margin_pct'     => $item['item_revenue'] > 0 ? 
                                ( floatval( $item['item_profit'] ) / floatval( $item['item_revenue'] ) ) * 100 : 0,
            ];
        }

        $this->security->json_success( $details );
    }

    /**
     * Get gateway fee via AJAX.
     */
    public function get_gateway_fee() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $gateway_id = isset( $_POST['gateway_id'] ) ? absint( $_POST['gateway_id'] ) : 0;

        if ( ! $gateway_id ) {
            $this->security->json_error( __( 'Invalid gateway ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $gateway = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $gateway_id ),
            ARRAY_A
        );

        if ( $gateway ) {
            $this->security->json_success( $gateway );
        } else {
            $this->security->json_error( __( 'Gateway fee not found', 'quarkcode-neuralcommerce-lite' ) );
        }
    }

    /**
     * Save gateway fee via AJAX.
     */
    public function save_gateway_fee() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $id = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $gateway_id = isset( $_POST['gateway_id'] ) ? sanitize_text_field( wp_unslash( $_POST['gateway_id'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $gateway_name = isset( $_POST['gateway_name'] ) ? sanitize_text_field( wp_unslash( $_POST['gateway_name'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $fee_type = isset( $_POST['fee_type'] ) ? sanitize_text_field( wp_unslash( $_POST['fee_type'] ) ) : 'percentage';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $percentage_fee = isset( $_POST['percentage_fee'] ) ? floatval( $_POST['percentage_fee'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $fixed_fee = isset( $_POST['fixed_fee'] ) ? floatval( $_POST['fixed_fee'] ) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $currency = isset( $_POST['currency'] ) ? sanitize_text_field( wp_unslash( $_POST['currency'] ) ) : get_woocommerce_currency();
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $is_active = isset( $_POST['is_active'] ) ? 1 : 0;

        if ( empty( $gateway_id ) || empty( $gateway_name ) ) {
            $this->security->json_error( __( 'Gateway ID and name are required', 'quarkcode-neuralcommerce-lite' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        $data = [
            'gateway_id'      => $gateway_id,
            'gateway_name'    => $gateway_name,
            'fee_type'        => $fee_type,
            'percentage_fee'  => $percentage_fee,
            'fixed_fee'       => $fixed_fee,
            'currency'        => $currency,
            'is_active'       => $is_active,
            'updated_at'      => current_time( 'mysql' ),
        ];

        if ( $id > 0 ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->update(
                $table,
                $data,
                [ 'id' => $id ],
                [ '%s', '%s', '%s', '%f', '%f', '%s', '%d', '%s' ],
                [ '%d' ]
            );
            $message = __( 'Gateway fee updated successfully', 'quarkcode-neuralcommerce-lite' );
        } else {
            // Insert new
            $data['created_at'] = current_time( 'mysql' );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->insert(
                $table,
                $data,
                [ '%s', '%s', '%s', '%f', '%f', '%s', '%d', '%s', '%s' ]
            );
            $message = __( 'Gateway fee added successfully', 'quarkcode-neuralcommerce-lite' );
        }

        if ( false !== $result ) {
            $this->security->json_success( [ 'message' => $message ] );
        } else {
            $this->security->json_error( __( 'Failed to save gateway fee', 'quarkcode-neuralcommerce-lite' ) );
        }
    }

    /**
     * Delete gateway fee via AJAX.
     */
    public function delete_gateway_fee() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $gateway_id = isset( $_POST['gateway_id'] ) ? absint( $_POST['gateway_id'] ) : 0;

        if ( ! $gateway_id ) {
            $this->security->json_error( __( 'Invalid gateway ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_payment_gateway_fees';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->delete( $table, [ 'id' => $gateway_id ], [ '%d' ] );

        if ( $result ) {
            $this->security->json_success( [ 'message' => __( 'Gateway fee deleted successfully', 'quarkcode-neuralcommerce-lite' ) ] );
        } else {
            $this->security->json_error( __( 'Failed to delete gateway fee', 'quarkcode-neuralcommerce-lite' ) );
        }
    }

    /**
     * Get cost history for a product via AJAX.
     */
    public function get_cost_history() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;

        if ( ! $product_id ) {
            $this->security->json_error( __( 'Invalid product ID', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Get product info
        $product = wc_get_product( $product_id );
        
        if ( ! $product ) {
            $this->security->json_error( __( 'Product not found', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Get cost history
        $history = $this->cost_manager->get_cost_history( $product_id, null, 50 );

        if ( empty( $history ) ) {
            $this->security->json_error( __( 'No cost history found for this product', 'quarkcode-neuralcommerce-lite' ) );
        }

        // Enrich with user data
        foreach ( $history as &$item ) {
            if ( $item['created_by'] ) {
                $user = get_userdata( $item['created_by'] );
                $item['created_by'] = $user ? $user->display_name : 'User #' . $item['created_by'];
            }
        }

        $this->security->json_success( [
            'product_id'   => $product_id,
            'product_name' => $product->get_name(),
            'sku'          => $product->get_sku(),
            'history'      => $history,
        ] );
    }

    /**
     * Cleanup duplicate current costs via AJAX.
     */
    public function cleanup_duplicate_costs() {
        if ( ! $this->security->verify_request( 'qcnc_admin_nonce', 'manage_woocommerce' ) ) {
            $this->security->json_error( __( 'Security check failed', 'quarkcode-neuralcommerce-lite' ), 403 );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'qcnc_product_costs';

        // Step 1: Delete invalid historical records where effective_from = effective_to
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $deleted_invalid = $wpdb->query(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "DELETE FROM {$table}
            WHERE effective_from = effective_to
            AND is_current = 0"
        );

        // Step 2: Find products with multiple current costs
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $duplicates = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT product_id, variation_id, COUNT(*) as count FROM {$table} WHERE is_current = 1
            GROUP BY product_id, variation_id
            HAVING count > 1",
            ARRAY_A
        );

        $cleaned = 0;
        foreach ( $duplicates as $dup ) {
            $this->cost_manager->cleanup_duplicate_current_costs( 
                $dup['product_id'], 
                $dup['variation_id'] 
            );
            $cleaned++;
        }

        $this->security->json_success( [
            'message' => sprintf( 
                /* translators: 1: Number of invalid records, 2: Number of products with duplicate current costs */
                __( 'Cleanup complete: Removed %1$d invalid records and fixed %2$d products with duplicate current costs', 'quarkcode-neuralcommerce-lite' ), 
                $deleted_invalid,
                $cleaned 
            ),
        ] );
    }

    /**
     * Get advanced dashboard data
     */
    public function get_advanced_dashboard_data() {
        check_ajax_referer('qcnc_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $date_from = isset($_POST['date_from']) ? sanitize_text_field(wp_unslash($_POST['date_from'])) : wp_date('Y-m-d', strtotime('-30 days'));
        $date_to = isset($_POST['date_to']) ? sanitize_text_field(wp_unslash($_POST['date_to'])) : wp_date('Y-m-d');
        $comparison_period = isset($_POST['comparison_period']) ? sanitize_text_field(wp_unslash($_POST['comparison_period'])) : 'previous_period';

        // ✅ USE CORE SINGLETON INSTEAD OF NEW INSTANCES
        $core = \QuarkcodeNeuralCommerce\QCNC_Core::instance();
        $report_generator = $core->report_generator;
        $profit_calculator = $core->order_profit_calculator;

        // Get KPIs
        $kpis = $report_generator->get_dashboard_metrics($date_from, $date_to);
        
        // Get Comparison Data
        $comparison = $report_generator->get_comparison_metrics($date_from, $date_to, $comparison_period);
        
        // ✅ Get summary for stat boxes
        $summary = $report_generator->get_dashboard_summary($date_from, $date_to);

        // Get trends data
        $trends = $this->get_trends_data($date_from, $date_to);

        // Get top/bottom products
        $top_products = $this->get_top_performing_products($date_from, $date_to, 10);
        $bottom_products = $this->get_bottom_performing_products($date_from, $date_to, 10);

        // Get margin distribution
        // $margin_distribution = $this->get_margin_distribution();
        $margin_distribution = $this->get_margin_distribution($date_from, $date_to);

        // Get cost breakdown
        $cost_breakdown = $this->get_cost_breakdown($date_from, $date_to);

        // Get sparkline data
        $sparklines = $this->get_sparkline_data($date_from, $date_to);

        // Get activity feed
        $activity = $this->get_recent_activity(10);

        // Get profit leaks
        $profit_leaks = $this->detect_profit_leaks($date_from, $date_to);

        // Get product heatmap data
        $product_heatmap = $this->get_product_heatmap($date_from, $date_to);

        // Get currency settings
        $currency_symbol = get_woocommerce_currency_symbol();
        $currency_code = get_woocommerce_currency();

        // ✅ SINGLE wp_send_json_success with all data including summary
        wp_send_json_success(array(
            'kpis' => $kpis,
            'comparison' => $comparison,
            'summary' => $summary,
            'trends' => $trends,
            'top_products' => $top_products,
            'bottom_products' => $bottom_products,
            'margin_distribution' => $margin_distribution,
            'cost_breakdown' => $cost_breakdown,
            'sparklines' => $sparklines,
            'activity' => $activity,
            'profit_leaks' => $profit_leaks,
            'product_heatmap' => $product_heatmap,
            'currency' => array(
                'symbol' => $currency_symbol,
                'code' => $currency_code
            )
        ));
    }


    /**
     * Get profit forecast from AI
     */
    public function get_profit_forecast() {
        check_ajax_referer('qcnc_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $period = isset($_POST['period']) ? intval($_POST['period']) : 30;

        // Get core instance
        $core = \QuarkcodeNeuralCommerce\QCNC_Core::instance();
        $ai_insights = $core->ai_insights_engine;
        
        $forecast = $ai_insights->predict_profit($period);

        // ✅ Check for error
        if (isset($forecast['error'])) {
            // Return empty but valid data structure
            wp_send_json_success(array(
                'labels' => array(),
                'actual' => array(),
                'predicted' => array(),
                'confidence_upper' => array(),
                'total_predicted' => 0,
                'confidence' => 0,
                'growth_rate' => 0,
                'message' => $forecast['error']
            ));
            return;
        }

        // Format data for Chart.js
        $labels = array();
        $actual = array();
        $predicted = array();
        $confidence_upper = array();

        // Add historical data (last 30 days)
        global $wpdb;
        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        $profit_table = $wpdb->prefix . 'qcnc_order_profit';
        $orders_table = $wpdb->prefix . 'wc_orders';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $historical = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT DATE(o.date_created_gmt) as date, SUM(op.net_profit) as profit FROM {$profit_table} op INNER JOIN {$orders_table} o ON op.order_id = o.id
            WHERE o.date_created_gmt >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(o.date_created_gmt)
            ORDER BY date ASC
        ");

        foreach ($historical as $row) {
            if ($row->date) { // ✅ Check if date is valid
                $labels[] = wp_date('M j', strtotime($row->date));
                $actual[] = floatval($row->profit);
                $predicted[] = null;
                $confidence_upper[] = null;
            }
        }


        foreach ($historical as $row) {
            $labels[] = wp_date('M j', strtotime($row->date));
            $actual[] = floatval($row->profit);
            $predicted[] = null;
            $confidence_upper[] = null;
        }

        // Add predictions
        if (!empty($forecast['predictions'])) {
            foreach ($forecast['predictions'] as $prediction) {
                $labels[] = wp_date('M j', strtotime($prediction['date']));
                $actual[] = null;
                $predicted[] = floatval($prediction['predicted_profit']);
                $confidence_upper[] = floatval($prediction['confidence_upper']);
            }
        }

        wp_send_json_success(array(
            'labels' => $labels,
            'actual' => $actual,
            'predicted' => $predicted,
            'confidence_upper' => $confidence_upper,
            'total_predicted' => floatval($forecast['total_predicted'] ?? 0),
            'confidence' => intval($forecast['confidence'] ?? 0),
            'growth_rate' => floatval($forecast['growth_rate'] ?? 0)
        ));
    }


    /**
     * Get real-time stats
     */
    public function get_realtime_stats() {
        check_ajax_referer('qcnc_ajax_nonce', 'nonce');

        global $wpdb;
        $table = $wpdb->prefix . 'qcncorderprofits';
    
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") != $table) {
            wp_send_json_success(['alerts' => [], 'stats' => []]);
            return;
        }

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }

        $activity = $this->get_recent_activity(5);
        $alerts = $this->check_for_alerts();

        wp_send_json_success(array(
            'activity' => $activity,
            'alerts' => $alerts
        ));
    }

    /**
     * Export dashboard data
     */
    public function export_dashboard() {
        check_ajax_referer('qcnc_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_die('Insufficient permissions');
        }

        $date_from = isset($_GET['date_from']) ? sanitize_text_field(wp_unslash($_GET['date_from'])) : wp_date('Y-m-d', strtotime('-30 days'));
        $date_to = isset($_GET['date_to']) ? sanitize_text_field(wp_unslash($_GET['date_to'])) : wp_date('Y-m-d');

        // ✅ USE CORE SINGLETON
        $core = \QuarkcodeNeuralCommerce\QCNC_Core::instance();
        $report_generator = $core->report_generator;
        
        $data = $report_generator->get_dashboard_metrics($date_from, $date_to);

        // Generate CSV
        $filename = 'neural-commerce-dashboard-' . wp_date('Y-m-d') . '.csv';

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');

        fputcsv($output, array('Metric', 'Value'));
        fputcsv($output, array('Total Revenue', $data['total_revenue']));
        fputcsv($output, array('Total Costs', $data['total_costs']));
        fputcsv($output, array('Net Profit', $data['net_profit']));
        fputcsv($output, array('Profit Margin', $data['profit_margin'] . '%'));
        fputcsv($output, array('Average Order Value', $data['avg_order_value']));
        fputcsv($output, array('Profit Per Order', $data['profit_per_order']));

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($output);
        exit;
    }


    /**
     * Helper: Get trends data
     */
    private function get_trends_data($date_from, $date_to) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_profit';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results($wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT DATE(calculated_at) as date, SUM(revenue) as revenue, SUM(cogs_total + COALESCE(shipping_cost, 0) + COALESCE(payment_fee, 0)) as costs, SUM(net_profit) as profit, AVG(margin_pct) as margin FROM {$table_name}
            WHERE calculated_at BETWEEN %s AND %s
            GROUP BY DATE(calculated_at)
            ORDER BY date ASC
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        $labels = array();
        $revenue = array();
        $costs = array();
        $profit = array();
        $margin = array();
        
        foreach ($results as $row) {
            $labels[] = wp_date('M j', strtotime($row->date));
            $revenue[] = floatval($row->revenue);
            $costs[] = floatval($row->costs);
            $profit[] = floatval($row->profit);
            $margin[] = floatval($row->margin);
        }
        
        return array(
            'labels' => $labels,
            'revenue' => $revenue,
            'costs' => $costs,
            'profit' => $profit,
            'margin' => $margin
        );
    }


    /**
     * Helper: Get top performing products
     */
    private function get_top_performing_products($date_from, $date_to, $limit = 10) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_items_profit';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results($wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT product_id, SUM(item_profit) as total_profit, AVG((item_profit / NULLIF(item_revenue, 0)) * 100) as avg_margin, SUM(quantity) as total_quantity FROM {$table_name} ip INNER JOIN {$wpdb->prefix}qcnc_order_profit op ON ip.order_id = op.order_id
            WHERE op.calculated_at BETWEEN %s AND %s
            GROUP BY product_id
            HAVING total_profit > 0
            ORDER BY total_profit DESC
            LIMIT %d
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59', $limit));
        
        $products = array();
        foreach ($results as $row) {
            $product = wc_get_product($row->product_id);
            if ($product) {
                $products[] = array(
                    'id' => $row->product_id,
                    'name' => $product->get_name(),
                    'profit' => floatval($row->total_profit),
                    'margin' => floatval($row->avg_margin)
                );
            }
        }
        
        return $products;
    }

    /**
     * Helper: Get bottom performing products
     */
    private function get_bottom_performing_products($date_from, $date_to, $limit = 10) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_items_profit';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results($wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT product_id, SUM(item_profit) as total_profit, AVG((item_profit / NULLIF(item_revenue, 0)) * 100) as avg_margin, SUM(quantity) as total_quantity FROM {$table_name} ip
            INNER JOIN {$wpdb->prefix}qcnc_order_profit op ON ip.order_id = op.order_id
            WHERE op.calculated_at BETWEEN %s AND %s
            GROUP BY product_id
            HAVING total_profit < 0
            ORDER BY total_profit ASC
            LIMIT %d
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59', $limit));
        
        $products = array();
        foreach ($results as $row) {
            $product = wc_get_product($row->product_id);
            if ($product) {
                $products[] = array(
                    'id' => $row->product_id,
                    'name' => $product->get_name(),
                    'profit' => floatval($row->total_profit),
                    'margin' => floatval($row->avg_margin)
                );
            }
        }
        
        return $products;
    }

    /**
     * Helper: Get margin distribution
    */
    private function get_margin_distribution($date_from, $date_to) {
        global $wpdb;
        
        $ranges = array(
            '<0%' => array(-999, 0),
            '0-10%' => array(0, 10),
            '10-20%' => array(10, 20),
            '20-30%' => array(20, 30),
            '30-40%' => array(30, 40),
            '40-50%' => array(40, 50),
            '>50%' => array(50, 999)
        );
        
        $distribution = array();
        
        foreach ($ranges as $label => $range) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $count = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(DISTINCT ip.product_id)
                FROM {$wpdb->prefix}qcnc_order_items_profit ip
                INNER JOIN {$wpdb->prefix}qcnc_order_profit op ON ip.order_id = op.order_id
                WHERE op.calculated_at BETWEEN %s AND %s
                    AND ((ip.item_profit / NULLIF(ip.item_revenue, 0)) * 100) >= %f
                    AND ((ip.item_profit / NULLIF(ip.item_revenue, 0)) * 100) < %f
            ", $date_from . ' 00:00:00', $date_to . ' 23:59:59', $range[0], $range[1]));
            
            $distribution[] = intval($count ?? 0);
        }
        
        return $distribution;
    }




    /**
     * Helper: Get cost breakdown - ACTUAL data only
     */
    private function get_cost_breakdown($date_from, $date_to) {
        global $wpdb;
        
        // Get all order costs
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $costs = $wpdb->get_row($wpdb->prepare("
            SELECT 
                SUM(cogs_total) as cogs,
                SUM(COALESCE(shipping_cost, 0)) as shipping,
                SUM(COALESCE(payment_fee, 0)) as payment_fee,
                SUM(COALESCE(tax_amount, 0)) as tax,
                SUM(COALESCE(discount_amount, 0)) as discount
            FROM {$wpdb->prefix}qcnc_order_profit
            WHERE calculated_at BETWEEN %s AND %s
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        // Try to get itemized additional costs
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $additional = $wpdb->get_row($wpdb->prepare("
            SELECT 
                SUM(COALESCE(pc.packaging_cost, 0) * oip.quantity) as packaging,
                SUM(COALESCE(pc.handling_cost, 0) * oip.quantity) as handling,
                SUM(COALESCE(pc.warehousing_cost, 0) * oip.quantity) as warehousing,
                SUM(COALESCE(pc.labor_cost, 0) * oip.quantity) as labor
            FROM {$wpdb->prefix}qcnc_order_items_profit oip
            INNER JOIN {$wpdb->prefix}qcnc_order_profit op ON oip.order_id = op.order_id
            LEFT JOIN {$wpdb->prefix}qcnc_product_costs pc 
                ON oip.product_id = pc.product_id 
                AND pc.is_current = 1
            WHERE op.calculated_at BETWEEN %s AND %s
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        $breakdown = array(
            floatval($costs->cogs ?? 0),                           // Product Cost
            floatval($additional->warehousing ?? 0),               // Warehousing
            floatval($additional->labor ?? 0),                     // Labor
            floatval($additional->packaging ?? 0) + floatval($additional->handling ?? 0), // Packaging
            floatval($costs->shipping ?? 0),                       // Shipping
            floatval($costs->payment_fee ?? 0)                     // Other (payment fees)
        );
        
        // ✅ Debug log to see actual values
        // error_log('Cost Breakdown Data: ' . print_r($breakdown, true));
        
        return $breakdown;
    }




    /**
     * Helper: Get sparkline data for KPIs
     */
    private function get_sparkline_data($date_from, $date_to) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qcnc_order_profit'; // ← FIXED
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results($wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            "SELECT DATE(calculated_at) as date, SUM(revenue) as revenue, SUM(cogs_total + COALESCE(shipping_cost, 0) + COALESCE(payment_fee, 0)) as costs, SUM(net_profit) as profit, AVG(margin_pct) as margin FROM {$table_name}
            WHERE calculated_at BETWEEN %s AND %s
            GROUP BY DATE(calculated_at)
            ORDER BY date ASC
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        return array(
            'revenue' => array_map(function($r) { return floatval($r->revenue); }, $results),
            'costs' => array_map(function($r) { return floatval($r->costs); }, $results),
            'profit' => array_map(function($r) { return floatval($r->profit); }, $results),
            'margin' => array_map(function($r) { return floatval($r->margin); }, $results),
            'aov' => array_map(function($r) { return floatval($r->revenue); }, $results),
            'ppo' => array_map(function($r) { return floatval($r->profit); }, $results)
        );
    }


    /**
     * Helper: Get recent activity
     */
    private function get_recent_activity($limit = 10) {
        $args = array(
            'limit' => $limit,
            'orderby' => 'date',
            'order' => 'DESC',
            'status' => array('processing', 'completed')
        );

        $orders = wc_get_orders($args);
        $activity = array();

        // ✅ USE EXISTING CALCULATOR INSTANCE
        foreach ($orders as $order) {
            $profit_data = $this->calculator->get_order_profit($order->get_id());
            
            if (!$profit_data) {
                continue; // Skip if no profit data
            }

            $type = $profit_data['net_profit'] >= 0 ? 'profit' : 'loss';

            $activity[] = array(
                'time' => human_time_diff($order->get_date_created()->getTimestamp()) . ' ago',
                'type' => $type,
                'text' => sprintf(
                    'Order #%s: %s (Profit: %s)',
                    $order->get_id(),
                    $order->get_formatted_order_total(),
                    wc_price($profit_data['net_profit'])
                )
            );
        }

        return $activity;
    }


    /**
     * Helper: Check for alerts
     */
    private function check_for_alerts() {
        $alerts = array();

        // Check for low margin orders in last hour
        global $wpdb;
        $table_name = $wpdb->prefix . 'qcnc_order_profit';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $low_margin_count = $wpdb->get_var(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
            "SELECT COUNT(*) FROM {$table_name}
            WHERE profit_margin < 15
                AND order_date >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");

        if ($low_margin_count > 0) {
            $alerts[] = array(
                'type' => 'warning',
                'message' => sprintf(
                    /* translators: %d: Number of orders */
                    __('%d orders with margins below 15%% in the last hour!', 'quarkcode-neuralcommerce-lite'),
                    $low_margin_count
                )
            );
        }

        return $alerts;
    }

    /**
     * Helper: Detect profit leaks
     */
    private function detect_profit_leaks($date_from, $date_to) {
        $leaks = array();
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'qcnc_order_profit'; // ← FIXED
        
        // High shipping costs
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $high_shipping = $wpdb->get_var($wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            " SELECT SUM(shipping_cost) FROM {$table_name}
            WHERE calculated_at BETWEEN %s AND %s
                AND revenue > 0
                AND (shipping_cost / revenue) > 0.15
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        if ($high_shipping > 100) {
            $leaks[] = array(
                'title' => __('High Shipping Costs', 'quarkcode-neuralcommerce-lite'),
                'description' => __('Some orders have shipping costs exceeding 15% of revenue.', 'quarkcode-neuralcommerce-lite'),
                'amount' => floatval($high_shipping),
                'severity' => 'high'
            );
        }
        
        return $leaks;
    }

    /**
     * Helper: Get product heatmap data
     */
    private function get_product_heatmap($date_from, $date_to) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $results = $wpdb->get_results($wpdb->prepare("
            SELECT 
                ip.product_id,
                SUM(ip.item_profit) as total_profit,
                SUM(ip.quantity) as total_quantity,
                AVG((ip.item_profit / NULLIF(ip.item_revenue, 0)) * 100) as avg_margin
            FROM {$wpdb->prefix}qcnc_order_items_profit ip
            INNER JOIN {$wpdb->prefix}qcnc_order_profit op ON ip.order_id = op.order_id
            WHERE op.calculated_at BETWEEN %s AND %s
            GROUP BY ip.product_id
            ORDER BY total_profit DESC
            LIMIT 20
        ", $date_from . ' 00:00:00', $date_to . ' 23:59:59'));
        
        $heatmap = array();
        foreach ($results as $row) {
            $product = wc_get_product($row->product_id);
            if ($product) {
                $heatmap[] = array(
                    'name' => substr($product->get_name(), 0, 30), // Truncate for display
                    'profit' => floatval($row->total_profit),
                    'quantity' => intval($row->total_quantity),
                    'margin' => floatval($row->avg_margin)
                );
            }
        }
        
        return $heatmap;
    }

    /**
     * Save product costs and prices from inline editor
     */
    public function save_products_inline() {
        check_ajax_referer( 'qcnc_save_products', 'qcnc_products_nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ) );
        }

        // ✅ FIXED: Properly sanitize and unslash POST arrays
        $costs = array();
        if ( isset( $_POST['cost'] ) && is_array( $_POST['cost'] ) ) {
            $costs = array_map( 'floatval', wp_unslash( $_POST['cost'] ) );
        }

        $prices = array();
        if ( isset( $_POST['price'] ) && is_array( $_POST['price'] ) ) {
            $prices = array_map( 'floatval', wp_unslash( $_POST['price'] ) );
        }

        if ( empty( $costs ) && empty( $prices ) ) {
            wp_send_json_error( array( 'message' => 'No data to save' ) );
        }

        $updated = 0;
        $cost_updated = 0;
        $price_updated = 0;
        $product_ids = array_unique( array_merge( array_keys( $costs ), array_keys( $prices ) ) );

        foreach ( $product_ids as $product_id ) {
            $product_id = intval( $product_id );
            if ( $product_id <= 0 ) {
                continue;
            }

            $product = wc_get_product( $product_id );
            if ( ! $product ) {
                continue;
            }

            // Update cost if provided
            if ( isset( $costs[ $product_id ] ) ) {
                $new_cost = floatval( $costs[ $product_id ] );
                if ( $new_cost >= 0 ) {
                    $result = $this->cost_manager->set_product_cost(
                        $product_id,
                        $new_cost,
                        null, // variation_id
                        0.0,  // packaging_cost
                        0.0,  // handling_cost
                        '',   // currency
                        array() // metadata
                    );

                    if ( $result ) {
                        $cost_updated++;
                        $updated++;
                    }
                }
            }

            // Update price if provided
            if ( isset( $prices[ $product_id ] ) ) {
                $new_price = floatval( $prices[ $product_id ] );
                if ( $new_price >= 0 ) {
                    $product->set_regular_price( $new_price );
                    $product->set_price( $new_price );
                    $product->save();
                    $price_updated++;
                    $updated++;
                }
            }
        }

        // Clear relevant caches
        if ( method_exists( $this, 'cache_manager' ) && $this->cache_manager ) {
            foreach ( $product_ids as $product_id ) {
                $this->cache_manager->delete_product_cache( $product_id );
            }
        }

        wp_send_json_success( array(
            'updated'       => $updated,
            'cost_updated'  => $cost_updated,
            'price_updated' => $price_updated,
            'message'       => sprintf( 'Successfully updated %d products', $updated ),
        ) );
    }

    /**
     * Send digest email manually via AJAX
     */
    public function send_digest_now() {
        check_ajax_referer( 'qcnc_send_digest', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array( 
                'message' => __( 'Permission denied', 'quarkcode-neuralcommerce-lite' ) 
            ) );
        }

        // Get the alert system instance
        $alert_system = new \QuarkcodeNeuralCommerce\Core\QCNC_Margin_Alert_System();

        // Check if there are queued alerts
        $queue = get_option( 'qcnc_alert_digest_queue', array() );

        if ( empty( $queue ) ) {
            wp_send_json_error( array( 
                'message' => __( 'No alerts in queue to send', 'quarkcode-neuralcommerce-lite' ) 
            ) );
        }

        // Send the digest
        $sent = $alert_system->send_digest_email();

        if ( $sent ) {
            wp_send_json_success( array( 
                'message' => sprintf(
                    /* translators: %d: Number of alerts in the digest */
                    __( 'Digest email sent successfully with %d alerts', 'quarkcode-neuralcommerce-lite' ),
                    count( $queue )
                )
            ) );
        } else {
            wp_send_json_error( array( 
                'message' => __( 'Failed to send digest email', 'quarkcode-neuralcommerce-lite' ) 
            ) );
        }
    }

}
