/**
 * Reports JavaScript
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        if ($('.qcnc-reports').length) {
            initReportsPage();
        }
    });

    /**
     * Initialize reports page
     */
    function initReportsPage() {
        initDateRangePicker();
        initReportFilters();
        initExport();
        initCharts();
        initReportTabs();
    }

    /**
     * Initialize date range picker
     */
    function initDateRangePicker() {
        // Quick date ranges
        $('.qcnc-quick-range').on('click', function(e) {
            e.preventDefault();
            const range = $(this).data('range');
            setDateRange(range);
        });

        // Add quick range buttons if not exist
        if ($('.qcnc-quick-ranges').length === 0) {
            const html = `
                <div class="qcnc-quick-ranges">
                    <span>Quick Select:</span>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="today">Today</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="yesterday">Yesterday</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="last7days">Last 7 Days</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="last30days">Last 30 Days</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="thismonth">This Month</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="lastmonth">Last Month</a>
                    <a href="#" class="qcnc-quick-range button button-small" data-range="thisyear">This Year</a>
                </div>
            `;
            $('.qcnc-report-filters form').before(html);
        }
    }

    /**
     * Set date range
     */
    function setDateRange(range) {
        const today = new Date();
        let dateFrom, dateTo;

        switch (range) {
            case 'today':
                dateFrom = dateTo = formatDate(today);
                break;
            
            case 'yesterday':
                const yesterday = new Date(today);
                yesterday.setDate(yesterday.getDate() - 1);
                dateFrom = dateTo = formatDate(yesterday);
                break;
            
            case 'last7days':
                dateTo = formatDate(today);
                const last7 = new Date(today);
                last7.setDate(last7.getDate() - 7);
                dateFrom = formatDate(last7);
                break;
            
            case 'last30days':
                dateTo = formatDate(today);
                const last30 = new Date(today);
                last30.setDate(last30.getDate() - 30);
                dateFrom = formatDate(last30);
                break;
            
            case 'thismonth':
                dateFrom = formatDate(new Date(today.getFullYear(), today.getMonth(), 1));
                dateTo = formatDate(today);
                break;
            
            case 'lastmonth':
                const lastMonth = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                const lastMonthEnd = new Date(today.getFullYear(), today.getMonth(), 0);
                dateFrom = formatDate(lastMonth);
                dateTo = formatDate(lastMonthEnd);
                break;
            
            case 'thisyear':
                dateFrom = formatDate(new Date(today.getFullYear(), 0, 1));
                dateTo = formatDate(today);
                break;
        }

        $('#date_from').val(dateFrom);
        $('#date_to').val(dateTo);

        // Auto-submit form
        $('.qcnc-report-filters form').submit();
    }

    /**
     * Format date as YYYY-MM-DD
     */
    function formatDate(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    /**
     * Initialize report filters
     */
    function initReportFilters() {
        // Filter by category
        $(document).on('change', '#qcnc-filter-category', function() {
            filterReport();
        });

        // Filter by product
        $(document).on('change', '#qcnc-filter-product', function() {
            filterReport();
        });

        // Sort columns
        $(document).on('click', '.qcnc-sortable', function() {
            const column = $(this).data('column');
            const currentOrder = $(this).data('order') || 'asc';
            const newOrder = currentOrder === 'asc' ? 'desc' : 'asc';

            sortReportTable(column, newOrder);

            // Update sort indicator
            $('.qcnc-sortable').removeClass('sorted-asc sorted-desc');
            $(this).addClass('sorted-' + newOrder).data('order', newOrder);
        });
    }

    /**
     * Filter report
     */
    function filterReport() {
        const filters = {
            category: $('#qcnc-filter-category').val(),
            product: $('#qcnc-filter-product').val()
        };

        // Apply filters to current table
        $('table tbody tr').each(function() {
            let show = true;

            if (filters.category && $(this).data('category') != filters.category) {
                show = false;
            }

            if (filters.product && $(this).data('product') != filters.product) {
                show = false;
            }

            $(this).toggle(show);
        });
    }

    /**
     * Sort report table
     */
    function sortReportTable(column, order) {
        const $table = $('#qcnc-report-content table');
        const $tbody = $table.find('tbody');
        const $rows = $tbody.find('tr').toArray();

        $rows.sort(function(a, b) {
            const aValue = $(a).find(`[data-sort="${column}"]`).text();
            const bValue = $(b).find(`[data-sort="${column}"]`).text();

            // Try to parse as number
            const aNum = parseFloat(aValue.replace(/[^0-9.-]/g, ''));
            const bNum = parseFloat(bValue.replace(/[^0-9.-]/g, ''));

            if (!isNaN(aNum) && !isNaN(bNum)) {
                return order === 'asc' ? aNum - bNum : bNum - aNum;
            }

            // String comparison
            if (order === 'asc') {
                return aValue.localeCompare(bValue);
            } else {
                return bValue.localeCompare(aValue);
            }
        });

        $tbody.empty().append($rows);
    }

    /**
     * Initialize export functionality
     */
    function initExport() {
        $('#qcnc-export-report').on('click', function(e) {
            e.preventDefault();
            exportReport('csv');
        });

        // Add export options dropdown
        if ($('#qcnc-export-options').length === 0) {
            const html = `
                <div class="qcnc-export-dropdown" style="display: inline-block; position: relative;">
                    <button type="button" class="button" id="qcnc-export-toggle">
                        Export ▼
                    </button>
                    <div id="qcnc-export-options" class="qcnc-dropdown-menu" style="display: none;">
                        <a href="#" data-format="csv">Export as CSV</a>
                        <a href="#" data-format="pdf">Export as PDF</a>
                        <a href="#" data-format="excel">Export as Excel</a>
                    </div>
                </div>
            `;
            $('#qcnc-export-report').replaceWith(html);
        }

        // Toggle export dropdown
        $(document).on('click', '#qcnc-export-toggle', function(e) {
            e.stopPropagation();
            $('#qcnc-export-options').toggle();
        });

        // Export format selection
        $(document).on('click', '#qcnc-export-options a', function(e) {
            e.preventDefault();
            const format = $(this).data('format');
            exportReport(format);
            $('#qcnc-export-options').hide();
        });

        // Close dropdown when clicking outside
        $(document).on('click', function() {
            $('#qcnc-export-options').hide();
        });
    }

    /**
     * Export report
     */
    function exportReport(format) {
        const reportType = new URLSearchParams(window.location.search).get('report') || 'product';
        const dateFrom = $('#date_from').val();
        const dateTo = $('#date_to').val();

        const url = qcncAdmin.ajaxUrl + 
            '?action=qcnc_export_report' +
            '&qcnc_nonce=' + qcncAdmin.nonce +
            '&report_type=' + reportType +
            '&format=' + format +
            '&date_from=' + dateFrom +
            '&date_to=' + dateTo;

        window.location.href = url;
    }

    /**
     * Initialize charts
     */
    function initCharts() {
        // Check if Chart.js is loaded
        if (typeof Chart === 'undefined') {
            return;
        }

        renderProfitChart();
        renderMarginChart();
    }

    /**
     * Render profit trend chart
     */
    function renderProfitChart() {
        const canvas = document.getElementById('qcnc-profit-chart');
        if (!canvas) return;

        const ctx = canvas.getContext('2d');
        const data = JSON.parse(canvas.dataset.chartData || '[]');

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: data.map(item => item.period),
                datasets: [{
                    label: 'Net Profit',
                    data: data.map(item => item.total_profit),
                    borderColor: '#28a745',
                    backgroundColor: 'rgba(40, 167, 69, 0.1)',
                    tension: 0.4
                }, {
                    label: 'Revenue',
                    data: data.map(item => item.total_revenue),
                    borderColor: '#007bff',
                    backgroundColor: 'rgba(0, 123, 255, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }

    /**
     * Render margin distribution chart
     */
    function renderMarginChart() {
        const canvas = document.getElementById('qcnc-margin-chart');
        if (!canvas) return;

        const ctx = canvas.getContext('2d');
        const data = JSON.parse(canvas.dataset.chartData || '[]');

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: data.map(item => item.category_name || item.product_name),
                datasets: [{
                    label: 'Margin %',
                    data: data.map(item => item.avg_margin_pct),
                    backgroundColor: data.map(item => {
                        const margin = parseFloat(item.avg_margin_pct);
                        if (margin >= 15) return '#28a745';
                        if (margin >= 0) return '#ffc107';
                        return '#dc3545';
                    })
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Margin %'
                        }
                    }
                }
            }
        });
    }

    /**
     * Initialize report tabs
     */
    function initReportTabs() {
        $(document).on('click', '.qcnc-report-tab', function(e) {
            e.preventDefault();
            
            const reportType = $(this).data('report');
            const url = new URL(window.location.href);
            url.searchParams.set('report', reportType);
            
            window.location.href = url.toString();
        });
    }

    /**
     * Auto-refresh report data
     */
    let refreshInterval;
    
    function startAutoRefresh(interval = 300000) { // 5 minutes default
        if ($('#qcnc-auto-refresh').is(':checked')) {
            refreshInterval = setInterval(function() {
                location.reload();
            }, interval);
        }
    }

    function stopAutoRefresh() {
        if (refreshInterval) {
            clearInterval(refreshInterval);
        }
    }

    $(document).on('change', '#qcnc-auto-refresh', function() {
        if ($(this).is(':checked')) {
            startAutoRefresh();
        } else {
            stopAutoRefresh();
        }
    });

})(jQuery);
