/**
 * QCNC Admin Views - Consolidated Inline Scripts  
 * Extracted from view files for WordPress.org compliance
 * 
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.5
 */

jQuery(document).ready(function($) {
    'use strict';

    /* =========================================
       Gateway Modal Logic
       ========================================= */

    // Open Add Gateway Fee Modal
    $('#qcnc-add-gateway-fee').on('click', function() {
        $('#qcnc-gateway-form')[0].reset();
        $('#qcnc-gateway-id').val('');
        $('#qcnc-gateway-modal-title').text(qcncAdmin.i18n.addGatewayFee || 'Add Gateway Fee');
        $('#qcnc-gateway-modal').fadeIn(200);
    });

    // Close Modal
    $('.qcnc-modal-close').on('click', function() {
        $('.qcnc-modal').fadeOut(200);
    });

    // Auto-populate gateway name from dropdown
    $('#qcnc-gateway-select').on('change', function() {
        const text = $(this).find('option:selected').text().split('(')[0].trim();
        if(text) $('#qcnc-gateway-name').val(text);
    });

    /* =========================================
       AI Integration Logic
       ========================================= */

    // Default AI models for each provider
    const defaultModels = {
        'openai': 'gpt-4o-mini',
        'claude': 'claude-4-sonnet-20250924',
        'gemini': 'gemini-2.5-flash',
        'perplexity': 'sonar-pro'
    };

    // Toggle API key visibility
    $('#qcnc-toggle-api-key').on('click', function() {
        var $input = $('#qcnc_ai_api_key');
        var $icon = $(this).find('.dashicons');
        var $text = $(this).contents().filter(function() {
            return this.nodeType === 3;
        });

        if ($input.attr('type') === 'password') {
            $input.attr('type', 'text');
            $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            if ($text[0]) $text[0].textContent = ' Hide';
        } else {
            $input.attr('type', 'password');
            $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            if ($text[0]) $text[0].textContent = ' Show';
        }
    });

    // Update model visibility based on provider
    function updateModelVisibility() {
        var provider = $('#qcnc_ai_provider').val();

        // Hide all model option groups
        $('.ai-model-openai, .ai-model-claude, .ai-model-gemini, .ai-model-perplexity').hide();

        // Show and select correct model for provider
        if (provider === 'openai') {
            $('.ai-model-openai').show();
            $('#qcnc_ai_model').val(defaultModels.openai);
        } else if (provider === 'claude') {
            $('.ai-model-claude').show();
            $('#qcnc_ai_model').val(defaultModels.claude);
        } else if (provider === 'gemini') {
            $('.ai-model-gemini').show();
            $('#qcnc_ai_model').val(defaultModels.gemini);
        } else if (provider === 'perplexity') {
            $('.ai-model-perplexity').show();
            $('#qcnc_ai_model').val(defaultModels.perplexity);
        }
    }

    // Update models when provider changes
    $('#qcnc_ai_provider').on('change', function() {
        updateModelVisibility();
    });

    // Run on page load
    updateModelVisibility();

    // Test AI Connection
    $('#qcnc-test-ai-connection').on('click', function() {
        var $btn = $(this);
        var $result = $('#qcnc-ai-test-result');

        $btn.prop('disabled', true).text('Testing...');
        $result.removeClass('success error').text('');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_test_ai_connection',
                nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    $result.addClass('success').html(
                        '<span style="color:#28a745; font-size:13px;">✓ ' + 
                        response.data.message + ' (' + response.data.provider + ' - ' + 
                        response.data.model + ')</span>'
                    );
                } else {
                    $result.addClass('error').html(
                        '<span style="color:#ef4444; font-size:13px;">✗ ' + 
                        response.data + '</span>'
                    );
                }
            },
            error: function() {
                $result.addClass('error').html(
                    '<span style="color:#ef4444; font-size:13px;">✗ Connection test failed</span>'
                );
            },
            complete: function() {
                $btn.prop('disabled', false).text('Test Connection');
            }
        });
    });

    /* =========================================
       Cleanup Duplicate Costs
       ========================================= */

    $('#qcnc-cleanup-duplicates').on('click', function() {
        if(!confirm(qcncAdmin.i18n.confirmCleanup || 'Clean up duplicate costs?')) return;
        
        $.post(ajaxurl, { 
            action: 'qcnc_cleanup_duplicate_costs', 
            qcnc_nonce: qcncAdmin.nonce 
        }, function(res) {
            alert(res.data.message);
        });
    });

    /* =========================================
       Send Digest Now
       ========================================= */

    $('#qcnc-send-digest-now').on('click', function(e) {
        e.preventDefault();

        var $button = $(this);
        var $result = $('#qcnc-digest-result');

        // Disable button and show loading
        $button.prop('disabled', true);
        
        var originalText = $button.html();
        $button.html('<span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear;"></span> Sending...');

        $result.html('');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_send_digest_now',
                nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    $result.html(
                        '<div class="qcnc-success-box">' +
                        '<span class="dashicons dashicons-yes-alt"></span>' +
                        '<div><strong>Success!</strong> ' + response.data.message + '</div>' +
                        '</div>'
                    );

                    // Update queue count in description
                    var $queueDesc = $('#qcnc-send-digest-now').closest('.qcnc-form-row').find('.description').first();
                    $queueDesc.text('No alerts currently ' +
'');

                } else {
                    $result.html(
                        '<div class="qcnc-error-box">' +
                        '<span class="dashicons dashicons-warning"></span>' +
                        '<div><strong>Error:</strong> ' + response.data.message + '</div>' +
                        '</div>'
                    );
                }
            },
            error: function(xhr, status, error) {
                $result.html(
                    '<div class="qcnc-error-box">' +
                    '<span class="dashicons dashicons-warning"></span>' +
                    '<div><strong>Error:</strong> Failed to send digest. Please try again.</div>' +
                    '</div>'
                );
                console.error('AJAX Error:', error);
            },
            complete: function() {
                $button.prop('disabled', false);
                $button.html(originalText);
            }
        });
    });

    /* =========================================
       Add CSS Animation for Loading Spinner
       ========================================= */

    if (!$('#qcnc-rotation-animation').length) {
        $('<style id="qcnc-rotation-animation">')
            .text('@keyframes rotation { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }')
            .appendTo('head');
    }

    /* =========================================
       Alert Test Buttons
       ========================================= */

    $('.qcnc-test-alert-btn').on('click', function(e) {
        e.preventDefault();
        
        var $btn = $(this);
        var alertType = $btn.data('alert-type');
        var originalHtml = $btn.html();
        
        // Disable button and show loading
        $btn.prop('disabled', true);
        $btn.html('<span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear;"></span> Sending...');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_test_alert',
                alert_type: alertType,
                nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    $btn.html('<span class="dashicons dashicons-yes"></span> Sent!');
                    setTimeout(function() {
                        $btn.html(originalHtml).prop('disabled', false);
                    }, 2000);
                } else {
                    $btn.html('<span class="dashicons dashicons-no"></span> Failed');
                    setTimeout(function() {
                        $btn.html(originalHtml).prop('disabled', false);
                    }, 2000);
                    alert(response.data || 'Failed to send test alert');
                }
            },
            error: function() {
                $btn.html('<span class="dashicons dashicons-no"></span> Error');
                setTimeout(function() {
                    $btn.html(originalHtml).prop('disabled', false);
                }, 2000);
            }
        });
    });

    /* =========================================
       Gateway Fee Form Handlers
       ========================================= */

    // Edit Gateway
    $('.qcnc-edit-gateway').on('click', function() {
        var gatewayId = $(this).data('id');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_get_gateway_fee',
                gateway_id: gatewayId,
                nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    var data = response.data;
                    $('#qcnc-gateway-id').val(data.id);
                    $('#qcnc-gateway-name').val(data.gateway_name);
                    $('#qcnc-fee-type').val(data.fee_type);
                    $('#qcnc-percentage-fee').val(data.percentage_fee);
                    $('#qcnc-fixed-fee').val(data.fixed_fee);
                    $('#qcnc-gateway-currency').val(data.currency);
                    $('#qcnc-gateway-modal-title').text('Edit Gateway Fee');
                    $('#qcnc-gateway-modal').fadeIn(200);
                }
            }
        });
    });

    // Delete Gateway
    $('.qcnc-delete-gateway').on('click', function() {
        if (!confirm('Are you sure you want to delete this gateway fee?')) return;
        
        var gatewayId = $(this).data('id');
        var $row = $(this).closest('tr');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_delete_gateway_fee',
                gateway_id: gatewayId,
                nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    $row.fadeOut(300, function() {
                        $(this).remove();
                    });
                }
            }
        });
    });

    // Save Gateway Fee
    $('#qcnc-gateway-form').on('submit', function(e) {
        e.preventDefault();
        
        var formData = $(this).serializeArray();
        formData.push({ name: 'action', value: 'qcnc_save_gateway_fee' });
        formData.push({ name: 'nonce', value: qcncAdmin.nonce });
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: $.param(formData),
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data || 'Failed to save gateway fee');
                }
            }
        });
    });

    // Fee type field visibility
    $('#qcnc-fee-type').on('change', function() {
        var feeType = $(this).val();
        $('#qcnc-percentage-field').toggle(feeType === 'percentage' || feeType === 'mixed');
        $('#qcnc-fixed-field').toggle(feeType === 'fixed' || feeType === 'mixed');
    }).trigger('change');

    /* =========================================
       Product Costs Page Initialization
       ========================================= */
    
    // Load costs on page load (if product costs page)
    if ($('#qcnc-costs-table').length) {
        qcncLoadCosts(1);
    }

    /* =========================================
       Order Profitability Page Initialization
       ========================================= */
    
    // Load orders on page load (if order profitability page)
    if ($('#qcnc-orders-table').length) {
        qcncLoadOrders(1);

        // Filter button
        $('#qcnc-filter-orders').on('click', function() {
            qcncLoadOrders(1);
        });
    }

});

/* =========================================
   CLV Calculation Functions (Global Scope)
   Used in user profiles and order meta boxes
   ========================================= */

/**
 * Calculate CLV for a specific user
 * @param {number} userId - User ID
 */
function qcncCalculateUserCLV(userId) {
    if (!confirm('Recalculate CLV for this customer?')) return;

    jQuery.post(ajaxurl, {
        action: 'qcnc_calculate_customer_clv',
        nonce: qcncAdmin.nonce,
        customer_id: userId
    }, function(response) {
        if (response.success) {
            alert('CLV calculated successfully!');
            location.reload();
        } else {
            alert('Failed to calculate CLV.');
        }
    });
}

/**
 * Calculate CLV for a customer from order page
 * @param {number} customerId - Customer ID
 */
function qcncCalculateCustomerCLV(customerId) {
    jQuery.post(ajaxurl, {
        action: 'qcnc_calculate_customer_clv',
        nonce: qcncAdmin.nonce,
        customer_id: customerId
    }, function(response) {
        if (response.success) {
            alert('CLV calculated!');
            location.reload();
        }
    });
}
