/**
 * Product Costs JavaScript
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        if ($('.qcnc-product-costs').length) {
            initProductCostsPage();
        }
    });

    /**
     * Initialize product costs page
     */
    function initProductCostsPage() {
        initInlineEditing();
        initQuickEdit();
        initBulkActions();
        initSearch();
    }

    /**
     * Initialize inline editing
     */
    function initInlineEditing() {
        // Make cost cells editable
        $(document).on('dblclick', '.qcnc-editable-cost', function() {
            const $cell = $(this);
            const currentValue = $cell.text().trim();
            const productId = $cell.closest('tr').data('product-id');
            const costType = $cell.data('cost-type');

            // Replace cell with input
            $cell.html('<input type="number" class="qcnc-inline-input" value="' + currentValue + '" step="0.01" min="0" />');
            
            const $input = $cell.find('.qcnc-inline-input');
            $input.focus().select();

            // Save on blur or enter
            $input.on('blur keypress', function(e) {
                if (e.type === 'blur' || (e.type === 'keypress' && e.which === 13)) {
                    const newValue = $(this).val();
                    saveInlineCost(productId, costType, newValue, $cell);
                }
            });

            // Cancel on escape
            $input.on('keypress', function(e) {
                if (e.which === 27) {
                    $cell.html(currentValue);
                }
            });
        });
    }

    /**
     * Save inline cost
     */
    function saveInlineCost(productId, costType, value, $cell) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_update_inline_cost',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId,
                cost_type: costType,
                value: value
            },
            beforeSend: function() {
                $cell.html('<span class="spinner is-active" style="float:none;"></span>');
            },
            success: function(response) {
                if (response.success) {
                    $cell.html(parseFloat(value).toFixed(2));
                    $cell.addClass('qcnc-updated');
                    
                    // Update total cost
                    updateRowTotals($cell.closest('tr'));
                    
                    setTimeout(function() {
                        $cell.removeClass('qcnc-updated');
                    }, 2000);
                } else {
                    $cell.html(value);
                    alert(response.data.message || 'Failed to update cost');
                }
            },
            error: function() {
                $cell.html(value);
                alert('Error updating cost');
            }
        });
    }

    /**
     * Update row totals after inline edit
     */
    function updateRowTotals($row) {
        const cost = parseFloat($row.find('[data-cost-type="cost"]').text()) || 0;
        const packaging = parseFloat($row.find('[data-cost-type="packaging_cost"]').text()) || 0;
        const handling = parseFloat($row.find('[data-cost-type="handling_cost"]').text()) || 0;
        const price = parseFloat($row.find('.qcnc-product-price').text()) || 0;

        const totalCost = cost + packaging + handling;
        const margin = price > 0 ? ((price - totalCost) / price * 100) : 0;

        $row.find('.qcnc-total-cost').html('<strong>' + totalCost.toFixed(2) + '</strong>');
        
        const $marginCell = $row.find('.qcnc-margin');
        $marginCell.text(margin.toFixed(2) + '%');
        
        // Update margin color
        $marginCell.removeClass('qcnc-text-success qcnc-text-warning qcnc-text-danger');
        if (margin >= 15) {
            $marginCell.addClass('qcnc-text-success');
        } else if (margin >= 0) {
            $marginCell.addClass('qcnc-text-warning');
        } else {
            $marginCell.addClass('qcnc-text-danger');
        }
    }

    /**
     * Initialize quick edit functionality
     */
    function initQuickEdit() {
        $(document).on('click', '.qcnc-quick-edit', function() {
            const $row = $(this).closest('tr');
            const productId = $row.data('product-id');
            
            // Hide normal row
            $row.hide();

            // Show quick edit row
            const $quickEditRow = createQuickEditRow($row);
            $row.after($quickEditRow);
            $quickEditRow.show();
        });

        // Cancel quick edit
        $(document).on('click', '.qcnc-cancel-quick-edit', function() {
            const $quickEditRow = $(this).closest('.qcnc-quick-edit-row');
            const $normalRow = $quickEditRow.prev('tr');
            
            $quickEditRow.remove();
            $normalRow.show();
        });

        // Save quick edit
        $(document).on('click', '.qcnc-save-quick-edit', function() {
            const $quickEditRow = $(this).closest('.qcnc-quick-edit-row');
            const productId = $quickEditRow.data('product-id');
            
            const formData = {
                action: 'qcnc_quick_edit_save',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId,
                cost: $quickEditRow.find('[name="qe_cost"]').val(),
                packaging_cost: $quickEditRow.find('[name="qe_packaging_cost"]').val(),
                handling_cost: $quickEditRow.find('[name="qe_handling_cost"]').val()
            };

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                beforeSend: function() {
                    $quickEditRow.find('.qcnc-save-quick-edit').prop('disabled', true).text('Saving...');
                },
                success: function(response) {
                    if (response.success) {
                        // Reload the row
                        qcncLoadCosts(window.qcncCurrentPage);
                    } else {
                        alert(response.data.message || 'Failed to save');
                        $quickEditRow.find('.qcnc-save-quick-edit').prop('disabled', false).text('Update');
                    }
                },
                error: function() {
                    alert('Error saving changes');
                    $quickEditRow.find('.qcnc-save-quick-edit').prop('disabled', false).text('Update');
                }
            });
        });
    }

    /**
     * Create quick edit row
     */
    function createQuickEditRow($row) {
        const productId = $row.data('product-id');
        const cost = parseFloat($row.find('[data-cost-type="cost"]').text()) || 0;
        const packaging = parseFloat($row.find('[data-cost-type="packaging_cost"]').text()) || 0;
        const handling = parseFloat($row.find('[data-cost-type="handling_cost"]').text()) || 0;

        const html = `
            <tr class="qcnc-quick-edit-row" data-product-id="${productId}">
                <td colspan="10">
                    <div class="qcnc-quick-edit-container">
                        <h3>Quick Edit: ${$row.find('.qcnc-product-name').text()}</h3>
                        <div class="qcnc-quick-edit-fields">
                            <label>
                                Cost:
                                <input type="number" name="qe_cost" value="${cost}" step="0.01" min="0" />
                            </label>
                            <label>
                                Packaging Cost:
                                <input type="number" name="qe_packaging_cost" value="${packaging}" step="0.01" min="0" />
                            </label>
                            <label>
                                Handling Cost:
                                <input type="number" name="qe_handling_cost" value="${handling}" step="0.01" min="0" />
                            </label>
                        </div>
                        <div class="qcnc-quick-edit-actions">
                            <button type="button" class="button button-primary qcnc-save-quick-edit">Update</button>
                            <button type="button" class="button qcnc-cancel-quick-edit">Cancel</button>
                        </div>
                    </div>
                </td>
            </tr>
        `;

        return $(html);
    }

    /**
     * Initialize bulk actions
     */
    function initBulkActions() {
        // Select all checkbox
        $(document).on('change', '#qcnc-select-all', function() {
            $('.qcnc-product-checkbox').prop('checked', $(this).is(':checked'));
            updateBulkActionsBar();
        });

        // Individual checkbox
        $(document).on('change', '.qcnc-product-checkbox', function() {
            updateBulkActionsBar();
        });

        // Apply bulk action
        $(document).on('click', '#qcnc-apply-bulk', function() {
            const action = $('#qcnc-bulk-action').val();
            const selectedIds = getSelectedProductIds();

            if (!action) {
                alert('Please select an action');
                return;
            }

            if (selectedIds.length === 0) {
                alert('Please select products');
                return;
            }

            applyBulkAction(action, selectedIds);
        });

        // Bulk increase/decrease
        $(document).on('click', '.qcnc-bulk-adjust', function() {
            const type = $(this).data('type'); // increase or decrease
            const percentage = prompt('Enter percentage (e.g., 10 for 10%):');
            
            if (percentage === null || percentage === '') {
                return;
            }

            const selectedIds = getSelectedProductIds();
            if (selectedIds.length === 0) {
                alert('Please select products');
                return;
            }

            bulkAdjustCosts(selectedIds, type, parseFloat(percentage));
        });
    }

    /**
     * Get selected product IDs
     */
    function getSelectedProductIds() {
        const ids = [];
        $('.qcnc-product-checkbox:checked').each(function() {
            ids.push($(this).val());
        });
        return ids;
    }

    /**
     * Update bulk actions bar
     */
    function updateBulkActionsBar() {
        const count = $('.qcnc-product-checkbox:checked').length;
        
        if (count > 0) {
            if ($('#qcnc-bulk-actions-bar').length === 0) {
                const html = `
                    <div id="qcnc-bulk-actions-bar" class="qcnc-bulk-bar">
                        <span class="qcnc-bulk-count">${count} selected</span>
                        <button type="button" class="button qcnc-bulk-adjust" data-type="increase">Increase by %</button>
                        <button type="button" class="button qcnc-bulk-adjust" data-type="decrease">Decrease by %</button>
                        <button type="button" class="button qcnc-bulk-delete">Delete Selected</button>
                        <button type="button" class="button" id="qcnc-clear-selection">Clear</button>
                    </div>
                `;
                $('.qcnc-toolbar').after(html);
            } else {
                $('#qcnc-bulk-actions-bar .qcnc-bulk-count').text(count + ' selected');
            }
        } else {
            $('#qcnc-bulk-actions-bar').remove();
        }
    }

    /**
     * Bulk adjust costs
     */
    function bulkAdjustCosts(productIds, type, percentage) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_bulk_adjust_costs',
                qcnc_nonce: qcncAdmin.nonce,
                product_ids: productIds,
                adjustment_type: type,
                percentage: percentage
            },
            beforeSend: function() {
                $('#qcnc-bulk-actions-bar').html('<span>Processing...</span>');
            },
            success: function(response) {
                if (response.success) {
                    alert(response.data.message);
                    qcncLoadCosts(window.qcncCurrentPage);
                } else {
                    alert(response.data.message || 'Failed to adjust costs');
                }
            },
            error: function() {
                alert('Error adjusting costs');
            },
            complete: function() {
                $('#qcnc-bulk-actions-bar').remove();
            }
        });
    }

    /**
     * Initialize search functionality
     */
    function initSearch() {
        let searchTimeout;

        $(document).on('keyup', '#qcnc-search-products', function() {
            clearTimeout(searchTimeout);
            const query = $(this).val();

            searchTimeout = setTimeout(function() {
                searchProducts(query);
            }, 500);
        });
    }

    /**
     * Search products
     */
    function searchProducts(query) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_search_products',
                qcnc_nonce: qcncAdmin.nonce,
                query: query
            },
            success: function(response) {
                if (response.success) {
                    renderCostsTable(response.data);
                }
            }
        });
    }

    /**
     * Clear selection
     */
    $(document).on('click', '#qcnc-clear-selection', function() {
        $('.qcnc-product-checkbox').prop('checked', false);
        $('#qcnc-select-all').prop('checked', false);
        updateBulkActionsBar();
    });

})(jQuery);
