/**
 * Admin JavaScript
 *
 * @package QuarkcodeNeuralCommerce
 * @since 1.0.0
 */

(function($) {
    'use strict';

    // Global variables
    window.qcncCurrentPage = 1;

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        initModals();
        initProductCosts();
        initOrderProfitability();
        initBulkOrderProcessing();
        initGatewayFees();
    });

    /**
     * Initialize modal functionality
     */
    function initModals() {
        // Close modal when clicking X or Cancel
        $('.qcnc-modal-close').on('click', function() {
            $(this).closest('.qcnc-modal').hide();
        });

        // Close modal when clicking outside
        $('.qcnc-modal').on('click', function(e) {
            if ($(e.target).hasClass('qcnc-modal')) {
                $(this).hide();
            }
        });
    }

    /**
     * Initialize product costs functionality
     */
    function initProductCosts() {
        // Open cost modal
        $(document).on('click', '.qcnc-edit-cost', function() {
            const productId = $(this).data('product-id');
            const cost = $(this).data('cost');
            const packagingCost = $(this).data('packaging-cost');
            const handlingCost = $(this).data('handling-cost');
            const warehousingCost = $(this).data('warehousing-cost');
            const laborCost = $(this).data('labor-cost');
            const shippingMaterialCost = $(this).data('shipping-material-cost');
            const customCost1 = $(this).data('custom-cost-1');
            const customCost1Label = $(this).data('custom-cost-1-label');
            const customCost2 = $(this).data('custom-cost-2');
            const customCost2Label = $(this).data('custom-cost-2-label');
            const customCost3 = $(this).data('custom-cost-3');
            const customCost3Label = $(this).data('custom-cost-3-label');

            $('#qcnc-product-id').val(productId);
            $('#qcnc-cost').val(cost);
            $('#qcnc-packaging-cost').val(packagingCost);
            $('#qcnc-handling-cost').val(handlingCost);
            $('#qcnc-warehousing-cost').val(warehousingCost || '0.00');
            $('#qcnc-labor-cost').val(laborCost || '0.00');
            $('#qcnc-shipping-material-cost').val(shippingMaterialCost || '0.00');
            $('#qcnc-custom-cost-1').val(customCost1 || '0.00');
            $('#qcnc-custom-cost-1-label').val(customCost1Label || '');
            $('#qcnc-custom-cost-2').val(customCost2 || '0.00');
            $('#qcnc-custom-cost-2-label').val(customCost2Label || '');
            $('#qcnc-custom-cost-3').val(customCost3 || '0.00');
            $('#qcnc-custom-cost-3-label').val(customCost3Label || '');
            
            // Load product details to show product info and notes
            loadProductDetails(productId);
            
            // Calculate totals
            calculateTotalCost();
            
            $('#qcnc-cost-modal').show();
        });


        // Add New Cost button (update existing handler)
        $('#qcnc-add-cost').on('click', function() {
            // Clear all form fields
            $('#qcnc-product-id').val('');
            $('#qcnc-cost').val('0.00');
            $('#qcnc-packaging-cost').val('0.00');
            $('#qcnc-handling-cost').val('0.00');
            $('#qcnc-warehousing-cost').val('0.00');
            $('#qcnc-labor-cost').val('0.00');
            $('#qcnc-shipping-material-cost').val('0.00');
            $('#qcnc-custom-cost-1').val('0.00');
            $('#qcnc-custom-cost-1-label').val('');
            $('#qcnc-custom-cost-2').val('0.00');
            $('#qcnc-custom-cost-2-label').val('');
            $('#qcnc-custom-cost-3').val('0.00');
            $('#qcnc-custom-cost-3-label').val('');
            $('#qcnc-notes').val('');
            
            // Hide product info
            $('#qcnc-product-info').hide();
            
            // Change modal title
            $('#qcnc-cost-modal h2').text('Add New Cost');
            
            // Add product selector if it doesn't exist
            if ($('#qcnc-product-selector-wrapper').length === 0) {
                const productSelectorHtml = `
                    <p id="qcnc-product-selector-wrapper">
                        <label for="qcnc-product-selector">Select Product: *</label>
                        <select id="qcnc-product-selector" style="width: 100%;" required>
                            <option value="">Loading products...</option>
                        </select>
                    </p>
                `;
                $('#qcnc-cost-form').prepend(productSelectorHtml);
                
                // Load products list
                loadProductsList();
            } else {
                // Reset selector if it already exists
                $('#qcnc-product-selector').val('');
            }
            
            // Show product selector, hide product ID
            $('#qcnc-product-selector-wrapper').show();
            
            // Calculate initial totals
            calculateTotalCost();
            
            $('#qcnc-cost-modal').show();
        });


        // When editing, hide product selector
        $(document).on('click', '.qcnc-edit-cost', function() {
            $('#qcnc-product-selector-wrapper').hide();
            $('#qcnc-cost-modal h2').text('Edit Product Cost');
        });

        // Product selector change handler (keep this)
        $(document).on('change', '#qcnc-product-selector', function() {
            const productId = $(this).val();
            $('#qcnc-product-id').val(productId);
            
            if (productId) {
                loadProductDetails(productId);  // Just call it, don't define it here
            } else {
                $('#qcnc-product-info').hide();
            }
        });

        // Add this inside initProductCosts() function
        $('#qcnc-cost, #qcnc-packaging-cost, #qcnc-handling-cost, #qcnc-warehousing-cost, #qcnc-labor-cost, #qcnc-shipping-material-cost, #qcnc-custom-cost-1, #qcnc-custom-cost-2, #qcnc-custom-cost-3').on('input', function() {
            calculateTotalCost();
        });

        // Add this to initProductCosts() function
        $('#qcnc-cost, #qcnc-packaging-cost, #qcnc-handling-cost').on('input', function() {
            calculateMargin();
        });




        // Save cost form
        $('#qcnc-cost-form').on('submit', function(e) {
            e.preventDefault();
            
            // Validate product ID is selected
            const productId = $('#qcnc-product-id').val();
            if (!productId || productId == 0) {
                showModalNotice('Please select a product before saving.', 'error');
                return false;
            }
            
            const formData = {
                action: 'qcnc_update_product_cost',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId,
                cost: $('#qcnc-cost').val(),
                packaging_cost: $('#qcnc-packaging-cost').val(),
                handling_cost: $('#qcnc-handling-cost').val(),
                warehousing_cost: $('#qcnc-warehousing-cost').val(),
                labor_cost: $('#qcnc-labor-cost').val(),
                shipping_material_cost: $('#qcnc-shipping-material-cost').val(),
                custom_cost_1: $('#qcnc-custom-cost-1').val(),
                custom_cost_1_label: $('#qcnc-custom-cost-1-label').val(),
                custom_cost_2: $('#qcnc-custom-cost-2').val(),
                custom_cost_2_label: $('#qcnc-custom-cost-2-label').val(),
                custom_cost_3: $('#qcnc-custom-cost-3').val(),
                custom_cost_3_label: $('#qcnc-custom-cost-3-label').val(),
                notes: $('#qcnc-notes').val()
            };
            
            console.log('Submitting cost data:', formData);
            
            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                beforeSend: function() {
                    // Remove any existing modal notices
                    $('.qcnc-modal-notice').remove();
                    $('#qcnc-cost-form button[type="submit"]').prop('disabled', true).text(qcncAdmin.i18n.saving || 'Saving...');
                },
                success: function(response) {
                    console.log('Save response:', response);
                    
                    if (response.success) {
                        // Success - close modal and show page notice
                        $('#qcnc-cost-modal').hide();
                        qcncLoadCosts(window.qcncCurrentPage);
                        showNotice(response.data.message || 'Cost saved successfully!', 'success');
                    } else {
                        // Error - show inside modal
                        const errorMsg = response.data && response.data.message ? response.data.message : (qcncAdmin.i18n.error || 'An error occurred');
                        showModalNotice(errorMsg, 'error');
                        console.error('Server error:', response);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', { status: status, error: error, responseText: xhr.responseText });
                    
                    // Try to parse error response
                    let errorMsg = qcncAdmin.i18n.error || 'An error occurred';
                    try {
                        const response = JSON.parse(xhr.responseText);
                        if (response.data && response.data.message) {
                            errorMsg = response.data.message;
                        }
                    } catch (e) {
                        // If not JSON, use the status text
                        if (xhr.statusText && xhr.statusText !== 'error') {
                            errorMsg = xhr.statusText;
                        }
                    }
                    showModalNotice(errorMsg, 'error');
                },
                complete: function() {
                    $('#qcnc-cost-form button[type="submit"]').prop('disabled', false).text('Save Cost');
                }
            });
        });



        // Import CSV button
        $('#qcnc-import-csv').on('click', function() {
            $('#qcnc-import-modal').show();
        });

        // Import CSV form
        $('#qcnc-import-form').on('submit', function(e) {
            e.preventDefault();

            const formData = new FormData();
            formData.append('action', 'qcnc_bulk_import_costs');
            formData.append('qcnc_nonce', qcncAdmin.nonce);
            formData.append('csv_file', $('#qcnc-csv-file')[0].files[0]);

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                beforeSend: function() {
                    $('#qcnc-import-progress').show();
                    $('#qcnc-import-form').hide();
                },
                success: function(response) {
                    if (response.success) {
                        $('#qcnc-import-modal').hide();
                        qcncLoadCosts(1);
                        showNotice(response.data.message, 'success');
                    } else {
                        showNotice(response.data.message || qcncAdmin.i18n.error, 'error');
                        if (response.data.errors) {
                            console.error('Import errors:', response.data.errors);
                        }
                    }
                },
                error: function() {
                    showNotice(qcncAdmin.i18n.error, 'error');
                },
                complete: function() {
                    $('#qcnc-import-progress').hide();
                    $('#qcnc-import-form').show();
                }
            });
        });

        // Delete cost
        $(document).on('click', '.qcnc-delete-cost', function() {
            if (!confirm(qcncAdmin.i18n.confirmDelete)) {
                return;
            }

            const costId = $(this).data('cost-id');

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'qcnc_delete_cost',
                    qcnc_nonce: qcncAdmin.nonce,
                    cost_id: costId
                },
                success: function(response) {
                    if (response.success) {
                        qcncLoadCosts(window.qcncCurrentPage);
                        showNotice(response.data.message, 'success');
                    } else {
                        showNotice(response.data.message || qcncAdmin.i18n.error, 'error');
                    }
                },
                error: function() {
                    showNotice(qcncAdmin.i18n.error, 'error');
                }
            });
        });

        // View cost history
        $(document).on('click', '.qcnc-view-history', function() {
            const productId = $(this).data('product-id');
            loadCostHistory(productId);
        });

    }

    /**
     * Load products list for selector
     */
    function loadProductsList() {
        console.log('Loading products list...');
        
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_products_list',
                qcnc_nonce: qcncAdmin.nonce,
                limit: 500
            },
            beforeSend: function() {
                console.log('AJAX request sent');
                $('#qcnc-product-selector').html('<option value="">Loading products...</option>');
            },
            success: function(response) {
                console.log('AJAX Response:', response);
                
                if (response.success && response.data && response.data.length > 0) {
                    console.log('Products loaded:', response.data.length);
                    let options = '<option value="">Select a product...</option>';
                    response.data.forEach(function(product) {
                        options += `<option value="${product.id}">${product.name} ${product.sku ? '(' + product.sku + ')' : ''}</option>`;
                    });
                    $('#qcnc-product-selector').html(options);
                } else {
                    console.error('No products found or invalid response');
                    $('#qcnc-product-selector').html('<option value="">No products found</option>');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error:', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText
                });
                $('#qcnc-product-selector').html('<option value="">Error loading products</option>');
            }
        });
    }

    /**
     * Load and display product details
     */
    function loadProductDetails(productId) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_product_details',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId
            },
            beforeSend: function() {
                $('#qcnc-product-info').show().html('<p style="margin: 10px;">Loading product details...</p>');
            },
            success: function(response) {
                console.log('Product details response:', response);
                
                if (response.success && response.data) {
                    const product = response.data;
                    
                    // Build the info HTML
                    let infoHtml = `
                        <p style="margin: 5px 0;">
                            <strong>Product:</strong> ${product.name}
                        </p>
                        <p style="margin: 5px 0;">
                            <strong>SKU:</strong> ${product.sku || 'N/A'}
                        </p>
                        <p style="margin: 5px 0;">
                            <strong>Regular Price:</strong> 
                            <span style="color: #666;">${product.regular_price_formatted}</span>
                        </p>
                    `;
                    
                    // Show sale price if exists
                    if (product.sale_price && parseFloat(product.sale_price) > 0) {
                        infoHtml += `
                            <p style="margin: 5px 0;">
                                <strong>Sale Price:</strong> 
                                <span style="color: #d63638; font-weight: bold;">${product.sale_price_formatted}</span>
                            </p>
                        `;
                    }
                    
                    infoHtml += `
                        <p style="margin: 5px 0;">
                            <strong>Current Price:</strong> 
                            <span id="qcnc-info-current-price" style="color: #2271b1; font-weight: bold; font-size: 1.1em;">${product.price_formatted}</span>
                        </p>
                    `;
                    
                    $('#qcnc-product-info').html(infoHtml);
                    
                    // REPLACE THE SECTION BELOW ↓↓↓
                    // Load existing cost if available
                    if (product.existing_cost) {
                        $('#qcnc-cost').val(product.existing_cost.cost || '0.00');
                        $('#qcnc-packaging-cost').val(product.existing_cost.packaging_cost || '0.00');
                        $('#qcnc-handling-cost').val(product.existing_cost.handling_cost || '0.00');
                        $('#qcnc-warehousing-cost').val(product.existing_cost.warehousing_cost || '0.00');
                        $('#qcnc-labor-cost').val(product.existing_cost.labor_cost || '0.00');
                        $('#qcnc-shipping-material-cost').val(product.existing_cost.shipping_material_cost || '0.00');
                        $('#qcnc-custom-cost-1').val(product.existing_cost.custom_cost_1 || '0.00');
                        $('#qcnc-custom-cost-1-label').val(product.existing_cost.custom_cost_1_label || '');
                        $('#qcnc-custom-cost-2').val(product.existing_cost.custom_cost_2 || '0.00');
                        $('#qcnc-custom-cost-2-label').val(product.existing_cost.custom_cost_2_label || '');
                        $('#qcnc-custom-cost-3').val(product.existing_cost.custom_cost_3 || '0.00');
                        $('#qcnc-custom-cost-3-label').val(product.existing_cost.custom_cost_3_label || '');
                        $('#qcnc-notes').val(product.existing_cost.notes || '');
                    } else {
                        // Reset to default values
                        $('#qcnc-cost').val('0.00');
                        $('#qcnc-packaging-cost').val('0.00');
                        $('#qcnc-handling-cost').val('0.00');
                        $('#qcnc-warehousing-cost').val('0.00');
                        $('#qcnc-labor-cost').val('0.00');
                        $('#qcnc-shipping-material-cost').val('0.00');
                        $('#qcnc-custom-cost-1').val('0.00');
                        $('#qcnc-custom-cost-1-label').val('');
                        $('#qcnc-custom-cost-2').val('0.00');
                        $('#qcnc-custom-cost-2-label').val('');
                        $('#qcnc-custom-cost-3').val('0.00');
                        $('#qcnc-custom-cost-3-label').val('');
                        $('#qcnc-notes').val('');
                    }
                    
                    // Calculate totals after loading costs
                    calculateTotalCost();
                    // ↑↑↑ END OF REPLACEMENT SECTION
                    
                } else {
                    console.error('Invalid response:', response);
                    $('#qcnc-product-info').html('<p style="color: #d63638;">Error loading product details</p>');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error:', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText
                });
                $('#qcnc-product-info').html('<p style="color: #d63638;">Error loading product details</p>');
            }
        });
    }


    
    /**
     * Load existing product cost data
     */
    function loadProductCostData(productId) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_product_cost',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId
            },
            success: function(response) {
                if (response.success && response.data) {
                    $('#qcnc-cost').val(response.data.cost || '0.00');
                    $('#qcnc-packaging-cost').val(response.data.packaging_cost || '0.00');
                    $('#qcnc-handling-cost').val(response.data.handling_cost || '0.00');
                }
            }
        });
    }


    /**
     * Load product costs with pagination
     */
    window.qcncLoadCosts = function(page) {
        window.qcncCurrentPage = page;

        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_product_costs_list',
                qcnc_nonce: qcncAdmin.nonce,
                page: page,
                per_page: 20
            },
            beforeSend: function() {
                $('#qcnc-costs-tbody').html('<tr><td colspan="10" class="qcnc-loading">Loading...</td></tr>');
            },
            success: function(response) {
                if (response.success) {
                    renderCostsTable(response.data);
                } else {
                    $('#qcnc-costs-tbody').html('<tr><td colspan="10" class="qcnc-text-center">Error loading data</td></tr>');
                }
            },
            error: function() {
                $('#qcnc-costs-tbody').html('<tr><td colspan="10" class="qcnc-text-center">Error loading data</td></tr>');
            }
        });
    };

    /**
     * Render costs table
     */
    function renderCostsTable(data) {
        let html = '';

        if (data.costs.length === 0) {
            html = '<tr><td colspan="10" class="qcnc-text-center">No costs found</td></tr>';
        } else {
            data.costs.forEach(function(cost) {
                const price = parseFloat(cost.price) || 0;
                
                // Calculate total cost with ALL cost components
                const totalCost = parseFloat(cost.cost || 0) + 
                                parseFloat(cost.packaging_cost || 0) + 
                                parseFloat(cost.handling_cost || 0) +
                                parseFloat(cost.warehousing_cost || 0) +
                                parseFloat(cost.labor_cost || 0) +
                                parseFloat(cost.shipping_material_cost || 0) +
                                parseFloat(cost.custom_cost_1 || 0) +
                                parseFloat(cost.custom_cost_2 || 0) +
                                parseFloat(cost.custom_cost_3 || 0);
                
                const margin = price > 0 ? ((price - totalCost) / price * 100) : 0;
                const marginClass = margin >= 15 ? 'qcnc-text-success' : (margin >= 0 ? 'qcnc-text-warning' : 'qcnc-text-danger');

                // ADD THIS SECTION HERE ↓↓↓
                // Build cost breakdown text for tooltip
                const costBreakdown = 
                    'Base: $' + parseFloat(cost.cost || 0).toFixed(2) + '\n' +
                    'Packaging: $' + parseFloat(cost.packaging_cost || 0).toFixed(2) + '\n' +
                    'Handling: $' + parseFloat(cost.handling_cost || 0).toFixed(2) + '\n' +
                    'Warehousing: $' + parseFloat(cost.warehousing_cost || 0).toFixed(2) + '\n' +
                    'Labor: $' + parseFloat(cost.labor_cost || 0).toFixed(2) + '\n' +
                    'Shipping Mat.: $' + parseFloat(cost.shipping_material_cost || 0).toFixed(2) + '\n' +
                    (cost.custom_cost_1 ? 'Custom 1: $' + parseFloat(cost.custom_cost_1).toFixed(2) + '\n' : '') +
                    (cost.custom_cost_2 ? 'Custom 2: $' + parseFloat(cost.custom_cost_2).toFixed(2) + '\n' : '') +
                    (cost.custom_cost_3 ? 'Custom 3: $' + parseFloat(cost.custom_cost_3).toFixed(2) : '');
                // ↑↑↑ END OF ADDED SECTION

                // Add class for missing cost
                const rowClass = totalCost === 0 ? 'qcnc-missing-cost' : '';
                
                html += '<tr class="' + rowClass + '">';
                html += '<td>' + cost.product_id + '</td>';
                html += '<td>' + (cost.product_name || 'N/A') + '</td>';
                html += '<td>' + (cost.sku || 'N/A') + '</td>';
                html += '<td>$' + price.toFixed(2) + '</td>';
                html += '<td>$' + parseFloat(cost.cost).toFixed(2) + '</td>';
                html += '<td>$' + parseFloat(cost.packaging_cost || 0).toFixed(2) + '</td>';
                html += '<td>$' + parseFloat(cost.handling_cost || 0).toFixed(2) + '</td>';
                
                // REPLACE THIS LINE ↓↓↓
                // OLD: html += '<td><strong>$' + totalCost.toFixed(2) + '</strong></td>';
                // NEW:
                html += '<td title="' + costBreakdown + '" style="cursor: help;"><strong>$' + totalCost.toFixed(2) + '</strong></td>';
                // ↑↑↑ END OF REPLACEMENT
                
                html += '<td class="' + marginClass + '">' + margin.toFixed(2) + '%</td>';
                html += '<td>';
                html += '<button class="qcnc-action-btn qcnc-edit-cost" ';
                html += 'data-product-id="' + cost.product_id + '" ';
                html += 'data-cost="' + cost.cost + '" ';
                html += 'data-packaging-cost="' + cost.packaging_cost + '" ';
                html += 'data-handling-cost="' + cost.handling_cost + '" ';
                html += 'data-warehousing-cost="' + (cost.warehousing_cost || 0) + '" ';
                html += 'data-labor-cost="' + (cost.labor_cost || 0) + '" ';
                html += 'data-shipping-material-cost="' + (cost.shipping_material_cost || 0) + '" ';
                html += 'data-custom-cost-1="' + (cost.custom_cost_1 || 0) + '" ';
                html += 'data-custom-cost-1-label="' + (cost.custom_cost_1_label || '') + '" ';
                html += 'data-custom-cost-2="' + (cost.custom_cost_2 || 0) + '" ';
                html += 'data-custom-cost-2-label="' + (cost.custom_cost_2_label || '') + '" ';
                html += 'data-custom-cost-3="' + (cost.custom_cost_3 || 0) + '" ';
                html += 'data-custom-cost-3-label="' + (cost.custom_cost_3_label || '') + '">Edit</button>';
                html += '<button class="qcnc-action-btn qcnc-secondary qcnc-view-history" data-product-id="' + cost.product_id + '">History</button>';
                html += '<button class="qcnc-action-btn qcnc-danger qcnc-delete-cost" data-cost-id="' + cost.id + '">Delete</button>';
                html += '</td>';
                html += '</tr>';
            });
        }

        $('#qcnc-costs-tbody').html(html);

        // Render pagination
        renderPagination(data, 'qcnc-pagination', 'qcncLoadCosts');
    }




    /**
     * Initialize order profitability functionality
     */
    function initOrderProfitability() {
        // Recalculate order profit
        $(document).on('click', '.qcnc-recalc-order', function() {
            const orderId = $(this).data('order-id');

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'qcnc_recalculate_order',
                    qcnc_nonce: qcncAdmin.nonce,
                    order_id: orderId
                },
                beforeSend: function() {
                    $(this).prop('disabled', true);
                },
                success: function(response) {
                    if (response.success) {
                        qcncLoadOrders(window.qcncCurrentPage);
                        showNotice(response.data.message, 'success');
                    } else {
                        showNotice(response.data.message || qcncAdmin.i18n.error, 'error');
                    }
                },
                error: function() {
                    showNotice(qcncAdmin.i18n.error, 'error');
                }
            });
        });

        // View order details
        $(document).on('click', '.qcnc-view-order-details', function() {
            const orderId = $(this).data('order-id');
            loadOrderDetails(orderId);
        });
    }

    /**
     * Load order details and display in modal
     */
    function loadOrderDetails(orderId) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_order_details',
                qcnc_nonce: qcncAdmin.nonce,
                order_id: orderId
            },
            beforeSend: function() {
                $('#qcnc-order-details-modal').show();
                $('#qcnc-order-details-content').html('<p style="text-align: center; padding: 20px;">Loading...</p>');
            },
            success: function(response) {
                if (response.success) {
                    renderOrderDetails(response.data);
                } else {
                    $('#qcnc-order-details-content').html(
                        '<p style="color: #d63638; padding: 20px;">' + 
                        (response.data.message || 'Error loading order details') + 
                        '</p>'
                    );
                }
            },
            error: function() {
                $('#qcnc-order-details-content').html(
                    '<p style="color: #d63638; padding: 20px;">Error loading order details</p>'
                );
            }
        });
    }

    /**
     * Render order details in modal
     */
    function renderOrderDetails(data) {
        const info = data.order_info;
        const summary = data.profit_summary;
        const items = data.items;
        
        let html = '';
        
        // Order Info Section
        html += '<div class="qcnc-order-info" style="background: #f0f0f1; padding: 15px; margin-bottom: 20px;">';
        html += '<h3 style="margin-top: 0;">Order #' + info.order_number + '</h3>';
        html += '<p><strong>Date:</strong> ' + info.order_date + '</p>';
        html += '<p><strong>Status:</strong> <span class="order-status status-' + info.order_status + '">' + info.order_status + '</span></p>';
        html += '<p><strong>Customer:</strong> ' + info.customer_name + ' (' + info.customer_email + ')</p>';
        html += '</div>';
        
        // Profit Summary
        html += '<div class="qcnc-profit-summary">';
        html += '<h3>Profit Summary</h3>';
        html += '<table class="widefat" style="margin-bottom: 20px;">';
        html += '<tbody>';
        html += '<tr><td><strong>Revenue (excl. tax):</strong></td><td>' + formatPrice(summary.revenue, summary.currency) + '</td></tr>';
        html += '<tr><td><strong>Cost of Goods Sold:</strong></td><td style="color: #d63638;">-' + formatPrice(summary.cogs_total, summary.currency) + '</td></tr>';
        html += '<tr><td><strong>Shipping Cost:</strong></td><td style="color: #d63638;">-' + formatPrice(summary.shipping_cost, summary.currency) + '</td></tr>';
        html += '<tr><td><strong>Payment Gateway Fee:</strong></td><td style="color: #d63638;">-' + formatPrice(summary.payment_fee, summary.currency) + '</td></tr>';
        if (summary.discount_amount > 0) {
            html += '<tr><td><strong>Discount:</strong></td><td style="color: #d63638;">-' + formatPrice(summary.discount_amount, summary.currency) + '</td></tr>';
        }
        html += '<tr style="border-top: 2px solid #ddd;"><td><strong>Gross Profit:</strong></td><td><strong>' + formatPrice(summary.gross_profit, summary.currency) + '</strong></td></tr>';
        html += '<tr style="border-top: 2px solid #000; font-size: 1.1em;"><td><strong>Net Profit:</strong></td><td><strong style="color: ' + (summary.net_profit >= 0 ? '#00a32a' : '#d63638') + ';">' + formatPrice(summary.net_profit, summary.currency) + '</strong></td></tr>';
        html += '<tr><td><strong>Margin:</strong></td><td><strong style="color: ' + getMarginColor(summary.margin_pct) + ';">' + summary.margin_pct.toFixed(2) + '%</strong></td></tr>';
        html += '</tbody>';
        html += '</table>';
        html += '</div>';
        
        // Items Breakdown
        html += '<div class="qcnc-items-breakdown">';
        html += '<h3>Items Breakdown</h3>';
        html += '<table class="widefat striped">';
        html += '<thead>';
        html += '<tr>';
        html += '<th>Product</th>';
        html += '<th>SKU</th>';
        html += '<th>Qty</th>';
        html += '<th>Cost/Unit</th>';
        html += '<th>Revenue</th>';
        html += '<th>COGS</th>';
        html += '<th>Profit</th>';
        html += '<th>Margin %</th>';
        html += '</tr>';
        html += '</thead>';
        html += '<tbody>';
        
        items.forEach(function(item) {
            html += '<tr>';
            html += '<td>' + item.product_name + '</td>';
            html += '<td>' + (item.sku || 'N/A') + '</td>';
            html += '<td>' + item.quantity + '</td>';
            html += '<td>' + formatPrice(item.cost_per_unit, summary.currency) + '</td>';
            html += '<td>' + formatPrice(item.revenue, summary.currency) + '</td>';
            html += '<td>' + formatPrice(item.cogs, summary.currency) + '</td>';
            html += '<td style="color: ' + (item.profit >= 0 ? '#00a32a' : '#d63638') + ';"><strong>' + formatPrice(item.profit, summary.currency) + '</strong></td>';
            html += '<td style="color: ' + getMarginColor(item.margin_pct) + ';">' + item.margin_pct.toFixed(2) + '%</td>';
            html += '</tr>';
        });
        
        html += '</tbody>';
        html += '</table>';
        html += '</div>';
        
        $('#qcnc-order-details-content').html(html);
    }

    /**
     * Format price
     */
    function formatPrice(amount, currency) {
        const symbol = currency === 'USD' ? '$' : currency;
        return symbol + parseFloat(amount).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
    }

    /**
     * Get color based on margin
     */
    function getMarginColor(margin) {
        if (margin >= 15) return '#00a32a';
        if (margin >= 0) return '#dba617';
        return '#d63638';
    }


    /**
     * Load orders with pagination
     */
    window.qcncLoadOrders = function(page) {
        window.qcncCurrentPage = page;

        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_orders_profit',
                qcnc_nonce: qcncAdmin.nonce,
                page: page,
                per_page: 20
            },
            beforeSend: function() {
                $('#qcnc-orders-tbody').html('<tr><td colspan="10" class="qcnc-loading">Loading...</td></tr>');
            },
            success: function(response) {
                if (response.success) {
                    renderOrdersTable(response.data);
                } else {
                    $('#qcnc-orders-tbody').html('<tr><td colspan="10" class="qcnc-text-center">Error loading data</td></tr>');
                }
            },
            error: function() {
                $('#qcnc-orders-tbody').html('<tr><td colspan="10" class="qcnc-text-center">Error loading data</td></tr>');
            }
        });
    };

    /**
     * Render orders table
     */
    function renderOrdersTable(data) {
        let html = '';

        if (data.orders.length === 0) {
            html = '<tr><td colspan="10" class="qcnc-text-center">No orders found</td></tr>';
        } else {
            data.orders.forEach(function(order) {
                const netProfit = parseFloat(order.net_profit);
                const margin = parseFloat(order.margin_pct);
                const profitClass = netProfit >= 0 ? 'qcnc-text-success' : 'qcnc-text-danger';
                const marginClass = margin >= 15 ? 'qcnc-text-success' : (margin >= 0 ? 'qcnc-text-warning' : 'qcnc-text-danger');

                html += '<tr>';
                html += '<td>#' + (order.order_number || order.order_id) + '</td>';
                html += '<td>' + (order.order_date || 'N/A') + '</td>';
                html += '<td>' + (order.customer_name || 'Guest') + '</td>';
                html += '<td>' + (order.order_status || 'N/A') + '</td>';
                html += '<td>' + parseFloat(order.revenue).toFixed(2) + '</td>';
                html += '<td>' + parseFloat(order.cogs_total).toFixed(2) + '</td>';
                html += '<td>' + (parseFloat(order.payment_fee) + parseFloat(order.shipping_cost)).toFixed(2) + '</td>';
                html += '<td class="' + profitClass + '"><strong>' + netProfit.toFixed(2) + '</strong></td>';
                html += '<td class="' + marginClass + '">' + margin.toFixed(2) + '%</td>';
                html += '<td>';
                html += '<button class="qcnc-action-btn qcnc-secondary qcnc-recalc-order" data-order-id="' + order.order_id + '">Recalc</button>';
                html += '<button class="qcnc-action-btn qcnc-view-order-details" data-order-id="' + order.order_id + '">Details</button>';
                html += '</td>';
                html += '</tr>';
            });
        }

        $('#qcnc-orders-tbody').html(html);

        // Render pagination
        renderPagination(data, 'qcnc-orders-pagination', 'qcncLoadOrders');
    }

    /**
     * Render pagination
     */
    function renderPagination(data, containerId, loadFunction) {
        let html = '';

        html += '<button class="qcnc-pagination-btn" onclick="' + loadFunction + '(1)" ' + (data.page <= 1 ? 'disabled' : '') + '>&laquo; First</button>';
        html += '<button class="qcnc-pagination-btn" onclick="' + loadFunction + '(' + (data.page - 1) + ')" ' + (data.page <= 1 ? 'disabled' : '') + '>&lsaquo; Prev</button>';
        html += '<span class="qcnc-page-info">Page ' + data.page + ' of ' + data.total_pages + '</span>';
        html += '<button class="qcnc-pagination-btn" onclick="' + loadFunction + '(' + (data.page + 1) + ')" ' + (data.page >= data.total_pages ? 'disabled' : '') + '>Next &rsaquo;</button>';
        html += '<button class="qcnc-pagination-btn" onclick="' + loadFunction + '(' + data.total_pages + ')" ' + (data.page >= data.total_pages ? 'disabled' : '') + '>Last &raquo;</button>';

        $('#' + containerId).html(html);
    }

    /**
     * Show notice
     */
    function showNotice(message, type) {
        const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        const notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');

        $('.wrap > h1').after(notice);

        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }

    /**
     * Show notice inside modal (for form errors)
     */
    function showModalNotice(message, type) {
        // Remove any existing modal notices
        $('.qcnc-modal-notice').remove();
        
        const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        const iconClass = type === 'success' ? 'dashicons-yes' : 'dashicons-warning';
        
        const notice = $('<div class="notice qcnc-modal-notice ' + noticeClass + '" style="margin: 15px 0; padding: 10px 15px; border-left-width: 4px;">' +
            '<p style="margin: 0; display: flex; align-items: center;">' +
            '<span class="dashicons ' + iconClass + '" style="margin-right: 8px;"></span>' +
            message +
            '</p>' +
            '</div>');
        
        // Insert at the top of the modal content, before the form
        $('#qcnc-cost-form').before(notice);
        
        setTimeout(function() {
            // Find the scrollable modal container (try multiple selectors)
            const modalContent = $('.qcnc-modal-content');
            const modalBody = $('.qcnc-modal-body');
            const modal = $('.qcnc-modal.qcnc-cost-modal');
            
            // Scroll the modal content to top
            if (modalContent.length) {
                modalContent.animate({ scrollTop: 0 }, 400);
            }
            if (modalBody.length) {
                modalBody.animate({ scrollTop: 0 }, 400);
            }
            if (modal.length) {
                modal.find('.qcnc-modal-content, .qcnc-modal-body').animate({ scrollTop: 0 }, 400);
            }
            
            // Also scroll the notice itself into view
            notice[0].scrollIntoView({ 
                behavior: 'smooth', 
                block: 'start',
                inline: 'nearest'
            });
            
            // Add a pulse animation to draw attention
            notice.css({
                animation: 'qcnc-notice-pulse 0.6s ease-in-out'
            });
        }, 100); // Small delay to ensure DOM is updated
        
        // Auto-dismiss success messages after 5 seconds
        if (type === 'success') {
            setTimeout(function() {
                notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
       
    }


        /**
     * Initialize bulk order processing
     */
    function initBulkOrderProcessing() {
        // Get order count on page load
        getOrderCount();

        // Process all orders
        $('#qcnc-bulk-process-all').on('click', function() {
            if (confirm('This will recalculate profit for ALL orders. Continue?')) {
                startBulkProcessing('all');
            }
        });

        // Process missing orders only
        $('#qcnc-bulk-process-missing').on('click', function() {
            if (confirm('This will calculate profit for orders without existing data. Continue?')) {
                startBulkProcessing('missing');
            }
        });

        // Reset all data button
        $('#qcnc-reset-all-data').on('click', function() {
            if (!confirm('⚠️ WARNING: This will delete ALL profit calculations and you will need to reprocess orders. Continue?')) {
                return;
            }
            
            if (!confirm('Are you absolutely sure? This cannot be undone!')) {
                return;
            }

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'qcnc_reset_profit_data',
                    qcnc_nonce: qcncAdmin.nonce
                },
                beforeSend: function() {
                    $('#qcnc-reset-all-data').prop('disabled', true).text('Deleting...');
                },
                success: function(response) {
                    if (response.success) {
                        alert('✅ All profit data deleted. Please click "Process All Orders" to recalculate.');
                        location.reload();
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to reset data'));
                    }
                },
                error: function() {
                    alert('Error resetting data');
                },
                complete: function() {
                    $('#qcnc-reset-all-data').prop('disabled', false).text('Reset All Profit Data');
                }
            });
        });

    }

    /**
     * Get order count
     */
    function getOrderCount() {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_order_count',
                qcnc_nonce: qcncAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    $('#qcnc-order-count').text(
                        'Total: ' + response.data.total + ' orders | ' +
                        'Processed: ' + response.data.processed + ' | ' +
                        'Missing: ' + response.data.missing
                    );
                }
            }
        });
    }

    /**
     * Start bulk processing
     */
    function startBulkProcessing(mode) {
        $('#qcnc-bulk-process-all, #qcnc-bulk-process-missing').prop('disabled', true);
        $('#qcnc-bulk-progress').show();
        $('#qcnc-bulk-results').hide();

        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_start_bulk_processing',
                qcnc_nonce: qcncAdmin.nonce,
                mode: mode
            },
            success: function(response) {
                if (response.success) {
                    processBatch(0, response.data.total, response.data.batch_size, mode);
                } else {
                    alert(response.data.message || 'Failed to start processing');
                    resetBulkProcessing();
                }
            },
            error: function() {
                alert('Error starting bulk processing');
                resetBulkProcessing();
            }
        });
    }

    /**
     * Process batch of orders
     */
    function processBatch(offset, total, batchSize, mode) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_process_order_batch',
                qcnc_nonce: qcncAdmin.nonce,
                offset: offset,
                batch_size: batchSize,
                mode: mode
            },
            success: function(response) {
                if (response.success) {
                    const processed = offset + response.data.processed;
                    const percentage = Math.min(100, Math.round((processed / total) * 100));

                    // Update progress
                    $('#qcnc-progress-bar').css('width', percentage + '%');
                    $('#qcnc-progress-text').text(processed + ' / ' + total + ' orders processed');
                    $('#qcnc-progress-status').text('Processing orders ' + (offset + 1) + ' to ' + (offset + response.data.processed) + '...');

                    // Continue or finish
                    if (processed < total) {
                        processBatch(processed, total, batchSize, mode);
                    } else {
                        finishBulkProcessing(response.data);
                    }
                } else {
                    alert(response.data.message || 'Error processing batch');
                    resetBulkProcessing();
                }
            },
            error: function() {
                alert('Error processing orders. Please try again.');
                resetBulkProcessing();
            }
        });
    }

    /**
     * Finish bulk processing
     */
    function finishBulkProcessing(data) {
        $('#qcnc-bulk-progress').hide();
        $('#qcnc-bulk-results').show();
        
        let summary = 'Successfully processed ' + data.total_processed + ' orders. ';
        if (data.errors > 0) {
            summary += data.errors + ' errors occurred.';
        }
        
        $('#qcnc-results-summary').text(summary);
        
        // Refresh order count and table
        getOrderCount();
        qcncLoadOrders(1);
        
        // Re-enable buttons
        setTimeout(function() {
            resetBulkProcessing();
        }, 3000);
    }

    /**
     * Reset bulk processing UI
     */
    function resetBulkProcessing() {
        $('#qcnc-bulk-process-all, #qcnc-bulk-process-missing').prop('disabled', false);
        $('#qcnc-progress-bar').css('width', '0%');
    }


        /**
     * Clear dashboard cache
     */
    $('#qcnc-clear-dashboard-cache').on('click', function() {
        if (!confirm('Clear dashboard cache and reload?')) {
            return;
        }

        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_clear_dashboard_cache',
                qcnc_nonce: qcncAdmin.nonce
            },
            beforeSend: function() {
                $('#qcnc-clear-dashboard-cache').prop('disabled', true).text('Clearing...');
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message, 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showNotice(response.data.message || 'Error clearing cache', 'error');
                }
            },
            error: function() {
                showNotice('Error clearing cache', 'error');
            },
            complete: function() {
                $('#qcnc-clear-dashboard-cache').prop('disabled', false).text('Clear Cache & Refresh');
            }
        });
    });

        /**
     * Initialize gateway fee management
     */
    function initGatewayFees() {
        // Add gateway fee
        $('#qcnc-add-gateway-fee').on('click', function() {
            $('#qcnc-gateway-id').val('');
            $('#qcnc-gateway-form')[0].reset();
            $('#qcnc-gateway-modal-title').text('Add Gateway Fee');
            $('#qcnc-fee-type').trigger('change');
            $('#qcnc-gateway-modal').show();
        });

        // Edit gateway fee
        $(document).on('click', '.qcnc-edit-gateway', function() {
            const gatewayId = $(this).data('id');
            
            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'qcnc_get_gateway_fee',
                    qcnc_nonce: qcncAdmin.nonce,
                    gateway_id: gatewayId
                },
                success: function(response) {
                    if (response.success) {
                        const data = response.data;
                        $('#qcnc-gateway-id').val(data.id);
                        $('#qcnc-gateway-select').val(data.gateway_id);
                        $('#qcnc-gateway-name').val(data.gateway_name);
                        $('#qcnc-fee-type').val(data.fee_type).trigger('change');
                        $('#qcnc-percentage-fee').val(data.percentage_fee);
                        $('#qcnc-fixed-fee').val(data.fixed_fee);
                        $('#qcnc-gateway-currency').val(data.currency);
                        $('#qcnc-gateway-active').prop('checked', data.is_active == 1);
                        
                        $('#qcnc-gateway-modal-title').text('Edit Gateway Fee');
                        $('#qcnc-gateway-modal').show();
                    }
                }
            });
        });

        // Save gateway fee
        $('#qcnc-gateway-form').on('submit', function(e) {
            e.preventDefault();

            const formData = {
                action: 'qcnc_save_gateway_fee',
                qcnc_nonce: qcncAdmin.nonce,
                id: $('#qcnc-gateway-id').val(),
                gateway_id: $('#qcnc-gateway-select').val(),
                gateway_name: $('#qcnc-gateway-name').val(),
                fee_type: $('#qcnc-fee-type').val(),
                percentage_fee: $('#qcnc-percentage-fee').val(),
                fixed_fee: $('#qcnc-fixed-fee').val(),
                currency: $('#qcnc-gateway-currency').val(),
                is_active: $('#qcnc-gateway-active').is(':checked') ? 1 : 0
            };

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        $('#qcnc-gateway-modal').hide();
                        showNotice(response.data.message, 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotice(response.data.message || 'Error saving gateway fee', 'error');
                    }
                },
                error: function() {
                    showNotice('Error saving gateway fee', 'error');
                }
            });
        });

        // Delete gateway fee
        $(document).on('click', '.qcnc-delete-gateway', function() {
            if (!confirm('Are you sure you want to delete this gateway fee?')) {
                return;
            }

            const gatewayId = $(this).data('id');

            $.ajax({
                url: qcncAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'qcnc_delete_gateway_fee',
                    qcnc_nonce: qcncAdmin.nonce,
                    gateway_id: gatewayId
                },
                success: function(response) {
                    if (response.success) {
                        showNotice(response.data.message, 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotice(response.data.message || 'Error deleting gateway fee', 'error');
                    }
                },
                error: function() {
                    showNotice('Error deleting gateway fee', 'error');
                }
            });
        });
    }



    /**
     * Load cost history for a product
     */
    function loadCostHistory(productId) {
        $.ajax({
            url: qcncAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'qcnc_get_cost_history',
                qcnc_nonce: qcncAdmin.nonce,
                product_id: productId
            },
            beforeSend: function() {
                $('#qcnc-history-modal').show();
                $('#qcnc-history-content').html('<p style="text-align: center; padding: 20px;">Loading history...</p>');
            },
            success: function(response) {
                if (response.success) {
                    renderCostHistory(response.data);
                } else {
                    $('#qcnc-history-content').html(
                        '<p style="color: #d63638; padding: 20px;">' + 
                        (response.data.message || 'No history found') + 
                        '</p>'
                    );
                }
            },
            error: function() {
                $('#qcnc-history-content').html(
                    '<p style="color: #d63638; padding: 20px;">Error loading history</p>'
                );
            }
        });
    }

    /**
     * Render cost history in modal
     */
    function renderCostHistory(data) {
        if (!data.history || data.history.length === 0) {
            $('#qcnc-history-content').html('<p>No historical cost data found for this product.</p>');
            return;
        }

        let html = '';
        
        // Product info
        html += '<div style="background: #f0f0f1; padding: 15px; margin-bottom: 20px; border-left: 4px solid #2271b1;">';
        html += '<h3 style="margin-top: 0;">' + data.product_name + '</h3>';
        html += '<p><strong>Product ID:</strong> ' + data.product_id + '</p>';
        if (data.sku) {
            html += '<p><strong>SKU:</strong> ' + data.sku + '</p>';
        }
        html += '</div>';

        // History table
        html += '<table class="widefat striped">';
        html += '<thead>';
        html += '<tr>';
        html += '<th>Status</th>';
        html += '<th>Cost</th>';
        html += '<th>Packaging</th>';
        html += '<th>Handling</th>';
        html += '<th>Total Cost</th>';
        html += '<th>Effective From</th>';
        html += '<th>Effective To</th>';
        html += '<th>Created By</th>';
        html += '</tr>';
        html += '</thead>';
        html += '<tbody>';

        data.history.forEach(function(item) {
            const isCurrent = item.is_current == 1;
            const totalCost = parseFloat(item.cost) + parseFloat(item.packaging_cost) + parseFloat(item.handling_cost);
            
            html += '<tr' + (isCurrent ? ' style="background: #d7f8e0;"' : '') + '>';
            html += '<td>';
            if (isCurrent) {
                html += '<span style="color: #00a32a; font-weight: bold;">✓ Current</span>';
            } else {
                html += '<span style="color: #999;">Historical</span>';
            }
            html += '</td>';
            html += '<td>$' + parseFloat(item.cost).toFixed(2) + '</td>';
            html += '<td>$' + parseFloat(item.packaging_cost).toFixed(2) + '</td>';
            html += '<td>$' + parseFloat(item.handling_cost).toFixed(2) + '</td>';
            html += '<td><strong>$' + totalCost.toFixed(2) + '</strong></td>';
            html += '<td>' + formatDateTime(item.effective_from) + '</td>';
            html += '<td>' + (item.effective_to ? formatDateTime(item.effective_to) : '<em>Present</em>') + '</td>';
            html += '<td>' + (item.created_by || 'N/A') + '</td>';
            html += '</tr>';
        });

        html += '</tbody>';
        html += '</table>';

        // Summary
        html += '<div style="margin-top: 20px; padding: 15px; background: #f9f9f9; border-left: 4px solid #2271b1;">';
        html += '<h4 style="margin-top: 0;">ℹ️ How Historical Costs Work</h4>';
        html += '<p>When calculating profit for past orders, the system uses the cost that was effective on the order date.</p>';
        html += '<ul>';
        html += '<li><strong>Current cost:</strong> Used for new orders and orders placed after this cost was set</li>';
        html += '<li><strong>Historical costs:</strong> Used for orders placed during their effective period</li>';
        html += '</ul>';
        html += '<p>Example: If you changed cost from $10 to $15 on Nov 20, orders before Nov 20 use $10, orders after use $15.</p>';
        html += '</div>';

        $('#qcnc-history-content').html(html);
    }

    /**
     * Format datetime for display
     */
    function formatDateTime(datetime) {
        if (!datetime) return 'N/A';
        const date = new Date(datetime);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }

    /**
     * Calculate and display total costs
     */
    function calculateTotalCost() {
        const cost = parseFloat($('#qcnc-cost').val()) || 0;
        const packaging = parseFloat($('#qcnc-packaging-cost').val()) || 0;
        const handling = parseFloat($('#qcnc-handling-cost').val()) || 0;
        const warehousing = parseFloat($('#qcnc-warehousing-cost').val()) || 0;
        const labor = parseFloat($('#qcnc-labor-cost').val()) || 0;
        const shippingMaterial = parseFloat($('#qcnc-shipping-material-cost').val()) || 0;
        const custom1 = parseFloat($('#qcnc-custom-cost-1').val()) || 0;
        const custom2 = parseFloat($('#qcnc-custom-cost-2').val()) || 0;
        const custom3 = parseFloat($('#qcnc-custom-cost-3').val()) || 0;
        
        const additional = packaging + handling + warehousing + labor + shippingMaterial;
        const custom = custom1 + custom2 + custom3;
        const total = cost + additional + custom;
        
        $('#qcnc-display-cost').text('$' + cost.toFixed(2));
        $('#qcnc-display-additional').text('$' + additional.toFixed(2));
        $('#qcnc-display-custom').text('$' + custom.toFixed(2));
        $('#qcnc-display-total').text('$' + total.toFixed(2));
        
        // Update margin if product info is visible
        if ($('#qcnc-product-info').is(':visible')) {
            calculateMargin();
        }
    }

    /**
     * Calculate and display margin
     */
    function calculateMargin() {
        const cost = parseFloat($('#qcnc-cost').val()) || 0;
        const packaging = parseFloat($('#qcnc-packaging-cost').val()) || 0;
        const handling = parseFloat($('#qcnc-handling-cost').val()) || 0;
    
        const warehousing = parseFloat($('#qcnc-warehousing-cost').val()) || 0;
        const labor = parseFloat($('#qcnc-labor-cost').val()) || 0;
        const shippingMaterial = parseFloat($('#qcnc-shipping-material-cost').val()) || 0;
        const custom1 = parseFloat($('#qcnc-custom-cost-1').val()) || 0;
        const custom2 = parseFloat($('#qcnc-custom-cost-2').val()) || 0;
        const custom3 = parseFloat($('#qcnc-custom-cost-3').val()) || 0;
        
        const additional = packaging + handling + warehousing + labor + shippingMaterial;
        const custom = custom1 + custom2 + custom3;
        
        const totalCost = cost + additional + custom;
        
        const priceText = $('#qcnc-info-current-price').text();
        const price = parseFloat(priceText.replace(/[^0-9.]/g, '')) || 0;
        
        if (price > 0) {
            const margin = ((price - totalCost) / price * 100);
            const marginColor = margin >= 15 ? '#00a32a' : (margin >= 0 ? '#dba617' : '#d63638');
            
            // Add or update margin display
            if ($('#qcnc-margin-display').length === 0) {
                $('#qcnc-product-info').append(`
                    <p style="margin: 5px 0; padding-top: 10px; border-top: 1px solid #ddd;">
                        <strong>Estimated Margin:</strong> 
                        <span id="qcnc-margin-display" style="font-size: 1.2em; font-weight: bold;"></span>
                    </p>
                `);
            }
            
            $('#qcnc-margin-display')
                .text(margin.toFixed(2) + '%')
                .css('color', marginColor);
        }
    }

    // ===================================================================
    // AI INSIGHTS - AUTO SAVE & HISTORY
    // ===================================================================

    /**
     * Save AI insight automatically
     */
    function saveAIInsight(content, type = 'dashboard_insight', meta = {}) {
        if (!content || content.trim() === '') {
            console.log('Empty insight, not saving');
            return;
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_save_ai_insight',
                nonce: qcnc_ajax.nonce, // Make sure this is localized
                content: content,
                type: type,
                meta: meta
            },
            success: function(response) {
                if (response.success) {
                    console.log('✅ Insight saved:', response.data);

                    // Update insights counter if exists
                    if ($('#qcnc-insights-count').length) {
                        $('#qcnc-insights-count').text(response.data.count);
                    }

                    // Show saved indicator
                    showInsightSavedIndicator();
                } else {
                    console.error('Failed to save insight:', response.data);
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error saving insight:', error);
            }
        });
    }

    /**
     * Show a subtle "saved" indicator
     */
    function showInsightSavedIndicator() {
        var $indicator = $('<div class="qcnc-insight-saved">')
            .html('<span class="dashicons dashicons-saved"></span> Insight saved')
            .appendTo('body');

        setTimeout(function() {
            $indicator.fadeOut(400, function() {
                $(this).remove();
            });
        }, 2000);
    }

    /**
     * Monitor AI insights generation
     * This function watches for AI insights being generated in the dashboard
     */
    function monitorAIInsights() {
        // Backend now handles saving insights automatically (in get_ai_insights).
        // Disabling client-side monitoring to prevent duplicate saving.
        return;
    }

    /**
     * Load insights history
     */
    function loadInsightsHistory(limit = 10) {
        return $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_get_insights_history',
                nonce: qcnc_ajax.nonce,
                limit: limit
            }
        });
    }

    /**
     * Show insights history modal
     */
    function showInsightsHistory() {
        var $modal = $('#qcnc-insights-history-modal');

        if (!$modal.length) {
            console.error('Insights history modal not found');
            return;
        }

        // Show loading
        $modal.find('.qcnc-modal-body').html('<p class="qcnc-loading">Loading insights...</p>');
        $modal.fadeIn(200);

        // Load insights
        loadInsightsHistory().done(function(response) {
            if (response.success && response.data.insights) {
                renderInsightsHistory(response.data.insights);
            } else {
                $modal.find('.qcnc-modal-body').html('<p>No insights found.</p>');
            }
        }).fail(function() {
            $modal.find('.qcnc-modal-body').html('<p class="error">Failed to load insights.</p>');
        });
    }

    /**
     * Render insights history in modal
     */
    function renderInsightsHistory(insights) {
        var $modalBody = $('#qcnc-insights-history-modal .qcnc-modal-body');

        if (insights.length === 0) {
            $modalBody.html('<p>No insights saved yet.</p>');
            return;
        }

        var html = '<div class="qcnc-insights-list">';

        insights.forEach(function(insight) {
            html += '<div class="qcnc-insight-item" data-id="' + insight.id + '">';
            html += '  <div class="qcnc-insight-header">';
            html += '    <span class="qcnc-insight-time">' + insight.time_ago + '</span>';
            html += '    <span class="qcnc-insight-meta">' + insight.provider + ' - ' + insight.model + '</span>';
            html += '    <button class="qcnc-insight-delete" data-id="' + insight.id + '" title="Delete">';
            html += '      <span class="dashicons dashicons-trash"></span>';
            html += '    </button>';
            html += '  </div>';
            html += '  <div class="qcnc-insight-content">' + insight.content + '</div>';
            html += '</div>';
        });

        html += '</div>';
        $modalBody.html(html);
    }

    /**
     * Delete single insight
     */
    function deleteInsight(insightId) {
        if (!confirm('Are you sure you want to delete this insight?')) {
            return;
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qcnc_delete_insight',
                nonce: qcnc_ajax.nonce,
                insight_id: insightId
            },
            success: function(response) {
                if (response.success) {
                    // Remove from DOM
                    $('.qcnc-insight-item[data-id="' + insightId + '"]').fadeOut(300, function() {
                        $(this).remove();

                        // If no more insights, show message
                        if ($('.qcnc-insight-item').length === 0) {
                            $('.qcnc-insights-list').html('<p>No insights remaining.</p>');
                        }
                    });

                    // Update counter
                    if ($('#qcnc-insights-count').length) {
                        $('#qcnc-insights-count').text(response.data.count);
                    }
                } else {
                    alert('Failed to delete insight: ' + response.data);
                }
            },
            error: function() {
                alert('Error deleting insight');
            }
        });
    }

    // ===================================================================
    // EVENT HANDLERS
    // ===================================================================

    // Open insights history modal
    $(document).on('click', '#qcnc-view-insights-history, .qcnc-view-insights', function(e) {
        e.preventDefault();
        showInsightsHistory();
    });

    // Close insights modal
    $(document).on('click', '.qcnc-insights-modal-close, #qcnc-insights-history-modal .qcnc-modal-overlay', function() {
        $('#qcnc-insights-history-modal').fadeOut(200);
    });

    // Delete insight
    $(document).on('click', '.qcnc-insight-delete', function(e) {
        e.preventDefault();
        var insightId = $(this).data('id');
        deleteInsight(insightId);
    });

    // Manual save button (if you add one)
    $(document).on('click', '#qcnc-save-current-insight', function(e) {
        e.preventDefault();
        var content = $('#qcnc-ai-insights-content, .qcnc-ai-insight-content').text().trim();
        if (content) {
            saveAIInsight(content, 'dashboard_insight');
        } else {
            alert('No insight to save');
        }
    });

    // ===================================================================
    // INITIALIZATION
    // ===================================================================

    // Start monitoring AI insights on dashboard page
    if ($('.qcnc-dashboard-page').length || $('#qcnc-ai-insights').length) {
        monitorAIInsights();
        console.log('AI Insights auto-save initialized');
    }

    // Load initial count
    if ($('#qcnc-insights-count').length) {
        loadInsightsHistory(1).done(function(response) {
            if (response.success) {
                $('#qcnc-insights-count').text(response.data.count);
            }
        });
    }



    
})(jQuery);
