<?php
/**
 * Plugin Name: QMSpace Business Facts for AI Assistants
 * Description: Make your business visible to AI assistants like ChatGPT, Claude, and Gemini. Outputs structured JSON-LD data that AI can understand.
 * Version: 1.3.2
 * Author: QMSpace
 * Author URI: https://qmspace.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: qmspace-business-facts
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

// Plugin constants
define('QMSBF_VERSION', '1.3.2');
define('QMSBF_PLUGIN_FILE', __FILE__);
define('QMSBF_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('QMSBF_PLUGIN_URL', plugin_dir_url(__FILE__));

// Lead capture webhook endpoint
define('QMSBF_LEAD_WEBHOOK', 'https://qmspace.com/wp-json/leads/v1/capture');

class QMSpace_Business_Facts {
    
    private $version = '1.3.2';
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', array($this, 'init_plugin'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init_plugin() {
        if (!did_action('init')) {
            return;
        }
        
        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Admin assets
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // JSON endpoints
        $this->register_endpoints();
        add_action('template_redirect', array($this, 'handle_json_endpoint'));
        add_action('parse_request', array($this, 'handle_wellknown_fallback'), 1);
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // AJAX handlers
        add_action('wp_ajax_qmsbf_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_qmsbf_create_page', array($this, 'ajax_create_page'));
        add_action('wp_ajax_qmsbf_auto_populate', array($this, 'ajax_auto_populate'));
        add_action('wp_ajax_qmsbf_reset_all_data', array($this, 'ajax_reset_all_data'));
        add_action('wp_ajax_qmsbf_flush_cache', array($this, 'ajax_flush_cache'));
        add_action('wp_ajax_qmsbf_run_diagnostics', array($this, 'ajax_run_diagnostics'));
        
        // JSON-LD in head
        add_action('wp_head', array($this, 'output_json_ld_in_head'));
        
        // Shortcode
        add_shortcode('qmsbf_business_profile', array($this, 'render_profile_shortcode'));
        
        // Admin notices
        add_action('admin_notices', array($this, 'page_exists_notice'));
        
        // Check if we need to flush rewrite rules (after reset)
        add_action('init', array($this, 'maybe_flush_rewrite_rules'));
        
        // Setup wizard
        add_action('wp_ajax_qmsbf_wizard_skip', array($this, 'ajax_wizard_skip'));
        add_action('wp_ajax_qmsbf_wizard_complete', array($this, 'ajax_wizard_complete'));
    }
    
    /**
     * Check if setup wizard should be shown
     */
    public function should_show_wizard() {
        return get_option('qmsbf_wizard_completed') !== 'yes';
    }
    
    /**
     * Check if we need to flush rewrite rules (called on init)
     */
    public function maybe_flush_rewrite_rules() {
        if (get_option('qmsbf_flush_rewrite_rules')) {
            delete_option('qmsbf_flush_rewrite_rules');
            flush_rewrite_rules(true);
        }
    }
    
    /**
     * AJAX: Skip wizard
     */
    public function ajax_wizard_skip() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        update_option('qmsbf_wizard_completed', 'yes');
        wp_send_json_success('Skipped');
    }
    
    /**
     * AJAX: Complete wizard with all business data
     */
    public function ajax_wizard_complete() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array, individual fields sanitized below
        $data = isset( $_POST['wizard_data'] ) ? wp_unslash( $_POST['wizard_data'] ) : array();
        
        // Send lead notification to vendor if consent was given
        $email = sanitize_email($data['email'] ?? '');
        $consent = !empty($data['consent']);
        
        if ($email && $consent) {
            $this->send_lead_notification($email, 'Setup Wizard');
        }
        
        // Save business info (use business_email for public display, not lead email)
        $business_email = sanitize_email($data['business_email'] ?? '');
        $business = array(
            'name' => sanitize_text_field($data['business_name'] ?? ''),
            'description' => sanitize_textarea_field($data['description'] ?? ''),
            'website' => esc_url_raw($data['website'] ?? home_url()),
            'email' => $business_email,
            'phone' => sanitize_text_field($data['phone'] ?? '')
        );
        update_option('qmsbf_business', $business);
        
        // Save address
        $address = array(
            'street' => sanitize_text_field($data['address'] ?? ''),
            'city' => sanitize_text_field($data['city'] ?? ''),
            'state' => sanitize_text_field($data['state'] ?? ''),
            'postal' => sanitize_text_field($data['zip'] ?? ''),
            'country' => '',
            'map_url' => ''
        );
        update_option('qmsbf_address', $address);
        
        // Save settings with business type
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        $settings['business_type'] = sanitize_text_field($data['business_type'] ?? 'LocalBusiness');
        update_option('qmsbf_settings', $settings);
        
        // Mark wizard as completed
        update_option('qmsbf_wizard_completed', 'yes');
        update_option('qmsbf_last_updated', time());
        
        wp_send_json_success('Setup complete');
    }
    
    /**
     * Send lead notification to vendor via webhook (QMSpace)
     * More reliable than wp_mail() which fails on many hosts
     */
    private function send_lead_notification($lead_email, $source) {
        // Webhook endpoint
        $webhook_url = defined('QMSBF_LEAD_WEBHOOK') ? QMSBF_LEAD_WEBHOOK : '';
        $webhook_url = apply_filters('qmsbf_lead_webhook_url', $webhook_url);
        
        if (empty($webhook_url)) {
            return false;
        }
        
        $site_url = home_url();
        $site_host = wp_parse_url( $site_url, PHP_URL_HOST );
        
        $payload = array(
            'email' => $lead_email,
            'source' => $source,
            'site_url' => $site_url,
            'site_name' => get_bloginfo('name'),
            'timestamp' => current_time('c'),
            'plugin_version' => QMSBF_VERSION
        );
        
        // Verification hash
        $payload['hash'] = hash('sha256', $lead_email . $site_url . 'qmsbf_lead_2024');
        
        $response = wp_remote_post($webhook_url, array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-AIBF-Source' => $site_host
            ),
            'body' => wp_json_encode($payload)
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        if (!get_option('qmsbf_settings')) {
            update_option('qmsbf_settings', $this->get_default_settings());
        }
        if (!get_option('qmsbf_display_fields')) {
            update_option('qmsbf_display_fields', $this->get_default_display_fields());
        }
        
        // Register endpoints
        $this->register_endpoints();
        flush_rewrite_rules();
        
        // Check for existing profile page
        $this->check_existing_profile_page();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    /**
     * Check if this is Pro version
     */
    public function is_pro() {
        return false;
    }
    
    /**
     * Get Pro upgrade URL
     */
    public function get_pro_url() {
        return 'https://qmspace.com/buy/a-i-business-facts-pro';
    }
    
    /**
     * Default settings
     */
    private function get_default_settings() {
        return array(
            'business_type' => 'LocalBusiness',
            'show_footer' => false,  // Credits default to hidden per WordPress.org guideline 10
            'allow_cross_origin' => true,
            'data_reminder' => 30,
            'delete_data_on_uninstall' => false
        );
    }
    
    /**
     * Default display fields
     */
    private function get_default_display_fields() {
        return array(
            'name' => true,
            'description' => true,
            'website' => true,
            'email' => true,
            'phone' => true,
            'address' => true,
            'map' => true,
            'hours' => true
        );
    }
    
    /**
     * Get normalized display fields with all keys guaranteed to exist
     * @return array Display fields with all keys set (missing = false)
     */
    private function get_display_fields() {
        // All possible display field keys for Lite
        $all_keys = array('name', 'description', 'website', 'email', 'phone', 'address', 'map', 'hours');
        
        // Check if display settings have ever been saved
        $display_raw = get_option('qmsbf_display_fields', 'not_configured');
        
        if ($display_raw === 'not_configured') {
            // Option never saved - fresh install, default to showing everything
            return array_fill_keys($all_keys, true);
        }
        
        // Option was saved - normalize to ensure all keys exist
        $display = is_array($display_raw) ? $display_raw : array();
        
        // Fill in missing keys with false (unchecked = don't show)
        foreach ($all_keys as $key) {
            if (!isset($display[$key])) {
                $display[$key] = false;
            }
        }
        
        return $display;
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        $icon_svg = 'data:image/svg+xml;base64,' . base64_encode(
            '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                <g fill="#a0a5aa">
                    <circle cx="10" cy="10" r="8" fill="none" stroke="currentColor" stroke-width="1.5"/>
                    <ellipse cx="10" cy="10" rx="3" ry="8" fill="none" stroke="currentColor" stroke-width="1.5"/>
                    <line x1="2" y1="10" x2="18" y2="10" stroke="currentColor" stroke-width="1.5"/>
                </g>
            </svg>'
        );
        
        add_menu_page(
            'QMSpace Business Facts',
            'Business Facts',
            'manage_options',
            'qmspace-business-facts',
            array($this, 'render_admin_page'),
            $icon_svg,
            30
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'qmspace-business-facts') === false) return;
        
        wp_enqueue_media();
        wp_enqueue_style('dashicons');
        wp_enqueue_style('qmsbf-admin', QMSBF_PLUGIN_URL . 'admin-enterprise.css', array(), $this->version);
        wp_enqueue_script('qmsbf-admin', QMSBF_PLUGIN_URL . 'admin-enterprise.js', array('jquery'), $this->version, true);
        
        // Enqueue wizard assets (for setup wizard)
        wp_enqueue_style('qmsbf-wizard', QMSBF_PLUGIN_URL . 'admin-wizard.css', array(), $this->version);
        wp_enqueue_script('qmsbf-wizard', QMSBF_PLUGIN_URL . 'admin-wizard.js', array('jquery'), $this->version, true);
        
        wp_localize_script('qmsbf-admin', 'qmsbf_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('qmsbf_ajax_nonce'),
            'admin_url' => admin_url(),
            'home_url' => home_url(),
            'page_id' => get_option('qmsbf_page_id')
        ));
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        include(QMSBF_PLUGIN_DIR . 'templates/admin.php');
    }
    
    /**
     * Register rewrite endpoints
     */
    public function register_endpoints() {
        add_rewrite_rule('^\.well-known/ai-business\.json$', 'index.php?qmsbf_endpoint=ai-business', 'top');
        add_rewrite_rule('^\.well-known/ai-plugin\.json$', 'index.php?qmsbf_endpoint=ai-plugin', 'top');
        add_rewrite_rule('^\.well-known/llms\.txt$', 'index.php?qmsbf_endpoint=llms', 'top');
        add_rewrite_rule('^\.well-known/qmsbf-openapi\.json$', 'index.php?qmsbf_endpoint=openapi', 'top');
        add_filter('query_vars', function($vars) {
            $vars[] = 'qmsbf_endpoint';
            return $vars;
        });
    }
    
    /**
     * Handle JSON endpoints
     */
    public function handle_json_endpoint() {
        $endpoint = get_query_var('qmsbf_endpoint');
        if (empty($endpoint)) return;
        
        // Add CORS headers if enabled
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        if ($settings['allow_cross_origin'] ?? true) {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type');
        }
        
        // Handle CORS preflight requests
        if ( isset( $_SERVER['REQUEST_METHOD'] ) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS' ) {
            if ($settings['allow_cross_origin'] ?? true) {
                header('Access-Control-Max-Age: 86400');
            }
            http_response_code(204);
            exit;
        }
        
        switch ($endpoint) {
            case 'ai-business':
                $this->output_json_endpoint();
                break;
            case 'ai-plugin':
                $this->output_ai_plugin_json();
                break;
            case 'llms':
                $this->output_llms_txt();
                break;
            case 'openapi':
                $this->output_openapi_spec();
                break;
        }
        exit;
    }
    
    /**
     * Fallback handler for .well-known paths
     */
    public function handle_wellknown_fallback($wp) {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- URL path, sanitized via wp_parse_url
        $request = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
        $path = wp_parse_url( $request, PHP_URL_PATH );
        
        // Guard against parse_url returning false/null
        if (!is_string($path)) {
            return;
        }
        
        // Define plugin endpoints - use suffix matching for subdirectory installs
        $plugin_endpoints = array(
            '/.well-known/ai-business.json',
            '/.well-known/ai-plugin.json',
            '/.well-known/llms.txt',
            '/.well-known/qmsbf-openapi.json'
        );
        
        // Check if this request is for one of our endpoints
        $is_plugin_endpoint = false;
        $matched_endpoint = '';
        foreach ($plugin_endpoints as $endpoint) {
            if (substr($path, -strlen($endpoint)) === $endpoint) {
                $is_plugin_endpoint = true;
                $matched_endpoint = $endpoint;
                break;
            }
        }
        
        // Only proceed if this is one of our endpoints
        if (!$is_plugin_endpoint) {
            return;
        }
        
        // Handle CORS preflight requests ONLY for our endpoints
        if ( isset( $_SERVER['REQUEST_METHOD'] ) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS' ) {
            $settings = get_option('qmsbf_settings', $this->get_default_settings());
            if ($settings['allow_cross_origin'] ?? true) {
                header('Access-Control-Allow-Origin: *');
                header('Access-Control-Allow-Methods: GET, OPTIONS');
                header('Access-Control-Allow-Headers: Content-Type');
                header('Access-Control-Max-Age: 86400');
            }
            http_response_code(204);
            exit;
        }
        
        // Handle GET requests
        switch ($matched_endpoint) {
            case '/.well-known/ai-business.json':
                $this->output_json_endpoint();
                exit;
            case '/.well-known/ai-plugin.json':
                $this->output_ai_plugin_json();
                exit;
            case '/.well-known/llms.txt':
                $this->output_llms_txt();
                exit;
            case '/.well-known/qmsbf-openapi.json':
                $this->output_openapi_spec();
                exit;
        }
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('qmsbf/v1', '/business', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_get_business'),
            'permission_callback' => '__return_true'
        ));
        
        register_rest_route('qmsbf/v1', '/openapi', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_get_openapi'),
            'permission_callback' => '__return_true'
        ));
    }
    
    /**
     * REST API callback
     */
    public function rest_get_business() {
        return rest_ensure_response($this->get_business_data());
    }
    
    /**
     * REST API callback for OpenAPI spec
     */
    public function rest_get_openapi() {
        return rest_ensure_response($this->get_openapi_spec());
    }
    
    /**
     * Output JSON endpoint
     */
    private function output_json_endpoint() {
        header('Content-Type: application/json; charset=utf-8');
        header('X-Robots-Tag: noindex');
        
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        if ($settings['allow_cross_origin'] ?? true) {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type');
        }
        
        echo wp_json_encode($this->get_business_data(), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Output OpenAPI spec
     */
    private function output_openapi_spec() {
        header('Content-Type: application/json; charset=utf-8');
        
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        if ($settings['allow_cross_origin'] ?? true) {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type');
        }
        
        echo wp_json_encode($this->get_openapi_spec(), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Generate OpenAPI 3.0 spec
     */
    private function get_openapi_spec() {
        $business = get_option('qmsbf_business', array());
        $display = $this->get_display_fields();
        
        $business_name = ($display['name'] && !empty($business['name'])) ? $business['name'] : get_bloginfo('name');
        $business_description = ($display['description'] && !empty($business['description'])) ? $business['description'] : get_bloginfo('description');
        
        return array(
            'openapi' => '3.0.0',
            'info' => array(
                'title' => $business_name . ' Business API',
                'description' => 'API to retrieve business information for ' . $business_name . '. ' . $business_description,
                'version' => '1.0.0'
            ),
            'servers' => array(
                array(
                    'url' => home_url(),
                    'description' => 'Production server'
                )
            ),
            'paths' => array(
                '/.well-known/ai-business.json' => array(
                    'get' => array(
                        'summary' => 'Get business information',
                        'description' => 'Returns structured business data in JSON-LD format including name, contact info, address, and hours.',
                        'operationId' => 'getBusinessInfo',
                        'responses' => array(
                            '200' => array(
                                'description' => 'Business information in JSON-LD format',
                                'content' => array(
                                    'application/json' => array(
                                        'schema' => array(
                                            '$ref' => '#/components/schemas/BusinessInfo'
                                        )
                                    )
                                )
                            )
                        )
                    )
                ),
                '/wp-json/qmsbf/v1/business' => array(
                    'get' => array(
                        'summary' => 'Get business information via REST API',
                        'description' => 'Returns business data via WordPress REST API.',
                        'operationId' => 'getBusinessInfoRest',
                        'responses' => array(
                            '200' => array(
                                'description' => 'Business information',
                                'content' => array(
                                    'application/json' => array(
                                        'schema' => array(
                                            '$ref' => '#/components/schemas/BusinessInfo'
                                        )
                                    )
                                )
                            )
                        )
                    )
                )
            ),
            'components' => array(
                'schemas' => array(
                    'BusinessInfo' => array(
                        'type' => 'object',
                        'properties' => array(
                            '@context' => array('type' => 'string'),
                            '@type' => array('type' => 'string'),
                            'name' => array('type' => 'string'),
                            'description' => array('type' => 'string'),
                            'url' => array('type' => 'string', 'format' => 'uri'),
                            'email' => array('type' => 'string', 'format' => 'email'),
                            'telephone' => array('type' => 'string'),
                            'address' => array(
                                'type' => 'object',
                                'properties' => array(
                                    '@type' => array('type' => 'string'),
                                    'streetAddress' => array('type' => 'string'),
                                    'addressLocality' => array('type' => 'string'),
                                    'addressRegion' => array('type' => 'string'),
                                    'postalCode' => array('type' => 'string'),
                                    'addressCountry' => array('type' => 'string')
                                )
                            ),
                            'openingHoursSpecification' => array(
                                'type' => 'array',
                                'items' => array(
                                    'type' => 'object',
                                    'properties' => array(
                                        '@type' => array('type' => 'string'),
                                        'dayOfWeek' => array('type' => 'string'),
                                        'opens' => array('type' => 'string'),
                                        'closes' => array('type' => 'string')
                                    )
                                )
                            )
                        )
                    )
                )
            )
        );
    }
    
    /**
     * Output ai-plugin.json
     */
    private function output_ai_plugin_json() {
        header('Content-Type: application/json; charset=utf-8');
        
        $business = get_option('qmsbf_business', array());
        $settings = get_option('qmsbf_settings', array());
        $display = $this->get_display_fields();
        
        // Get contact email - only if email display is enabled
        $contact_email = '';
        if ($display['email']) {
            if (!empty($settings['manifest_email'])) {
                $contact_email = $settings['manifest_email'];
            } elseif (!empty($business['email'])) {
                $contact_email = $business['email'];
            }
        }
        // Apply filter for customization
        $contact_email = apply_filters('qmsbf_manifest_contact_email', $contact_email);
        
        // Get privacy policy URL - prefer WordPress configured URL, fallback to common path
        $privacy_url = function_exists('get_privacy_policy_url') ? get_privacy_policy_url() : '';
        if (empty($privacy_url)) {
            $privacy_url = home_url('/privacy-policy/');
        }
        
        // Name and description respect display settings
        $name = ($display['name'] && !empty($business['name'])) ? $business['name'] : get_bloginfo('name');
        $description = ($display['description'] && !empty($business['description'])) ? $business['description'] : get_bloginfo('description');
        
        $data = array(
            'schema_version' => 'v1',
            'name_for_human' => $name,
            'name_for_model' => sanitize_title($name),
            'description_for_human' => $description,
            'description_for_model' => 'Get accurate business information including hours, location, contact details.',
            'api' => array(
                'type' => 'openapi',
                'url' => home_url('/.well-known/qmsbf-openapi.json')
            ),
            'legal_info_url' => $privacy_url
        );
        
        // Only include contact_email if it's set and display is enabled
        if (!empty($contact_email)) {
            $data['contact_email'] = $contact_email;
        }
        
        echo wp_json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    }
    
    /**
     * Output llms.txt
     */
    private function output_llms_txt() {
        header('Content-Type: text/plain; charset=utf-8');
        
        $business = get_option('qmsbf_business', array());
        $address = get_option('qmsbf_address', array());
        $hours = get_option('qmsbf_hours', array());
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        $display = $this->get_display_fields();
        
        $name = ($display['name'] && !empty($business['name'])) ? $business['name'] : get_bloginfo('name');
        $type = $settings['business_type'] ?? 'LocalBusiness';
        
        $output = "# {$name}\n\n";
        if ($display['description'] && !empty($business['description'])) {
            $output .= "> {$business['description']}\n\n";
        }
        $output .= "## Business Information\n\n";
        $output .= "- Type: {$type}\n";
        
        if ($display['website'] && !empty($business['website'])) {
            $output .= "- Website: {$business['website']}\n";
        }
        if ($display['email'] && !empty($business['email'])) {
            $output .= "- Email: {$business['email']}\n";
        }
        if ($display['phone'] && !empty($business['phone'])) {
            $output .= "- Phone: {$business['phone']}\n";
        }
        
        // Address - only if display setting is enabled
        if ($display['address'] && !empty($address['street'])) {
            $addr_parts = array_filter(array(
                $address['street'] ?? '',
                $address['city'] ?? '',
                $address['state'] ?? '',
                $address['postal'] ?? '',
                $address['country'] ?? ''
            ));
            $output .= "- Address: " . implode(', ', $addr_parts) . "\n";
        }
        
        // Hours - only if display setting is enabled
        if ($display['hours'] && !empty($hours)) {
            if (!empty($hours['is_24_7'])) {
                $output .= "\n## Business Hours\n\nOpen 24/7\n";
            } else {
                $has_hours = false;
                $hours_text = "\n## Business Hours\n\n";
                $days = array('monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday');
                foreach ($days as $day) {
                    if (!empty($hours[$day])) {
                        $has_hours = true;
                        if (!empty($hours[$day]['closed'])) {
                            $hours_text .= "- " . ucfirst($day) . ": Closed\n";
                        } elseif (!empty($hours[$day]['open']) && !empty($hours[$day]['close'])) {
                            $hours_text .= "- " . ucfirst($day) . ": {$hours[$day]['open']} - {$hours[$day]['close']}\n";
                        }
                    }
                }
                if ($has_hours) {
                    $output .= $hours_text;
                }
            }
            
            // Special hours note
            if (!empty($hours['special'])) {
                $output .= "\nNote: " . $hours['special'] . "\n";
            }
        }
        
        $output .= "\n## Data Source\n\n";
        $output .= "For structured data, visit: " . esc_url( home_url( '/.well-known/ai-business.json' ) ) . "\n";
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Plain text output, data sanitized on input
        echo $output;
    }
    
    /**
     * Get business data as structured array
     */
    public function get_business_data() {
        $business = get_option('qmsbf_business', array());
        $address = get_option('qmsbf_address', array());
        $hours = get_option('qmsbf_hours', array());
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        $display = $this->get_display_fields();
        
        $data = array(
            '@context' => 'https://schema.org',
            '@type' => $settings['business_type'] ?? 'LocalBusiness'
        );
        
        // Basic info - only include if display setting is enabled
        if ($display['name'] && !empty($business['name'])) {
            $data['name'] = $business['name'];
        }
        if ($display['description'] && !empty($business['description'])) {
            $data['description'] = $business['description'];
        }
        if ($display['website'] && !empty($business['website'])) {
            $data['url'] = $business['website'];
        }
        if ($display['email'] && !empty($business['email'])) {
            $data['email'] = $business['email'];
        }
        if ($display['phone'] && !empty($business['phone'])) {
            $data['telephone'] = $business['phone'];
        }
        
        // Address - only include if display setting is enabled
        if ($display['address'] && (!empty($address['street']) || !empty($address['city']))) {
            $data['address'] = array(
                '@type' => 'PostalAddress'
            );
            if (!empty($address['street'])) $data['address']['streetAddress'] = $address['street'];
            if (!empty($address['city'])) $data['address']['addressLocality'] = $address['city'];
            if (!empty($address['state'])) $data['address']['addressRegion'] = $address['state'];
            if (!empty($address['postal'])) $data['address']['postalCode'] = $address['postal'];
            if (!empty($address['country'])) $data['address']['addressCountry'] = $address['country'];
        }
        
        // Hours - only include if display setting is enabled
        if ($display['hours'] && !empty($hours)) {
            if (!empty($hours['is_24_7'])) {
                $data['openingHoursSpecification'] = array(
                    array(
                        '@type' => 'OpeningHoursSpecification',
                        'dayOfWeek' => array('Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'),
                        'opens' => '00:00',
                        'closes' => '23:59'
                    )
                );
            } else {
                $hours_spec = array();
                $days_map = array(
                    'monday' => 'Monday',
                    'tuesday' => 'Tuesday',
                    'wednesday' => 'Wednesday',
                    'thursday' => 'Thursday',
                    'friday' => 'Friday',
                    'saturday' => 'Saturday',
                    'sunday' => 'Sunday'
                );
                
                foreach ($days_map as $key => $day) {
                    if (!empty($hours[$key]) && empty($hours[$key]['closed']) && !empty($hours[$key]['open']) && !empty($hours[$key]['close'])) {
                        $hours_spec[] = array(
                            '@type' => 'OpeningHoursSpecification',
                            'dayOfWeek' => $day,
                            'opens' => $hours[$key]['open'],
                            'closes' => $hours[$key]['close']
                        );
                    }
                }
                
                if (!empty($hours_spec)) {
                    $data['openingHoursSpecification'] = $hours_spec;
                }
            }
        }
        
        // Map link - only include if display setting is enabled
        if ($display['map'] && !empty($address['map_url'])) {
            $data['hasMap'] = $address['map_url'];
        }
        
        return $data;
    }
    
    /**
     * Output JSON-LD in page head
     */
    public function output_json_ld_in_head() {
        if (is_admin()) return;
        
        $data = $this->get_business_data();
        if (empty($data) || empty($data['name'])) return;
        
        $json = wp_json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_HEX_TAG | JSON_HEX_AMP);
        
        if ($json) {
            echo "\n<!-- AI Business Facts JSON-LD -->\n";
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON encoded with JSON_HEX_TAG | JSON_HEX_AMP for script safety
            echo '<script type="application/ld+json">' . $json . '</script>' . "\n";
        }
    }
    
    /**
     * Render profile shortcode
     */
    public function render_profile_shortcode($atts) {
        // Enqueue frontend profile styles
        wp_enqueue_style('qmsbf-profile', QMSBF_PLUGIN_URL . 'frontend-profile.css', array(), $this->version);
        
        $business = get_option('qmsbf_business', array());
        $address = get_option('qmsbf_address', array());
        $hours = get_option('qmsbf_hours', array());
        $settings = get_option('qmsbf_settings', $this->get_default_settings());
        $display = $this->get_display_fields();
        
        ob_start();
        include(QMSBF_PLUGIN_DIR . 'templates/profile.php');
        return ob_get_clean();
    }
    
    /**
     * AJAX: Save settings
     */
    public function ajax_save_settings() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Unslash POST data - individual fields sanitized below
        // phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $post_settings = isset( $_POST['settings'] ) ? wp_unslash( $_POST['settings'] ) : array();
        $post_business = isset( $_POST['business'] ) ? wp_unslash( $_POST['business'] ) : array();
        $post_address = isset( $_POST['address'] ) ? wp_unslash( $_POST['address'] ) : array();
        $post_hours = isset( $_POST['hours'] ) ? wp_unslash( $_POST['hours'] ) : array();
        $post_display = isset( $_POST['display'] ) ? wp_unslash( $_POST['display'] ) : array();
        // phpcs:enable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        
        // Settings - handle JS booleans sent as strings
        $settings = array(
            'business_type' => sanitize_text_field( $post_settings['business_type'] ?? 'LocalBusiness' ),
            'show_footer' => filter_var( $post_settings['show_footer'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'allow_cross_origin' => filter_var( $post_settings['allow_cross_origin'] ?? true, FILTER_VALIDATE_BOOLEAN ),
            'data_reminder' => absint( $post_settings['data_reminder'] ?? 30 )
        );
        update_option('qmsbf_settings', $settings);
        
        // Business info
        $business = array(
            'name' => sanitize_text_field( $post_business['name'] ?? '' ),
            'description' => sanitize_textarea_field( $post_business['description'] ?? '' ),
            'website' => esc_url_raw( $post_business['website'] ?? '' ),
            'email' => sanitize_email( $post_business['email'] ?? '' ),
            'phone' => sanitize_text_field( $post_business['phone'] ?? '' )
        );
        update_option('qmsbf_business', $business);
        
        // Address
        $address = array(
            'street' => sanitize_text_field( $post_address['street'] ?? '' ),
            'city' => sanitize_text_field( $post_address['city'] ?? '' ),
            'state' => sanitize_text_field( $post_address['state'] ?? '' ),
            'postal' => sanitize_text_field( $post_address['postal'] ?? '' ),
            'country' => sanitize_text_field( $post_address['country'] ?? '' ),
            'map_url' => esc_url_raw( $post_address['map_url'] ?? '' )
        );
        update_option('qmsbf_address', $address);
        
        // Hours
        $hours = array();
        if ( ! empty( $post_hours['is_24_7'] ) ) {
            $hours['is_24_7'] = true;
        } else {
            $days = array('monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday');
            foreach ($days as $day) {
                if ( isset( $post_hours[ $day ] ) ) {
                    $hours[$day] = array(
                        'open' => sanitize_text_field( $post_hours[ $day ]['open'] ?? '09:00' ),
                        'close' => sanitize_text_field( $post_hours[ $day ]['close'] ?? '17:00' ),
                        'closed' => ! empty( $post_hours[ $day ]['closed'] )
                    );
                }
            }
        }
        
        // Special hours note (always save, even if empty, so it can be cleared)
        $hours['special'] = sanitize_textarea_field( $post_hours['special'] ?? '' );
        
        update_option('qmsbf_hours', $hours);
        
        // Display fields - handle JS booleans sent as strings
        $display = array(
            'name' => filter_var( $post_display['name'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'description' => filter_var( $post_display['description'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'website' => filter_var( $post_display['website'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'email' => filter_var( $post_display['email'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'phone' => filter_var( $post_display['phone'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'address' => filter_var( $post_display['address'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'map' => filter_var( $post_display['map'] ?? false, FILTER_VALIDATE_BOOLEAN ),
            'hours' => filter_var( $post_display['hours'] ?? false, FILTER_VALIDATE_BOOLEAN )
        );
        update_option('qmsbf_display_fields', $display);
        
        // Update timestamp
        update_option('qmsbf_last_updated', time());
        
        wp_send_json_success('Settings saved');
    }
    
    /**
     * AJAX: Create profile page
     */
    public function ajax_create_page() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $business = get_option('qmsbf_business', array());
        $page_title = !empty($business['name']) ? $business['name'] : 'Business Profile';
        
        $page_id = wp_insert_post(array(
            'post_title' => $page_title,
            'post_content' => '[qmsbf_business_profile]',
            'post_status' => 'publish',
            'post_type' => 'page'
        ));
        
        if ($page_id && !is_wp_error($page_id)) {
            update_option('qmsbf_page_id', $page_id);
            
            // Schedule rewrite rules flush for next page load
            update_option('qmsbf_flush_rewrite_rules', true);
            
            // Use the plain permalink format which always works immediately
            // Pretty permalinks will work after next page load when rewrite rules flush
            $url = add_query_arg('page_id', $page_id, home_url('/'));
            
            wp_send_json_success(array(
                'page_id' => $page_id,
                'page_url' => $url
            ));
        } else {
            wp_send_json_error('Failed to create page');
        }
    }
    
    /**
     * AJAX: Auto-populate from WordPress
     */
    public function ajax_auto_populate() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Note: We intentionally don't auto-populate email because:
        // 1. admin_email is often a personal email, not a public business email
        // 2. This email will be publicly visible in JSON endpoints and profile page
        $data = array(
            'name' => get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'website' => home_url(),
            'email' => '' // User should manually enter a public-facing business email
        );
        
        wp_send_json_success($data);
    }
    
    /**
     * AJAX: Reset all plugin data
     */
    public function ajax_reset_all_data() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Get page ID before deleting
        $page_id = get_option('qmsbf_page_id');
        
        // Delete the profile page if it exists
        if ($page_id) {
            wp_delete_post($page_id, true);
        }
        
        // Delete all plugin options
        delete_option('qmsbf_settings');
        delete_option('qmsbf_business');
        delete_option('qmsbf_address');
        delete_option('qmsbf_hours');
        delete_option('qmsbf_social');
        delete_option('qmsbf_display_fields');
        delete_option('qmsbf_page_id');
        delete_option('qmsbf_leads');
        delete_option('qmsbf_wizard_completed');
        delete_option('qmsbf_last_updated');
        delete_option('qmsbf_analytics');
        
        // Delete transients
        delete_transient('qmsbf_license_status_cache');
        delete_transient('qmsbf_page_exists_notice');
        
        // Flush rewrite rules to ensure clean state for new page creation
        flush_rewrite_rules(true);
        
        // Set flag to flush again on next page load (belt and suspenders)
        update_option('qmsbf_flush_rewrite_rules', true);
        
        wp_send_json_success('All data has been reset');
    }
    
    /**
     * AJAX: Flush all caches
     */
    public function ajax_flush_cache() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $flushed = array();
        
        // Clear plugin transients
        delete_transient('qmsbf_json_cache');
        delete_transient('qmsbf_business_data');
        $flushed[] = 'Plugin cache';
        
        // Clear WordPress object cache
        wp_cache_flush();
        $flushed[] = 'WordPress object cache';
        
        // Clear rewrite rules
        flush_rewrite_rules();
        $flushed[] = 'Rewrite rules';
        
        // Try to clear popular caching plugins
        // WP Super Cache
        if (function_exists('wp_cache_clear_cache')) {
            wp_cache_clear_cache();
            $flushed[] = 'WP Super Cache';
        }
        
        // W3 Total Cache
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all();
            $flushed[] = 'W3 Total Cache';
        }
        
        // WP Rocket
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain();
            $flushed[] = 'WP Rocket';
        }
        
        // LiteSpeed Cache
        if (class_exists('LiteSpeed_Cache_API') && method_exists('LiteSpeed_Cache_API', 'purge_all')) {
            LiteSpeed_Cache_API::purge_all();
            $flushed[] = 'LiteSpeed Cache';
        }
        
        // Autoptimize
        if (class_exists('autoptimizeCache') && method_exists('autoptimizeCache', 'clearall')) {
            autoptimizeCache::clearall();
            $flushed[] = 'Autoptimize';
        }
        
        wp_send_json_success(array(
            'message' => 'Cache flushed successfully!',
            'flushed' => $flushed
        ));
    }
    
    /**
     * AJAX: Run diagnostics
     */
    public function ajax_run_diagnostics() {
        check_ajax_referer('qmsbf_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $diagnostics = array();
        
        // SSL verification - enabled by default, filterable for local dev environments
        $sslverify = apply_filters('qmsbf_diagnostics_sslverify', true);
        
        // Check JSON endpoint
        $json_url = home_url('/.well-known/ai-business.json');
        $response = wp_remote_get($json_url, array('timeout' => 10, 'sslverify' => $sslverify));
        
        if (is_wp_error($response)) {
            $diagnostics['json_endpoint'] = array(
                'status' => 'error',
                'message' => 'JSON endpoint not accessible: ' . $response->get_error_message()
            );
        } else {
            $code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $json = json_decode($body, true);
            
            if ($code === 200 && $json) {
                $diagnostics['json_endpoint'] = array(
                    'status' => 'success',
                    'message' => 'JSON endpoint working correctly'
                );
            } else {
                $diagnostics['json_endpoint'] = array(
                    'status' => 'warning',
                    'message' => 'JSON endpoint returned code ' . $code
                );
            }
        }
        
        // Check profile page
        $page_id = get_option('qmsbf_page_id');
        if ($page_id && get_post($page_id)) {
            $page = get_post($page_id);
            $diagnostics['profile_page'] = array(
                'status' => $page->post_status === 'publish' ? 'success' : 'warning',
                'message' => 'Profile page exists (Status: ' . $page->post_status . ')',
                'url' => get_permalink($page_id)
            );
        } else {
            $diagnostics['profile_page'] = array(
                'status' => 'warning',
                'message' => 'No profile page created yet'
            );
        }
        
        // Check business data
        $business = get_option('qmsbf_business', array());
        $filled_fields = 0;
        $fields_to_check = array('name', 'description', 'website', 'email', 'phone');
        foreach ($fields_to_check as $field) {
            if (!empty($business[$field])) $filled_fields++;
        }
        
        $diagnostics['business_data'] = array(
            'status' => $filled_fields >= 3 ? 'success' : ($filled_fields > 0 ? 'warning' : 'error'),
            'message' => $filled_fields . ' of ' . count($fields_to_check) . ' key business fields filled'
        );
        
        // Check REST API
        $rest_url = rest_url('qmsbf/v1/business');
        $rest_response = wp_remote_get($rest_url, array('timeout' => 10, 'sslverify' => $sslverify));
        
        if (is_wp_error($rest_response)) {
            $diagnostics['rest_api'] = array(
                'status' => 'error',
                'message' => 'REST API not accessible'
            );
        } else {
            $diagnostics['rest_api'] = array(
                'status' => 'success',
                'message' => 'REST API endpoint working'
            );
        }
        
        // Check OpenAPI spec
        $openapi_url = home_url('/.well-known/qmsbf-openapi.json');
        $openapi_response = wp_remote_get($openapi_url, array('timeout' => 10, 'sslverify' => $sslverify));
        
        if (is_wp_error($openapi_response)) {
            $diagnostics['openapi_spec'] = array(
                'status' => 'error',
                'message' => 'OpenAPI spec not accessible'
            );
        } else {
            $openapi_code = wp_remote_retrieve_response_code($openapi_response);
            $openapi_body = wp_remote_retrieve_body($openapi_response);
            $openapi_json = json_decode($openapi_body, true);
            
            if ($openapi_code === 200 && $openapi_json && isset($openapi_json['openapi'])) {
                $diagnostics['openapi_spec'] = array(
                    'status' => 'success',
                    'message' => 'OpenAPI 3.0 spec working correctly'
                );
            } else {
                $diagnostics['openapi_spec'] = array(
                    'status' => 'warning',
                    'message' => 'OpenAPI spec returned code ' . $openapi_code
                );
            }
        }
        
        // PHP version
        $diagnostics['php_version'] = array(
            'status' => version_compare(PHP_VERSION, '7.4', '>=') ? 'success' : 'warning',
            'message' => 'PHP ' . PHP_VERSION
        );
        
        // WordPress version
        $diagnostics['wp_version'] = array(
            'status' => 'success',
            'message' => 'WordPress ' . get_bloginfo('version')
        );
        
        wp_send_json_success($diagnostics);
    }
    
    /**
     * Check for existing profile page on activation
     */
    private function check_existing_profile_page() {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time activation check, no caching needed
        $page_id = $wpdb->get_var(
            "SELECT ID FROM {$wpdb->posts} 
             WHERE post_content LIKE '%[qmsbf_business_profile]%' 
             AND post_status = 'publish' 
             AND post_type = 'page' 
             LIMIT 1"
        );
        
        if ($page_id) {
            update_option('qmsbf_page_id', $page_id);
            set_transient('qmsbf_page_exists_notice', array(
                'page_id' => $page_id,
                'message' => 'Found existing page with the business profile shortcode.'
            ), 30);
        }
    }
    
    /**
     * Show notice about existing page
     */
    public function page_exists_notice() {
        $notice = get_transient('qmsbf_page_exists_notice');
        if (!$notice) return;
        
        delete_transient('qmsbf_page_exists_notice');
        ?>
        <div class="notice notice-info is-dismissible">
            <p><strong>AI Business Facts:</strong> <?php echo esc_html($notice['message']); ?></p>
            <p>
                <a href="<?php echo esc_url(get_edit_post_link($notice['page_id'])); ?>" class="button button-primary">Edit Page</a>
                <a href="<?php echo esc_url(get_permalink($notice['page_id'])); ?>" class="button" target="_blank">View Page</a>
            </p>
        </div>
        <?php
    }
}

// Initialize plugin
QMSpace_Business_Facts::getInstance();
