// QMSpace Business Facts - Enterprise Admin JavaScript v5.11.0

jQuery(document).ready(function($) {
    
    // ==================== AI SCORE FUNCTIONALITY ====================
    var businessData = {
        name: '',
        description: '',
        phone: '',
        email: '',
        website: '',
        street: '',
        city: '',
        state: '',
        postal: '',
        country: '',
        hours: {},
        price_range: '',
        logo: ''
    };
    
    var dataPointsConfig = [
        { key: 'name', label: 'Business Name', weight: 15 },
        { key: 'description', label: 'Description', weight: 10 },
        { key: 'phone', label: 'Phone Number', weight: 12 },
        { key: 'email', label: 'Email Address', weight: 8 },
        { key: 'website', label: 'Website URL', weight: 10 },
        { key: 'street', label: 'Street Address', weight: 8 },
        { key: 'city', label: 'City', weight: 5 },
        { key: 'state', label: 'State/Province', weight: 3 },
        { key: 'postal', label: 'Postal Code', weight: 4 },
        { key: 'country', label: 'Country', weight: 3 },
        { key: 'hours', label: 'Business Hours', weight: 12 },
        { key: 'price_range', label: 'Price Range', weight: 5 }
    ];
    
    function collectBusinessData() {
        businessData.name = $('#business_name').val() || '';
        businessData.description = $('#business_description').val() || '';
        businessData.phone = $('#business_phone').val() || '';
        businessData.email = $('#business_email').val() || '';
        businessData.website = $('#business_website').val() || '';
        businessData.street = $('#address_street').val() || '';
        businessData.city = $('#address_city').val() || '';
        businessData.state = $('#address_state').val() || '';
        businessData.postal = $('#address_postal').val() || '';
        businessData.country = $('#address_country').val() || '';
        businessData.price_range = $('#price_range').val() || '';
        businessData.logo = $('#logo_url').val() || '';
        
        // Collect hours
        var days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        var hasHours = false;
        businessData.hours = {};
        days.forEach(function(day) {
            var open = $('input[name="hours_' + day + '_open"]').val();
            var close = $('input[name="hours_' + day + '_close"]').val();
            var closed = $('input[name="hours_' + day + '_closed"]').is(':checked');
            if (open || close || closed) {
                hasHours = true;
                businessData.hours[day] = { open: open, close: close, closed: closed };
            }
        });
        if (!hasHours) businessData.hours = null;
    }
    
    function calculateScore() {
        collectBusinessData();
        
        var totalWeight = 0;
        var earnedWeight = 0;
        var filledCount = 0;
        
        dataPointsConfig.forEach(function(point) {
            totalWeight += point.weight;
            var value = businessData[point.key];
            var isFilled = false;
            
            if (point.key === 'hours') {
                isFilled = value && Object.keys(value).length > 0;
            } else {
                isFilled = value && value.trim && value.trim() !== '';
            }
            
            if (isFilled) {
                earnedWeight += point.weight;
                filledCount++;
            }
        });
        
        return {
            score: Math.round((earnedWeight / totalWeight) * 100),
            filled: filledCount,
            total: dataPointsConfig.length
        };
    }
    
    function getScoreLabel(score) {
        if (score >= 80) return 'Excellent';
        if (score >= 60) return 'Good';
        if (score >= 40) return 'Fair';
        return 'Needs Work';
    }
    
    function getScoreClass(score) {
        if (score >= 80) return 'score-excellent';
        if (score >= 60) return 'score-good';
        if (score >= 40) return 'score-medium';
        return 'score-low';
    }
    
    function getScoreExplanation(score) {
        if (score >= 80) return "Excellent! AI assistants like ChatGPT, Claude, and Gemini will be able to accurately answer most questions about your business.";
        if (score >= 60) return "Good progress! AI assistants can answer basic questions about your business. Add more details to improve visibility.";
        if (score >= 40) return "Fair start. AI assistants have some information about your business, but may struggle with specific queries.";
        return "AI assistants have very limited information about your business. Add more details to improve your visibility.";
    }
    
    function updateScoreDisplay() {
        var result = calculateScore();
        var score = result.score;
        var scoreClass = getScoreClass(score);
        
        // Update score number with animation
        $({ val: 0 }).animate({ val: score }, {
            duration: 1500,
            easing: 'swing',
            step: function() {
                $('#score-number').text(Math.round(this.val));
            },
            complete: function() {
                $('#score-number').text(score);
            }
        });
        
        // Update progress ring
        var circumference = 339.3;
        var offset = circumference - (score / 100) * circumference;
        
        $('#score-ring').removeClass('score-low score-medium score-good score-excellent').addClass(scoreClass);
        setTimeout(function() {
            $('.qmsbf-score-progress').css('stroke-dashoffset', offset);
        }, 100);
        
        // Update label
        $('#score-label').text(getScoreLabel(score))
            .removeClass('score-low score-medium score-good score-excellent')
            .addClass(scoreClass);
        
        // Update explanation
        $('#score-explanation').text(getScoreExplanation(score));
        
        // Update datapoints list
        updateDataPointsList();
    }
    
    function updateDataPointsList() {
        collectBusinessData();
        var html = '';
        
        dataPointsConfig.forEach(function(point) {
            var value = businessData[point.key];
            var isFilled = false;
            
            if (point.key === 'hours') {
                isFilled = value && Object.keys(value).length > 0;
            } else {
                isFilled = value && value.trim && value.trim() !== '';
            }
            
            var filledClass = isFilled ? 'filled' : '';
            var icon = isFilled ? 'dashicons-yes-alt' : 'dashicons-dismiss';
            
            html += '<div class="qmsbf-datapoint ' + filledClass + '">';
            html += '<span class="dashicons ' + icon + '"></span>';
            html += point.label;
            html += '</div>';
        });
        
        $('#datapoints-list').html(html);
    }
    
    // Recalculate score button
    $('#recalculate-score').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).html('Calculating <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        setTimeout(function() {
            updateScoreDisplay();
            $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Recalculate Score');
            showNotification('Score updated!', 'success');
        }, 500);
    });
    
    // Update score when switching to AI Score tab
    $('.qmsbf-tab[data-tab="ai-score"]').on('click', function() {
        setTimeout(updateScoreDisplay, 200);
    });
    
    // ==================== AI TEST CHAT FUNCTIONALITY ====================
    var chatMessages = [];
    
    function getQuickQuestions() {
        collectBusinessData();
        var name = businessData.name || 'this business';
        return [
            'What are the hours for ' + name + '?',
            'Where is ' + name + ' located?',
            'How can I contact ' + name + '?',
            'What type of business is ' + name + '?'
        ];
    }
    
    function initQuickQuestions() {
        var questions = getQuickQuestions();
        var html = '';
        questions.forEach(function(q) {
            html += '<button type="button" class="qmsbf-quick-btn">' + escapeHtml(q) + '</button>';
        });
        $('#quick-questions').html(html);
    }
    
    function formatHours(hours) {
        if (!hours || Object.keys(hours).length === 0) return null;
        
        var dayMap = { 
            monday: 'Monday', tuesday: 'Tuesday', wednesday: 'Wednesday', 
            thursday: 'Thursday', friday: 'Friday', saturday: 'Saturday', sunday: 'Sunday' 
        };
        var days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        var parts = [];
        
        days.forEach(function(day) {
            var data = hours[day];
            if (data) {
                if (data.closed) {
                    parts.push(dayMap[day] + ': Closed');
                } else if (data.open && data.close) {
                    parts.push(dayMap[day] + ': ' + data.open + ' - ' + data.close);
                }
            }
        });
        
        return parts.length > 0 ? parts.join(', ') : null;
    }
    
    function simulateResponse(question) {
        collectBusinessData();
        var q = question.toLowerCase();
        var response = { text: '', found: false };
        var name = businessData.name || 'This business';
        
        if (q.indexOf('hour') >= 0 || q.indexOf('open') >= 0 || q.indexOf('close') >= 0) {
            var hoursText = formatHours(businessData.hours);
            if (hoursText) {
                response = { text: name + ' is open: ' + hoursText, found: true };
            } else {
                response = { text: "I don't have operating hours information for " + name + ". This data point is missing.", found: false };
            }
        } else if (q.indexOf('where') >= 0 || q.indexOf('location') >= 0 || q.indexOf('address') >= 0) {
            if (businessData.street && businessData.city) {
                var addr = businessData.street + ', ' + businessData.city;
                if (businessData.state) addr += ', ' + businessData.state;
                if (businessData.postal) addr += ' ' + businessData.postal;
                response = { text: name + ' is located at ' + addr + '.', found: true };
            } else {
                response = { text: "I don't have the full address for " + name + ".", found: false };
            }
        } else if (q.indexOf('contact') >= 0 || q.indexOf('phone') >= 0 || q.indexOf('call') >= 0 || q.indexOf('email') >= 0) {
            var contactParts = [];
            if (businessData.phone) contactParts.push('phone: ' + businessData.phone);
            if (businessData.email) contactParts.push('email: ' + businessData.email);
            if (contactParts.length > 0) {
                response = { text: 'You can reach ' + name + ' at ' + contactParts.join(' or ') + '.', found: true };
            } else {
                response = { text: "I don't have contact information for " + name + ".", found: false };
            }
        } else if (q.indexOf('type') >= 0 || q.indexOf('what kind') >= 0 || q.indexOf('what is') >= 0 || q.indexOf('about') >= 0) {
            if (businessData.description) {
                response = { text: name + ': ' + businessData.description, found: true };
            } else {
                response = { text: "I don't have detailed information about what type of business " + name + " is.", found: false };
            }
        } else if (q.indexOf('price') >= 0 || q.indexOf('cost') >= 0 || q.indexOf('expensive') >= 0) {
            if (businessData.price_range) {
                response = { text: name + ' has a price range of ' + businessData.price_range + '.', found: true };
            } else {
                response = { text: "I don't have pricing information for " + name + ".", found: false };
            }
        } else if (q.indexOf('website') >= 0 || q.indexOf('site') >= 0 || q.indexOf('url') >= 0) {
            if (businessData.website) {
                response = { text: 'You can visit ' + name + ' online at ' + businessData.website, found: true };
            } else {
                response = { text: "I don't have a website URL for " + name + ".", found: false };
            }
        } else {
            response = { text: "I can help answer questions about hours, location, contact info, prices, and business type. What would you like to know?", found: true };
        }
        
        return response;
    }
    
    // Helper function to escape HTML for XSS prevention
    function escapeHtml(text) {
        if (typeof text !== 'string') return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function addMessage(type, text, found) {
        var statusHtml = '';
        if (type === 'ai') {
            var statusClass = found ? 'found' : 'gap';
            var statusText = found ? '✓ Data Found' : '⚠ Data Gap';
            statusHtml = '<div class="qmsbf-message-status ' + statusClass + '">' + statusText + '</div>';
        }
        
        var html = '<div class="qmsbf-message ' + type + '">';
        html += '<div class="qmsbf-message-bubble">';
        html += statusHtml;
        html += escapeHtml(text); // XSS fix: escape user-provided text
        html += '</div></div>';
        
        // Remove empty state if present
        $('.qmsbf-chat-empty').remove();
        
        $('#chat-messages').append(html);
        
        // Scroll to bottom
        var container = $('#chat-messages');
        container.scrollTop(container[0].scrollHeight);
    }
    
    function showTypingIndicator() {
        var html = '<div class="qmsbf-message ai typing-indicator">';
        html += '<div class="qmsbf-message-bubble">';
        html += '<div class="qmsbf-chat-typing"><span></span><span></span><span></span></div>';
        html += '</div></div>';
        $('#chat-messages').append(html);
        
        var container = $('#chat-messages');
        container.scrollTop(container[0].scrollHeight);
    }
    
    function removeTypingIndicator() {
        $('.typing-indicator').remove();
    }
    
    function handleSendMessage(text) {
        var message = text || $('#chat-input').val().trim();
        if (!message) return;
        
        // Add user message
        addMessage('user', message);
        $('#chat-input').val('');
        
        // Show typing indicator
        showTypingIndicator();
        
        // Simulate response delay
        setTimeout(function() {
            removeTypingIndicator();
            var response = simulateResponse(message);
            addMessage('ai', response.text, response.found);
        }, 1000 + Math.random() * 500);
    }
    
    // Send message on button click
    $('#send-message').on('click', function() {
        handleSendMessage();
    });
    
    // Send message on Enter key
    $('#chat-input').on('keydown', function(e) {
        if (e.key === 'Enter') {
            handleSendMessage();
        }
    });
    
    // Quick question buttons
    $(document).on('click', '.qmsbf-quick-btn', function() {
        var question = $(this).text();
        handleSendMessage(question);
    });
    
    // Initialize quick questions when switching to AI Test tab
    $('.qmsbf-tab[data-tab="ai-test"]').on('click', function() {
        setTimeout(initQuickQuestions, 200);
    });
    
    // ==================== ORIGINAL FUNCTIONALITY ====================
    
    // Tab switching with smooth animation
    $('.qmsbf-tab').on('click', function() {
        var tabId = $(this).data('tab');
        
        $('.qmsbf-tab').removeClass('active');
        $(this).addClass('active');
        
        $('.qmsbf-tab-content').removeClass('active');
        setTimeout(() => {
            $('#' + tabId + '-tab').addClass('active');
        }, 100);
    });
    
    // ==================== MEDIA LIBRARY LOGO PICKER ====================
    var logoFrame = null;
    
    $(document).on('click', '#upload-logo-btn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        
        // Check if wp.media is available
        if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
            alert('WordPress Media Library is not available.\n\nPlease try:\n1. Refresh the page\n2. Make sure you are logged in as admin');
            return false;
        }
        
        // If the frame already exists, reopen it
        if (logoFrame) {
            logoFrame.open();
            return false;
        }
        
        // Create the media frame
        logoFrame = wp.media({
            title: 'Select or Upload Logo',
            button: {
                text: 'Use this logo'
            },
            multiple: false
        });
        
        // When an image is selected
        logoFrame.on('select', function() {
            var attachment = logoFrame.state().get('selection').first().toJSON();
            
            // Update hidden input
            $('#business_logo').val(attachment.url);
            
            // Update preview
            $('#logo-preview').show().find('img').attr('src', attachment.url);
            
            // Update button text
            $('#upload-logo-btn').html('<span class="dashicons dashicons-upload"></span> Change Logo');
            
            // Show remove button
            $('#remove-logo-btn').show();
        });
        
        // Open the frame
        logoFrame.open();
        return false;
    });
    
    // Remove logo
    $(document).on('click', '#remove-logo-btn', function(e) {
        e.preventDefault();
        
        // Clear hidden input
        $('#business_logo').val('');
        
        // Hide preview
        $('#logo-preview').hide().find('img').attr('src', '');
        
        // Update button text
        $('#upload-logo-btn').html('<span class="dashicons dashicons-upload"></span> Select Logo');
        
        // Hide remove button
        $(this).hide();
        
        return false;
    });
    
    // ==================== GOOGLE MAPS INTEGRATION ====================
    
    // Generate embed URL from address (no API key required)
    function generateFreeMapUrl(address) {
        if (!address) return '';
        var encoded = encodeURIComponent(address);
        return 'https://maps.google.com/maps?q=' + encoded + '&t=&z=15&ie=UTF8&iwloc=&output=embed';
    }
    
    // Update map preview
    function updateMapPreview(url) {
        if (url) {
            $('#map_url').val(url);
            $('#map-preview').attr('src', url);
            $('#map-preview-container').show();
            $('#clear-map-container').show();
        } else {
            $('#map_url').val('');
            $('#map-preview').attr('src', '');
            $('#map-preview-container').hide();
            $('#clear-map-container').hide();
        }
    }
    
    // Get full address from form fields
    function getAddressFromFields() {
        var parts = [];
        var street = $('#address_street').val();
        var city = $('#address_city').val();
        var state = $('#address_state').val();
        var postal = $('#address_postal').val();
        var country = $('#address_country').val();
        
        if (street) parts.push(street);
        if (city) parts.push(city);
        if (state) parts.push(state);
        if (postal) parts.push(postal);
        if (country) parts.push(country);
        
        return parts.join(', ');
    }
    
    // Auto-populate search field when address tab is shown or on load
    function populateMapSearch() {
        var address = getAddressFromFields();
        if (address) {
            $('#map-search').val(address);
        }
    }
    
    // Call on page load
    populateMapSearch();
    
    // Also call when switching to address tab
    $(document).on('click', '.qmsbf-tab[data-tab="address"]', function() {
        setTimeout(populateMapSearch, 100);
    });
    
    // Live update map search field as address fields are typed - using delegated events
    $(document).on('input change keyup', '#address_street, #address_city, #address_state, #address_postal, #address_country', function() {
        populateMapSearch();
    });
    
    // Generate map from address fields
    $('#generate-map-btn').on('click', function(e) {
        e.preventDefault();
        
        var address = getAddressFromFields();
        
        if (!address) {
            alert('Please enter an address first in the fields above.');
            return false;
        }
        
        var url = generateFreeMapUrl(address);
        updateMapPreview(url);
        
        // Show success
        var $btn = $(this);
        var originalHtml = $btn.html();
        $btn.html('<span class="dashicons dashicons-yes"></span> Map Generated!');
        setTimeout(function() {
            $btn.html(originalHtml);
        }, 2000);
        
        return false;
    });
    
    // Search for location
    $('#search-map-btn').on('click', function(e) {
        e.preventDefault();
        
        var query = $('#map-search').val();
        
        if (!query) {
            alert('Please enter a location to search.');
            return false;
        }
        
        var url = generateFreeMapUrl(query);
        updateMapPreview(url);
        
        // Show success
        var $btn = $(this);
        var originalHtml = $btn.html();
        $btn.html('<span class="dashicons dashicons-yes"></span> Found!');
        setTimeout(function() {
            $btn.html(originalHtml);
        }, 2000);
        
        return false;
    });
    
    // Allow enter key in search field
    $('#map-search').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            $('#search-map-btn').click();
            return false;
        }
    });
    
    // Toggle manual URL section
    $('#toggle-manual-url').on('click', function(e) {
        e.preventDefault();
        var $section = $('#manual-url-section');
        var $arrow = $('#manual-url-arrow');
        
        if ($section.is(':visible')) {
            $section.slideUp();
            $arrow.removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
        } else {
            $section.slideDown();
            $arrow.removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
        }
        return false;
    });
    
    // Manual URL input - update preview on change
    $('#map_url').on('change blur', function() {
        var url = $(this).val();
        if (url) {
            $('#map-preview').attr('src', url);
            $('#map-preview-container').show();
            $('#clear-map-container').show();
        }
    });
    
    // Clear map
    $('#clear-map-btn').on('click', function(e) {
        e.preventDefault();
        updateMapPreview('');
        return false;
    });
    
    // ==================== GOOGLE PLACES AUTOCOMPLETE ====================
    
    // Initialize Google Places Autocomplete if API key is set
    function initGooglePlacesAutocomplete() {
        // Check if Google Places API is loaded
        if (typeof google === 'undefined' || typeof google.maps === 'undefined' || typeof google.maps.places === 'undefined') {
            $('#address-autocomplete-wrapper').hide();
            return;
        }
        
        $('#address-autocomplete-wrapper').show();
        
        var input = document.getElementById('address_autocomplete');
        if (!input) {
            return;
        }
        
        var autocomplete = new google.maps.places.Autocomplete(input, {
            types: ['address'],
            fields: ['address_components', 'formatted_address', 'geometry']
        });
        
        // Handle place selection
        autocomplete.addListener('place_changed', function() {
            var place = autocomplete.getPlace();
            
            if (!place.address_components) {
                return;
            }
            
            
            // Parse address components
            var streetNumber = '';
            var streetName = '';
            var city = '';
            var state = '';
            var postalCode = '';
            var country = '';
            
            place.address_components.forEach(function(component) {
                var types = component.types;
                
                if (types.includes('street_number')) {
                    streetNumber = component.long_name;
                }
                if (types.includes('route')) {
                    streetName = component.long_name;
                }
                if (types.includes('locality') || types.includes('sublocality') || types.includes('postal_town')) {
                    city = component.long_name;
                }
                if (types.includes('administrative_area_level_1')) {
                    state = component.short_name; // Use abbreviation for state
                }
                if (types.includes('postal_code')) {
                    postalCode = component.long_name;
                }
                if (types.includes('country')) {
                    country = component.long_name;
                }
            });
            
            // Fill in the form fields
            var streetAddress = streetNumber ? streetNumber + ' ' + streetName : streetName;
            
            $('#address_street').val(streetAddress).trigger('change');
            $('#address_city').val(city).trigger('change');
            $('#address_state').val(state).trigger('change');
            $('#address_postal').val(postalCode).trigger('change');
            $('#address_country').val(country).trigger('change');
            
            // Clear the autocomplete field
            $('#address_autocomplete').val('');
            
            // Update map search field
            populateMapSearch();
            
            // Auto-generate map
            var fullAddress = getAddressFromFields();
            if (fullAddress) {
                var mapUrl = generateFreeMapUrl(fullAddress);
                updateMapPreview(mapUrl);
            }
            
            // Show success notification
            showNotification('Address auto-filled!', 'success');
        });
    }
    
    // Initialize when Google API is loaded (may load after our script)
    if (typeof google !== 'undefined' && typeof google.maps !== 'undefined') {
        initGooglePlacesAutocomplete();
    } else {
        // Wait for Google API to load
        window.initGooglePlacesCallback = initGooglePlacesAutocomplete;
        
        // Check periodically if Google API has loaded
        var checkGoogleInterval = setInterval(function() {
            if (typeof google !== 'undefined' && typeof google.maps !== 'undefined' && typeof google.maps.places !== 'undefined') {
                clearInterval(checkGoogleInterval);
                initGooglePlacesAutocomplete();
            }
        }, 500);
        
        // Stop checking after 10 seconds
        setTimeout(function() {
            clearInterval(checkGoogleInterval);
        }, 10000);
    }
    
    // ==================== SOCIAL LINK VALIDATION ====================
    
    // Define allowed domains for each social platform
    var socialDomains = {
        'social_facebook': ['facebook.com', 'www.facebook.com', 'fb.com', 'www.fb.com', 'm.facebook.com'],
        'social_twitter': ['twitter.com', 'www.twitter.com', 'x.com', 'www.x.com', 'mobile.twitter.com'],
        'social_instagram': ['instagram.com', 'www.instagram.com', 'instagr.am'],
        'social_linkedin': ['linkedin.com', 'www.linkedin.com', 'uk.linkedin.com'],
        'social_youtube': ['youtube.com', 'www.youtube.com', 'youtu.be', 'm.youtube.com'],
        'social_tiktok': ['tiktok.com', 'www.tiktok.com', 'vm.tiktok.com'],
        'social_pinterest': ['pinterest.com', 'www.pinterest.com', 'pin.it', 'pinterest.co.uk'],
        'social_yelp': ['yelp.com', 'www.yelp.com', 'm.yelp.com'],
        'social_google': ['google.com', 'www.google.com', 'g.page', 'goo.gl', 'maps.google.com', 'business.google.com'],
        'social_other': null // Allow any URL for "other"
    };
    
    // Platform display names for error messages
    var socialNames = {
        'social_facebook': 'Facebook',
        'social_twitter': 'X (Twitter)',
        'social_instagram': 'Instagram',
        'social_linkedin': 'LinkedIn',
        'social_youtube': 'YouTube',
        'social_tiktok': 'TikTok',
        'social_pinterest': 'Pinterest',
        'social_yelp': 'Yelp',
        'social_google': 'Google Business',
        'social_other': 'Other'
    };
    
    // Extract domain from URL
    function getDomainFromUrl(url) {
        if (!url) return null;
        try {
            // Add protocol if missing
            if (!url.match(/^https?:\/\//i)) {
                url = 'https://' + url;
            }
            var urlObj = new URL(url);
            return urlObj.hostname.toLowerCase();
        } catch (e) {
            return null;
        }
    }
    
    // Validate social link
    function validateSocialLink(inputId, url) {
        var allowedDomains = socialDomains[inputId];
        var platformName = socialNames[inputId];
        
        // If empty, it's valid (not required)
        if (!url || url.trim() === '') {
            return { valid: true };
        }
        
        // "Other" allows any URL
        if (allowedDomains === null) {
            return { valid: true };
        }
        
        var domain = getDomainFromUrl(url);
        
        if (!domain) {
            return { 
                valid: false, 
                message: 'Please enter a valid URL'
            };
        }
        
        // Check if domain matches any allowed domain
        var isValid = allowedDomains.some(function(allowed) {
            return domain === allowed || domain.endsWith('.' + allowed);
        });
        
        if (!isValid) {
            return { 
                valid: false, 
                message: 'Please enter a valid ' + platformName + ' URL (e.g., ' + allowedDomains[0] + ')'
            };
        }
        
        return { valid: true };
    }
    
    // Show/hide validation error
    function showSocialError($input, message) {
        var $field = $input.closest('.qmsbf-field');
        $field.find('.qmsbf-social-error').remove();
        
        if (message) {
            $input.css({
                'border-color': '#dc2626',
                'background-color': '#fef2f2'
            });
            $field.append('<p class="qmsbf-social-error" style="color: #dc2626; font-size: 12px; margin-top: 5px; display: flex; align-items: center; gap: 5px;"><span class="dashicons dashicons-warning" style="font-size: 16px;"></span> ' + escapeHtml(message) + '</p>');
        } else {
            $input.css({
                'border-color': '',
                'background-color': ''
            });
        }
    }
    
    // Validate on blur/change/input - using delegated events
    $(document).on('blur change input', 'input[id^="social_"]', function() {
        var $input = $(this);
        var inputId = $input.attr('id');
        var url = $input.val().trim();
        
        // Skip validation if empty
        if (!url) {
            showSocialError($input, null);
            return;
        }
        
        var result = validateSocialLink(inputId, url);
        
        if (!result.valid) {
            showSocialError($input, result.message);
        } else {
            showSocialError($input, null);
            
            // Auto-add https:// if missing and URL is valid (only on blur)
            if (url && !url.match(/^https?:\/\//i)) {
                $input.val('https://' + url);
            }
        }
    });
    
    // Validate all social links before save
    function validateAllSocialLinks() {
        var allValid = true;
        var firstInvalid = null;
        var errorMessages = [];
        
        $('input[id^="social_"]').each(function() {
            var $input = $(this);
            var inputId = $input.attr('id');
            var url = $input.val().trim();
            
            // Skip empty fields
            if (!url) {
                showSocialError($input, null);
                return true; // continue
            }
            
            var result = validateSocialLink(inputId, url);
            
            if (!result.valid) {
                showSocialError($input, result.message);
                errorMessages.push(result.message);
                allValid = false;
                if (!firstInvalid) {
                    firstInvalid = $input;
                }
            } else {
                showSocialError($input, null);
            }
        });
        
        // Switch to social tab and focus first invalid field
        if (!allValid && firstInvalid) {
            $('.qmsbf-tab[data-tab="social"]').click();
            setTimeout(function() {
                firstInvalid.focus();
                // Scroll to the field
                $('html, body').animate({
                    scrollTop: firstInvalid.offset().top - 100
                }, 300);
            }, 150);
        }
        
        return allValid;
    }

    // Create page functionality
    $('#create-page-btn').on('click', function() {
        var $btn = $(this);
        var originalHtml = $btn.html();
        $btn.prop('disabled', true).html('Creating <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_create_page',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Replace button with view button
                    $btn.replaceWith(
                        '<a href="' + response.data.page_url + '" target="_blank" class="qmsbf-button secondary">' +
                        '<span class="dashicons dashicons-visibility"></span> View Page</a>'
                    );
                    
                    showNotification('Page created successfully!', 'success');
                }
            },
            error: function() {
                showNotification('Error creating page', 'error');
                $btn.prop('disabled', false).html(originalHtml);
            }
        });
    });
    
    // Regenerate page button (in settings tab)
    $('#regenerate-page-btn').on('click', function() {
        var $btn = $(this);
        var originalHtml = $btn.html();
        $btn.prop('disabled', true).html('Creating <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_create_page',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Hide the button and update the default page link
                    $btn.hide();
                    showNotification('Default profile page created! URL: ' + response.data.page_url, 'success');
                    // Reload to show updated header button
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                }
            },
            error: function() {
                showNotification('Error creating page', 'error');
                $btn.prop('disabled', false).html(originalHtml);
            }
        });
    });
    
    // Auto-populate functionality
    $('#auto-populate-btn').on('click', function() {
        var $btn = $(this);
        var originalText = $btn.html();
        $btn.prop('disabled', true).html('Loading <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_auto_populate',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    $('#business_name').val(response.data.business_name).addClass('highlight');
                    $('#business_website').val(response.data.website).addClass('highlight');
                    $('#business_email').val(response.data.email).addClass('highlight');
                    $('#business_description').val(response.data.description).addClass('highlight');
                    
                    setTimeout(() => {
                        $('.highlight').removeClass('highlight');
                    }, 1000);
                    
                    showNotification('Data populated successfully!', 'success');
                }
            },
            error: function() {
                showNotification('Error auto-populating data', 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    });
    
    // Flush Cache functionality
    $('#flush-cache-btn').on('click', function() {
        // Show warning confirmation
        if (!confirm('⚠️ This will flush ALL site caches including:\n\n• WordPress object cache (site-wide)\n• Rewrite rules\n• Popular caching plugins (if detected)\n\nThis may temporarily impact site performance.\n\nContinue?')) {
            return;
        }
        
        var $btn = $(this);
        var originalText = $btn.html();
        $btn.prop('disabled', true).html('Flushing <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_flush_cache',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    var message = response.data.message + '\n\nCleared:\n• ' + response.data.flushed.join('\n• ');
                    
                    if (response.data.notes && response.data.notes.length > 0) {
                        message += '\n\n' + response.data.notes.join('\n');
                    }
                    
                    showNotification(response.data.message, 'success');
                    
                    // Show detailed modal
                    var modalHtml = '<div class="qmsbf-cache-modal" style="position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.5);z-index:100000;display:flex;align-items:center;justify-content:center;">';
                    modalHtml += '<div style="background:white;border-radius:12px;padding:30px;max-width:500px;margin:20px;box-shadow:0 20px 40px rgba(0,0,0,0.3);">';
                    modalHtml += '<h3 style="margin:0 0 15px;color:#10b981;">Cache Flushed Successfully</h3>';
                    modalHtml += '<p style="margin:0 0 15px;color:#6b7280;"><strong>Cleared:</strong></p>';
                    modalHtml += '<ul style="margin:0 0 20px;padding-left:20px;color:#374151;">';
                    response.data.flushed.forEach(function(item) {
                        modalHtml += '<li>' + item + '</li>';
                    });
                    modalHtml += '</ul>';
                    if (response.data.notes && response.data.notes.length > 0) {
                        modalHtml += '<div style="background:#fef3c7;border-radius:8px;padding:15px;margin-bottom:20px;">';
                        response.data.notes.forEach(function(note) {
                            modalHtml += '<p style="margin:5px 0;font-size:13px;color:#92400e;">' + note + '</p>';
                        });
                        modalHtml += '</div>';
                    }
                    modalHtml += '<button onclick="jQuery(\'.qmsbf-cache-modal\').remove();" style="background:#1e1e1e;color:white;border:none;padding:10px 20px;border-radius:6px;cursor:pointer;font-size:14px;">Got it</button>';
                    modalHtml += '</div></div>';
                    
                    $('body').append(modalHtml);
                } else {
                    showNotification('Error flushing cache: ' + response.data, 'error');
                }
            },
            error: function() {
                showNotification('Error flushing cache', 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    });
    
    // Run Diagnostics functionality
    $('#run-diagnostics-btn').on('click', function() {
        var $btn = $(this);
        var originalText = $btn.html();
        $btn.prop('disabled', true).html('Running <span class="qmsbf-spinner" style="margin-left: 8px;"></span>');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_run_diagnostics',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    var html = '<div style="display: grid; gap: 12px;">';
                    
                    var tests = {
                        'json_endpoint': 'JSON Endpoint',
                        'profile_page': 'Profile Page',
                        'business_data': 'Business Data',
                        'rest_api': 'REST API',
                        'openapi_spec': 'OpenAPI Spec',
                        'php_version': 'PHP Version',
                        'wp_version': 'WordPress Version'
                    };
                    
                    for (var key in tests) {
                        if (response.data[key]) {
                            var item = response.data[key];
                            var color = item.status === 'success' ? '#10b981' : (item.status === 'warning' ? '#f59e0b' : '#ef4444');
                            var icon = item.status === 'success' ? '✓' : (item.status === 'warning' ? '⚠' : '✗');
                            
                            html += '<div style="display: flex; align-items: center; padding: 12px; background: #f9fafb; border-radius: 8px; border-left: 4px solid ' + color + ';">';
                            html += '<span style="color: ' + color + '; font-size: 18px; margin-right: 12px;">' + icon + '</span>';
                            html += '<div>';
                            html += '<strong style="color: #374151;">' + escapeHtml(tests[key]) + '</strong><br>';
                            html += '<span style="color: #6b7280; font-size: 13px;">' + escapeHtml(item.message) + '</span>';
                            if (item.url) {
                                // Sanitize URL - only allow http/https
                                var safeUrl = item.url;
                                if (!/^https?:\/\//i.test(safeUrl)) {
                                    safeUrl = '';
                                }
                                if (safeUrl) {
                                    html += '<br><a href="' + encodeURI(safeUrl) + '" target="_blank" rel="noopener" style="font-size: 12px;">View →</a>';
                                }
                            }
                            html += '</div></div>';
                        }
                    }
                    
                    html += '</div>';
                    
                    $('#diagnostics-content').html(html);
                    $('#diagnostics-results').slideDown();
                } else {
                    showNotification('Error running diagnostics: ' + response.data, 'error');
                }
            },
            error: function() {
                showNotification('Error running diagnostics', 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    });
    
    // Refresh Analytics button with loading animation
    $('#refresh-analytics-btn').on('click', function() {
        var $btn = $(this);
        var $icon = $btn.find('.dashicons');
        
        // Disable button and show loading state
        $btn.prop('disabled', true);
        $icon.addClass('spin');
        $btn.find('span:last').text('Refreshing...');
        
        // Add a small delay so the user sees the animation, then reload
        setTimeout(function() {
            window.location.reload(true);
        }, 600);
    });
    
    // Export Analytics dropdown toggle
    $('#export-analytics-btn').on('click', function(e) {
        e.stopPropagation();
        var $menu = $('#export-dropdown-menu');
        $menu.toggle();
    });
    
    // Close dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.qmsbf-export-dropdown').length) {
            $('#export-dropdown-menu').hide();
        }
    });
    
    // Export option click handlers
    $('.export-option').on('click', function() {
        var format = $(this).data('format');
        var $btn = $('#export-analytics-btn');
        var originalText = $btn.html();
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Exporting...');
        $('#export-dropdown-menu').hide();
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_export_analytics',
                nonce: qmsbf_ajax.nonce,
                format: format
            },
            success: function(response) {
                $btn.prop('disabled', false).html(originalText);
                
                if (response.success) {
                    // Decode base64 content
                    var content = atob(response.data.content);
                    
                    if (format === 'csv') {
                        // Download CSV directly
                        var blob = new Blob([content], { type: response.data.mime_type });
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = response.data.filename;
                        document.body.appendChild(a);
                        a.click();
                        window.URL.revokeObjectURL(url);
                        document.body.removeChild(a);
                        showNotification('CSV exported successfully!', 'success');
                    } else {
                        // For PDF - open in new window and trigger print dialog
                        var printWindow = window.open('', '_blank', 'width=900,height=700');
                        printWindow.document.write(content);
                        printWindow.document.close();
                        
                        // Wait for content to load, then trigger print
                        printWindow.onload = function() {
                            setTimeout(function() {
                                printWindow.print();
                            }, 250);
                        };
                        
                        showNotification('PDF report opened. Use "Save as PDF" in the print dialog.', 'success');
                    }
                } else {
                    showNotification('Error exporting analytics: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function() {
                $btn.prop('disabled', false).html(originalText);
                showNotification('Error exporting analytics', 'error');
            }
        });
    });
    
    // Hover effect for export dropdown options
    $(document).on('mouseenter', '.export-option', function() {
        $(this).css('background', '#f3f4f6');
    }).on('mouseleave', '.export-option', function() {
        $(this).css('background', 'none');
    });
    
    // Reset Analytics functionality
    $('#reset-analytics-btn').on('click', function() {
        if (!confirm('Are you sure you want to reset all analytics data? This cannot be undone.')) {
            return;
        }
        
        var $btn = $(this);
        var originalText = $btn.html();
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Resetting...');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_reset_analytics',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotification('Analytics reset successfully!', 'success');
                    // Reload the page to show cleared stats
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification('Error resetting analytics: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function() {
                showNotification('Error resetting analytics', 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    });
    
    // Test Tracking functionality - handles buttons under AI service logos
    $('.qmsbf-test-btn').on('click', function() {
        var $btn = $(this);
        var testType = $btn.data('service');
        var originalText = $btn.text();
        var $result = $('#test-tracking-result');
        
        // Disable button and show loading state
        $btn.prop('disabled', true).addClass('testing').text('Testing...');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_test_tracking',
                nonce: qmsbf_ajax.nonce,
                test_type: testType
            },
            success: function(response) {
                if (response.success) {
                    var data = response.data;
                    $btn.removeClass('testing').addClass('success').text('✓ Pass');
                    
                    // Trigger glow animation on the logo
                    var $logo = $btn.closest('.qmsbf-ai-service-item').find('.qmsbf-ai-service-logo');
                    $logo.removeClass('glow-success');
                    // Force reflow to restart animation
                    $logo[0].offsetWidth;
                    $logo.addClass('glow-success');
                    
                    // Remove glow class after animation completes
                    setTimeout(function() {
                        $logo.removeClass('glow-success');
                    }, 2000);
                    
                    // Show result card
                    var resultHtml = '<h3 style="margin: 0 0 10px; color: #065f46;">✓ Test Passed: ' + data.test_service + '</h3>';
                    resultHtml += '<p style="margin: 0; color: #047857;">';
                    resultHtml += 'Detected as: <strong>' + (data.detection_result.service || 'Unknown') + '</strong><br>';
                    resultHtml += 'Method: ' + data.detection_result.method + ' • Confidence: ' + data.detection_result.confidence + '<br>';
                    resultHtml += 'Endpoint requests: ' + data.before_count + ' → ' + data.after_count;
                    resultHtml += '</p>';
                    
                    $result.css('background', '#f0fdf4').css('border', '2px solid #10b981').html(resultHtml).show();
                    
                    // Reset button after 3 seconds
                    setTimeout(function() {
                        $btn.removeClass('success').text(originalText);
                    }, 3000);
                } else {
                    $btn.removeClass('testing').addClass('error').text('✗ Fail');
                    
                    var errorMsg = response.data && response.data.message ? response.data.message : 'Test failed';
                    $result.css('background', '#fef2f2').css('border', '2px solid #ef4444')
                           .html('<h3 style="margin: 0 0 10px; color: #991b1b;">✗ Test Failed</h3><p style="margin: 0; color: #b91c1c;">' + errorMsg + '</p>')
                           .show();
                    
                    setTimeout(function() {
                        $btn.removeClass('error').text(originalText);
                    }, 3000);
                }
            },
            error: function(xhr, status, error) {
                $btn.removeClass('testing').addClass('error').text('✗ Error');
                
                $result.css('background', '#fef2f2').css('border', '2px solid #ef4444')
                       .html('<h3 style="margin: 0 0 10px; color: #991b1b;">✗ Request Failed</h3><p style="margin: 0; color: #b91c1c;">Could not connect to server. Error: ' + error + '</p>')
                       .show();
                
                setTimeout(function() {
                    $btn.removeClass('error').text(originalText);
                }, 3000);
            },
            complete: function() {
                $btn.prop('disabled', false);
            }
        });
    });
    
    // Save all settings with enhanced feedback
    $('#save-settings').on('click', function() {
        var $btn = $(this);
        var originalText = $btn.text();
        
        // Validate social links first
        if (typeof validateAllSocialLinks === 'function' && !validateAllSocialLinks()) {
            showNotification('Please fix the invalid social links before saving.', 'error');
            return false;
        }
        
        // Add saving animation
        $btn.prop('disabled', true)
            .html('Saving <span class="qmsbf-spinner" style="margin-left: 8px;"></span>')
            .addClass('saving');
        
        // Collect all data
        var formData = {
            action: 'qmsbf_save_settings',
            nonce: qmsbf_ajax.nonce,
            settings: {},
            business: {},
            address: {},
            hours: {}
        };
        
        // Settings tab
        formData.settings.business_type = $('#business_type').val();
        formData.settings.show_footer = $('input[name="show_footer"]').is(':checked') ? 1 : 0;
        formData.settings.enable_analytics = $('input[name="enable_analytics"]').is(':checked') ? 1 : 0;
        formData.settings.enable_widget = $('input[name="enable_widget"]').is(':checked') ? 1 : 0;
        formData.settings.allow_cross_origin = $('input[name="allow_cross_origin"]').is(':checked') ? 1 : 0;
        formData.settings.data_reminder = $('#data_reminder').val();
        formData.settings.timezone = $('#plugin_timezone').val();
        formData.settings.custom_profile_url = $('#custom_profile_url').val();
        
        // Business info
        formData.business.name = $('#business_name').val();
        formData.business.legal_name = $('#legal_name').val();
        formData.business.description = $('#business_description').val();  // Fixed: was #description
        formData.business.logo = $('#business_logo').val();
        formData.business.price_range = $('#price_range').val();
        
        // Contact
        formData.business.website = $('#business_website').val();
        formData.business.email = $('#business_email').val();
        formData.business.phone = $('#business_phone').val();
        formData.business.fax = $('#business_fax').val();
        
        // Address
        formData.address.street = $('#address_street').val();
        formData.address.city = $('#address_city').val();
        formData.address.state = $('#address_state').val();
        formData.address.postal = $('#address_postal').val();
        formData.address.country = $('#address_country').val();
        formData.address.map_url = $('#map_url').val();
        
        // Hours
        formData.hours.is_24_7 = $('#open_24_7').is(':checked') ? 1 : 0;
        var days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        days.forEach(function(day) {
            formData.hours[day] = {
                open: $('input[name="hours_' + day + '_open"]').val(),
                close: $('input[name="hours_' + day + '_close"]').val(),
                closed: $('input[name="hours_' + day + '_closed"]').is(':checked') ? 1 : 0
            };
        });
        formData.hours.special = $('#special_hours').val();
        
        // Social links
        formData.social = {
            facebook: $('#social_facebook').val(),
            twitter: $('#social_twitter').val(),
            instagram: $('#social_instagram').val(),
            linkedin: $('#social_linkedin').val(),
            youtube: $('#social_youtube').val(),
            tiktok: $('#social_tiktok').val(),
            pinterest: $('#social_pinterest').val(),
            yelp: $('#social_yelp').val(),
            google: $('#social_google').val(),
            other: $('#social_other').val()
        };
        
        // Collect display fields
        formData.display = {
            name: $('input[name="display_name"]').is(':checked'),
            legal_name: $('input[name="display_legal_name"]').is(':checked'),
            description: $('input[name="display_description"]').is(':checked'),
            logo: $('input[name="display_logo"]').is(':checked'),
            website: $('input[name="display_website"]').is(':checked'),
            email: $('input[name="display_email"]').is(':checked'),
            phone: $('input[name="display_phone"]').is(':checked'),
            fax: $('input[name="display_fax"]').is(':checked'),
            address: $('input[name="display_address"]').is(':checked'),
            hours: $('input[name="display_hours"]').is(':checked'),
            price_range: $('input[name="display_price_range"]').is(':checked'),
            map: $('input[name="display_map"]').is(':checked'),
            special_hours: $('input[name="display_special_hours"]').is(':checked'),
            social: $('input[name="display_social"]').is(':checked')
        };
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    $btn.removeClass('saving').addClass('success');
                    $btn.html('<span class="dashicons dashicons-yes"></span> Saved!');
                    
                    $('.qmsbf-save-status').text('✓ All settings saved successfully!').fadeIn();
                    
                    setTimeout(function() {
                        $btn.removeClass('success')
                            .prop('disabled', false)
                            .text(originalText);
                        $('.qmsbf-save-status').fadeOut();
                    }, 3000);
                }
            },
            error: function() {
                $btn.removeClass('saving');
                showNotification('Error saving settings. Please try again.', 'error');
                $btn.prop('disabled', false).text(originalText);
            }
        });
    });
    
    // Reset All Data button handler
    $('#qmsbf-reset-all-data').on('click', function() {
        if (!confirm('Are you sure you want to delete ALL plugin data?\n\nThis will remove:\n• All business information\n• All settings\n• Collected email leads\n• The profile page\n\nThis cannot be undone!')) {
            return;
        }
        
        if (!confirm('FINAL WARNING: Click OK to permanently delete all data.')) {
            return;
        }
        
        var $btn = $(this);
        $btn.prop('disabled', true).html('<span class="qmsbf-spinner"></span> Resetting...');
        
        $.ajax({
            url: qmsbf_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'qmsbf_reset_all_data',
                nonce: qmsbf_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    alert('All data has been reset. The page will now reload.');
                    window.location.reload();
                } else {
                    alert('Error: ' + (response.data || 'Could not reset data'));
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Reset All Data');
                }
            },
            error: function() {
                alert('Connection error. Please try again.');
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Reset All Data');
            }
        });
    });
    
    // License handlers are in admin-license.js - not duplicated here
    
    // Open 24/7 checkbox handling
    $('#open_24_7').on('change', function() {
        var $wrapper = $('#daily-hours-wrapper');
        if ($(this).is(':checked')) {
            $wrapper.css({
                'opacity': '0.5',
                'pointer-events': 'none'
            });
        } else {
            $wrapper.css({
                'opacity': '1',
                'pointer-events': 'auto'
            });
        }
    });
    
    // Hours - closed checkbox handling with smooth animation
    $('.closed-checkbox').on('change', function() {
        var $row = $(this).closest('.qmsbf-hours-row');
        var $timeInputs = $row.find('input[type="time"]');
        
        if ($(this).is(':checked')) {
            $timeInputs.prop('disabled', true).addClass('disabled');
            $row.addClass('day-closed');
        } else {
            $timeInputs.prop('disabled', false).removeClass('disabled');
            $row.removeClass('day-closed');
        }
    });
    
    // Media uploader with preview
    $('.upload-btn').on('click', function(e) {
        e.preventDefault();
        
        var $input = $(this).prev('input');
        var $btn = $(this);
        
        if (typeof wp.media !== 'undefined') {
            var mediaFrame = wp.media({
                title: 'Select or Upload Logo',
                button: {
                    text: 'Use this logo'
                },
                multiple: false,
                library: {
                    type: 'image'
                }
            });
            
            mediaFrame.on('select', function() {
                var attachment = mediaFrame.state().get('selection').first().toJSON();
                $input.val(attachment.url).addClass('highlight');
                
                // Show preview if possible
                if (attachment.sizes && attachment.sizes.thumbnail) {
                    var preview = '<img src="' + attachment.sizes.thumbnail.url + '" class="logo-preview">';
                    $btn.after(preview);
                }
                
                setTimeout(() => {
                    $input.removeClass('highlight');
                }, 1000);
            });
            
            mediaFrame.open();
        } else {
            alert('Media library not available. Please enter the URL manually.');
        }
    });
    
    // Form validation with visual feedback
    $('#business_name').on('blur', function() {
        if ($(this).val() === '') {
            $(this).addClass('error');
            showFieldError($(this), 'Business name is required');
        } else {
            $(this).removeClass('error');
            hideFieldError($(this));
        }
    });
    
    $('#email').on('blur', function() {
        var email = $(this).val();
        if (email && !isValidEmail(email)) {
            $(this).addClass('error');
            showFieldError($(this), 'Please enter a valid email address');
        } else {
            $(this).removeClass('error');
            hideFieldError($(this));
        }
    });
    
    // Helper functions
    function showFieldError($field, message) {
        var $error = $field.next('.field-error');
        if ($error.length === 0) {
            $error = $('<span class="field-error"></span>');
            $field.after($error);
        }
        $error.text(message).fadeIn();
    }
    
    function hideFieldError($field) {
        $field.next('.field-error').fadeOut();
    }
    
    function isValidEmail(email) {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
    }
    
    // Premium notification function
    function showNotification(message, type) {
        var icon = type === 'success' ? 'yes-alt' : 'warning';
        var $notification = $(
            '<div class="qmsbf-notification ' + type + '">' +
            '<span class="dashicons dashicons-' + icon + '"></span>' +
            '<span>' + escapeHtml(message) + '</span>' +
            '</div>'
        );
        
        $('body').append($notification);
        
        setTimeout(function() {
            $notification.addClass('show');
        }, 100);
        
        setTimeout(function() {
            $notification.removeClass('show');
            setTimeout(function() {
                $notification.remove();
            }, 300);
        }, 3000);
    }
    
    // Keyboard shortcuts
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + S to save
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 83) {
            e.preventDefault();
            $('#save-settings').click();
        }
    });
    
    // Auto-save draft (every 60 seconds)
    // Autosave functionality removed - use manual save button
    // The qmsbf_save_draft handler was not implemented, causing failed AJAX requests
    
    function collectFormData() {
        // Reusable function to collect all form data
        return {
            nonce: qmsbf_ajax.nonce,
            // ... collect all form fields
        };
    }
    
    // Store original values for change detection
    $('input, textarea, select').each(function() {
        $(this).data('original', $(this).val());
    });
    
    // Select/Deselect all display fields
    $('#select-all-fields').on('click', function() {
        $('#display-tab input[type="checkbox"]').prop('checked', true);
        showNotification('All fields selected', 'success');
    });
    
    $('#deselect-all-fields').on('click', function() {
        $('#display-tab input[type="checkbox"]').prop('checked', false);
        showNotification('All fields deselected', 'success');
    });
});

// Add dynamic styles for new features
if (!document.getElementById('qmsbf-dynamic-styles')) {
    var styles = `
        <style id="qmsbf-dynamic-styles">
            .qmsbf-display-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
                gap: 40px;
                margin-top: 30px;
            }
            
            .qmsbf-display-section {
                background: linear-gradient(135deg, #fafafa 0%, #f4f4f5 100%);
                border: 1px solid #e5e7eb;
                border-radius: 12px;
                padding: 24px;
            }
            
            .qmsbf-display-section h3 {
                font-size: 16px;
                font-weight: 600;
                color: #111827;
                margin: 0 0 20px 0;
                padding-bottom: 12px;
                border-bottom: 1px solid #e5e7eb;
            }
            
            .qmsbf-display-section .qmsbf-field {
                margin-bottom: 16px;
            }
            
            .qmsbf-display-section .qmsbf-field:last-child {
                margin-bottom: 0;
            }
            
            .qmsbf-display-section .qmsbf-checkbox {
                font-size: 14px;
            }
            
            .qmsbf-display-section .qmsbf-button {
                margin-right: 10px;
                margin-top: 10px;
            }
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            
            .dashicons.spin {
                animation: spin 1s linear infinite;
            }
            
            .spinner {
                display: inline-block;
                width: 16px;
                height: 16px;
                border: 2px solid rgba(255,255,255,0.3);
                border-top-color: white;
                border-radius: 50%;
                animation: spin 0.8s linear infinite;
            }
            
            .qmsbf-button.saving {
                background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
            }
            
            .qmsbf-button.success {
                background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            }
            
            .qmsbf-notification {
                position: fixed;
                top: 50px;
                right: -400px;
                background: white;
                padding: 16px 24px;
                border-radius: 12px;
                box-shadow: 0 10px 40px rgba(0,0,0,0.2);
                z-index: 100000;
                transition: right 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
                max-width: 400px;
                border-left: 4px solid #000;
                display: flex;
                align-items: center;
                gap: 12px;
            }
            
            .qmsbf-notification.show {
                right: 20px;
            }
            
            .qmsbf-notification.success {
                border-left-color: #10b981;
                background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 100%);
            }
            
            .qmsbf-notification.error {
                border-left-color: #ef4444;
                background: linear-gradient(135deg, #ffffff 0%, #fef2f2 100%);
            }
            
            .qmsbf-notification .dashicons {
                font-size: 24px;
            }
            
            .qmsbf-notification.success .dashicons {
                color: #10b981;
            }
            
            .qmsbf-notification.error .dashicons {
                color: #ef4444;
            }
            
            .highlight {
                background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%) !important;
                transition: background 0.3s;
            }
            
            input.error {
                border-color: #ef4444 !important;
                background: #fef2f2 !important;
            }
            
            .field-error {
                display: none;
                color: #ef4444;
                font-size: 12px;
                margin-top: 4px;
            }
            
            .qmsbf-hours-row.day-closed {
                opacity: 0.6;
                background: #f9fafb;
            }
            
            .logo-preview {
                width: 60px;
                height: 60px;
                object-fit: cover;
                border-radius: 8px;
                margin-left: 12px;
                border: 2px solid #e5e7eb;
            }
            
            /* License page styles */
            .qmsbf-license-status {
                display: flex;
                align-items: center;
                gap: 16px;
                padding: 20px;
                border-radius: 12px;
                margin-bottom: 24px;
            }
            
            .qmsbf-license-status.active {
                background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 100%);
                border: 2px solid #10b981;
            }
            
            .qmsbf-license-status.inactive {
                background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
                border: 2px solid #ef4444;
            }
            
            .qmsbf-license-status .dashicons {
                font-size: 32px;
            }
            
            .qmsbf-license-status.active .dashicons {
                color: #10b981;
            }
            
            .qmsbf-license-status.inactive .dashicons {
                color: #ef4444;
            }
            
            .qmsbf-license-input {
                display: flex;
                gap: 12px;
                align-items: center;
            }
            
            .qmsbf-license-input input {
                flex: 1;
                max-width: 400px;
            }
            
            .qmsbf-features {
                list-style: none;
                padding: 0;
                display: grid;
                grid-template-columns: repeat(2, 1fr);
                gap: 12px;
            }
            
            .qmsbf-features li {
                padding: 8px 0;
                color: #4b5563;
            }
        </style>
    `;
    document.head.insertAdjacentHTML('beforeend', styles);
}

// ==================== VISIT DETAIL MODAL - FIXED VERSION ====================

(function() {
    'use strict';
    
    // Wait for everything to be ready
    function initWhenReady() {
        // Make sure jQuery is available (WordPress admin uses it)
        if (typeof jQuery === 'undefined') {
            setTimeout(initWhenReady, 100);
            return;
        }
        
        jQuery(document).ready(function($) {
            initVisitDetailModal($);
            initBotBlocking($);
            
            // Re-init when analytics tab is clicked (in case it lazy loads)
            $(document).on('click', '.qmsbf-tab[data-tab="analytics"]', function() {
                setTimeout(function() {
                    initVisitDetailModal($);
                }, 200);
            });
        });
    }
    
    function initVisitDetailModal($) {
        // Create modal if it doesn't exist
        if ($('#qmsbf-visit-detail-modal').length === 0) {
            var modalHTML = '<div id="qmsbf-visit-detail-modal" class="qmsbf-visit-modal">' +
                '<div class="qmsbf-visit-modal-content">' +
                '<div class="qmsbf-visit-modal-header">' +
                '<h3><span class="dashicons dashicons-visibility"></span> Visit Details</h3>' +
                '<button type="button" class="qmsbf-visit-modal-close">&times;</button>' +
                '</div>' +
                '<div class="qmsbf-visit-modal-body" id="qmsbf-visit-detail-content"></div>' +
                '</div>' +
                '</div>';
            $('body').append(modalHTML);
        }
        
        // Use event delegation for click handling (works even if elements added later)
        $(document).off('click.qmsbfVisit').on('click.qmsbfVisit', '.qmsbf-request-row[data-visit-index]', function(e) {
            e.preventDefault();
            var index = parseInt($(this).attr('data-visit-index'), 10);
            showVisitDetails(index);
        });
        
        // Add clickable class and hint to rows
        $('.qmsbf-request-row[data-visit-index]').each(function() {
            var $row = $(this);
            if (!$row.hasClass('clickable')) {
                $row.addClass('clickable');
            }
            if ($row.find('.qmsbf-view-details').length === 0) {
                $row.append('<span class="qmsbf-view-details">Click for details →</span>');
            }
        });
        
        // Close modal handlers
        $(document).off('click.qmsbfModalClose').on('click.qmsbfModalClose', '.qmsbf-visit-modal-close', function() {
            closeVisitModal();
        });
        
        $('#qmsbf-visit-detail-modal').off('click.qmsbfBackdrop').on('click.qmsbfBackdrop', function(e) {
            if (e.target === this) {
                closeVisitModal();
            }
        });
        
        $(document).off('keydown.qmsbfEsc').on('keydown.qmsbfEsc', function(e) {
            if (e.key === 'Escape' || e.keyCode === 27) {
                closeVisitModal();
            }
        });
    }
    
    function showVisitDetails(index) {
        // Get data from the global variable set by PHP
        var visitData = window.qmsbfVisitData || [];
        var visit = visitData[index];
        
        if (!visit) {
            console.log('AIBF Debug: No visit data at index', index, 'Data:', visitData);
            alert('Visit details not available. Please refresh the page.');
            return;
        }
        
        var serviceLower = (visit.service || 'unknown').toLowerCase();
        var serviceClass = (['openai', 'anthropic', 'meta', 'google'].indexOf(serviceLower) !== -1) ? serviceLower : 'default';
        
        // Confidence is stored as text: 'high', 'medium', 'low', 'none'
        var confidenceText = (visit.confidence || 'none').toLowerCase();
        var confidenceMap = {
            'high': { percent: 95, label: 'High (95%)', class: 'high' },
            'medium': { percent: 70, label: 'Medium (70%)', class: 'medium' },
            'low': { percent: 40, label: 'Low (40%)', class: 'low' },
            'none': { percent: 10, label: 'Minimal (10%)', class: 'low' }
        };
        var confidenceInfo = confidenceMap[confidenceText] || confidenceMap['none'];
        var confidence = confidenceInfo.percent;
        var confidenceClass = confidenceInfo.class;
        var confidenceLabel = confidenceInfo.label;
        
        var methodLabels = {
            'user_agent': 'User-Agent String Match',
            'ip_range': 'Known IP Range',
            'pattern': 'Request Pattern Analysis'
        };
        
        var content = '<div class="qmsbf-visit-detail-grid">' +
            '<div class="qmsbf-visit-detail-item">' +
            '<label>AI Service</label>' +
            '<div class="value"><span class="qmsbf-visit-service-badge ' + serviceClass + '">' + 
            escapeHtml(visit.service || 'Unknown').toUpperCase() + '</span></div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>Page Visited</label>' +
            '<div class="value highlight">/' + escapeHtml(visit.page || '') + '</div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>Timestamp</label>' +
            '<div class="value">' + escapeHtml(visit.time || 'Unknown') + '</div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>Detection Method</label>' +
            '<div class="value">' + escapeHtml(methodLabels[visit.method] || visit.method || 'Unknown') + '</div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>Detection Confidence</label>' +
            '<div class="value">' + confidenceLabel +
            '<div class="qmsbf-confidence-bar"><div class="qmsbf-confidence-fill ' + confidenceClass + '" style="width:' + confidence + '%"></div></div>' +
            '</div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>IP Address (Masked)</label>' +
            '<div class="value mono">' + escapeHtml(visit.ip_masked || 'Not recorded') + '</div>' +
            '</div>' +
            
            '<div class="qmsbf-visit-detail-item">' +
            '<label>User Agent</label>' +
            '<div class="value mono" style="font-size:11px;line-height:1.6;word-break:break-all;">' + 
            escapeHtml(visit.user_agent || 'Not recorded') + '</div>' +
            '</div>' +
            '</div>';
        
        document.getElementById('qmsbf-visit-detail-content').innerHTML = content;
        document.getElementById('qmsbf-visit-detail-modal').classList.add('active');
    }
    
    function closeVisitModal() {
        var modal = document.getElementById('qmsbf-visit-detail-modal');
        if (modal) {
            modal.classList.remove('active');
        }
    }
    
    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function initBotBlocking($) {
        // Toggle handlers for bot blocking
        $(document).off('change.qmsbfBot').on('change.qmsbfBot', '.qmsbf-bot-toggle input', function() {
            var $toggle = $(this);
            var botId = $toggle.data('bot-id');
            var isBlocked = $toggle.is(':checked');
            var $card = $toggle.closest('.qmsbf-bot-card');
            var $status = $card.find('.qmsbf-bot-status');
            
            // Update UI immediately
            if (isBlocked) {
                $card.addClass('blocked');
                $status.removeClass('allowed').addClass('blocked').text('🚫 Blocked');
            } else {
                $card.removeClass('blocked');
                $status.removeClass('blocked').addClass('allowed').text('✓ Allowed');
            }
            
            // Save to server
            saveBotBlockingSettings();
        });
        
        // Custom agents save button
        $(document).off('click.qmsbfCustomAgents').on('click.qmsbfCustomAgents', '#save-custom-agents-btn', function() {
            var agents = $('#custom_blocked_agents').val();
            
            $.post(ajaxurl, {
                action: 'qmsbf_save_custom_agents',
                nonce: (typeof qmsbf_ajax !== 'undefined' && qmsbf_ajax.nonce) ? qmsbf_ajax.nonce : '',
                agents: agents
            }, function(response) {
                if (response.success) {
                    showToast('Custom blocks saved!', 'success');
                } else {
                    showToast('Error saving. Please try again.', 'error');
                }
            }).fail(function() {
                showToast('Network error. Please check your connection.', 'error');
            });
        });
    }
    
    function saveBotBlockingSettings() {
        var blockedBots = [];
        jQuery('.qmsbf-bot-toggle input:checked').each(function() {
            blockedBots.push(jQuery(this).data('bot-id'));
        });
        
        jQuery.post(ajaxurl, {
            action: 'qmsbf_save_blocked_bots',
            nonce: (typeof qmsbf_ajax !== 'undefined' && qmsbf_ajax.nonce) ? qmsbf_ajax.nonce : '',
            blocked_bots: blockedBots
        }, function(response) {
            if (response.success) {
                showToast('Bot blocking settings saved!', 'success');
            } else {
                showToast('Error saving settings: ' + (response.data || 'Unknown error'), 'error');
            }
        }).fail(function() {
            showToast('Network error. Please check your connection.', 'error');
        });
    }
    
    function showToast(message, type) {
        // Remove existing toasts
        jQuery('.qmsbf-toast').remove();
        
        var bgColor = type === 'success' ? '#22c55e' : '#ef4444';
        var $toast = jQuery('<div class="qmsbf-toast">' + message + '</div>');
        $toast.css({
            position: 'fixed',
            bottom: '30px',
            right: '30px',
            padding: '14px 24px',
            background: bgColor,
            color: 'white',
            borderRadius: '10px',
            fontWeight: '500',
            fontSize: '14px',
            boxShadow: '0 10px 40px rgba(0,0,0,0.2)',
            zIndex: 100001,
            opacity: 0
        });
        
        jQuery('body').append($toast);
        $toast.animate({opacity: 1}, 200);
        
        setTimeout(function() {
            $toast.animate({opacity: 0}, 200, function() {
                $toast.remove();
            });
        }, 3000);
    }
    
    // Start initialization
    initWhenReady();
    
})();
