<?php
/**
 * Plugin Name:       Qeltra Gateway for SecureSoftPay
 * Description:       Integrates SecureSoftPay with WooCommerce, featuring server-to-server payment verification.
 * Version:           5.0.0
 * Author:            securesoftpay
 * Author URI:        https://profiles.wordpress.org/securesoftpay/
 * License:           GPL-2.0-or-later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       qeltra-gateway-for-securesoftpay
 * Domain Path:       /languages
 * Requires at least: 5.8
 * Tested up to:      6.9
 * Requires PHP:      7.4
 * WC requires at least: 7.0
 * WC tested up to: 10.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Declare HPOS (High-Performance Order Storage) compatibility.
 */
add_action( 'before_woocommerce_init', function () {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

/**
 * Initialize the gateway class after plugins are loaded.
 */
add_action( 'plugins_loaded', 'init_qeltra_ssp_gateway_class' );

/**
 * Initializes the gateway class.
 */
function init_qeltra_ssp_gateway_class() {
	if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
		return;
	}

	class WC_Gateway_Qeltra_SSP extends WC_Payment_Gateway {

		/**
		 * Constructor for the gateway.
		 */
		public function __construct() {
			$this->id                 = 'securesoftpay';
			$this->icon               = apply_filters( 'qeltra_ssp_icon', '' );
			$this->has_fields         = false;
			$this->method_title       = __( 'SecureSoftPay (via Qeltra)', 'qeltra-gateway-for-securesoftpay' );
			$this->method_description = __( 'Allows payments via SecureSoftPay using the Qeltra gateway with server-to-server verification.', 'qeltra-gateway-for-securesoftpay' );

			// Load the settings.
			$this->init_form_fields();
			$this->init_settings();

			// Define user-facing properties.
			$this->title          = $this->get_option( 'title' );
			$this->description    = $this->get_option( 'description' );
			$this->enabled        = $this->get_option( 'enabled' );

			// Define API settings from admin options.
			$this->api_key      = $this->get_option( 'api_key' );
			$this->base_url     = $this->get_option( 'base_url' );
			$this->checkout_url = $this->get_option( 'checkout_url' );

			// Define order status settings.
			$this->success_status = $this->get_option( 'success_status', 'wc-processing' );
			$this->cancel_status  = $this->get_option( 'cancel_status', 'wc-cancelled' );

			// Hooks.
			add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
			add_action( 'woocommerce_api_' . $this->id, array( $this, 'handle_gateway_response' ) );
		}

		/**
		 * Gateway settings in the admin area.
		 */
		public function init_form_fields() {
			$this->form_fields = array(
				'enabled'         => array(
					'title'   => __( 'Enable/Disable', 'qeltra-gateway-for-securesoftpay' ),
					'type'    => 'checkbox',
					'label'   => __( 'Enable Qeltra Gateway for SecureSoftPay', 'qeltra-gateway-for-securesoftpay' ),
					'default' => 'yes',
				),
				'title'           => array(
					'title'       => __( 'Title', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'text',
					'description' => __( 'This controls the title for the payment method shown to the customer during checkout.', 'qeltra-gateway-for-securesoftpay' ),
					'default'     => __( 'SecureSoftPay', 'qeltra-gateway-for-securesoftpay' ),
					'desc_tip'    => true,
				),
				'description'     => array(
					'title'       => __( 'Description', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'textarea',
					'description' => __( 'This controls the description which the user sees during checkout.', 'qeltra-gateway-for-securesoftpay' ),
					'default'     => __( 'Pay securely via SecureSoftPay.', 'qeltra-gateway-for-securesoftpay' ),
				),
				'api_settings'    => array(
					'title'       => __( 'API Settings', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'title',
					'description' => __( 'Enter your API credentials and URLs provided by your self-hosted payment server.', 'qeltra-gateway-for-securesoftpay' ),
				),
				'api_key'         => array(
					'title' => __( 'API Key', 'qeltra-gateway-for-securesoftpay' ),
					'type'  => 'text',
				),
				'base_url'        => array(
					'title'       => __( 'Base URL', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'text',
					'description' => __( 'The client base URL required by the API for verification. E.g., https://your-domain.com/api', 'qeltra-gateway-for-securesoftpay' ),
					'desc_tip'    => true,
				),
				'checkout_url'    => array(
					'title'       => __( 'Checkout URL', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'text',
					'description' => __( 'The payment initiation endpoint URL. E.g., https://your-domain.com/api/v1/checkout', 'qeltra-gateway-for-securesoftpay' ),
					'desc_tip'    => true,
				),
				'status_settings' => array(
					'title'       => __( 'Order Status Settings', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'title',
					'description' => __( 'Choose how order statuses are handled after payment attempts.', 'qeltra-gateway-for-securesoftpay' ),
				),
				'success_status'  => array(
					'title'       => __( 'Status for Successful Payments', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'select',
					'options'     => wc_get_order_statuses(),
					'default'     => 'wc-processing',
					'description' => __( 'Select the order status for a successfully completed payment.', 'qeltra-gateway-for-securesoftpay' ),
					'desc_tip'    => true,
				),
				'cancel_status'   => array(
					'title'       => __( 'Status for Failed/Cancelled Payments', 'qeltra-gateway-for-securesoftpay' ),
					'type'        => 'select',
					'options'     => wc_get_order_statuses(),
					'default'     => 'wc-cancelled',
					'description' => __( 'Select the order status when a customer cancels the payment or it fails.', 'qeltra-gateway-for-securesoftpay' ),
					'desc_tip'    => true,
				),
			);
		}

		/**
		 * Process the payment and return the result.
		 *
		 * @param int $order_id Order ID.
		 * @return array
		 */
		public function process_payment( $order_id ) {
			if ( empty( $this->checkout_url ) || empty( $this->base_url ) || empty( $this->api_key ) ) {
				wc_add_notice( __( 'Payment gateway is not configured correctly. Please contact the site administrator.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
				return array( 'result' => 'failure' );
			}

			$order        = wc_get_order( $order_id );
			$callback_url = WC()->api_request_url( $this->id );

			$body = array(
				'amount'           => (float) $order->get_total(),
				'client_base_url'  => $this->base_url,
				'customer_name'    => $order->get_formatted_billing_full_name(),
				'customer_email'   => $order->get_billing_email(),
				'success_url'      => add_query_arg( array( 'status' => 'success', 'order_key' => $order->get_order_key() ), $callback_url ),
				'cancel_url'       => add_query_arg( array( 'status' => 'cancel', 'order_key' => $order->get_order_key() ), $callback_url ),
				'meta_data'        => wp_json_encode( array( 'order_id' => $order_id ) ),
			);

			$args = array(
				'body'    => wp_json_encode( $body ),
				'headers' => array(
					'Authorization' => 'Bearer ' . $this->api_key,
					'Content-Type'  => 'application/json',
				),
				'timeout' => 45,
			);

			$response = wp_remote_post( $this->checkout_url, $args );

			if ( is_wp_error( $response ) ) {
				wc_add_notice( __( 'Payment error: Could not connect to the payment gateway.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
				return array( 'result' => 'failure' );
			}

			$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( isset( $response_body['status'] ) && 'success' === $response_body['status'] && ! empty( $response_body['payment_url'] ) && ! empty( $response_body['unique_id'] ) ) {
				$order->update_meta_data( '_qeltra_ssp_unique_id', $response_body['unique_id'] );
				$order->save();
				
				return array(
					'result'   => 'success',
					'redirect' => $response_body['payment_url'],
				);
			} else {
				$error_message = isset( $response_body['message'] ) ? esc_html( $response_body['message'] ) : __( 'An unknown error occurred while initiating the payment.', 'qeltra-gateway-for-securesoftpay' );
				wc_add_notice( 'Payment error: ' . $error_message, 'error' );
				return array( 'result' => 'failure' );
			}
		}

		/**
		 * Handle the response from the payment gateway.
		 */
		public function handle_gateway_response() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$status    = isset( $_GET['status'] ) ? wc_clean( wp_unslash( $_GET['status'] ) ) : '';
			$order_key = isset( $_GET['order_key'] ) ? wc_clean( wp_unslash( $_GET['order_key'] ) ) : '';
			// phpcs:enable

			if ( empty( $order_key ) ) {
				wp_safe_redirect( home_url() );
				exit;
			}

			$order_id = wc_get_order_id_by_order_key( $order_key );
			$order    = wc_get_order( $order_id );

			if ( ! $order ) {
				wc_add_notice( __( 'Invalid order.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
				wp_safe_redirect( wc_get_checkout_url() );
				exit;
			}
			
			if ( 'success' === $status ) {
				
				$unique_id = $order->get_meta( '_qeltra_ssp_unique_id' );

				if ( empty( $unique_id ) ) {
					wc_add_notice( __( 'Payment session is invalid or has expired. Please try again.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
					wp_safe_redirect( $order->get_checkout_payment_url() );
					exit;
				}

				$verification_url = rtrim( $this->checkout_url, '/' ) . '/' . $unique_id;
				
				$args = array(
					'headers' => array( 'Authorization' => 'Bearer ' . $this->api_key ),
					'timeout' => 30,
				);
				
				$response = wp_remote_get( $verification_url, $args );

				if ( is_wp_error( $response ) ) {
					wc_add_notice( __( 'Payment verification failed. Could not connect to the payment server.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
					wp_safe_redirect( $order->get_checkout_payment_url() );
					exit;
				}

				$response_body = json_decode( wp_remote_retrieve_body( $response ), true );
				
				if ( isset( $response_body['success'] ) && $response_body['success'] && isset( $response_body['status'] ) && $response_body['status'] === 'completed' ) {
					
					$gateway_transaction_id = $response_body['transaction_id'] ?? '';
					
					if ( ! $order->is_paid() ) {
						$order->payment_complete( $gateway_transaction_id );
						$success_status_slug = str_replace( 'wc-', '', $this->success_status );

						/* translators: %s: The transaction ID from the payment gateway. */
						$order->update_status( $success_status_slug, sprintf( __( 'Payment successful via SecureSoftPay. Transaction ID: %s', 'qeltra-gateway-for-securesoftpay' ), $gateway_transaction_id ) );
						
						WC()->cart->empty_cart();
					}

					wp_safe_redirect( $this->get_return_url( $order ) );
					exit;

				} else {
					$error_message = $response_body['message'] ?? __( 'Payment could not be verified. Please contact support or try again.', 'qeltra-gateway-for-securesoftpay' );
					wc_add_notice( $error_message, 'error' );
					wp_safe_redirect( $order->get_checkout_payment_url() );
					exit;
				}
				
			} else { // status === 'cancel' or anything else
				
				// ★★★ চূড়ান্ত সমাধান এখানে ★★★
				$cancel_status_slug = str_replace( 'wc-', '', $this->cancel_status );
				$order->update_status( $cancel_status_slug, __( 'The payment was not completed by the customer.', 'qeltra-gateway-for-securesoftpay' ) );

				if ( 'wc-cancelled' === $this->cancel_status ) {
					// Add a notice, empty the cart, and redirect to the cart page
					wc_add_notice( __( 'Your payment has been cancelled and the cart has been emptied.', 'qeltra-gateway-for-securesoftpay' ), 'notice' );
					
					if ( function_exists( 'WC' ) && WC()->cart ) {
						WC()->cart->empty_cart();
					}

					wp_safe_redirect( wc_get_cart_url() );
					exit;
				} else {
					// For any other status (e.g., 'failed', 'pending'), redirect back to payment page
					wc_add_notice( __( 'Your payment was not completed. Please try again.', 'qeltra-gateway-for-securesoftpay' ), 'error' );
					wp_safe_redirect( $order->get_checkout_payment_url() );
					exit;
				}
			}
		}
	}
}

/**
 * Add the Qeltra SSP Gateway to WooCommerce.
 *
 * @param array $gateways Existing gateways.
 * @return array
 */
function add_qeltra_ssp_gateway_class( $gateways ) {
	$gateways[] = 'WC_Gateway_Qeltra_SSP';
	return $gateways;
}
add_filter( 'woocommerce_payment_gateways', 'add_qeltra_ssp_gateway_class' );