<?php
/**
 * Plugin Name: Puntr
 * Plugin URI: https://puntr.me/wordpress
 * Description: A WordPress plugin for managing provider profiles and services
 * Version: 1.2.3
 * Author: Puntr
 * Author URI: https://puntr.me
 * License: GPL v2 or later
 * Text Domain: puntr
 * Domain Path: /languages
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Define plugin constants
define('PUNTR_VERSION', '1.2.3');
define('PUNTR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PUNTR_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Include required files
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-post-types.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-meta-boxes.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-roster-meta-boxes.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-taxonomies.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-settings.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-calendar.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-rest-api.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-templates.php';
require_once PUNTR_PLUGIN_PATH . 'includes/class-puntr-hooks.php';
require_once PUNTR_PLUGIN_PATH . 'includes/puntr-header.php';

/**
 * Main Puntr class
 */
class Puntr {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance of this class
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'), 20); // Later priority to ensure taxonomies are registered
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        add_action('wp_enqueue_scripts', array($this, 'frontend_scripts'));
        add_action('admin_notices', array($this, 'add_provider_header'));
        add_filter('parent_file', array($this, 'fix_admin_menu_parent'));
        
        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Get SVG icon and convert to base64
        $svg_icon = file_get_contents(PUNTR_PLUGIN_PATH . 'assets/img/puntr-marker.svg');
        $svg_icon_base64 = 'data:image/svg+xml;base64,' . base64_encode($svg_icon);
        
        // Add main menu item
        add_menu_page(
            __('Puntr', 'puntr'),
            __('Puntr', 'puntr'),
            'manage_options',
            'puntr',
            array($this, 'render_dashboard_page'),
            $svg_icon_base64,
            20
        );
        
        // Add dashboard submenu
        add_submenu_page(
            'puntr',
            __('Dashboard', 'puntr'),
            __('Dashboard', 'puntr'),
            'manage_options',
            'puntr',
            array($this, 'render_dashboard_page')
        );
        
        // Add calendar submenu
        add_submenu_page(
            'puntr',
            __('Roster', 'puntr'),
            __('Roster', 'puntr'),
            'manage_options',
            'puntr-calendar',
            array($this, 'render_calendar_page')
        );
        
        // Move custom post types under Puntr menu
        global $submenu;
        
        // Add Providers submenu
        add_submenu_page(
            'puntr',
            __('Providers', 'puntr'),
            __('Providers', 'puntr'),
            'edit_posts',
            'edit.php?post_type=puntr_provider'
        );
        
        // Add Services taxonomy submenu
        add_submenu_page(
            'puntr',
            __('Services', 'puntr'),
            __('Services', 'puntr'),
            'manage_categories',
            'edit-tags.php?post_type=puntr_provider&taxonomy=puntr_service'
        );
        
        // Add Settings submenu
        add_submenu_page(
            'puntr',
            __('Settings', 'puntr'),
            __('Settings', 'puntr'),
            'manage_options',
            'puntr-settings',
            array('Puntr_Settings', 'render_settings_page')
        );
    }
    
    /**
     * Render dashboard page
     */
    public function render_dashboard_page() {
        ?>
        <div class="wrap">
            <?php puntr_render_header(__('Dashboard', 'puntr')); ?>
            
            <div class="puntr-dashboard">
                <div class="puntr-welcome">
                    <h2><?php esc_html_e('Welcome to Puntr', 'puntr'); ?></h2>
                    <p><?php esc_html_e('The professional provider management system for WordPress. Manage your providers, rosters, and schedules all in one place.', 'puntr'); ?></p>
                </div>
                
                <div class="puntr-stats">
                    <h3><?php esc_html_e('Quick Stats', 'puntr'); ?></h3>
                    <?php
                    $provider_count = 0;
                    $roster_count = 0;
                    
                    if (post_type_exists('puntr_provider')) {
                        $provider_posts = wp_count_posts('puntr_provider');
                        $provider_count = isset($provider_posts->publish) ? $provider_posts->publish : 0;
                    }
                    
                    if (post_type_exists('puntr_roster')) {
                        $roster_posts = wp_count_posts('puntr_roster');
                        $roster_count = isset($roster_posts->publish) ? $roster_posts->publish : 0;
                    }
                    ?>
                    <ul>
                        <li><?php 
                            /* translators: %d: Number of providers */
                            printf(esc_html__('Total Providers: %d', 'puntr'), intval($provider_count)); 
                        ?></li>
                        <li><?php 
                            /* translators: %d: Number of roster entries */
                            printf(esc_html__('Total Roster Entries: %d', 'puntr'), intval($roster_count)); 
                        ?></li>
                    </ul>
                </div>
                
                <div class="puntr-quick-links">
                    <h3><?php esc_html_e('Quick Links', 'puntr'); ?></h3>
                    <ul>
                        <li><a href="<?php echo esc_url(admin_url('admin.php?page=puntr-calendar')); ?>"><?php esc_html_e('View Roster', 'puntr'); ?></a></li>
                        <li><a href="<?php echo esc_url(admin_url('post-new.php?post_type=puntr_provider')); ?>"><?php esc_html_e('Add New Provider', 'puntr'); ?></a></li>
                        <li><a href="<?php echo esc_url(admin_url('post-new.php?post_type=puntr_roster')); ?>"><?php esc_html_e('Add New Roster Entry', 'puntr'); ?></a></li>
                        <li><a href="<?php echo esc_url(admin_url('admin.php?page=puntr-settings')); ?>"><?php esc_html_e('Plugin Settings', 'puntr'); ?></a></li>
                    </ul>
                </div>
                
            </div>
        </div>
        <?php
    }
    
    /**
     * Render calendar page
     */
    public function render_calendar_page() {
        ?>
        <div class="wrap">
            <?php puntr_render_header(__('Roster', 'puntr')); ?>
            
            <div class="puntr-info-box">
                <h3><?php esc_html_e('How to use the Roster', 'puntr'); ?></h3>
                <ul>
                    <li><?php esc_html_e('Click and drag on any time slot to create a new roster entry', 'puntr'); ?></li>
                    <li><?php esc_html_e('Drag events to move them to different times', 'puntr'); ?></li>
                    <li><?php esc_html_e('Resize events by dragging their edges', 'puntr'); ?></li>
                    <li><?php esc_html_e('Click the eye icon next to a provider name to edit their profile', 'puntr'); ?></li>
                    <li><?php esc_html_e('Use the zoom controls to adjust the calendar view', 'puntr'); ?></li>
                </ul>
            </div>
            
            <?php echo do_shortcode('[puntr_roster_calendar height="700"]'); ?>
        </div>
        <?php
    }
    
    /**
     * Add header to provider pages
     */
    public function add_provider_header() {
        global $post_type, $pagenow, $taxonomy;
        
        // Check if we're on a provider post type page
        if (($post_type === 'puntr_provider' && in_array($pagenow, array('edit.php', 'post.php', 'post-new.php', "edit-tags.php", "term.php")))) {
            // Determine the appropriate title

            $title = __('Providers', 'puntr');
            if ($pagenow === 'post-new.php') {
                $title = __('Add New Provider', 'puntr');
            } elseif ($pagenow === 'post.php') {
                $title = __('Edit Provider', 'puntr');
            }
            
            if ($pagenow === 'edit-tags.php' && $taxonomy == 'puntr_service') {
                $title = __("Add new service", 'puntr');
            } elseif ($pagenow === 'term.php' && $taxonomy == 'puntr_service') {
                $title = __("Edit Service", 'puntr');
            }
            
            // Output the header
            echo '<div class="puntr-provider-header-wrapper">';
            puntr_render_header_on_posttype($title);
            echo '</div>';
        }
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Register post types immediately
        Puntr_Post_Types::register_post_types();
        
        // Initialize post types
        Puntr_Post_Types::init();
        
        // Register taxonomies immediately
        Puntr_Taxonomies::register_taxonomies();
        
        // Initialize taxonomies
        Puntr_Taxonomies::init();
        
        // Initialize meta boxes
        if (is_admin()) {
            Puntr_Meta_Boxes::init();
            Puntr_Roster_Meta_Boxes::init();
            Puntr_Settings::init();
        }
        
        // Initialize calendar
        Puntr_Calendar::init();
        
        // Initialize REST API
        Puntr_REST_API::init();
        
        // Initialize templates
        Puntr_Templates::init();
        
        // Initialize hooks
        Puntr_Hooks::init();
        
        // Text domain is automatically loaded by WordPress for plugins hosted on WordPress.org
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function admin_scripts($hook) {
        global $post_type;
        
        // Load on all Puntr admin pages
        $puntr_pages = array(
            'toplevel_page_puntr',
            'puntr_page_puntr-calendar',
            'puntr_page_puntr-settings',
            'edit.php',
            'post.php',
            'post-new.php'
        );
        
        // Check if we're on a Puntr page or editing our post types
        if (in_array($hook, $puntr_pages) || 
            in_array($post_type, array('puntr_provider', 'puntr_roster')) ||
            (strpos($hook, 'puntr') !== false)) {
            
            wp_enqueue_style('puntr-admin', PUNTR_PLUGIN_URL . 'assets/css/admin.css', array(), PUNTR_VERSION);
            wp_enqueue_script('puntr-admin', PUNTR_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), PUNTR_VERSION, true);
        }
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function frontend_scripts() {
        if (is_singular('puntr_provider') || is_post_type_archive('puntr_provider')) {
            // Minimal template styles
            wp_enqueue_style('puntr-templates', PUNTR_PLUGIN_URL . 'assets/css/templates.css', array(), PUNTR_VERSION);
            
            // Gallery styles for product gallery
            wp_enqueue_style('puntr-provider-archive', PUNTR_PLUGIN_URL . 'assets/css/provider-archive.css', array(), PUNTR_VERSION);
            
            // JavaScript for gallery
            wp_enqueue_script('puntr-frontend', PUNTR_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), PUNTR_VERSION, true);
        }
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Create custom post types
        Puntr_Post_Types::init();
        
        // Create taxonomies
        Puntr_Taxonomies::init();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Fix parent menu highlighting for taxonomy pages
     */
    public function fix_admin_menu_parent($parent_file) {
        global $current_screen, $submenu_file, $pagenow;
        
        // Check if we're on the puntr_service taxonomy page
        if ($current_screen && $current_screen->taxonomy === 'puntr_service' && $current_screen->post_type === 'puntr_provider') {
            $parent_file = 'puntr';
            // Only set submenu_file if we're on the taxonomy listing page, not when editing a term
            if ($pagenow === 'edit-tags.php') {
                $submenu_file = 'edit-tags.php?post_type=puntr_provider&taxonomy=puntr_service';
            }
        }
        
        // Check if we're on provider post type pages
        elseif ($current_screen && $current_screen->post_type === 'puntr_provider' && !isset($_GET['taxonomy'])) {
            $parent_file = 'puntr';
            $submenu_file = 'edit.php?post_type=puntr_provider';
        }
        
        return $parent_file;
    }
}

// Initialize the plugin
Puntr::get_instance();