<?php
/**
 * Templates for Puntr plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_Templates {
    
    /**
     * Initialize templates
     */
    public static function init() {
        // Template loading
        add_filter('template_include', array(__CLASS__, 'template_loader'));
        
        // Modify provider archive
        add_action('pre_get_posts', array(__CLASS__, 'modify_provider_archive_query'));
        
        // Add gallery to single provider
        add_filter('the_content', array(__CLASS__, 'add_gallery_to_provider_page'), 10);
        
        // Remove featured image from single provider pages
        add_action('wp', array(__CLASS__, 'remove_featured_image_from_provider'));
        
        // Hook gallery and roster to template actions when using custom template
        add_action('puntr_before_provider_content', array(__CLASS__, 'output_provider_gallery'));
        add_action('puntr_after_provider_content', array(__CLASS__, 'output_provider_roster'));
    }
    
    /**
     * Modify provider archive query
     */
    public static function modify_provider_archive_query($query) {
        if (!is_admin() && $query->is_main_query() && is_post_type_archive('puntr_provider')) {
            // Only show active providers
            $query->set('meta_query', array(
                array(
                    'key' => '_puntr_is_active',
                    'value' => '1',
                    'compare' => '='
                )
            ));
            
            // Order by name
            $query->set('orderby', 'title');
            $query->set('order', 'ASC');
        }
    }
    
    /**
     * Add provider details to content on archive pages
     */
    public static function add_provider_details_to_content($content) {
        if (is_post_type_archive('puntr_provider') && in_the_loop() && is_main_query()) {
            $provider_id = get_the_ID();
            
            // Filter: Allow developers to skip provider details
            if (apply_filters('puntr_skip_archive_provider_details', false, $provider_id)) {
                return $content;
            }
            
            // Get provider meta data
            $headline = get_post_meta($provider_id, '_puntr_headline', true);
            $age = get_post_meta($provider_id, '_puntr_age', true);
            $ethnicity = get_post_meta($provider_id, '_puntr_ethnicity', true);
            $body_type = get_post_meta($provider_id, '_puntr_body_type', true);
            $height = get_post_meta($provider_id, '_puntr_height_cm', true);
            $dress_size = get_post_meta($provider_id, '_puntr_dress_size', true);
            $cup_size = get_post_meta($provider_id, '_puntr_cup_size', true);
            
            // Get pricing
            $price_30 = get_post_meta($provider_id, '_puntr_price_30_mins', true);
            $price_60 = get_post_meta($provider_id, '_puntr_price_60_mins', true);
            
            // Get services
            $services = wp_get_post_terms($provider_id, 'puntr_service', array('fields' => 'names'));
            if (is_wp_error($services)) {
                $services = array();
            }
            
            // Filter: Allow modification of provider data before display
            $provider_data = apply_filters('puntr_archive_provider_data', array(
                'headline' => $headline,
                'age' => $age,
                'ethnicity' => $ethnicity,
                'body_type' => $body_type,
                'height' => $height,
                'dress_size' => $dress_size,
                'cup_size' => $cup_size,
                'price_30' => $price_30,
                'price_60' => $price_60,
                'services' => $services
            ), $provider_id);
            
            // Extract filtered data
            extract($provider_data);
            
            // Build provider info
            $provider_info = '<div class="puntr-provider-info">';
            
            if ($headline) {
                $headline_html = '<p class="puntr-headline">' . esc_html($headline) . '</p>';
                $provider_info .= apply_filters('puntr_archive_headline_html', $headline_html, $headline, $provider_id);
            }
            
            // Attributes row
            $attributes = array();
            /* translators: %s: Provider age */
            if ($age) $attributes[] = sprintf(__('Age: %s', 'puntr'), $age);
            /* translators: %s: Provider ethnicity */
            if ($ethnicity) $attributes[] = sprintf(__('Ethnicity: %s', 'puntr'), $ethnicity);
            /* translators: %s: Provider body type */
            if ($body_type) $attributes[] = sprintf(__('Body: %s', 'puntr'), ucfirst($body_type));
            /* translators: %s: Height in centimeters */
            if ($height) $attributes[] = sprintf(__('Height: %scm', 'puntr'), $height);
            /* translators: %s: Dress size */
            if ($dress_size) $attributes[] = sprintf(__('Dress: %s', 'puntr'), $dress_size);
            /* translators: %s: Cup size */
            if ($cup_size) $attributes[] = sprintf(__('Cup: %s', 'puntr'), $cup_size);
            
            // Filter: Allow modification of attributes array
            $attributes = apply_filters('puntr_archive_provider_attributes', $attributes, $provider_id);
            
            if (!empty($attributes)) {
                $attributes_html = '<div class="puntr-attributes">' . implode(' | ', $attributes) . '</div>';
                $provider_info .= apply_filters('puntr_archive_attributes_html', $attributes_html, $attributes, $provider_id);
            }
            
            // Services
            if (!empty($services) && is_array($services)) {
                $services_html = '<div class="puntr-services"><strong>' . __('Services:', 'puntr') . '</strong> ' . implode(', ', $services) . '</div>';
                $provider_info .= apply_filters('puntr_archive_services_html', $services_html, $services, $provider_id);
            }
            
            // Pricing
            $pricing = array();
            /* translators: %s: Price for 30 minutes */
            if ($price_30) $pricing[] = sprintf(__('30min: $%s', 'puntr'), $price_30);
            /* translators: %s: Price for 60 minutes */
            if ($price_60) $pricing[] = sprintf(__('60min: $%s', 'puntr'), $price_60);
            
            // Filter: Allow modification of pricing display
            $pricing = apply_filters('puntr_archive_pricing_array', $pricing, $provider_data, $provider_id);
            
            if (!empty($pricing)) {
                $pricing_html = '<div class="puntr-pricing"><strong>' . __('Rates:', 'puntr') . '</strong> ' . implode(' | ', $pricing) . '</div>';
                $provider_info .= apply_filters('puntr_archive_pricing_html', $pricing_html, $pricing, $provider_id);
            }
            
            $provider_info .= '</div>';
            
            // Filter: Allow modification of complete provider info HTML
            $provider_info = apply_filters('puntr_provider_archive_info', $provider_info, $provider_id);
            
            // Filter: Allow custom content positioning
            $content_position = apply_filters('puntr_archive_content_position', 'before', $provider_id);
            
            // Add to content
            if ($content_position === 'after') {
                $content = $content . $provider_info;
            } else {
                $content = $provider_info . $content;
            }
        }
        
        return $content;
    }
    
    /**
     * Add classes to provider posts
     */
    public static function add_provider_post_classes($classes, $class, $post_id) {
        if (get_post_type($post_id) === 'puntr_provider') {
            $classes[] = 'puntr-provider-item';
            
            // Add body type class if available
            $body_type = get_post_meta($post_id, '_puntr_body_type', true);
            if ($body_type) {
                $classes[] = 'puntr-body-' . sanitize_html_class($body_type);
            }
        }
        
        return $classes;
    }
    
    /**
     * Add gallery to single provider page
     */
    public static function add_gallery_to_provider_page($content) {
        if (is_singular('puntr_provider') && in_the_loop() && is_main_query()) {
            $provider_id = get_the_ID();
            
            // Filter: Allow developers to skip gallery display
            if (apply_filters('puntr_skip_provider_gallery', false, $provider_id)) {
                return $content;
            }
            
            $gallery_images = get_post_meta($provider_id, '_puntr_gallery_images', true);
            $featured_image_id = get_post_thumbnail_id($provider_id);
            
            // Combine featured image with gallery images
            $all_images = array();
            if ($featured_image_id) {
                $all_images[] = $featured_image_id;
            }
            if (!empty($gallery_images) && is_array($gallery_images)) {
                $all_images = array_merge($all_images, $gallery_images);
            }
            
            // Remove duplicates
            $all_images = array_unique($all_images);
            
            // Filter: Allow modification of gallery images
            $all_images = apply_filters('puntr_gallery_images', $all_images, $provider_id);
            
            // Remove featured image from content since we're showing it in gallery
            remove_filter('post_thumbnail_html', '__return_empty_string', 10);
            add_filter('post_thumbnail_html', '__return_empty_string', 10, 4);
            
            if (!empty($all_images)) {
                // Filter: Gallery wrapper classes
                $gallery_classes = apply_filters('puntr_gallery_wrapper_classes', 'puntr-product-gallery', $provider_id);
                $gallery_html = '<div class="' . esc_attr($gallery_classes) . '">';
                
                // Main image display
                $gallery_html .= '<div class="puntr-gallery-main">';
                $first_image = reset($all_images);
                $main_image = wp_get_attachment_image_src($first_image, 'large');
                if ($main_image) {
                    $gallery_html .= sprintf(
                        '<img src="%s" alt="" class="puntr-main-image" data-zoom-image="%s" />',
                        esc_url($main_image[0]),
                        esc_url($main_image[0])
                    );
                }
                $gallery_html .= '</div>';
                
                // Thumbnail navigation
                if (count($all_images) > 1) {
                    $gallery_html .= '<div class="puntr-gallery-thumbs">';
                    foreach ($all_images as $index => $image_id) {
                        $thumb = wp_get_attachment_image_src($image_id, 'thumbnail');
                        $large = wp_get_attachment_image_src($image_id, 'large');
                        if ($thumb && $large) {
                            $active_class = $index === 0 ? ' active' : '';
                            $gallery_html .= sprintf(
                                '<div class="puntr-thumb-item%s" data-image="%s" data-zoom-image="%s">%s</div>',
                                $active_class,
                                esc_url($large[0]),
                                esc_url($large[0]),
                                wp_get_attachment_image($image_id, 'thumbnail', false, array('class' => 'puntr-thumb-image'))
                            );
                        }
                    }
                    $gallery_html .= '</div>';
                }
                
                $gallery_html .= '</div>';
                
                // Filter: Allow modification of complete gallery HTML
                $gallery_html = apply_filters('puntr_gallery_html', $gallery_html, $all_images, $provider_id);
                
                // Action: Before gallery output
                do_action('puntr_before_gallery', $provider_id);
                
                // Filter: Gallery position relative to content
                $gallery_position = apply_filters('puntr_gallery_position', 'before', $provider_id);
                
                // Add gallery to content
                if ($gallery_position === 'after') {
                    $content = $content . $gallery_html;
                } else {
                    $content = $gallery_html . $content;
                }
                
                // Action: After gallery output
                do_action('puntr_after_gallery', $provider_id);
            }
        }
        
        return $content;
    }
    
    /**
     * Get provider details box
     */
    private static function get_provider_details_box($provider_id) {
        // Filter: Allow skipping details box
        if (apply_filters('puntr_skip_provider_details_box', false, $provider_id)) {
            return '';
        }
        
        // Get provider meta data
        $meta_data = array(
            'headline' => get_post_meta($provider_id, '_puntr_headline', true),
            'age' => get_post_meta($provider_id, '_puntr_age', true),
            'ethnicity' => get_post_meta($provider_id, '_puntr_ethnicity', true),
            'body_type' => get_post_meta($provider_id, '_puntr_body_type', true),
            'height' => get_post_meta($provider_id, '_puntr_height_cm', true),
            'dress_size' => get_post_meta($provider_id, '_puntr_dress_size', true),
            'cup_size' => get_post_meta($provider_id, '_puntr_cup_size', true),
            'hair' => get_post_meta($provider_id, '_puntr_hair_description', true),
            'eye_color' => get_post_meta($provider_id, '_puntr_eye_color', true)
        );
        
        // Filter: Allow modification of meta data
        $meta_data = apply_filters('puntr_provider_details_meta_data', $meta_data, $provider_id);
        
        // Extract for easier use
        extract($meta_data);
        
        // Get pricing
        $prices = array(
            '30' => get_post_meta($provider_id, '_puntr_price_30_mins', true),
            '45' => get_post_meta($provider_id, '_puntr_price_45_mins', true),
            '60' => get_post_meta($provider_id, '_puntr_price_60_mins', true),
            '90' => get_post_meta($provider_id, '_puntr_price_90_mins', true),
            '120' => get_post_meta($provider_id, '_puntr_price_120_mins', true),
            '180' => get_post_meta($provider_id, '_puntr_price_180_mins', true)
        );
        
        // Get services
        $services = wp_get_post_terms($provider_id, 'puntr_service', array('fields' => 'names'));
        if (is_wp_error($services)) {
            $services = array();
        }
        
        $html = '<div class="puntr-provider-details-box">';
        
        if ($headline) {
            $html .= '<p class="puntr-provider-headline">' . esc_html($headline) . '</p>';
        }
        
        // Quick stats
        $html .= '<div class="puntr-provider-stats">';
        if ($age) $html .= '<div class="stat-item"><span class="stat-label">' . __('Age', 'puntr') . ':</span> ' . esc_html($age) . '</div>';
        if ($ethnicity) $html .= '<div class="stat-item"><span class="stat-label">' . __('Ethnicity', 'puntr') . ':</span> ' . esc_html($ethnicity) . '</div>';
        if ($body_type) $html .= '<div class="stat-item"><span class="stat-label">' . __('Body Type', 'puntr') . ':</span> ' . esc_html(ucfirst($body_type)) . '</div>';
        if ($height) $html .= '<div class="stat-item"><span class="stat-label">' . __('Height', 'puntr') . ':</span> ' . esc_html($height) . 'cm</div>';
        if ($dress_size) $html .= '<div class="stat-item"><span class="stat-label">' . __('Dress', 'puntr') . ':</span> ' . esc_html($dress_size) . '</div>';
        if ($cup_size) $html .= '<div class="stat-item"><span class="stat-label">' . __('Cup', 'puntr') . ':</span> ' . esc_html($cup_size) . '</div>';
        if ($hair) $html .= '<div class="stat-item"><span class="stat-label">' . __('Hair', 'puntr') . ':</span> ' . esc_html($hair) . '</div>';
        if ($eye_color) $html .= '<div class="stat-item"><span class="stat-label">' . __('Eyes', 'puntr') . ':</span> ' . esc_html(ucfirst($eye_color)) . '</div>';
        $html .= '</div>';
        
        // Pricing table
        $html .= '<div class="puntr-pricing-table">';
        $html .= '<h4>' . __('Rates', 'puntr') . '</h4>';
        $html .= '<table>';
        foreach ($prices as $duration => $price) {
            if ($price) {
                $html .= '<tr>';
                $html .= '<td>' . $duration . ' ' . __('mins', 'puntr') . '</td>';
                $html .= '<td>$' . esc_html($price) . '</td>';
                $html .= '</tr>';
            }
        }
        $html .= '</table>';
        $html .= '</div>';
        
        // Services
        if (!empty($services) && is_array($services)) {
            $html .= '<div class="puntr-services-list">';
            $html .= '<h4>' . __('Services', 'puntr') . '</h4>';
            $html .= '<div class="service-tags">';
            foreach ($services as $service) {
                if (is_string($service)) {
                    $html .= '<span class="service-tag">' . esc_html($service) . '</span>';
                }
            }
            $html .= '</div>';
            $html .= '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Remove featured image from single provider pages
     */
    public static function remove_featured_image_from_provider() {
        if (is_singular('puntr_provider')) {
            // Remove post thumbnail from the content
            add_filter('post_thumbnail_html', function($html, $post_id, $post_thumbnail_id, $size) {
                if (get_post_type($post_id) === 'puntr_provider') {
                    return '';
                }
                return $html;
            }, 10, 4);
        }
    }
    
    /**
     * Load custom templates
     */
    public static function template_loader($template) {
        if (is_post_type_archive('puntr_provider')) {
            $custom_template = puntr_locate_template('archive-puntr_provider.php');
            if ($custom_template) {
                return $custom_template;
            }
        }
        
        if (is_singular('puntr_provider')) {
            $custom_template = puntr_locate_template('single-puntr_provider.php');
            if ($custom_template) {
                // Disable auto-add roster when using custom template
                add_filter('puntr_auto_add_roster', '__return_false');
                return $custom_template;
            }
        }
        
        return $template;
    }
    
    /**
     * Output provider gallery for template action
     */
    public static function output_provider_gallery($provider_id) {
        // Get the content with gallery
        $content = '';
        $content = apply_filters('the_content', $content);
    }
    
    /**
     * Output provider roster for template action
     */
    public static function output_provider_roster($provider_id) {
        echo do_shortcode('[puntr_provider_roster]');
    }
}

/**
 * Template loading functions
 */

/**
 * Get template part
 */
function puntr_get_template_part($slug, $name = null) {
    $templates = array();
    
    if (isset($name)) {
        $templates[] = "{$slug}-{$name}.php";
    }
    
    $templates[] = "{$slug}.php";
    
    // Look in theme first
    foreach ($templates as $template) {
        $theme_file = get_stylesheet_directory() . '/puntr/' . $template;
        if (file_exists($theme_file)) {
            load_template($theme_file, false);
            return;
        }
    }
    
    // Then in plugin
    foreach ($templates as $template) {
        $plugin_file = PUNTR_PLUGIN_PATH . 'templates/' . $template;
        if (file_exists($plugin_file)) {
            load_template($plugin_file, false);
            return;
        }
    }
}

/**
 * Locate a template
 */
function puntr_locate_template($template_name) {
    // Look in theme first
    $theme_file = get_stylesheet_directory() . '/puntr/' . $template_name;
    if (file_exists($theme_file)) {
        return $theme_file;
    }
    
    // Look in parent theme
    $parent_theme_file = get_template_directory() . '/puntr/' . $template_name;
    if (file_exists($parent_theme_file)) {
        return $parent_theme_file;
    }
    
    // Then in plugin
    $plugin_file = PUNTR_PLUGIN_PATH . 'templates/' . $template_name;
    if (file_exists($plugin_file)) {
        return $plugin_file;
    }
    
    return false;
}