<?php
/**
 * Plugin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_Settings {
    
    /**
     * Initialize settings
     */
    public static function init() {
        add_action('admin_menu', array(__CLASS__, 'add_admin_menu'));
        add_action('admin_init', array(__CLASS__, 'register_settings'));
    }
    
    /**
     * Add admin menu
     */
    public static function add_admin_menu() {
        // Settings page is now added in the main plugin class under the Puntr menu
    }
    
    /**
     * Register settings
     */
    public static function register_settings() {
        // Register setting
        register_setting('puntr_settings', 'puntr_options', array(
            'sanitize_callback' => array(__CLASS__, 'sanitize_options')
        ));
        
        // General Section
        add_settings_section(
            'puntr_general_section',
            __('General Settings', 'puntr'),
            array(__CLASS__, 'render_general_section'),
            'puntr_settings'
        );
        
        // Currency setting
        add_settings_field(
            'currency',
            __('Currency', 'puntr'),
            array(__CLASS__, 'render_currency_field'),
            'puntr_settings',
            'puntr_general_section'
        );
        
        // Provider archive settings
        add_settings_field(
            'providers_per_page',
            __('Providers per page', 'puntr'),
            array(__CLASS__, 'render_providers_per_page_field'),
            'puntr_settings',
            'puntr_general_section'
        );
        
        // Display Section
        add_settings_section(
            'puntr_display_section',
            __('Display Settings', 'puntr'),
            array(__CLASS__, 'render_display_section'),
            'puntr_settings'
        );
        
        // Show pricing
        add_settings_field(
            'show_pricing',
            __('Show Pricing', 'puntr'),
            array(__CLASS__, 'render_show_pricing_field'),
            'puntr_settings',
            'puntr_display_section'
        );
        
        // Show physical details
        add_settings_field(
            'show_physical_details',
            __('Show Physical Details', 'puntr'),
            array(__CLASS__, 'render_show_physical_details_field'),
            'puntr_settings',
            'puntr_display_section'
        );
    }
    
    /**
     * Render settings page
     */
    public static function render_settings_page() {
        ?>
        <div class="wrap">
            <?php 
            // Display any settings errors/notices before the header
            settings_errors();
            
            puntr_render_header(__('Settings', 'puntr')); 
            ?>
            
            <div class="puntr-settings-content">
                <form action="options.php" method="post">
                    <?php
                    // This will output hidden fields but NOT duplicate notices since we already called settings_errors()
                    settings_fields('puntr_settings');
                    ?>
                    
                    <div class="puntr-settings-sections">
                        <?php do_settings_sections('puntr_settings'); ?>
                        
                        <h2><?php esc_html_e('How to Use Puntr', 'puntr'); ?></h2>
                        <div class="puntr-help-section">
                            <h3><?php esc_html_e('Quick Start Guide', 'puntr'); ?></h3>
                            <ol>
                                <li><strong><?php esc_html_e('Add Providers', 'puntr'); ?></strong> - <?php esc_html_e('Go to Puntr → Providers and click "Add New Provider"', 'puntr'); ?></li>
                                <li><strong><?php esc_html_e('Create Roster Entries', 'puntr'); ?></strong> - <?php esc_html_e('Visit Puntr → Roster and click-drag on the calendar', 'puntr'); ?></li>
                                <li><strong><?php esc_html_e('Display on Website', 'puntr'); ?></strong> - <?php esc_html_e('Use shortcodes to show rosters on your pages', 'puntr'); ?></li>
                            </ol>
                            
                            <h3><?php esc_html_e('Shortcode Examples', 'puntr'); ?></h3>
                            <p><strong><?php esc_html_e('Basic daily roster:', 'puntr'); ?></strong></p>
                            <code>[puntr_daily_roster]</code>
                            
                            <p><strong><?php esc_html_e('With photos and details:', 'puntr'); ?></strong></p>
                            <code>[puntr_daily_roster show_photo="true" photo_size="60" show_age="true" columns="photo,name,age,location,start,end"]</code>
                            
                            <p><strong><?php esc_html_e('Full calendar view:', 'puntr'); ?></strong></p>
                            <code>[puntr_roster_calendar height="700"]</code>
                            
                            <h3><?php esc_html_e('Tips', 'puntr'); ?></h3>
                            <ul>
                                <li><?php esc_html_e('Overnight shifts automatically appear on both days', 'puntr'); ?></li>
                                <li><?php esc_html_e('Click the eye icon next to provider names to edit', 'puntr'); ?></li>
                                <li><?php esc_html_e('Use the zoom controls to see more/less hours at once', 'puntr'); ?></li>
                                <li><?php esc_html_e('Provider names in rosters link to their full profiles', 'puntr'); ?></li>
                            </ul>
                            
                            <p><a href="https://puntr.me/wordpress" target="_blank" class="button"><?php esc_html_e('View Full Documentation', 'puntr'); ?></a></p>
                        </div>
                    </div>
                    
                    <div class="puntr-settings-footer">
                        <?php submit_button(__('Save Settings', 'puntr'), 'primary large'); ?>
                        <p class="puntr-version">
                            <?php 
                            /* translators: %s: Plugin version number */
                            printf(esc_html__('Puntr Version %s', 'puntr'), esc_html(PUNTR_VERSION)); 
                            ?> | 
                            <a href="https://puntr.me" target="_blank"><?php esc_html_e('Visit puntr.me', 'puntr'); ?></a>
                        </p>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render general section
     */
    public static function render_general_section() {
        echo '<p>' . esc_html__('Configure general plugin settings.', 'puntr') . '</p>';
    }
    
    /**
     * Render display section
     */
    public static function render_display_section() {
        echo '<p>' . esc_html__('Configure how provider information is displayed.', 'puntr') . '</p>';
    }
    
    
    /**
     * Render currency field
     */
    public static function render_currency_field() {
        $options = get_option('puntr_options');
        $currency = isset($options['currency']) ? $options['currency'] : 'USD';
        ?>
        <select name="puntr_options[currency]">
            <option value="USD" <?php selected($currency, 'USD'); ?>>USD ($)</option>
            <option value="EUR" <?php selected($currency, 'EUR'); ?>>EUR (€)</option>
            <option value="GBP" <?php selected($currency, 'GBP'); ?>>GBP (£)</option>
            <option value="AUD" <?php selected($currency, 'AUD'); ?>>AUD ($)</option>
            <option value="CAD" <?php selected($currency, 'CAD'); ?>>CAD ($)</option>
        </select>
        <?php
    }
    
    /**
     * Render providers per page field
     */
    public static function render_providers_per_page_field() {
        $options = get_option('puntr_options');
        $per_page = isset($options['providers_per_page']) ? $options['providers_per_page'] : 12;
        ?>
        <input type="number" name="puntr_options[providers_per_page]" value="<?php echo esc_attr($per_page); ?>" min="1" max="100" />
        <p class="description"><?php esc_html_e('Number of providers to show per page on archive pages.', 'puntr'); ?></p>
        <?php
    }
    
    /**
     * Render show pricing field
     */
    public static function render_show_pricing_field() {
        $options = get_option('puntr_options');
        $show_pricing = isset($options['show_pricing']) ? $options['show_pricing'] : '1';
        ?>
        <label>
            <input type="checkbox" name="puntr_options[show_pricing]" value="1" <?php checked($show_pricing, '1'); ?> />
            <?php esc_html_e('Display pricing information on provider profiles', 'puntr'); ?>
        </label>
        <?php
    }
    
    /**
     * Render show physical details field
     */
    public static function render_show_physical_details_field() {
        $options = get_option('puntr_options');
        $show_physical = isset($options['show_physical_details']) ? $options['show_physical_details'] : '1';
        ?>
        <label>
            <input type="checkbox" name="puntr_options[show_physical_details]" value="1" <?php checked($show_physical, '1'); ?> />
            <?php esc_html_e('Display physical details on provider profiles', 'puntr'); ?>
        </label>
        <?php
    }
    
    
    /**
     * Sanitize options
     */
    public static function sanitize_options($input) {
        $sanitized = array();
        
        // Sanitize providers per page
        if (isset($input['providers_per_page'])) {
            $sanitized['providers_per_page'] = absint($input['providers_per_page']);
            if ($sanitized['providers_per_page'] < 1) {
                $sanitized['providers_per_page'] = 12;
            }
        }
        
        // Sanitize checkboxes
        $checkboxes = array(
            'show_pricing_on_archive',
            'show_attributes_on_archive',
            'enable_public_api',
            'require_api_authentication'
        );
        
        foreach ($checkboxes as $checkbox) {
            $sanitized[$checkbox] = isset($input[$checkbox]) ? '1' : '0';
        }
        
        return $sanitized;
    }
}