<?php
/**
 * Meta Boxes for Roster
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_Roster_Meta_Boxes {
    
    /**
     * Initialize meta boxes
     */
    public static function init() {
        add_action('add_meta_boxes', array(__CLASS__, 'add_meta_boxes'));
        add_action('save_post_puntr_roster', array(__CLASS__, 'save_meta_boxes'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue scripts for roster meta boxes
     */
    public static function enqueue_scripts($hook) {
        global $post_type;
        
        // Only enqueue on roster edit screens
        if ($post_type === 'puntr_roster' && in_array($hook, array('post.php', 'post-new.php'))) {
            wp_enqueue_script('puntr-admin-metabox', PUNTR_PLUGIN_URL . 'assets/js/admin-metabox.js', array('jquery'), PUNTR_VERSION, true);
            
            // Use the same localized script object
            wp_localize_script('puntr-admin-metabox', 'puntrMetabox', array(
                'selectGalleryImages' => __('Select or Upload Gallery Images', 'puntr'),
                'addToGallery' => __('Add to Gallery', 'puntr')
            ));
        }
    }
    
    /**
     * Add meta boxes
     */
    public static function add_meta_boxes() {
        // Roster Details
        add_meta_box(
            'puntr_roster_details',
            __('Roster Details', 'puntr'),
            array(__CLASS__, 'render_roster_details_meta_box'),
            'puntr_roster',
            'normal',
            'high'
        );
    }
    
    /**
     * Render roster details meta box
     */
    public static function render_roster_details_meta_box($post) {
        wp_nonce_field('puntr_save_roster_meta_boxes', 'puntr_roster_meta_nonce');
        
        // Get saved values
        $provider_id = get_post_meta($post->ID, '_puntr_provider_id', true);
        $start_date = get_post_meta($post->ID, '_puntr_start_date', true);
        $end_date = get_post_meta($post->ID, '_puntr_end_date', true);
        $all_day = get_post_meta($post->ID, '_puntr_all_day', true);
        $status = get_post_meta($post->ID, '_puntr_roster_status', true);
        $notes = get_post_meta($post->ID, '_puntr_roster_notes', true);
        
        // Get all providers for dropdown
        $providers = get_posts(array(
            'post_type' => 'puntr_provider',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC'
        ));
        ?>
        <table class="form-table">
            <tr>
                <th><label for="puntr_provider_id"><?php esc_html_e('Provider', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_provider_id" name="puntr_provider_id" required>
                        <option value=""><?php esc_html_e('-- Select Provider --', 'puntr'); ?></option>
                        <?php foreach ($providers as $provider) : ?>
                            <option value="<?php echo esc_attr($provider->ID); ?>" <?php selected($provider_id, $provider->ID); ?>>
                                <?php echo esc_html($provider->post_title); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_start_date"><?php esc_html_e('Start Date/Time', 'puntr'); ?></label></th>
                <td>
                    <input type="datetime-local" id="puntr_start_date" name="puntr_start_date" value="<?php echo esc_attr($start_date); ?>" required />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_end_date"><?php esc_html_e('End Date/Time', 'puntr'); ?></label></th>
                <td>
                    <input type="datetime-local" id="puntr_end_date" name="puntr_end_date" value="<?php echo esc_attr($end_date); ?>" required />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_all_day"><?php esc_html_e('All Day Event', 'puntr'); ?></label></th>
                <td>
                    <label>
                        <input type="checkbox" id="puntr_all_day" name="puntr_all_day" value="1" <?php checked($all_day, '1'); ?> />
                        <?php esc_html_e('This is an all-day event', 'puntr'); ?>
                    </label>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_roster_status"><?php esc_html_e('Status', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_roster_status" name="puntr_roster_status">
                        <option value="scheduled" <?php selected($status, 'scheduled'); ?>><?php esc_html_e('Scheduled', 'puntr'); ?></option>
                        <option value="confirmed" <?php selected($status, 'confirmed'); ?>><?php esc_html_e('Confirmed', 'puntr'); ?></option>
                        <option value="cancelled" <?php selected($status, 'cancelled'); ?>><?php esc_html_e('Cancelled', 'puntr'); ?></option>
                        <option value="completed" <?php selected($status, 'completed'); ?>><?php esc_html_e('Completed', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_roster_notes"><?php esc_html_e('Notes', 'puntr'); ?></label></th>
                <td>
                    <textarea id="puntr_roster_notes" name="puntr_roster_notes" rows="4" class="large-text"><?php echo esc_textarea($notes); ?></textarea>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Save meta boxes
     */
    public static function save_meta_boxes($post_id) {
        // Check nonce
        if (!isset($_POST['puntr_roster_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['puntr_roster_meta_nonce'])), 'puntr_save_roster_meta_boxes')) {
            return;
        }
        
        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save fields
        $fields = array(
            'puntr_provider_id' => '_puntr_provider_id',
            'puntr_start_date' => '_puntr_start_date',
            'puntr_end_date' => '_puntr_end_date',
            'puntr_roster_status' => '_puntr_roster_status',
            'puntr_roster_notes' => '_puntr_roster_notes',
        );
        
        foreach ($fields as $field_name => $meta_key) {
            if (isset($_POST[$field_name])) {
                update_post_meta($post_id, $meta_key, sanitize_text_field(wp_unslash($_POST[$field_name])));
            }
        }
        
        // Handle checkbox
        update_post_meta($post_id, '_puntr_all_day', isset($_POST['puntr_all_day']) ? '1' : '0');
    }
}