<?php
/**
 * REST API functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_REST_API {
    
    /**
     * Initialize REST API
     */
    public static function init() {
        add_action('rest_api_init', array(__CLASS__, 'register_routes'));
    }
    
    /**
     * Register REST API routes
     */
    public static function register_routes() {
        $namespace = 'puntr/v1';
        
        // Providers endpoint
        register_rest_route($namespace, '/providers', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array(__CLASS__, 'get_providers'),
                'permission_callback' => '__return_true', // Public access
                'args' => array(
                    'per_page' => array(
                        'default' => 10,
                        'sanitize_callback' => 'absint',
                    ),
                    'page' => array(
                        'default' => 1,
                        'sanitize_callback' => 'absint',
                    ),
                    'service' => array(
                        'sanitize_callback' => 'sanitize_text_field',
                    ),
                    'active_only' => array(
                        'default' => true,
                        'sanitize_callback' => 'rest_sanitize_boolean',
                    ),
                ),
            ),
        ));
        
        // Single provider endpoint
        register_rest_route($namespace, '/providers/(?P<id>\d+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array(__CLASS__, 'get_provider'),
                'permission_callback' => '__return_true',
                'args' => array(
                    'id' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                ),
            ),
        ));
        
        // Roster events endpoint
        register_rest_route($namespace, '/roster', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array(__CLASS__, 'get_roster_events'),
                'permission_callback' => '__return_true',
                'args' => array(
                    'start_date' => array(
                        'required' => true,
                        'sanitize_callback' => 'sanitize_text_field',
                        'validate_callback' => function($param) {
                            return (bool) strtotime($param);
                        }
                    ),
                    'end_date' => array(
                        'required' => true,
                        'sanitize_callback' => 'sanitize_text_field',
                        'validate_callback' => function($param) {
                            return (bool) strtotime($param);
                        }
                    ),
                    'provider_id' => array(
                        'sanitize_callback' => 'absint',
                    ),
                ),
            ),
        ));
        
        // Daily roster endpoint
        register_rest_route($namespace, '/daily-roster', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array(__CLASS__, 'get_daily_roster'),
                'permission_callback' => '__return_true',
                'args' => array(
                    'date' => array(
                        'default' => wp_date('Y-m-d'),
                        'sanitize_callback' => 'sanitize_text_field',
                        'validate_callback' => function($param) {
                            return (bool) strtotime($param);
                        }
                    ),
                ),
            ),
        ));
    }
    
    /**
     * Get providers
     */
    public static function get_providers($request) {
        $args = array(
            'post_type' => 'puntr_provider',
            'post_status' => 'publish',
            'posts_per_page' => $request['per_page'],
            'paged' => $request['page'],
            'orderby' => 'title',
            'order' => 'ASC',
        );
        
        // Filter by active status
        if ($request['active_only']) {
            $args['meta_query'] = array(
                array(
                    'key' => '_puntr_is_active',
                    'value' => '1',
                    'compare' => '='
                )
            );
        }
        
        // Filter by service
        if (!empty($request['service'])) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'puntr_service',
                    'field' => 'slug',
                    'terms' => $request['service'],
                ),
            );
        }
        
        $query = new WP_Query($args);
        $providers = array();
        
        foreach ($query->posts as $provider) {
            $providers[] = self::format_provider($provider);
        }
        
        return new WP_REST_Response(array(
            'providers' => $providers,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'page' => $request['page'],
        ), 200);
    }
    
    /**
     * Get single provider
     */
    public static function get_provider($request) {
        $provider = get_post($request['id']);
        
        if (!$provider || $provider->post_type !== 'puntr_provider' || $provider->post_status !== 'publish') {
            return new WP_Error('not_found', 'Provider not found', array('status' => 404));
        }
        
        return new WP_REST_Response(self::format_provider($provider, true), 200);
    }
    
    /**
     * Get roster events
     */
    public static function get_roster_events($request) {
        $args = array(
            'post_type' => 'puntr_roster',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_puntr_start_date',
                    'value' => array($request['start_date'], $request['end_date']),
                    'compare' => 'BETWEEN',
                    'type' => 'DATETIME'
                )
            )
        );
        
        // Filter by provider
        if (!empty($request['provider_id'])) {
            $args['meta_query'][] = array(
                'key' => '_puntr_provider_id',
                'value' => $request['provider_id'],
                'compare' => '='
            );
        }
        
        
        $events = get_posts($args);
        $formatted_events = array();
        
        foreach ($events as $event) {
            $formatted_events[] = self::format_roster_event($event);
        }
        
        return new WP_REST_Response($formatted_events, 200);
    }
    
    /**
     * Get daily roster
     */
    public static function get_daily_roster($request) {
        $date = $request['date'];
        $date_start = $date . ' 00:00:00';
        $date_end = $date . ' 23:59:59';
        
        // Query roster entries including overnight shifts
        $args = array(
            'post_type' => 'puntr_roster',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => array(
                'relation' => 'OR',
                // Events that start on this date
                array(
                    'key' => '_puntr_start_date',
                    'value' => array($date_start, $date_end),
                    'compare' => 'BETWEEN',
                    'type' => 'DATETIME'
                ),
                // Events that started yesterday but end today
                array(
                    'relation' => 'AND',
                    array(
                        'key' => '_puntr_start_date',
                        'value' => $date_start,
                        'compare' => '<',
                        'type' => 'DATETIME'
                    ),
                    array(
                        'key' => '_puntr_end_date',
                        'value' => $date_start,
                        'compare' => '>=',
                        'type' => 'DATETIME'
                    )
                )
            ),
            'orderby' => 'meta_value',
            'meta_key' => '_puntr_start_date',
            'order' => 'ASC'
        );
        
        
        $events = get_posts($args);
        $roster = array();
        
        foreach ($events as $event) {
            $roster[] = self::format_daily_roster_entry($event, $date);
        }
        
        return new WP_REST_Response(array(
            'date' => $date,
            'roster' => $roster,
            'total' => count($roster)
        ), 200);
    }
    
    /**
     * Format provider for API response
     */
    private static function format_provider($provider, $detailed = false) {
        $data = array(
            'id' => $provider->ID,
            'name' => $provider->post_title,
            'slug' => $provider->post_name,
            'url' => get_permalink($provider->ID),
            'excerpt' => get_the_excerpt($provider->ID),
            'featured_image' => get_the_post_thumbnail_url($provider->ID, 'full'),
            'is_active' => get_post_meta($provider->ID, '_puntr_is_active', true) === '1',
        );
        
        if ($detailed) {
            // Add all meta fields
            $data['age'] = get_post_meta($provider->ID, '_puntr_age', true);
            $data['gender'] = get_post_meta($provider->ID, '_puntr_gender', true);
            $data['ethnicity'] = get_post_meta($provider->ID, '_puntr_ethnicity', true);
            $data['height'] = get_post_meta($provider->ID, '_puntr_height', true);
            $data['weight'] = get_post_meta($provider->ID, '_puntr_weight', true);
            $data['hair_color'] = get_post_meta($provider->ID, '_puntr_hair_color', true);
            $data['eye_color'] = get_post_meta($provider->ID, '_puntr_eye_color', true);
            $data['bust_size'] = get_post_meta($provider->ID, '_puntr_bust_size', true);
            $data['waist_size'] = get_post_meta($provider->ID, '_puntr_waist_size', true);
            $data['hip_size'] = get_post_meta($provider->ID, '_puntr_hip_size', true);
            $data['dress_size'] = get_post_meta($provider->ID, '_puntr_dress_size', true);
            $data['shoe_size'] = get_post_meta($provider->ID, '_puntr_shoe_size', true);
            $data['incall_rate'] = get_post_meta($provider->ID, '_puntr_incall_rate', true);
            $data['outcall_rate'] = get_post_meta($provider->ID, '_puntr_outcall_rate', true);
            $data['city'] = get_post_meta($provider->ID, '_puntr_city', true);
            $data['languages'] = get_post_meta($provider->ID, '_puntr_languages', true);
            $data['content'] = apply_filters('the_content', $provider->post_content);
            
            // Get taxonomies
            $data['services'] = wp_get_object_terms($provider->ID, 'puntr_service', array('fields' => 'all'));
        }
        
        return $data;
    }
    
    /**
     * Format roster event for API response
     */
    private static function format_roster_event($event) {
        $provider_id = get_post_meta($event->ID, '_puntr_provider_id', true);
        $provider = null;
        $provider_data = array(
            'id' => $provider_id,
            'name' => 'Unknown Provider',
            'url' => ''
        );
        
        if ($provider_id) {
            $provider = get_post($provider_id);
            if ($provider && $provider->post_type === 'puntr_provider') {
                $provider_data = array(
                    'id' => $provider_id,
                    'name' => $provider->post_title,
                    'url' => get_permalink($provider_id)
                );
            }
        }
        
        return array(
            'id' => $event->ID,
            'provider' => $provider_data,
            'start' => get_post_meta($event->ID, '_puntr_start_date', true),
            'end' => get_post_meta($event->ID, '_puntr_end_date', true),
            'status' => get_post_meta($event->ID, '_puntr_roster_status', true),
            'notes' => get_post_meta($event->ID, '_puntr_roster_notes', true),
        );
    }
    
    /**
     * Format daily roster entry for API response
     */
    private static function format_daily_roster_entry($event, $display_date) {
        $provider_id = get_post_meta($event->ID, '_puntr_provider_id', true);
        $start_date = get_post_meta($event->ID, '_puntr_start_date', true);
        $end_date = get_post_meta($event->ID, '_puntr_end_date', true);
        
        // Initialize provider data with defaults
        $provider_data = array(
            'id' => $provider_id,
            'name' => 'Unknown Provider',
            'url' => '',
            'ethnicity' => '',
            'age' => '',
            'location' => '',
        );
        
        // Get provider if exists
        if ($provider_id) {
            $provider = get_post($provider_id);
            if ($provider && $provider->post_type === 'puntr_provider') {
                $provider_data = array(
                    'id' => $provider_id,
                    'name' => $provider->post_title,
                    'url' => get_permalink($provider_id),
                    'ethnicity' => get_post_meta($provider_id, '_puntr_ethnicity', true),
                    'age' => get_post_meta($provider_id, '_puntr_age', true),
                    'location' => get_post_meta($provider_id, '_puntr_city', true),
                );
            }
        }
        
        // Format times
        $start_time = wp_date('g:i A', strtotime($start_date));
        $end_time = wp_date('g:i A', strtotime($end_date));
        
        // Check if overnight shift
        $start_day = wp_date('Y-m-d', strtotime($start_date));
        $end_day = wp_date('Y-m-d', strtotime($end_date));
        $is_overnight = ($start_day != $end_day);
        
        return array(
            'id' => $event->ID,
            'provider' => $provider_data,
            'start_time' => $start_time,
            'end_time' => $end_time,
            'start_datetime' => $start_date,
            'end_datetime' => $end_date,
            'is_overnight' => $is_overnight,
            'status' => get_post_meta($event->ID, '_puntr_roster_status', true),
        );
    }
}