<?php
/**
 * Meta Boxes for Provider Custom Post Type
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_Meta_Boxes {
    
    /**
     * Initialize meta boxes
     */
    public static function init() {
        add_action('add_meta_boxes', array(__CLASS__, 'add_meta_boxes'));
        add_action('save_post_puntr_provider', array(__CLASS__, 'save_meta_boxes'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue scripts for meta boxes
     */
    public static function enqueue_scripts($hook) {
        global $post_type;
        
        // Only enqueue on provider edit screens
        if ($post_type === 'puntr_provider' && in_array($hook, array('post.php', 'post-new.php'))) {
            wp_enqueue_media();
            wp_enqueue_script('puntr-admin-metabox', PUNTR_PLUGIN_URL . 'assets/js/admin-metabox.js', array('jquery', 'jquery-ui-sortable'), PUNTR_VERSION, true);
            
            // Localize script for translations
            wp_localize_script('puntr-admin-metabox', 'puntrMetabox', array(
                'selectGalleryImages' => __('Select or Upload Gallery Images', 'puntr'),
                'addToGallery' => __('Add to Gallery', 'puntr')
            ));
        }
    }
    
    /**
     * Add meta boxes
     */
    public static function add_meta_boxes() {
        // Basic Information
        add_meta_box(
            'puntr_provider_basic_info',
            __('Basic Information', 'puntr'),
            array(__CLASS__, 'render_basic_info_meta_box'),
            'puntr_provider',
            'normal',
            'high'
        );
        
        // Physical Details
        add_meta_box(
            'puntr_provider_physical_details',
            __('Physical Details', 'puntr'),
            array(__CLASS__, 'render_physical_details_meta_box'),
            'puntr_provider',
            'normal',
            'default'
        );
        
        // Pricing
        add_meta_box(
            'puntr_provider_pricing',
            __('Pricing', 'puntr'),
            array(__CLASS__, 'render_pricing_meta_box'),
            'puntr_provider',
            'normal',
            'default'
        );
        
        // Additional Info
        add_meta_box(
            'puntr_provider_additional_info',
            __('Additional Information', 'puntr'),
            array(__CLASS__, 'render_additional_info_meta_box'),
            'puntr_provider',
            'normal',
            'default'
        );
        
        // Photo Gallery
        add_meta_box(
            'puntr_provider_gallery',
            __('Photo Gallery', 'puntr'),
            array(__CLASS__, 'render_gallery_meta_box'),
            'puntr_provider',
            'normal',
            'default'
        );
    }
    
    /**
     * Render basic info meta box
     */
    public static function render_basic_info_meta_box($post) {
        wp_nonce_field('puntr_save_meta_boxes', 'puntr_meta_nonce');
        
        // Get saved values
        $headline = get_post_meta($post->ID, '_puntr_headline', true);
        $gender = get_post_meta($post->ID, '_puntr_gender', true);
        $sexuality = get_post_meta($post->ID, '_puntr_sexuality', true);
        $age = get_post_meta($post->ID, '_puntr_age', true);
        $ethnicity = get_post_meta($post->ID, '_puntr_ethnicity', true);
        $language = get_post_meta($post->ID, '_puntr_language', true);
        ?>
        <table class="form-table">
            <tr>
                <th><label for="puntr_headline"><?php esc_html_e('Headline', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_headline" name="puntr_headline" value="<?php echo esc_attr($headline); ?>" class="regular-text" />
                    <p class="description"><?php esc_html_e('A short headline for the provider profile', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_gender"><?php esc_html_e('Gender', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_gender" name="puntr_gender">
                        <option value=""><?php esc_html_e('-- Select --', 'puntr'); ?></option>
                        <option value="female" <?php selected($gender, 'female'); ?>><?php esc_html_e('Female', 'puntr'); ?></option>
                        <option value="male" <?php selected($gender, 'male'); ?>><?php esc_html_e('Male', 'puntr'); ?></option>
                        <option value="other" <?php selected($gender, 'other'); ?>><?php esc_html_e('Other', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_sexuality"><?php esc_html_e('Sexuality', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_sexuality" name="puntr_sexuality">
                        <option value=""><?php esc_html_e('-- Select --', 'puntr'); ?></option>
                        <option value="heterosexual" <?php selected($sexuality, 'heterosexual'); ?>><?php esc_html_e('Heterosexual', 'puntr'); ?></option>
                        <option value="homosexual" <?php selected($sexuality, 'homosexual'); ?>><?php esc_html_e('Homosexual', 'puntr'); ?></option>
                        <option value="bisexual" <?php selected($sexuality, 'bisexual'); ?>><?php esc_html_e('Bisexual', 'puntr'); ?></option>
                        <option value="pansexual" <?php selected($sexuality, 'pansexual'); ?>><?php esc_html_e('Pansexual', 'puntr'); ?></option>
                        <option value="asexual" <?php selected($sexuality, 'asexual'); ?>><?php esc_html_e('Asexual', 'puntr'); ?></option>
                        <option value="questioning" <?php selected($sexuality, 'questioning'); ?>><?php esc_html_e('Questioning', 'puntr'); ?></option>
                        <option value="other" <?php selected($sexuality, 'other'); ?>><?php esc_html_e('Other', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_age"><?php esc_html_e('Age', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_age" name="puntr_age" value="<?php echo esc_attr($age); ?>" min="18" max="99" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_ethnicity"><?php esc_html_e('Ethnicity', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_ethnicity" name="puntr_ethnicity" value="<?php echo esc_attr($ethnicity); ?>" class="regular-text" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_language"><?php esc_html_e('Language', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_language" name="puntr_language" value="<?php echo esc_attr($language); ?>" class="regular-text" />
                    <p class="description"><?php esc_html_e('Languages spoken (e.g., English, Spanish)', 'puntr'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render physical details meta box
     */
    public static function render_physical_details_meta_box($post) {
        // Get saved values
        $body_type = get_post_meta($post->ID, '_puntr_body_type', true);
        $cup_size = get_post_meta($post->ID, '_puntr_cup_size', true);
        $height_cm = get_post_meta($post->ID, '_puntr_height_cm', true);
        $weight_kg = get_post_meta($post->ID, '_puntr_weight_kg', true);
        $dress_size = get_post_meta($post->ID, '_puntr_dress_size', true);
        $eye_color = get_post_meta($post->ID, '_puntr_eye_color', true);
        $hair_description = get_post_meta($post->ID, '_puntr_hair_description', true);
        $skin_tone = get_post_meta($post->ID, '_puntr_skin_tone', true);
        ?>
        <table class="form-table">
            <tr>
                <th><label for="puntr_body_type"><?php esc_html_e('Body Type', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_body_type" name="puntr_body_type">
                        <option value=""><?php esc_html_e('-- Select --', 'puntr'); ?></option>
                        <option value="slim" <?php selected($body_type, 'slim'); ?>><?php esc_html_e('Slim', 'puntr'); ?></option>
                        <option value="petite" <?php selected($body_type, 'petite'); ?>><?php esc_html_e('Petite', 'puntr'); ?></option>
                        <option value="athletic" <?php selected($body_type, 'athletic'); ?>><?php esc_html_e('Athletic', 'puntr'); ?></option>
                        <option value="average" <?php selected($body_type, 'average'); ?>><?php esc_html_e('Average', 'puntr'); ?></option>
                        <option value="curvy" <?php selected($body_type, 'curvy'); ?>><?php esc_html_e('Curvy', 'puntr'); ?></option>
                        <option value="curvaceous" <?php selected($body_type, 'curvaceous'); ?>><?php esc_html_e('Curvaceous', 'puntr'); ?></option>
                        <option value="bbw" <?php selected($body_type, 'bbw'); ?>><?php esc_html_e('BBW', 'puntr'); ?></option>
                        <option value="muscular" <?php selected($body_type, 'muscular'); ?>><?php esc_html_e('Muscular', 'puntr'); ?></option>
                        <option value="other" <?php selected($body_type, 'other'); ?>><?php esc_html_e('Other', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_cup_size"><?php esc_html_e('Cup Size', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_cup_size" name="puntr_cup_size" value="<?php echo esc_attr($cup_size); ?>" class="small-text" />
                    <p class="description"><?php esc_html_e('e.g., C', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_height_cm"><?php esc_html_e('Height (cm)', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_height_cm" name="puntr_height_cm" value="<?php echo esc_attr($height_cm); ?>" min="120" max="220" />
                    <p class="description"><?php esc_html_e('e.g., 157', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_weight_kg"><?php esc_html_e('Weight (kg)', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_weight_kg" name="puntr_weight_kg" value="<?php echo esc_attr($weight_kg); ?>" min="30" max="200" />
                    <p class="description"><?php esc_html_e('e.g., 46', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_dress_size"><?php esc_html_e('Dress Size', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_dress_size" name="puntr_dress_size" value="<?php echo esc_attr($dress_size); ?>" class="small-text" />
                    <p class="description"><?php esc_html_e('e.g., 6', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_eye_color"><?php esc_html_e('Eye Color', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_eye_color" name="puntr_eye_color">
                        <option value=""><?php esc_html_e('-- Select --', 'puntr'); ?></option>
                        <option value="brown" <?php selected($eye_color, 'brown'); ?>><?php esc_html_e('Brown', 'puntr'); ?></option>
                        <option value="blue" <?php selected($eye_color, 'blue'); ?>><?php esc_html_e('Blue', 'puntr'); ?></option>
                        <option value="green" <?php selected($eye_color, 'green'); ?>><?php esc_html_e('Green', 'puntr'); ?></option>
                        <option value="hazel" <?php selected($eye_color, 'hazel'); ?>><?php esc_html_e('Hazel', 'puntr'); ?></option>
                        <option value="black" <?php selected($eye_color, 'black'); ?>><?php esc_html_e('Black', 'puntr'); ?></option>
                        <option value="grey" <?php selected($eye_color, 'grey'); ?>><?php esc_html_e('Grey', 'puntr'); ?></option>
                        <option value="other" <?php selected($eye_color, 'other'); ?>><?php esc_html_e('Other', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_hair_description"><?php esc_html_e('Hair Description', 'puntr'); ?></label></th>
                <td>
                    <input type="text" id="puntr_hair_description" name="puntr_hair_description" value="<?php echo esc_attr($hair_description); ?>" class="regular-text" />
                    <p class="description"><?php esc_html_e('e.g., Long Black', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_skin_tone"><?php esc_html_e('Skin Tone', 'puntr'); ?></label></th>
                <td>
                    <select id="puntr_skin_tone" name="puntr_skin_tone">
                        <option value=""><?php esc_html_e('-- Select --', 'puntr'); ?></option>
                        <option value="fair" <?php selected($skin_tone, 'fair'); ?>><?php esc_html_e('Fair', 'puntr'); ?></option>
                        <option value="light" <?php selected($skin_tone, 'light'); ?>><?php esc_html_e('Light', 'puntr'); ?></option>
                        <option value="medium" <?php selected($skin_tone, 'medium'); ?>><?php esc_html_e('Medium', 'puntr'); ?></option>
                        <option value="olive" <?php selected($skin_tone, 'olive'); ?>><?php esc_html_e('Olive', 'puntr'); ?></option>
                        <option value="brown" <?php selected($skin_tone, 'brown'); ?>><?php esc_html_e('Brown', 'puntr'); ?></option>
                        <option value="dark" <?php selected($skin_tone, 'dark'); ?>><?php esc_html_e('Dark', 'puntr'); ?></option>
                        <option value="white" <?php selected($skin_tone, 'white'); ?>><?php esc_html_e('White', 'puntr'); ?></option>
                        <option value="other" <?php selected($skin_tone, 'other'); ?>><?php esc_html_e('Other', 'puntr'); ?></option>
                    </select>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render pricing meta box
     */
    public static function render_pricing_meta_box($post) {
        // Get saved values
        $price_30 = get_post_meta($post->ID, '_puntr_price_30_mins', true);
        $price_45 = get_post_meta($post->ID, '_puntr_price_45_mins', true);
        $price_60 = get_post_meta($post->ID, '_puntr_price_60_mins', true);
        $price_90 = get_post_meta($post->ID, '_puntr_price_90_mins', true);
        $price_120 = get_post_meta($post->ID, '_puntr_price_120_mins', true);
        $price_180 = get_post_meta($post->ID, '_puntr_price_180_mins', true);
        $min_booking = get_post_meta($post->ID, '_puntr_min_booking_mins', true);
        ?>
        <table class="form-table">
            <tr>
                <th><label for="puntr_price_30_mins"><?php esc_html_e('30 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_30_mins" name="puntr_price_30_mins" value="<?php echo esc_attr($price_30); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_price_45_mins"><?php esc_html_e('45 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_45_mins" name="puntr_price_45_mins" value="<?php echo esc_attr($price_45); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_price_60_mins"><?php esc_html_e('60 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_60_mins" name="puntr_price_60_mins" value="<?php echo esc_attr($price_60); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_price_90_mins"><?php esc_html_e('90 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_90_mins" name="puntr_price_90_mins" value="<?php echo esc_attr($price_90); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_price_120_mins"><?php esc_html_e('120 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_120_mins" name="puntr_price_120_mins" value="<?php echo esc_attr($price_120); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_price_180_mins"><?php esc_html_e('180 Minutes', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_price_180_mins" name="puntr_price_180_mins" value="<?php echo esc_attr($price_180); ?>" min="0" step="1" />
                </td>
            </tr>
            <tr>
                <th><label for="puntr_min_booking_mins"><?php esc_html_e('Minimum Booking (minutes)', 'puntr'); ?></label></th>
                <td>
                    <input type="number" id="puntr_min_booking_mins" name="puntr_min_booking_mins" value="<?php echo esc_attr($min_booking); ?>" min="0" step="15" />
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render additional info meta box
     */
    public static function render_additional_info_meta_box($post) {
        // Get saved values
        $external_url = get_post_meta($post->ID, '_puntr_external_url', true);
        $photos_real = get_post_meta($post->ID, '_puntr_photos_real', true);
        $is_active = get_post_meta($post->ID, '_puntr_is_active', true);
        
        // Default to active if not set
        if ($is_active === '') {
            $is_active = '1';
        }
        ?>
        <table class="form-table">
            <tr>
                <th><label for="puntr_external_url"><?php esc_html_e('External URL', 'puntr'); ?></label></th>
                <td>
                    <input type="url" id="puntr_external_url" name="puntr_external_url" value="<?php echo esc_attr($external_url); ?>" class="large-text" />
                    <p class="description"><?php esc_html_e('External website or profile URL', 'puntr'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_photos_real"><?php esc_html_e('Photos Real?', 'puntr'); ?></label></th>
                <td>
                    <label>
                        <input type="checkbox" id="puntr_photos_real" name="puntr_photos_real" value="1" <?php checked($photos_real, '1'); ?> />
                        <?php esc_html_e('Yes, the photos are real', 'puntr'); ?>
                    </label>
                </td>
            </tr>
            <tr>
                <th><label for="puntr_is_active"><?php esc_html_e('Active', 'puntr'); ?></label></th>
                <td>
                    <label>
                        <input type="checkbox" id="puntr_is_active" name="puntr_is_active" value="1" <?php checked($is_active, '1'); ?> />
                        <?php esc_html_e('Provider is currently active', 'puntr'); ?>
                    </label>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Render gallery meta box
     */
    public static function render_gallery_meta_box($post) {
        $gallery_images = get_post_meta($post->ID, '_puntr_gallery_images', true);
        if (!is_array($gallery_images)) {
            $gallery_images = array();
        }
        ?>
        <div id="puntr-gallery-container">
            <ul id="puntr-gallery-images">
                <?php foreach ($gallery_images as $image_id) : 
                    $image_url = wp_get_attachment_image_src($image_id, 'thumbnail');
                    if ($image_url) : ?>
                    <li class="gallery-image" data-image-id="<?php echo esc_attr($image_id); ?>">
                        <img src="<?php echo esc_url($image_url[0]); ?>" />
                        <input type="hidden" name="puntr_gallery_images[]" value="<?php echo esc_attr($image_id); ?>" />
                        <a href="#" class="remove-gallery-image">&times;</a>
                    </li>
                    <?php endif; ?>
                <?php endforeach; ?>
            </ul>
            <p class="hide-if-no-js">
                <a href="#" id="puntr-add-gallery-images" class="button"><?php esc_html_e('Add Images to Gallery', 'puntr'); ?></a>
            </p>
            <p class="description"><?php esc_html_e('Add multiple photos to showcase this provider. The featured image will be shown first. Recommended size: 800x1200px or larger.', 'puntr'); ?></p>
            <p class="description" style="margin-top: 10px;"><strong><?php esc_html_e('Note:', 'puntr'); ?></strong> <?php esc_html_e('Set a featured image for the main provider photo. Gallery images will be shown as additional photos.', 'puntr'); ?></p>
        </div>
        <?php
    }
    
    /**
     * Save meta boxes
     */
    public static function save_meta_boxes($post_id) {
        // Check if this is a provider post type
        if (get_post_type($post_id) !== 'puntr_provider') {
            return;
        }
        
        // Check nonce
        if (!isset($_POST['puntr_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['puntr_meta_nonce'])), 'puntr_save_meta_boxes')) {
            return;
        }
        
        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Prepare sanitized data
        $sanitized_data = array();
        
        // Collect and sanitize all data first
        $all_fields = array(
            'puntr_headline' => 'sanitize_text_field',
            'puntr_gender' => 'sanitize_text_field',
            'puntr_sexuality' => 'sanitize_text_field',
            'puntr_age' => 'intval',
            'puntr_ethnicity' => 'sanitize_text_field',
            'puntr_language' => 'sanitize_text_field',
            'puntr_body_type' => 'sanitize_text_field',
            'puntr_cup_size' => 'sanitize_text_field',
            'puntr_height_cm' => 'intval',
            'puntr_weight_kg' => 'intval',
            'puntr_dress_size' => 'sanitize_text_field',
            'puntr_eye_color' => 'sanitize_text_field',
            'puntr_hair_description' => 'sanitize_text_field',
            'puntr_skin_tone' => 'sanitize_text_field',
            'puntr_price_30_mins' => 'intval',
            'puntr_price_45_mins' => 'intval',
            'puntr_price_60_mins' => 'intval',
            'puntr_price_90_mins' => 'intval',
            'puntr_price_120_mins' => 'intval',
            'puntr_price_180_mins' => 'intval',
            'puntr_min_booking_mins' => 'intval',
        );
        
        // Sanitize all fields
        foreach ($all_fields as $field_name => $sanitize_function) {
            if (isset($_POST[$field_name])) {
                $sanitized_data[$field_name] = call_user_func($sanitize_function, wp_unslash($_POST[$field_name]));
            }
        }
        
        // Handle special fields
        if (isset($_POST['puntr_external_url'])) {
            $sanitized_data['puntr_external_url'] = esc_url_raw(wp_unslash($_POST['puntr_external_url']));
        }
        $sanitized_data['puntr_photos_real'] = isset($_POST['puntr_photos_real']) ? '1' : '0';
        $sanitized_data['puntr_is_active'] = isset($_POST['puntr_is_active']) ? '1' : '0';
        if (isset($_POST['puntr_gallery_images'])) {
            $sanitized_data['puntr_gallery_images'] = array_map('intval', wp_unslash($_POST['puntr_gallery_images']));
        }
        
        // Hook: Before save provider
        do_action('puntr_before_save_provider', $post_id, $sanitized_data);
        
        // Basic Information fields
        $basic_fields = array(
            'puntr_headline' => '_puntr_headline',
            'puntr_gender' => '_puntr_gender',
            'puntr_sexuality' => '_puntr_sexuality',
            'puntr_age' => '_puntr_age',
            'puntr_ethnicity' => '_puntr_ethnicity',
            'puntr_language' => '_puntr_language',
        );
        
        foreach ($basic_fields as $field_name => $meta_key) {
            if (isset($sanitized_data[$field_name])) {
                update_post_meta($post_id, $meta_key, $sanitized_data[$field_name]);
            }
        }
        
        // Physical Details fields
        $physical_fields = array(
            'puntr_body_type' => '_puntr_body_type',
            'puntr_cup_size' => '_puntr_cup_size',
            'puntr_height_cm' => '_puntr_height_cm',
            'puntr_weight_kg' => '_puntr_weight_kg',
            'puntr_dress_size' => '_puntr_dress_size',
            'puntr_eye_color' => '_puntr_eye_color',
            'puntr_hair_description' => '_puntr_hair_description',
            'puntr_skin_tone' => '_puntr_skin_tone',
        );
        
        foreach ($physical_fields as $field_name => $meta_key) {
            if (isset($sanitized_data[$field_name])) {
                update_post_meta($post_id, $meta_key, $sanitized_data[$field_name]);
            }
        }
        
        // Pricing fields
        $pricing_fields = array(
            'puntr_price_30_mins' => '_puntr_price_30_mins',
            'puntr_price_45_mins' => '_puntr_price_45_mins',
            'puntr_price_60_mins' => '_puntr_price_60_mins',
            'puntr_price_90_mins' => '_puntr_price_90_mins',
            'puntr_price_120_mins' => '_puntr_price_120_mins',
            'puntr_price_180_mins' => '_puntr_price_180_mins',
            'puntr_min_booking_mins' => '_puntr_min_booking_mins',
        );
        
        foreach ($pricing_fields as $field_name => $meta_key) {
            if (isset($sanitized_data[$field_name])) {
                update_post_meta($post_id, $meta_key, $sanitized_data[$field_name]);
            }
        }
        
        // Additional fields
        if (isset($sanitized_data['puntr_external_url'])) {
            update_post_meta($post_id, '_puntr_external_url', $sanitized_data['puntr_external_url']);
        }
        
        // Checkbox fields
        update_post_meta($post_id, '_puntr_photos_real', $sanitized_data['puntr_photos_real']);
        update_post_meta($post_id, '_puntr_is_active', $sanitized_data['puntr_is_active']);
        
        // Gallery images
        if (isset($sanitized_data['puntr_gallery_images'])) {
            update_post_meta($post_id, '_puntr_gallery_images', $sanitized_data['puntr_gallery_images']);
        } else {
            delete_post_meta($post_id, '_puntr_gallery_images');
        }
        
        // Hook: After save provider
        do_action('puntr_after_save_provider', $post_id, $sanitized_data);
    }
}