<?php
/**
 * Calendar functionality using DayPilot
 */

if (!defined('ABSPATH')) {
    exit;
}

class Puntr_Calendar {
    
    /**
     * Initialize calendar
     */
    public static function init() {
        add_shortcode('puntr_roster_calendar', array(__CLASS__, 'render_calendar_shortcode'));
        add_shortcode('puntr_daily_roster', array(__CLASS__, 'render_daily_roster_shortcode'));
        add_shortcode('puntr_provider_roster', array(__CLASS__, 'render_provider_roster_shortcode'));
        
        // Public AJAX actions (no authentication required)
        add_action('wp_ajax_puntr_get_events', array(__CLASS__, 'ajax_get_events'));
        add_action('wp_ajax_nopriv_puntr_get_events', array(__CLASS__, 'ajax_get_events'));
        add_action('wp_ajax_puntr_get_providers', array(__CLASS__, 'ajax_get_providers'));
        add_action('wp_ajax_nopriv_puntr_get_providers', array(__CLASS__, 'ajax_get_providers'));
        
        // Admin-only AJAX actions
        add_action('wp_ajax_puntr_save_event', array(__CLASS__, 'ajax_save_event'));
        add_action('wp_ajax_puntr_update_event', array(__CLASS__, 'ajax_update_event'));
        add_action('wp_ajax_puntr_delete_event', array(__CLASS__, 'ajax_delete_event'));
        
        // Automatically add roster to provider pages
        add_filter('the_content', array(__CLASS__, 'add_roster_to_provider_page'));
    }
    
    /**
     * Render calendar shortcode
     */
    public static function render_calendar_shortcode($atts) {
        $atts = shortcode_atts(array(
            'view' => 'scheduler', // scheduler, month, week
            'height' => '600',
            'readonly' => 'auto' // auto, true, false
        ), $atts);
        
        // Determine if calendar should be read-only
        $is_admin = current_user_can('edit_posts');
        // On frontend, always readonly unless explicitly set to false
        if (!is_admin()) {
            $readonly = ($atts['readonly'] === 'false') ? false : true;
        } else {
            // In admin area, allow editing by default for admins
            $readonly = $atts['readonly'] === 'auto' ? !$is_admin : ($atts['readonly'] === 'true');
        }
        
        // Enqueue scripts
        wp_enqueue_script('daypilot', PUNTR_PLUGIN_URL . 'assets/js/daypilot-all.min.js', array(), PUNTR_VERSION, true);
        wp_enqueue_script('puntr-calendar', PUNTR_PLUGIN_URL . 'assets/js/calendar.js', array('jquery', 'daypilot'), PUNTR_VERSION, true);
        
        // Load styles based on context
        if (is_admin()) {
            // Full styles in admin
            wp_enqueue_style('puntr-calendar', PUNTR_PLUGIN_URL . 'assets/css/calendar.css', array(), PUNTR_VERSION);
        } elseif ($is_admin && !is_admin()) {
            // Minimal admin styles on frontend
            wp_enqueue_style('puntr-calendar-admin', PUNTR_PLUGIN_URL . 'assets/css/calendar-admin-frontend.css', array(), PUNTR_VERSION);
        }
        
        // Enqueue dashicons for the eye icon (only for admins)
        if ($is_admin) {
            wp_enqueue_style('dashicons');
        }
        
        // Localize script
        wp_localize_script('puntr-calendar', 'puntr_calendar', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'admin_url' => admin_url(),
            'site_url' => home_url(),
            'nonce' => wp_create_nonce('puntr_calendar_nonce'),
            'view' => $atts['view'],
            'is_admin' => $is_admin ? 'true' : 'false',
            'readonly' => $readonly ? 'true' : 'false'
        ));
        
        ob_start();
        ?>
        <div class="puntr-calendar-wrapper">
            <div class="puntr-calendar-header">
                <?php if (!is_admin()) : ?>
                    <h2><?php esc_html_e('Weekly Roster', 'puntr'); ?></h2>
                <?php endif; ?>
            </div>
            <div id="puntr-calendar" style="height: <?php echo esc_attr($atts['height']); ?>px;"></div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get providers via AJAX
     */
    public static function ajax_get_providers() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'puntr_calendar_nonce')) {
            wp_die('Security check failed');
        }
        
        $query_args = array(
            'post_type' => 'puntr_provider',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'title',
            'order' => 'ASC'
        );
        
        // Allow filtering of provider query args
        $query_args = apply_filters('puntr_calendar_providers_query', $query_args);
        
        $providers = get_posts($query_args);
        
        $resources = array();
        foreach ($providers as $provider) {
            $is_active = get_post_meta($provider->ID, '_puntr_is_active', true);
            $name = $provider->post_title;
            
            // Add (Inactive) suffix only if explicitly set to '0'
            // Empty or '1' are considered active (default behavior)
            if ($is_active === '0') {
                $name .= ' (Inactive)';
            }
            
            $resources[] = array(
                'id' => $provider->ID,
                'name' => $name,
                'active' => ($is_active !== '0') // Active unless explicitly set to '0'
            );
        }
        
        wp_send_json_success($resources);
    }
    
    /**
     * Get events via AJAX
     */
    public static function ajax_get_events() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'puntr_calendar_nonce')) {
            wp_die('Security check failed');
        }
        
        $start = isset($_POST['start']) ? sanitize_text_field(wp_unslash($_POST['start'])) : '';
        $end = isset($_POST['end']) ? sanitize_text_field(wp_unslash($_POST['end'])) : '';
        
        
        // Query roster entries - try without date filter first to see all events
        $args = array(
            'post_type' => 'puntr_roster',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        );
        
        // Get all events first
        $all_events = get_posts($args);
        
        
        $calendar_events = array();
        
        foreach ($all_events as $event) {
            $provider_id = get_post_meta($event->ID, '_puntr_provider_id', true);
            $provider = get_post($provider_id);
            $status = get_post_meta($event->ID, '_puntr_roster_status', true);
            $start_date = get_post_meta($event->ID, '_puntr_start_date', true);
            $end_date = get_post_meta($event->ID, '_puntr_end_date', true);
            
            
            // Add all events for now to debug
            $calendar_events[] = array(
                'id' => $event->ID,
                'text' => $provider ? $provider->post_title : $event->post_title,
                'start' => $start_date,
                'end' => $end_date,
                'resource' => intval($provider_id), // Ensure it's an integer
                'status' => $status ?: 'scheduled',
                'allDay' => get_post_meta($event->ID, '_puntr_all_day', true) === '1',
                'backColor' => self::get_status_color($status ?: 'scheduled'),
                'notes' => get_post_meta($event->ID, '_puntr_roster_notes', true)
            );
        }
        
        wp_send_json_success($calendar_events);
    }
    
    /**
     * Save new event via AJAX
     */
    public static function ajax_save_event() {
        // Check nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'puntr_calendar_nonce')) {
            wp_die('Security check failed');
        }
        
        if (!current_user_can('edit_posts')) {
            wp_die('Permission denied');
        }
        
        $provider_id = isset($_POST['provider_id']) ? intval($_POST['provider_id']) : 0;
        $start = isset($_POST['start']) ? sanitize_text_field(wp_unslash($_POST['start'])) : '';
        $end = isset($_POST['end']) ? sanitize_text_field(wp_unslash($_POST['end'])) : '';
        $notes = isset($_POST['notes']) ? sanitize_textarea_field(wp_unslash($_POST['notes'])) : '';
        
        if (!$provider_id || !$start || !$end) {
            wp_send_json_error('Missing required fields');
        }
        
        $provider = get_post($provider_id);
        if (!$provider) {
            wp_send_json_error('Invalid provider');
        }
        
        // Hook: Before create roster event
        $event_data = array(
            'provider_id' => $provider_id,
            'start' => $start,
            'end' => $end,
            'notes' => $notes
        );
        do_action('puntr_before_create_roster_event', $event_data);
        
        // Create roster entry
        $roster_id = wp_insert_post(array(
            'post_type' => 'puntr_roster',
            'post_title' => $provider->post_title . ' - ' . wp_date('Y-m-d', strtotime($start)),
            'post_status' => 'publish'
        ));
        
        if ($roster_id) {
            update_post_meta($roster_id, '_puntr_provider_id', $provider_id);
            update_post_meta($roster_id, '_puntr_start_date', $start);
            update_post_meta($roster_id, '_puntr_end_date', $end);
            update_post_meta($roster_id, '_puntr_roster_status', 'scheduled');
            update_post_meta($roster_id, '_puntr_roster_notes', $notes);
            update_post_meta($roster_id, '_puntr_all_day', '0');
            
            // Hook: After create roster event
            do_action('puntr_after_create_roster_event', $roster_id, $event_data);
            
            wp_send_json_success(array(
                'id' => $roster_id,
                'message' => __('Roster entry created successfully', 'puntr')
            ));
        } else {
            wp_send_json_error('Failed to create roster entry');
        }
    }
    
    /**
     * Update event via AJAX
     */
    public static function ajax_update_event() {
        // Check nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'puntr_calendar_nonce')) {
            wp_die('Security check failed');
        }
        
        if (!current_user_can('edit_posts')) {
            wp_die('Permission denied');
        }
        
        $event_id = isset($_POST['event_id']) ? intval($_POST['event_id']) : 0;
        $start = isset($_POST['start']) ? sanitize_text_field(wp_unslash($_POST['start'])) : '';
        $end = isset($_POST['end']) ? sanitize_text_field(wp_unslash($_POST['end'])) : '';
        $resource = isset($_POST['resource']) ? intval($_POST['resource']) : 0;
        
        if (!$event_id || !$start || !$end) {
            wp_send_json_error('Missing required fields');
        }
        
        // Update roster entry
        update_post_meta($event_id, '_puntr_start_date', $start);
        update_post_meta($event_id, '_puntr_end_date', $end);
        
        if ($resource) {
            update_post_meta($event_id, '_puntr_provider_id', $resource);
        }
        
        wp_send_json_success(array(
            'message' => __('Roster entry updated successfully', 'puntr')
        ));
    }
    
    /**
     * Delete event via AJAX
     */
    public static function ajax_delete_event() {
        // Check nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'puntr_calendar_nonce')) {
            wp_die('Security check failed');
        }
        
        if (!current_user_can('edit_posts')) {
            wp_die('Permission denied');
        }
        
        $event_id = isset($_POST['event_id']) ? intval($_POST['event_id']) : 0;
        
        if (!$event_id) {
            wp_send_json_error('Invalid event ID');
        }
        
        // Actually delete the post
        if (wp_delete_post($event_id, true)) {
            wp_send_json_success(array(
                'message' => __('Roster entry deleted successfully', 'puntr')
            ));
        } else {
            wp_send_json_error('Failed to delete roster entry');
        }
    }
    
    /**
     * Get status color
     */
    private static function get_status_color($status) {
        $colors = array(
            'scheduled' => '#2196F3',
            'confirmed' => '#4CAF50',
            'cancelled' => '#f44336',
            'completed' => '#9E9E9E'
        );
        
        // Filter: Allow customization of calendar event colors
        $colors = apply_filters('puntr_calendar_event_colors', $colors);
        
        return isset($colors[$status]) ? $colors[$status] : '#2196F3';
    }
    
    /**
     * Render daily roster shortcode
     */
    public static function render_daily_roster_shortcode($atts) {
        $atts = shortcode_atts(array(
            'date' => '', // Empty means today
            'show_photo' => 'false',
            'photo_size' => '50',
            'show_age' => 'false',
            'show_height' => 'false',
            'show_dress_size' => 'false',
            'show_bust' => 'false',
            'show_location' => 'false',
            'show_services' => 'false',
            'show_rates' => 'false',
            'columns' => 'ethnicity,name,start,end' // Customizable columns
        ), $atts);
        
        // Only enqueue styles in admin
        if (is_admin()) {
            wp_enqueue_style('puntr-roster-table', PUNTR_PLUGIN_URL . 'assets/css/roster-table.css', array(), PUNTR_VERSION);
        }
        
        // Get the date to display
        $display_date = !empty($atts['date']) ? $atts['date'] : wp_date('Y-m-d');
        $date_start = $display_date . ' 00:00:00';
        $date_end = $display_date . ' 23:59:59';
        
        // Query roster entries that either start on this date OR overlap from the previous day
        $prev_date_end = wp_date('Y-m-d', strtotime($display_date . ' -1 day')) . ' 23:59:59';
        
        $args = array(
            'post_type' => 'puntr_roster',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => array(
                'relation' => 'OR',
                // Events that start on this date
                array(
                    'key' => '_puntr_start_date',
                    'value' => array($date_start, $date_end),
                    'compare' => 'BETWEEN',
                    'type' => 'DATETIME'
                ),
                // Events that started yesterday but end today (overnight shifts)
                array(
                    'relation' => 'AND',
                    array(
                        'key' => '_puntr_start_date',
                        'value' => $date_start,
                        'compare' => '<',
                        'type' => 'DATETIME'
                    ),
                    array(
                        'key' => '_puntr_end_date',
                        'value' => $date_start,
                        'compare' => '>=',
                        'type' => 'DATETIME'
                    )
                )
            ),
            'orderby' => 'meta_value',
            'meta_key' => '_puntr_start_date',
            'order' => 'ASC'
        );
        
        
        $roster_entries = get_posts($args);
        
        ob_start();
        ?>
        <div class="puntr-daily-roster">
            <h3><?php 
            /* translators: %s: Formatted date (e.g., Monday, January 15, 2025) */
            echo esc_html(sprintf(__('Roster for %s', 'puntr'), wp_date('l, F j, Y', strtotime($display_date)))); 
            ?></h3>
            
            <?php if (empty($roster_entries)) : ?>
                <p><?php esc_html_e('No roster entries for this date.', 'puntr'); ?></p>
            <?php else : ?>
                <?php
                // Parse columns to display
                $columns = array_map('trim', explode(',', $atts['columns']));
                $column_headers = array(
                    'photo' => __('Photo', 'puntr'),
                    'ethnicity' => __('Ethnicity', 'puntr'),
                    'name' => __('Provider Name', 'puntr'),
                    'age' => __('Age', 'puntr'),
                    'height' => __('Height', 'puntr'),
                    'dress_size' => __('Dress Size', 'puntr'),
                    'bust' => __('Bust', 'puntr'),
                    'location' => __('Location', 'puntr'),
                    'services' => __('Services', 'puntr'),
                    'rates' => __('Rates', 'puntr'),
                    'start' => __('Start Time', 'puntr'),
                    'end' => __('End Time', 'puntr')
                );
                
                // Add photo column if requested
                if ($atts['show_photo'] === 'true' && !in_array('photo', $columns)) {
                    array_unshift($columns, 'photo');
                }
                ?>
                <table class="puntr-roster-table">
                    <thead>
                        <tr>
                            <?php foreach ($columns as $column) : ?>
                                <?php if (isset($column_headers[$column])) : ?>
                                    <th><?php echo esc_html($column_headers[$column]); ?></th>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($roster_entries as $entry) : 
                            $provider_id = get_post_meta($entry->ID, '_puntr_provider_id', true);
                            $provider = get_post($provider_id);
                            
                            // Get all provider meta data
                            $provider_data = array(
                                'ethnicity' => get_post_meta($provider_id, '_puntr_ethnicity', true),
                                'age' => get_post_meta($provider_id, '_puntr_age', true),
                                'height' => get_post_meta($provider_id, '_puntr_height', true),
                                'dress_size' => get_post_meta($provider_id, '_puntr_dress_size', true),
                                'bust' => get_post_meta($provider_id, '_puntr_bust_size', true),
                                'location' => get_post_meta($provider_id, '_puntr_city', true),
                                'incall_rate' => get_post_meta($provider_id, '_puntr_incall_rate', true),
                                'outcall_rate' => get_post_meta($provider_id, '_puntr_outcall_rate', true)
                            );
                            
                            // Get services
                            $services = wp_get_post_terms($provider_id, 'puntr_service', array('fields' => 'names'));
                            
                            // Get times
                            $start_date = get_post_meta($entry->ID, '_puntr_start_date', true);
                            $end_date = get_post_meta($entry->ID, '_puntr_end_date', true);
                            
                            // Format times
                            $start_time = wp_date('g:i A', strtotime($start_date));
                            $end_time = wp_date('g:i A', strtotime($end_date));
                            
                            // Check if the entry spans multiple days
                            $start_day = wp_date('Y-m-d', strtotime($start_date));
                            $end_day = wp_date('Y-m-d', strtotime($end_date));
                            
                            // Add date indicator if it spans days
                            if ($start_day != $end_day) {
                                if ($start_day < $display_date) {
                                    $start_time = $start_time . ' (from ' . wp_date('M j', strtotime($start_day)) . ')';
                                }
                                if ($end_day > $display_date) {
                                    $end_time = $end_time . ' (next day)';
                                }
                            }
                        ?>
                            <tr class="<?php echo ($start_day < $display_date) ? 'overnight-shift' : ''; ?>">
                                <?php foreach ($columns as $column) : ?>
                                    <?php if ($column === 'photo' && $atts['show_photo'] === 'true') : ?>
                                        <td class="photo-cell">
                                            <?php if (has_post_thumbnail($provider_id)) : ?>
                                                <a href="<?php echo esc_url(get_permalink($provider_id)); ?>" target="_blank">
                                                    <?php echo get_the_post_thumbnail($provider_id, array($atts['photo_size'], $atts['photo_size']), array('class' => 'provider-photo')); ?>
                                                </a>
                                            <?php else : ?>
                                                <div class="no-photo">-</div>
                                            <?php endif; ?>
                                        </td>
                                    <?php elseif ($column === 'ethnicity') : ?>
                                        <td><?php echo esc_html($provider_data['ethnicity'] ?: '-'); ?></td>
                                    <?php elseif ($column === 'name') : ?>
                                        <td>
                                            <?php if ($provider) : ?>
                                                <a href="<?php echo esc_url(get_permalink($provider_id)); ?>" target="_blank">
                                                    <?php echo esc_html($provider->post_title); ?>
                                                </a>
                                            <?php else : ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                    <?php elseif ($column === 'age' && $atts['show_age'] === 'true') : ?>
                                        <td><?php echo esc_html($provider_data['age'] ?: '-'); ?></td>
                                    <?php elseif ($column === 'height' && $atts['show_height'] === 'true') : ?>
                                        <td><?php echo esc_html($provider_data['height'] ? $provider_data['height'] . 'cm' : '-'); ?></td>
                                    <?php elseif ($column === 'dress_size' && $atts['show_dress_size'] === 'true') : ?>
                                        <td><?php echo esc_html($provider_data['dress_size'] ?: '-'); ?></td>
                                    <?php elseif ($column === 'bust' && $atts['show_bust'] === 'true') : ?>
                                        <td><?php echo esc_html($provider_data['bust'] ?: '-'); ?></td>
                                    <?php elseif ($column === 'location' && $atts['show_location'] === 'true') : ?>
                                        <td><?php echo esc_html($provider_data['location'] ?: '-'); ?></td>
                                    <?php elseif ($column === 'services' && $atts['show_services'] === 'true') : ?>
                                        <td><?php echo esc_html(!empty($services) ? implode(', ', $services) : '-'); ?></td>
                                    <?php elseif ($column === 'rates' && $atts['show_rates'] === 'true') : ?>
                                        <td>
                                            <?php 
                                            $rates = array();
                                            if ($provider_data['incall_rate']) $rates[] = 'In: $' . $provider_data['incall_rate'];
                                            if ($provider_data['outcall_rate']) $rates[] = 'Out: $' . $provider_data['outcall_rate'];
                                            echo esc_html(!empty($rates) ? implode(' / ', $rates) : '-');
                                            ?>
                                        </td>
                                    <?php elseif ($column === 'start') : ?>
                                        <td><?php echo esc_html($start_time); ?></td>
                                    <?php elseif ($column === 'end') : ?>
                                        <td><?php echo esc_html($end_time); ?></td>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Render provider roster shortcode
     */
    public static function render_provider_roster_shortcode($atts) {
        $atts = shortcode_atts(array(
            'provider_id' => get_the_ID(), // Default to current post
            'days' => 7, // Show next 7 days
            'show_past' => 'false'
        ), $atts);
        
        // Only enqueue styles in admin
        if (is_admin()) {
            wp_enqueue_style('puntr-roster-table', PUNTR_PLUGIN_URL . 'assets/css/roster-table.css', array(), PUNTR_VERSION);
        }
        
        $provider_id = intval($atts['provider_id']);
        $provider = get_post($provider_id);
        
        if (!$provider || $provider->post_type !== 'puntr_provider') {
            return '';
        }
        
        // Calculate date range
        $start_date = wp_date('Y-m-d H:i:s');
        $end_date = wp_date('Y-m-d H:i:s', strtotime('+' . $atts['days'] . ' days'));
        
        // Query roster entries for this provider
        $args = array(
            'post_type' => 'puntr_roster',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_puntr_provider_id',
                    'value' => $provider_id,
                    'compare' => '='
                ),
                array(
                    'key' => '_puntr_start_date',
                    'value' => $start_date,
                    'compare' => '>=',
                    'type' => 'DATETIME'
                )
            ),
            'orderby' => 'meta_value',
            'meta_key' => '_puntr_start_date',
            'order' => 'ASC'
        );
        
        // If showing past events too
        if ($atts['show_past'] === 'true') {
            unset($args['meta_query'][1]);
        }
        
        $roster_entries = get_posts($args);
        
        ob_start();
        ?>
        <div class="puntr-provider-roster">
            <h3><?php esc_html_e('Availability', 'puntr'); ?></h3>
            
            <?php if (empty($roster_entries)) : ?>
                <p><?php esc_html_e('No upcoming availability scheduled.', 'puntr'); ?></p>
            <?php else : ?>
                <table class="puntr-provider-roster-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Date', 'puntr'); ?></th>
                            <th><?php esc_html_e('Day', 'puntr'); ?></th>
                            <th><?php esc_html_e('Time', 'puntr'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($roster_entries as $entry) : 
                            $start_date = get_post_meta($entry->ID, '_puntr_start_date', true);
                            $end_date = get_post_meta($entry->ID, '_puntr_end_date', true);
                            
                            // Format date and time
                            $date = wp_date('M j', strtotime($start_date));
                            $day = wp_date('l', strtotime($start_date));
                            $start_time = wp_date('g:i A', strtotime($start_date));
                            $end_time = wp_date('g:i A', strtotime($end_date));
                            
                            // Check if overnight
                            $start_day = wp_date('Y-m-d', strtotime($start_date));
                            $end_day = wp_date('Y-m-d', strtotime($end_date));
                            if ($start_day != $end_day) {
                                $end_time .= ' (' . __('next day', 'puntr') . ')';
                            }
                        ?>
                            <tr>
                                <td><?php echo esc_html($date); ?></td>
                                <td><?php echo esc_html($day); ?></td>
                                <td><?php echo esc_html($start_time . ' - ' . $end_time); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Automatically add roster to provider pages
     */
    public static function add_roster_to_provider_page($content) {
        if (is_singular('puntr_provider') && in_the_loop() && is_main_query()) {
            // Allow themes/templates to disable auto roster
            if (apply_filters('puntr_auto_add_roster', true)) {
                // Add roster after content
                $roster = do_shortcode('[puntr_provider_roster]');
                $content .= $roster;
            }
        }
        return $content;
    }
}