jQuery(document).ready(function($) {
    // Initialize DayPilot Scheduler
    console.log('Today according to DayPilot:', DayPilot.Date.today().toString());
    console.log('Today according to JS:', new Date().toISOString());
    
    // Force DayPilot to use the correct current date
    const today = new DayPilot.Date(new Date());
    
    console.log('Starting from today:', today.toString());
    
    const dp = new DayPilot.Scheduler("puntr-calendar", {
        timeHeaders: [
            {groupBy: "Day", format: "dddd, MMMM d"},
            {groupBy: "Hour", format: "H"}
        ],
        scale: "Hour",
        days: 7,
        startDate: today,
        businessBeginsHour: 0,
        businessEndsHour: 24,
        cellDuration: 60,
        cellWidth: 40,
        rowHeaderWidth: 150,
        eventHeight: 50,
        rowMinHeight: 60,
        
        // Custom row header rendering
        onBeforeRowHeaderRender: (args) => {
            if (puntr_calendar.is_admin === 'true' && puntr_calendar.admin_url) {
                const viewUrl = puntr_calendar.site_url + '/?p=' + args.row.id;
                const editUrl = puntr_calendar.admin_url + 'post.php?post=' + args.row.id + '&action=edit';
                args.row.html = '<div style="display: flex; align-items: center; width: 100%; height: 100%; position: relative;">' +
                    '<span style="flex: 1; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; padding-right: 10px;">' + args.row.name + '</span>' +
                    '<span style="display: flex; gap: 8px;">' +
                    '<a href="' + viewUrl + '" target="_blank" class="provider-view-link" title="View Provider" style="opacity: 0.5; color: #666; text-decoration: none;" onmouseover="this.style.opacity=\'1\'; this.style.color=\'#2271b1\'" onmouseout="this.style.opacity=\'0.5\'; this.style.color=\'#666\'"><span class="dashicons dashicons-visibility" style="font-size: 18px; width: 18px; height: 18px;"></span></a>' +
                    '<a href="' + editUrl + '" class="provider-edit-link" title="Edit Provider" style="opacity: 0.5; color: #666; text-decoration: none;" onmouseover="this.style.opacity=\'1\'; this.style.color=\'#135e96\'" onmouseout="this.style.opacity=\'0.5\'; this.style.color=\'#666\'"><span class="dashicons dashicons-edit" style="font-size: 18px; width: 18px; height: 18px;"></span></a>' +
                    '</span>' +
                    '</div>';
            } else {
                args.row.html = args.row.name;
            }
        },
        
        // Enable event creation by selecting time range (only in admin or if not readonly)
        timeRangeSelectedHandling: (puntr_calendar.is_admin === 'true' && puntr_calendar.readonly !== 'true') ? "Enabled" : "Disabled",
        
        // Event handlers
        onTimeRangeSelected: async (args) => {
            const scheduler = args.control;
            
            // Get the provider ID from the resource (row) that was clicked
            const provider_id = args.resource;
            
            console.log('Creating event:', {
                provider_id: provider_id,
                start: args.start.toString(),
                end: args.end.toString()
            });
            
            scheduler.clearSelection();
            
            // Save event via AJAX immediately
            $.post(puntr_calendar.ajax_url, {
                action: 'puntr_save_event',
                nonce: puntr_calendar.nonce,
                provider_id: provider_id,
                start: args.start.toDate().toISOString(),
                end: args.end.toDate().toISOString(),
                notes: ''
            }, function(response) {
                console.log('Response:', response);
                if (response.success) {
                    loadEvents();
                } else {
                    alert('Error creating roster entry: ' + response.data);
                }
            }).fail(function(xhr, status, error) {
                console.error('AJAX failed:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
                alert('Failed to create event: ' + error);
            });
        },
        
        // Enable event moving
        eventMoveHandling: (puntr_calendar.is_admin === 'true' && puntr_calendar.readonly !== 'true') ? "Update" : "Disabled",
        onEventMoved: (args) => {
            $.post(puntr_calendar.ajax_url, {
                action: 'puntr_update_event',
                nonce: puntr_calendar.nonce,
                event_id: args.e.id(),
                start: args.newStart.toDate().toISOString(),
                end: args.newEnd.toDate().toISOString(),
                resource: args.newResource
            }, function(response) {
                if (!response.success) {
                    alert('Error updating roster entry: ' + response.data);
                    loadEvents();
                }
            });
        },
        
        // Enable event resizing
        eventResizeHandling: (puntr_calendar.is_admin === 'true' && puntr_calendar.readonly !== 'true') ? "Update" : "Disabled",
        onEventResized: (args) => {
            $.post(puntr_calendar.ajax_url, {
                action: 'puntr_update_event',
                nonce: puntr_calendar.nonce,
                event_id: args.e.id(),
                start: args.newStart.toDate().toISOString(),
                end: args.newEnd.toDate().toISOString()
            }, function(response) {
                if (!response.success) {
                    alert('Error updating roster entry: ' + response.data);
                    loadEvents();
                }
            });
        },
        
        // Enable event deletion without confirmation
        eventDeleteHandling: (puntr_calendar.is_admin === 'true' && puntr_calendar.readonly !== 'true') ? "Update" : "Disabled",
        onEventDelete: (args) => {
            // Cancel the default delete behavior and handle it ourselves
            args.preventDefault();
            
            // Remove from calendar immediately
            dp.events.remove(args.e);
            
            // Delete from database
            $.post(puntr_calendar.ajax_url, {
                action: 'puntr_delete_event',
                nonce: puntr_calendar.nonce,
                event_id: args.e.id()
            }, function(response) {
                if (!response.success) {
                    alert('Error deleting roster entry: ' + response.data);
                    loadEvents();
                }
            });
        },
        
        // Disable event click handling (no popup)
        onEventClick: (args) => {
            // Do nothing on click - no popup
            args.preventDefault();
        },
        
        // Context menu
        contextMenu: (puntr_calendar.is_admin === 'true' && puntr_calendar.readonly !== 'true') ? new DayPilot.Menu({
            items: [
                {
                    text: "Delete",
                    onClick: (args) => {
                        // Remove from calendar immediately
                        dp.events.remove(args.source);
                        
                        // Delete from database
                        $.post(puntr_calendar.ajax_url, {
                            action: 'puntr_delete_event',
                            nonce: puntr_calendar.nonce,
                            event_id: args.source.id()
                        }, function(response) {
                            if (!response.success) {
                                alert('Error deleting roster entry: ' + response.data);
                                loadEvents();
                            }
                        });
                    }
                }
            ]
        }) : null
    });
    
    dp.init();
    
    // Load providers for resources
    async function loadProviders() {
        try {
            const response = await $.post(puntr_calendar.ajax_url, {
                action: 'puntr_get_providers',
                nonce: puntr_calendar.nonce
            });
            
            console.log('Providers loaded:', response);
            
            if (response.success && response.data.length > 0) {
                // Ensure provider IDs are integers
                const resources = response.data.map(p => ({
                    id: parseInt(p.id, 10),
                    name: p.name
                }));
                console.log('Resources for calendar:', resources);
                dp.update({resources: resources});
                return resources;
            } else {
                console.warn('No providers found or error loading providers');
                // Show a message if no providers
                if (response.data.length === 0) {
                    alert('No active providers found. Please create a provider first.');
                }
            }
        } catch (error) {
            console.error('Error loading providers:', error);
        }
        return [];
    }
    
    
    // Load events
    function loadEvents() {
        console.log('Loading events for date range:', dp.startDate.toString(), 'to', dp.startDate.addDays(dp.days).toString());
        
        $.post(puntr_calendar.ajax_url, {
            action: 'puntr_get_events',
            nonce: puntr_calendar.nonce,
            start: dp.startDate.toString(),
            end: dp.startDate.addDays(dp.days).toString()
        }, function(response) {
            console.log('Events response:', response);
            
            if (response.success) {
                // Ensure resource IDs are integers
                const events = response.data.map(e => ({
                    ...e,
                    resource: parseInt(e.resource, 10)
                }));
                
                console.log('Processed events:', events);
                dp.update({events: events});
                
                // Show event count in header
                const eventCount = events.length;
                if (eventCount > 0) {
                    $('.event-count').remove();
                    $('.puntr-calendar-header h2').append(' <span class="event-count">(' + eventCount + ' events)</span>');
                } else {
                    $('.event-count').text('(0 events)');
                }
            }
        });
    }
    
    // Add navigation buttons
    function addNavigationButtons() {
        const header = $('.puntr-calendar-header');
        if (!header.find('.calendar-navigation').length) {
            let navigationHtml = '<div class="calendar-navigation">' +
                '<button id="navigate-previous" class="button">← Previous Week</button>' +
                '<button id="navigate-today" class="button">Today</button>' +
                '<button id="navigate-next" class="button">Next Week →</button>' +
                '<input type="date" id="navigate-date" class="date-picker" title="Jump to date" />';
            
            // Only show zoom controls in admin
            if (puntr_calendar.is_admin === 'true') {
                navigationHtml += '<span class="zoom-controls">' +
                    '<button id="zoom-out" class="button" title="Zoom Out">−</button>' +
                    '<button id="zoom-in" class="button" title="Zoom In">+</button>' +
                    '</span>';
            }
            
            navigationHtml += '</div>';
            const navigation = $(navigationHtml);
            header.append(navigation);
            
            $('#navigate-previous').on('click', function() {
                dp.startDate = dp.startDate.addDays(-7);
                dp.update();
                loadEvents();
            });
            
            $('#navigate-today').on('click', function() {
                const today = new DayPilot.Date(new Date());
                dp.startDate = today;
                dp.update();
                loadEvents();
            });
            
            $('#navigate-next').on('click', function() {
                dp.startDate = dp.startDate.addDays(7);
                dp.update();
                loadEvents();
            });
            
            $('#navigate-date').on('change', function() {
                const selectedDate = new DayPilot.Date(this.value);
                dp.startDate = selectedDate;
                dp.update();
                loadEvents();
            });
            
            // Zoom controls (only if admin)
            if (puntr_calendar.is_admin === 'true') {
                let currentZoom = 40; // Default cell width
                
                $('#zoom-in').on('click', function() {
                    currentZoom = Math.min(currentZoom + 10, 100); // Max zoom 100px
                    dp.cellWidth = currentZoom;
                    dp.update();
                });
                
                $('#zoom-out').on('click', function() {
                    currentZoom = Math.max(currentZoom - 10, 20); // Min zoom 20px
                    dp.cellWidth = currentZoom;
                    dp.update();
                });
            }
        }
    }
    
    // Initial load
    loadProviders().then(() => {
        loadEvents();
        addNavigationButtons();
    });
    
});