<?php
/**
 * Plugin Name: Punchr Lite – PunchOut cXML Bridge for WooCommerce
 * Description: PunchOut (cXML) bridge for WooCommerce (Lite).
 * Version: 1.3.0
 * Author: punchr
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: punchr-lite
 * Requires PHP: 8.1
 * Requires at least: 6.2
 * Requires Plugins: woocommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once __DIR__ . '/includes/crypto.php';
require_once __DIR__ . '/includes/admin.php';

// ---- Lite Trial (Evaluation) ----
define( 'WCPOB_LITE_TRIAL_DAYS_DEFAULT', 14 );
define( 'WCPOB_LITE_OPT_ACTIVATED_AT', 'wcpob_lite_activated_at' );
define( 'WCPOB_LITE_OPT_TRIAL_DAYS', 'wcpob_lite_trial_days' );

function wcpob_lite_trial_days(): int {
    $days = (int) get_option( WCPOB_LITE_OPT_TRIAL_DAYS, WCPOB_LITE_TRIAL_DAYS_DEFAULT );
    if ( $days < 1 ) {
        $days = WCPOB_LITE_TRIAL_DAYS_DEFAULT;
    }
    // possibilité de filtrer
    $days = (int) apply_filters( 'wcpob_lite_trial_days', $days );
    return max( 1, $days );
}

function wcpob_lite_activated_at(): int {
    $ts = (int) get_option( WCPOB_LITE_OPT_ACTIVATED_AT, 0 );
    return $ts > 0 ? $ts : 0;
}

function wcpob_lite_trial_expires_at(): int {
    $activated_at = wcpob_lite_activated_at();
    if ( $activated_at <= 0 ) {
        return 0;
    }
    return $activated_at + ( wcpob_lite_trial_days() * DAY_IN_SECONDS );
}

function wcpob_lite_is_trial_expired(): bool {
    $exp = wcpob_lite_trial_expires_at();
    if ( $exp <= 0 ) {
        // si option manquante, on considère non-expiré (safe)
        return false;
    }
    return time() > $exp;
}

function wcpob_lite_trial_days_left(): int {
    $exp = wcpob_lite_trial_expires_at();
    if ( $exp <= 0 ) {
        return wcpob_lite_trial_days();
    }
    $left = (int) ceil( ( $exp - time() ) / DAY_IN_SECONDS );
    return max( 0, $left );
}

/**
 * Build a simple cXML Status response (used for expired evaluation).
 */
function wcpob_lite_cxml_status_response( int $code, string $text ): string {
    $payload_id = wp_generate_uuid4();
    $xml  = '<?xml version="1.0" encoding="UTF-8"?>';
    $xml .= '<cXML payloadID="' . esc_attr( $payload_id ) . '" timestamp="' . esc_attr( gmdate( 'c' ) ) . '">';
    $xml .= '<Response><Status code="' . (int) $code . '" text="' . esc_attr( $text ) . '"/></Response>';
    $xml .= '</cXML>';
    return $xml;
}



add_filter(
	'plugin_action_links_' . plugin_basename( __FILE__ ),
	function ( $links ) {
		$settings = '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite' ) ) . '">Settings</a>';
		$logs     = '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-logs' ) ) . '">Logs</a>';
		$upgrade  = '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-upgrade' ) ) . '">Upgrade</a>';

		array_unshift( $links, $settings, $logs, $upgrade );
		return $links;
	}
);

/**
 * Tables
 */
function wcpob_table_buyers(): string {
	global $wpdb;
	return $wpdb->prefix . 'wcpob_buyers';
}
function wcpob_table_sessions(): string {
	global $wpdb;
	return $wpdb->prefix . 'wcpob_sessions';
}
function wcpob_table_logs(): string {
	global $wpdb;
	return $wpdb->prefix . 'wcpob_logs';
}

/**
 * Logging (lite)
 */
function wcpob_log( array $data ): void {
	global $wpdb;

	$ip = null;
	if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
		$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
	}
	$ua = null;
	if ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
		$ua = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
	}

	$defaults = array(
		'buyer_id'         => null,
		'session_id'       => null,
		'direction'        => 'internal',
		'event'            => 'unknown',
		'severity'         => 'info',
		'http_status'      => null,
		'message'          => '',
		'payload_excerpt'  => null,
		'payload_hash'     => null,
		'ip'               => apply_filters( 'wcpob_lite_log_ip', $ip ),
		'ua'               => apply_filters( 'wcpob_lite_log_ua', $ua ),
	);

	$row = array_merge( $defaults, $data );

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	$wpdb->insert( wcpob_table_logs(), $row );
}

/**
 * Helpers cXML
 */
function wcpob_get_session_by_sid( string $sid ) {
	global $wpdb;

	$cache_key   = 'wcpob_session_' . md5( $sid );
	$cache_group = 'wcpob_lite';

	$cached = wp_cache_get( $cache_key, $cache_group );
	if ( false !== $cached ) {
		return $cached;
	}

	$t = wcpob_table_sessions();

	$row = $wpdb->get_row(
		$wpdb->prepare(
			'SELECT * FROM %i WHERE sid = %s',
			$t,
			$sid
		)
	);

	wp_cache_set( $cache_key, $row, $cache_group, 60 );
	return $row;
}

function wcpob_cxml_text( \DOMXPath $xp, string $q ): ?string {
	$n = $xp->query( $q )->item( 0 );
	if ( ! $n ) {
		return null;
	}
	$v = trim( (string) $n->textContent );
	return ( $v === '' ) ? null : $v;
}

function wcpob_cxml_attr( \DOMXPath $xp, string $q, string $attr ): ?string {
	$n = $xp->query( $q )->item( 0 );
	if ( ! $n || ! $n->attributes ) {
		return null;
	}
	$a = $n->attributes->getNamedItem( $attr );
	if ( ! $a ) {
		return null;
	}
	$v = trim( (string) $a->nodeValue );
	return ( $v === '' ) ? null : $v;
}

function wcpob_session_meta( $row ): array {
	$m = array();
	if ( ! empty( $row->payload_meta ) ) {
		$m = json_decode( (string) $row->payload_meta, true );
		if ( ! is_array( $m ) ) {
			$m = array();
		}
	}
	return $m;
}

function wcpob_build_order_message( string $sid, string $buyerCookie, array $items, float $total, string $currency ): string {
	$xml = '<?xml version="1.0" encoding="UTF-8"?>'
		. '<cXML payloadID="' . esc_attr( $sid ) . '" timestamp="' . esc_attr( gmdate( 'c' ) ) . '">'
		. '<Message><PunchOutOrderMessage>'
		. '<BuyerCookie>' . esc_html( $buyerCookie ) . '</BuyerCookie>'
		. '<PunchOutOrderMessageHeader operationAllowed="create">'
		. '<Total><Money currency="' . esc_attr( $currency ) . '">' . number_format( $total, 2, '.', '' ) . '</Money></Total>'
		. '</PunchOutOrderMessageHeader>';

	foreach ( $items as $it ) {
		$xml .= '<ItemIn quantity="' . (int) $it['qty'] . '">'
			. '<ItemID><SupplierPartID>' . esc_html( $it['sku'] ) . '</SupplierPartID></ItemID>'
			. '<ItemDetail>'
			. '<Description xml:lang="en">' . esc_html( $it['name'] ) . '</Description>'
			. '<UnitPrice><Money currency="' . esc_attr( $currency ) . '">' . number_format( (float) $it['price'], 2, '.', '' ) . '</Money></UnitPrice>'
			. '<UnitOfMeasure>EA</UnitOfMeasure>'
			. '</ItemDetail>'
			. '</ItemIn>';
	}

	$xml .= '</PunchOutOrderMessage></Message></cXML>';
	return $xml;
}

function wcpob_is_punchout(): bool {
	return function_exists( 'WC' ) && WC()->session && WC()->session->get( 'wcpob_mode' );
}

/**
 * Internal: return a safe, whitelisted table name.
 * (PluginCheck can't "prove" safety otherwise.)
 */
function wcpob_lite_get_table_name( string $key ): string {
	global $wpdb;

	switch ( $key ) {
		case 'buyers':
			return $wpdb->prefix . 'wcpob_buyers';
		case 'sessions':
			return $wpdb->prefix . 'wcpob_sessions';
		case 'logs':
			return $wpdb->prefix . 'wcpob_logs';
		default:
			return '';
	}
}

/**
 * Activation: create tables
 */
register_activation_hook(
	__FILE__,
	function () {
		// Store activation timestamp once (trial start)
        if ( ! get_option( WCPOB_LITE_OPT_ACTIVATED_AT ) ) {
            add_option( WCPOB_LITE_OPT_ACTIVATED_AT, time(), '', false );
        } else {
            $v = (int) get_option( WCPOB_LITE_OPT_ACTIVATED_AT, 0 );
            if ( $v <= 0 ) {
                update_option( WCPOB_LITE_OPT_ACTIVATED_AT, time(), false );
            }
        }
        if ( ! get_option( WCPOB_LITE_OPT_TRIAL_DAYS ) ) {
            add_option( WCPOB_LITE_OPT_TRIAL_DAYS, WCPOB_LITE_TRIAL_DAYS_DEFAULT, '', false );
        }

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		global $wpdb;

		$charset  = $wpdb->get_charset_collate();
		$buyers   = wcpob_table_buyers();
		$sessions = wcpob_table_sessions();
		$logs     = wcpob_table_logs();

		$sql1 = "CREATE TABLE $buyers (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			name VARCHAR(190) NOT NULL,
			is_enabled TINYINT(1) NOT NULL DEFAULT 1,
			token_id VARCHAR(64) NOT NULL,
			catalog_rules LONGTEXT NULL,
			deleted_at DATETIME NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			UNIQUE KEY token_id (token_id),
			PRIMARY KEY (id)
		) $charset;";

		$sql2 = "CREATE TABLE $sessions (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			buyer_id BIGINT UNSIGNED NOT NULL,
			sid CHAR(36) NOT NULL,
			status VARCHAR(20) NOT NULL DEFAULT 'created',
			expires_at DATETIME NOT NULL,
			payload_meta LONGTEXT NULL,
			return_url TEXT NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			UNIQUE KEY sid (sid),
			KEY buyer_id (buyer_id),
			KEY expires_at (expires_at),
			PRIMARY KEY (id)
		) $charset;";

		$sql3 = "CREATE TABLE $logs (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			buyer_id BIGINT UNSIGNED NULL,
			session_id BIGINT UNSIGNED NULL,
			direction VARCHAR(10) NOT NULL,
			event VARCHAR(64) NOT NULL,
			severity VARCHAR(16) NOT NULL,
			http_status SMALLINT NULL,
			message VARCHAR(255) NOT NULL,
			payload_excerpt LONGTEXT NULL,
			payload_hash CHAR(64) NULL,
			ip VARCHAR(45) NULL,
			ua VARCHAR(255) NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			KEY buyer_id (buyer_id),
			KEY session_id (session_id),
			KEY event (event),
			KEY created_at (created_at),
			PRIMARY KEY (id)
		) $charset;";

		dbDelta( $sql1 );
		dbDelta( $sql2 );
		dbDelta( $sql3 );
	}
);

/**
 * FRONT: consume ?wcpob_sid=...&wcpob_nonce=...
 * Init Woo session/cart and set punchout mode.
 */
add_action(
	'init',
	function () {
		if ( ! function_exists( 'WC' ) ) {
			return;
		}

		$sid   = isset( $_GET['wcpob_sid'] ) ? sanitize_text_field( wp_unslash( $_GET['wcpob_sid'] ) ) : '';
		$nonce = isset( $_GET['wcpob_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['wcpob_nonce'] ) ) : '';
		if ( ! $sid ) {
			return;
		}

		// Nonce recommended by plugin check (we tie it to sid).
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wcpob_sid_' . $sid ) ) {
			return;
		}

		$row = wcpob_get_session_by_sid( $sid );
		if ( ! $row ) {
			return;
		}
		if ( strtotime( $row->expires_at ) < time() ) {
			return;
		}

		if ( ! WC()->session && method_exists( WC(), 'initialize_session' ) ) {
			WC()->initialize_session();
		}
		if ( ! WC()->customer && method_exists( WC(), 'initialize_customer' ) ) {
			WC()->initialize_customer();
		}
		if ( ! WC()->cart && method_exists( WC(), 'initialize_cart' ) ) {
			WC()->initialize_cart();
		}
		if ( ! WC()->session ) {
			return;
		}

		WC()->session->set( 'wcpob_sid', $sid );
		WC()->session->set( 'wcpob_mode', 1 );

		$rt = wp_generate_password( 20, false, false );
		WC()->session->set( 'wcpob_return_token', $rt );

		if ( method_exists( WC()->session, 'set_customer_session_cookie' ) ) {
			WC()->session->set_customer_session_cookie( true );
		}
		if ( method_exists( WC()->session, 'save_data' ) ) {
			WC()->session->save_data();
		}

		wcpob_log(
			array(
				'direction'  => 'internal',
				'event'      => 'front_session_initialized',
				'message'    => 'PunchOut session applied on front for sid=' . $sid,
				'session_id' => $row->id ?? null,
				'buyer_id'   => $row->buyer_id ?? null,
			)
		);

		wp_cache_delete( 'wcpob_session_' . md5( $sid ), 'wcpob_lite' );

		wp_safe_redirect( remove_query_arg( array( 'wcpob_sid', 'wcpob_nonce' ) ) );
		exit;
	},
	1
);

function wcpob_validate_return_url( ?string $url ): ?string {
	$url = $url ? trim( $url ) : '';
	if ( $url === '' ) {
		return null;
	}

	$url = esc_url_raw( $url );
	if ( $url === '' ) {
		return null;
	}

	$parts = wp_parse_url( $url );
	if ( empty( $parts['scheme'] ) || empty( $parts['host'] ) ) {
		return null;
	}

	$scheme = strtolower( $parts['scheme'] );
	if ( ! in_array( $scheme, array( 'http', 'https' ), true ) ) {
		return null;
	}

	$host = strtolower( $parts['host'] );

	// block common internal TLDs.
	if ( preg_match( '/\.(local|internal|lan)$/i', $host ) ) {
		return null;
	}

	// optional: block non-standard ports.
	if ( ! empty( $parts['port'] ) && ! in_array( (int) $parts['port'], array( 80, 443 ), true ) ) {
		return null;
	}

	// block localhost + obvious local hosts.
	if ( in_array( $host, array( 'localhost' ), true ) ) {
		return null;
	}

	// block direct IPs (simple and safe for Lite).
	if ( filter_var( $host, FILTER_VALIDATE_IP ) ) {
		return null;
	}

	return $url;
}

/**
 * Basic Auth helpers (Lite)
 * Accepts:
 * - PHP_AUTH_USER / PHP_AUTH_PW (recommended)
 * - Fallback parse Authorization header
 */
function wcpob_lite_get_basic_auth(): array {

	$user = '';
	$pass = '';

	// Preferred: PHP auth vars (set by PHP when Basic Auth is present)
	if ( isset( $_SERVER['PHP_AUTH_USER'] ) ) {
		$user = sanitize_text_field( wp_unslash( $_SERVER['PHP_AUTH_USER'] ) );
	}
	if ( isset( $_SERVER['PHP_AUTH_PW'] ) ) {
		$pass = sanitize_text_field( wp_unslash( $_SERVER['PHP_AUTH_PW'] ) );
	}

	if ( $user !== '' || $pass !== '' ) {
		return array( $user, $pass );
	}

	// Fallback: Authorization header
	$auth = '';
	if ( isset( $_SERVER['HTTP_AUTHORIZATION'] ) ) {
		$auth = sanitize_text_field( wp_unslash( $_SERVER['HTTP_AUTHORIZATION'] ) );
	} elseif ( function_exists( 'getallheaders' ) ) {
		$headers = getallheaders();
		if ( is_array( $headers ) ) {
			foreach ( $headers as $k => $v ) {
				if ( strtolower( (string) $k ) === 'authorization' ) {
					$auth = sanitize_text_field( wp_unslash( (string) $v ) );
					break;
				}
			}
		}
	}

	if ( ! preg_match( '/^\s*Basic\s+(.+)\s*$/i', $auth, $m ) ) {
		return array( '', '' );
	}

	$decoded = base64_decode( trim( $m[1] ), true );
	if ( $decoded === false ) {
		return array( '', '' );
	}

	$parts = explode( ':', $decoded, 2 );
	if ( count( $parts ) !== 2 ) {
		return array( '', '' );
	}

	// Optional extra hardening: sanitize extracted pieces too
	$user = sanitize_text_field( $parts[0] );
	$pass = sanitize_text_field( $parts[1] );

	return array( (string) $user, (string) $pass );
}

/**
 * Constant-time check: provided password must match buyer secret.
 */
function wcpob_lite_authenticate_buyer($buyer, string $token, string $password): bool {
	if (!$buyer) return false;

	// Token must match
	if (!hash_equals((string)$buyer->token_id, (string)$token)) {
		return false;
	}

	// secret_enc in catalog_rules => decrypt and compare
	$secret = null;
	if (!empty($buyer->catalog_rules)) {
		$json = json_decode((string)$buyer->catalog_rules, true);
		if (is_array($json)) {
			$enc = $json['secret_enc'] ?? null;
			if ($enc) $secret = wcpob_decrypt($enc);
		}
	}
	if (!$secret) return false;

	return hash_equals((string)$secret, (string)$password);
}

/**
 * Extract cXML credentials from Header/Sender/Credential.
 * Returns [identity, sharedSecret]
 */
function wcpob_lite_get_cxml_auth( \DOMXPath $xp ): array {
	// Most common (ERP/procurement): Header/Sender/Credential
	$identity = wcpob_cxml_text( $xp, '//Header/Sender/Credential/Identity' );
	$secret   = wcpob_cxml_text( $xp, '//Header/Sender/Credential/SharedSecret' );

	// Optional fallback: some senders use From instead of Sender (rare, but seen)
	if ( ( $identity === null || $identity === '' ) ) {
		$identity = wcpob_cxml_text( $xp, '//Header/From/Credential/Identity' );
	}
	if ( ( $secret === null || $secret === '' ) ) {
		$secret = wcpob_cxml_text( $xp, '//Header/From/Credential/SharedSecret' );
	}

	$identity = $identity ? sanitize_text_field( $identity ) : '';
	$secret   = $secret ? sanitize_text_field( $secret ) : '';

	return array( $identity, $secret );
}

/**
 * Get credentials for buyer authentication:
 * 1) Prefer HTTP Basic Auth if present
 * 2) Else fallback to cXML Header credentials
 *
 * Returns [token, password, mode] where mode = 'basic'|'cxml'|'none'
 */
function wcpob_lite_get_credentials( string $raw_xml, ?\DOMXPath $xp = null ): array {
	// 1) Basic Auth first
	list( $token, $password ) = wcpob_lite_get_basic_auth();
	if ( $token !== '' && $password !== '' ) {
		return array( $token, $password, 'basic' );
	}

	// 2) cXML fallback
	if ( ! $xp ) {
		$dom = new \DOMDocument();
		if ( ! @ $dom->loadXML( $raw_xml ) ) {
			return array( '', '', 'none' );
		}
		$xp = new \DOMXPath( $dom );
	}

	list( $id2, $secret2 ) = wcpob_lite_get_cxml_auth( $xp );
	if ( $id2 !== '' && $secret2 !== '' ) {
		return array( $id2, $secret2, 'cxml' );
	}

	return array( '', '', 'none' );
}

/**
 * REST API
 */
add_action(
	'rest_api_init',
	function () {

		// PunchOut Setup.
		register_rest_route(
			'punchr/v1',
			'/setup',
			array(
				'methods'             => 'POST',
				'permission_callback' => '__return_true',
				'callback'            => function ( \WP_REST_Request $req ) {

					$raw = $req->get_body();
					if ( ! $raw ) {
						return new \WP_REST_Response( array( 'error' => 'empty body' ), 400 );
					}
					// Trial expiration: block PunchOut setup in a clean, explicit cXML way
					if ( wcpob_lite_is_trial_expired() ) {
						wcpob_log(
							array(
								'direction'   => 'in',
								'event'       => 'lite_trial_expired',
								'severity'    => 'warn',
								'http_status' => 401,
								'message'     => 'Punchr Lite evaluation period expired. Blocking /setup.',
							)
						);

						$payload = wcpob_lite_cxml_status_response(
							401,
							'Punchr Lite evaluation period expired. Upgrade to Punchr Pro to continue.'
						);

						nocache_headers();
						header( 'Content-Type: application/xml; charset=UTF-8', true, 401 );
						// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						echo $payload;
						exit;
					}

					// Parse XML FIRST (needed for cXML-auth fallback)
					$dom = new \DOMDocument();
					if ( ! @ $dom->loadXML( $raw ) ) {
						return new \WP_REST_Response( array( 'error' => 'invalid xml' ), 400 );
					}
					$xp = new \DOMXPath( $dom );

					// 1) Try HTTP Basic Auth; 2) fallback to cXML Header auth
					list( $token, $password, $auth_mode ) = wcpob_lite_get_credentials( $raw, $xp );

					if ( $token === '' || $password === '' ) {
						return new \WP_REST_Response(
							array(
								'error' => 'missing credentials',
								'hint'  => 'Provide either HTTP Basic Auth (username=TOKEN, password=SECRET) OR cXML Header Sender/Credential (Identity + SharedSecret).',
							),
							401
						);
					}

					global $wpdb;
					$buyers_table = wcpob_lite_get_table_name( 'buyers' );
					if ( '' === $buyers_table ) {
						return new \WP_REST_Response( array( 'error' => 'server configuration error' ), 500 );
					}

					// Lite = single buyer; but we still load by token for safety
					$buyer_cache_key = 'wcpob_buyer_' . md5($token);
					$buyer = wp_cache_get($buyer_cache_key, 'wcpob_lite');

					if (false === $buyer) {
						// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
						$buyer = $wpdb->get_row(
							$wpdb->prepare(
								'SELECT * FROM %i WHERE token_id = %s AND is_enabled = 1 AND deleted_at IS NULL',
								$buyers_table,
								$token
							)
						);
						wp_cache_set($buyer_cache_key, $buyer, 'wcpob_lite', 60);
					}

					if (!wcpob_lite_authenticate_buyer($buyer, $token, $password)) {
						return new \WP_REST_Response(array('error' => 'invalid credentials'), 401);
					}


					$buyerCookie   = wcpob_cxml_text( $xp, '//PunchOutSetupRequest/BuyerCookie' );
					$returnUrlRaw  = wcpob_cxml_text( $xp, '//BrowserFormPost/URL' );
					$returnUrl     = wcpob_validate_return_url( $returnUrlRaw );
					if ( ! $returnUrl ) {
						return new \WP_REST_Response( array( 'error' => 'invalid return_url' ), 400 );
					}

					$sid        = wp_generate_uuid4();
					$expires    = gmdate( 'Y-m-d H:i:s', time() + 30 * 60 );
					$startToken = wp_generate_password( 20, false, false );

					$sessions_table = wcpob_table_sessions();

					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->insert(
						$sessions_table,
						array(
							'buyer_id'     => (int) $buyer->id,
							'sid'          => $sid,
							'status'       => 'created',
							'expires_at'   => $expires,
							'return_url'   => $returnUrl,
							'payload_meta' => wp_json_encode(
								array(
									'buyer_cookie' => $buyerCookie,
									'start_token'  => $startToken,
								)
							),
						)
					);
					$session_id = (int) $wpdb->insert_id;

					wcpob_log(
						array(
							'buyer_id'   => (int) $buyer->id,
							'session_id' => $session_id,
							'direction'  => 'internal',
							'event'      => 'setup_session_created',
							'message'    => 'Session created: ' . $sid,
						)
					);

					// Add nonce to front start redirect chain (recommended).
					$front_nonce = wp_create_nonce( 'wcpob_sid_' . $sid );

					$startUrl = add_query_arg(
						array(
							'sid' => $sid,
							'st'  => $startToken,
						),
						home_url( '/wp-json/punchr/v1/start' )
					);

					$payload = '<?xml version="1.0" encoding="UTF-8"?>'
            . '<cXML payloadID="' . esc_attr( $sid ) . '" timestamp="' . esc_attr( gmdate( 'c' ) ) . '">'
            . '<Response><Status code="200" text="OK"/>'
            . '<PunchOutSetupResponse><StartPage><URL>' . esc_xml( $startUrl ) . '</URL></StartPage></PunchOutSetupResponse>'
            . '</Response></cXML>';

		   wcpob_log(
				array(
				  'buyer_id'   => (int) $buyer->id,
				  'direction'  => 'in',
				  'event'      => 'auth_ok',
				  'severity'   => 'info',
				  'message'    => 'Auth OK via ' . $auth_mode,
				)
		  );
          // IMPORTANT: on renvoie du XML brut (sinon WP REST l’encode en JSON string)
          nocache_headers();
          header( 'Content-Type: application/xml; charset=UTF-8', true, 200 );

          // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
          echo $payload;
          exit;
				},
			)
		);

		// Start.
		register_rest_route(
			'punchr/v1',
			'/start',
			array(
				'methods'             => 'GET',
				'permission_callback' => '__return_true',
				'callback'            => function ( \WP_REST_Request $req ) {

					$sid = (string) $req->get_param( 'sid' );
					if ( ! $sid ) {
						return new \WP_REST_Response( array( 'error' => 'missing sid' ), 400 );
					}
					if ( wcpob_lite_is_trial_expired() ) {
						$payload = wcpob_lite_cxml_status_response(
							401,
							'Punchr Lite evaluation period expired. Upgrade to Punchr Pro to continue.'
						);
					
						nocache_headers();
						header( 'Content-Type: application/xml; charset=UTF-8', true, 401 );
						// We intentionally output raw XML (cXML). Escaping would break the XML payload.
    					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						echo $payload;
						exit;
					}

					$row = wcpob_get_session_by_sid( $sid );
					if ( ! $row ) {
						return new \WP_REST_Response( array( 'error' => 'unknown sid' ), 404 );
					}
					if ( strtotime( $row->expires_at ) < time() ) {
						return new \WP_REST_Response( array( 'error' => 'expired' ), 410 );
					}

					$st       = (string) $req->get_param( 'st' );
					$meta     = wcpob_session_meta( $row );
					$expected = (string) ( $meta['start_token'] ?? '' );

					if ( $expected === '' || $st === '' || ! hash_equals( $expected, $st ) ) {
						wcpob_log(
							array(
								'buyer_id'   => (int) ( $row->buyer_id ?? 0 ),
								'session_id' => (int) ( $row->id ?? 0 ),
								'direction'  => 'internal',
								'event'      => 'start_token_invalid',
								'severity'   => 'warn',
								'message'    => 'Invalid start token for sid=' . $sid,
							)
						);
						return new \WP_REST_Response( array( 'error' => 'invalid start token' ), 403 );
					}

					global $wpdb;
					$t = wcpob_lite_get_table_name( 'sessions' );

					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->update( $t, array( 'status' => 'active' ), array( 'id' => $row->id ) );

					wp_cache_delete( 'wcpob_session_' . md5( $sid ), 'wcpob_lite' );

					$shop = function_exists( 'wc_get_page_permalink' ) ? wc_get_page_permalink( 'shop' ) : home_url( '/' );
					$shop = add_query_arg(
						array(
							'wcpob_sid'   => $sid,
							'wcpob_nonce' => wp_create_nonce( 'wcpob_sid_' . $sid ),
						),
						$shop
					);

					return new \WP_REST_Response( null, 302, array( 'Location' => $shop ) );
				},
			)
		);
	}
);

/**
 * PunchOut UX: block checkout + return button (classic + blocks)
 */
add_filter(
	'woocommerce_get_checkout_url',
	function ( $url ) {
		if ( wcpob_is_punchout() ) {
			return wc_get_cart_url();
		}
		return $url;
	},
	999
);

add_action(
	'template_redirect',
	function () {
		if ( ! wcpob_is_punchout() ) {
			return;
		}
		if ( function_exists( 'is_checkout' ) && is_checkout() ) {
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}
	},
	1
);

add_action(
	'wp_loaded',
	function () {
		if ( ! wcpob_is_punchout() ) {
			return;
		}
		remove_action( 'woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20 );
	},
	100
);

add_action(
	'woocommerce_cart_totals_after_order_total',
	function () {
		if ( ! wcpob_is_punchout() ) {
			return;
		}

		$sid = (string) WC()->session->get( 'wcpob_sid' );
		if ( ! $sid ) {
			return;
		}
		$rt = (string) WC()->session->get( 'wcpob_return_token' );

		$url = add_query_arg(
			array(
				'wcpob_return' => 1,
				'sid'          => $sid,
				'rt'           => $rt,
				'nonce'        => wp_create_nonce( 'wcpob_return_' . $sid ),
			),
			home_url( '/' )
		);

		echo '<tr class="wcpob-return-row"><th></th><td style="text-align:right">';
		echo '<a class="button alt" style="display:inline-block;background:#111;color:#fff;padding:12px 18px;" href="' . esc_url( $url ) . '">Return to Procurement</a>';
		echo '</td></tr>';
	},
	999
);

add_filter(
	'render_block',
	function ( $content, $block ) {
		if ( ! wcpob_is_punchout() ) {
			return $content;
		}
		if ( function_exists( 'is_cart' ) && ! is_cart() ) {
			return $content;
		}

		$sid = (string) WC()->session->get( 'wcpob_sid' );
		if ( ! $sid ) {
			return $content;
		}

		$rt  = (string) WC()->session->get( 'wcpob_return_token' );
		$url = add_query_arg(
			array(
				'wcpob_return' => 1,
				'sid'          => $sid,
				'rt'           => $rt,
				'nonce'        => wp_create_nonce( 'wcpob_return_' . $sid ),
			),
			home_url( '/' )
		);

		$btn = '<div class="wcpob-return-wrap" style="margin-top:12px; text-align:right;">'
			. '<a class="button alt wcpob-return-btn" style="display:inline-block;background:#111;color:#fff;padding:12px 18px;text-decoration:none;" href="' . esc_url( $url ) . '">Return to Procurement</a>'
			. '</div>';

		$name    = $block['blockName'] ?? '';
		$targets = array(
			'woocommerce/cart-totals-block',
			'woocommerce/cart-order-summary-block',
			'woocommerce/cart-totals',
			'woocommerce/cart-order-summary',
			'woocommerce/cart',
		);

		if ( in_array( $name, $targets, true ) && strpos( $content, 'wcpob-return-btn' ) === false ) {
			$content .= $btn;
		}

		return $content;
	},
	20,
	2
);

/**
 * Return handler (front) - posts PunchOutOrderMessage
 */
add_action(
	'template_redirect',
	function () {
		if ( ! isset( $_GET['wcpob_return'], $_GET['sid'] ) ) {
			return;
		}

		if ( ! function_exists( 'WC' ) ) {
			wp_send_json( array( 'error' => 'woocommerce not available' ), 500 );
		}

		$sid   = sanitize_text_field( wp_unslash( $_GET['sid'] ) );
		$rt    = sanitize_text_field( wp_unslash( $_GET['rt'] ?? '' ) );
		$nonce = sanitize_text_field( wp_unslash( $_GET['nonce'] ?? '' ) );

		// Nonce recommended by plugin check.
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wcpob_return_' . $sid ) ) {
			wp_send_json( array( 'error' => 'invalid nonce' ), 403 );
		}

		if ( ! WC()->session ) {
			wp_send_json( array( 'error' => 'no session' ), 403 );
		}
		if ( ! $rt || $rt !== (string) WC()->session->get( 'wcpob_return_token' ) ) {
			wp_send_json( array( 'error' => 'invalid return token' ), 403 );
		}

		$row = wcpob_get_session_by_sid( $sid );
		if ( ! $row ) {
			wp_send_json( array( 'error' => 'unknown sid' ), 404 );
		}
		if ( strtotime( $row->expires_at ) < time() ) {
			wp_send_json( array( 'error' => 'expired' ), 410 );
		}

		if ( ! WC()->cart && function_exists( 'wc_load_cart' ) ) {
			wc_load_cart();
		}
		if ( ! WC()->cart ) {
			wp_send_json( array( 'error' => 'woocommerce cart not available' ), 500 );
		}

		$items = array();
		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$p       = $cart_item['data'];
			$items[] = array(
				'sku'   => $p->get_sku() ?: ( 'product_' . $p->get_id() ),
				'name'  => $p->get_name(),
				'qty'   => (int) $cart_item['quantity'],
				'price' => (float) $p->get_price( 'edit' ),
			);
		}

		$meta       = wcpob_session_meta( $row );
		$buyerCookie = (string) ( $meta['buyer_cookie'] ?? '' );
		if ( $buyerCookie === '' ) {
			$buyerCookie = 'cookie-missing';
		}

		$currency = function_exists( 'get_woocommerce_currency' ) ? get_woocommerce_currency() : 'EUR';
		$total    = (float) WC()->cart->get_cart_contents_total()
			+ (float) WC()->cart->get_shipping_total()
			+ (float) WC()->cart->get_total_tax();

		$xml = wcpob_build_order_message( $sid, $buyerCookie, $items, $total, $currency );

		$returnUrl = (string) ( $row->return_url ?? '' );
		if ( ! $returnUrl ) {
			wp_send_json( array( 'error' => 'missing return_url' ), 500 );
		}

		wcpob_log(
			array(
				'buyer_id'       => (int) $row->buyer_id,
				'session_id'     => (int) $row->id,
				'direction'      => 'out',
				'event'          => 'cart_return',
				'message'        => 'Posting PunchOutOrderMessage',
				'payload_excerpt'=> null,
				'payload_hash'   => hash( 'sha256', $xml ),
			)
		);

		$resp = wp_remote_post(
			$returnUrl,
			array(
				'timeout'            => 20,
				'reject_unsafe_urls' => true,
				'headers'            => array( 'Content-Type' => 'text/xml; charset=UTF-8' ),
				'body'               => $xml,
			)
		);

		$status = 'error';
		$code   = 0;
		if ( ! is_wp_error( $resp ) ) {
			$code = (int) wp_remote_retrieve_response_code( $resp );
			if ( $code >= 200 && $code < 300 ) {
				$status = 'returned';
			}
		}

		global $wpdb;
		$t = wcpob_table_sessions();

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update( $t, array( 'status' => $status ), array( 'id' => $row->id ) );

		wp_cache_delete( 'wcpob_session_' . md5( $sid ), 'wcpob_lite' );

		$resp_body = is_wp_error( $resp ) ? '' : (string) wp_remote_retrieve_body( $resp );

		wcpob_log(
			array(
				'buyer_id'     => (int) $row->buyer_id,
				'session_id'   => (int) $row->id,
				'direction'    => 'in',
				'event'        => 'cart_return_response',
				'severity'     => ( $status === 'returned' ) ? 'info' : 'error',
				'http_status'  => ( $code ?: null ),
				'message'      => ( $status === 'returned' ) ? 'Return OK' : ( is_wp_error( $resp ) ? $resp->get_error_message() : 'Return failed' ),
				'payload_excerpt' => null,
				'payload_hash' => ( $resp_body !== '' ) ? hash( 'sha256', $resp_body ) : null,
			)
		);

		wp_send_json(
			array(
				'posted_to'    => $returnUrl,
				'http_status'  => $code,
				'status'       => $status,
				'items_count'  => count( $items ),
			),
			( $status === 'returned' ) ? 200 : 502
		);
	},
	1
);