<?php
/**
 * Punchr Lite - Admin
 *
 * @package punchr-lite
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin notice if WooCommerce missing
 */
add_action(
	'admin_notices',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( ! class_exists( 'WooCommerce' ) ) {
			echo '<div class="notice notice-error"><p>';
			echo '<strong>Punchr Lite</strong> requires <strong>WooCommerce</strong> to be installed and active.';
			echo '</p></div>';
		}
	}
);

/**
 * Internal: whitelist table names
 */
function wcpob_lite_is_allowed_table_name( string $table ): bool {
	global $wpdb;

	$allowed = array(
		$wpdb->prefix . 'wcpob_logs',
		$wpdb->prefix . 'wcpob_sessions',
		$wpdb->prefix . 'wcpob_buyers',
	);

	return in_array( $table, $allowed, true );
}

/**
 * Internal: safe buyers table name (whitelisted).
 */
function wcpob_lite_buyers_table(): string {
	$table = wcpob_lite_get_table_name( 'buyers' );
	if ( '' === $table ) {
		return '';
	}
	if ( ! wcpob_lite_is_allowed_table_name( $table ) ) {
		return '';
	}
	return $table;
}

function wcpob_generate_token(): string {
	return 'buyer_' . wp_generate_password( 16, false, false );
}

function wcpob_generate_secret(): string {
	// ERP / shell safe (hex)
	return bin2hex( random_bytes(16) ); // 32 hex chars
}

/**
 * Secret: display once (transient per admin user)
 */
function wcpob_lite_secret_transient_key( int $user_id ): string {
	return 'wcpob_lite_new_secret_' . $user_id;
}

function wcpob_lite_store_secret_once( string $secret ): void {
	$uid = get_current_user_id();
	if ( $uid <= 0 ) {
		return;
	}
	set_transient( wcpob_lite_secret_transient_key( $uid ), $secret, 2 * MINUTE_IN_SECONDS );
}

function wcpob_lite_pop_secret_once(): ?string {
	$uid = get_current_user_id();
	if ( $uid <= 0 ) {
		return null;
	}

	$key    = wcpob_lite_secret_transient_key( $uid );
	$secret = get_transient( $key );

	if ( false !== $secret ) {
		delete_transient( $key ); // one-shot
		return (string) $secret;
	}

	return null;
}

/**
 * Get (or create) the single buyer row
 */
function wcpob_get_or_create_single_buyer_id(): int {
	global $wpdb;

	$buyers_table = wcpob_lite_buyers_table();
	if ( '' === $buyers_table ) {
		return 0;
	}

	$cache_key   = 'wcpob_lite_single_buyer_id';
	$cache_group = 'wcpob_lite';

	$cached = wp_cache_get( $cache_key, $cache_group );
	if ( false !== $cached ) {
		return (int) $cached;
	}

	// Table name is whitelisted; safe to inline.
	$id = (int) $wpdb->get_var(
		$wpdb->prepare(
			'SELECT id FROM %i WHERE deleted_at IS NULL ORDER BY id ASC LIMIT 1',
			$buyers_table
		)
	);

	if ( $id > 0 ) {
		wp_cache_set( $cache_key, $id, $cache_group, 60 );
		return $id;
	}

	$token  = wcpob_generate_token();
	$secret = wcpob_generate_secret();

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	$wpdb->insert(
		$buyers_table,
		array(
			'name'          => 'Default Buyer (Lite)',
			'token_id'      => $token,
			'catalog_rules' => wp_json_encode(
				array(
					'secret_enc' => wcpob_encrypt( $secret ),
				)
			),
			'is_enabled'    => 1,
		)
	);

	$new_id = (int) $wpdb->insert_id;
	wp_cache_set( $cache_key, $new_id, $cache_group, 60 );
	return $new_id;
}

/**
 * Admin menus
 */
add_action(
	'admin_menu',
	function () {
		add_menu_page(
			'Punchr Lite',
			'Punchr Lite',
			'manage_options',
			'wcpob-lite',
			'wcpob_lite_admin_page',
			'dashicons-randomize',
			56
		);

		add_submenu_page(
			'wcpob-lite',
			'Logs',
			'Logs',
			'manage_options',
			'wcpob-lite-logs',
			'wcpob_lite_admin_logs_page'
		);

		add_submenu_page(
			'wcpob-lite',
			'Upgrade to Pro',
			'Upgrade to Pro',
			'manage_options',
			'wcpob-lite-upgrade',
			'wcpob_lite_admin_upgrade_page'
		);
	}
);

/**
 * Regenerate secret action (nonce protected)
 */
add_action(
	'admin_post_wcpob_lite_regen_secret',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Forbidden', 403 );
		}

		check_admin_referer( 'wcpob_lite_regen_secret' );

		global $wpdb;

		$buyers_table = wcpob_lite_buyers_table();
		if ( '' === $buyers_table ) {
		wp_die( 'Invalid buyers table', 500 );
		}

		// Single buyer is always resolved through the cached helper.
		$buyer_id = wcpob_get_or_create_single_buyer_id();


		$cache_key   = 'wcpob_lite_buyer_' . (int) $buyer_id;
		$cache_group = 'wcpob_lite';

		$buyer = wp_cache_get( $cache_key, $cache_group );
		if ( false === $buyer ) {
			// Custom plugin table query (no WP API available).
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$buyer = $wpdb->get_row(
				$wpdb->prepare(
					'SELECT * FROM %i WHERE id = %d',
					$buyers_table,
					$buyer_id
				)
			);
			wp_cache_set( $cache_key, $buyer, $cache_group, 60 );
		}

		if ( ! $buyer ) {
			wp_die( 'Buyer not found', 404 );
		}


		$catalog = array();
		if ( ! empty( $buyer->catalog_rules ) ) {
			$catalog = json_decode( (string) $buyer->catalog_rules, true );
			if ( ! is_array( $catalog ) ) {
				$catalog = array();
			}
		}

		$new_secret            = wcpob_generate_secret();
		$catalog['secret_enc'] = wcpob_encrypt( $new_secret );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update(
			$buyers_table,
			array(
				'catalog_rules' => wp_json_encode( $catalog ),
			),
			array( 'id' => $buyer_id )
		);

		// Invalidate caches (buyer + single buyer id)
		wp_cache_delete( 'wcpob_lite_single_buyer_id', $cache_group );
		wp_cache_delete( $cache_key, $cache_group );

		// Store secret once and redirect
		wcpob_lite_store_secret_once( $new_secret );

		$url = add_query_arg(
			array(
				'page'  => 'wcpob-lite',
				'regen' => 1,
			),
			admin_url( 'admin.php' )
		);

		wp_safe_redirect( $url );
		exit;
	}
);

/**
 * Settings page
 */
function wcpob_lite_admin_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	global $wpdb;

	$buyers_table = wcpob_lite_buyers_table();
  if ( '' === $buyers_table ) {
    echo '<div class="wrap"><h1>Punchr Lite</h1><p>Invalid buyers table.</p></div>';
    return;
  }

  // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
  $id = (int) $wpdb->get_var(
    $wpdb->prepare(
      'SELECT id FROM %i WHERE deleted_at IS NULL ORDER BY id ASC LIMIT 1',
      $buyers_table
    )
  );

	$buyer_id = wcpob_get_or_create_single_buyer_id();

	$cache_key   = 'wcpob_lite_buyer_' . (int) $buyer_id;
	$cache_group = 'wcpob_lite';

	$buyer = wp_cache_get( $cache_key, $cache_group );
	if ( false === $buyer ) {
		$buyer = $wpdb->get_row(
			$wpdb->prepare(
				'SELECT * FROM %i WHERE id = %d',
				$buyers_table,
				$buyer_id
			)
		);
	
		wp_cache_set( $cache_key, $buyer, $cache_group, 60 );
	}

	if ( ! $buyer ) {
		echo '<div class="wrap"><h1>Punchr Lite</h1><p>Buyer not found.</p></div>';
		return;
	}

	/**
	 * Save name + enabled only
	 */
	if ( isset( $_POST['wcpob_lite_save'] ) && check_admin_referer( 'wcpob_lite_save' ) ) {
		$name       = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : 'Default Buyer (Lite)';
		$is_enabled = isset( $_POST['is_enabled'] ) ? 1 : 0;

		$catalog = array();
		if ( ! empty( $buyer->catalog_rules ) ) {
			$catalog = json_decode( (string) $buyer->catalog_rules, true );
			if ( ! is_array( $catalog ) ) {
				$catalog = array();
			}
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update(
			$buyers_table,
			array(
				'name'          => $name,
				'is_enabled'    => $is_enabled,
				'catalog_rules' => wp_json_encode( $catalog ),
			),
			array( 'id' => $buyer_id )
		);

		// Refresh cache
		wp_cache_delete( $cache_key, $cache_group );

		// Re-fetch buyer (cached)
		$buyer = $wpdb->get_row(
			$wpdb->prepare(
				'SELECT * FROM %i WHERE id = %d',
				$buyers_table,
				$buyer_id
			)
		);

wp_cache_set( $cache_key, $buyer, $cache_group, 60 );

		echo '<div class="notice notice-success"><p>Saved.</p></div>';
	}

	echo '<div class="wrap"><h1>Punchr Lite</h1>';
	// Trial status banner
	$days_left = wcpob_lite_trial_days_left();
	$expired   = wcpob_lite_is_trial_expired();

	if ( $expired ) {
		echo '<div class="notice notice-error"><p><strong>Punchr Lite — Evaluation expired.</strong> ';
		echo 'PunchOut setup requests are blocked. ';
		echo '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-upgrade' ) ) . '">Upgrade to Punchr Pro</a> to reactivate instantly.</p></div>';
	} else {
		echo '<div class="notice notice-info"><p><strong>Punchr Lite — Evaluation mode.</strong> ';
		echo 'Days left: <strong>' . (int) $days_left . '</strong>. ';
		echo 'Not intended for production use. ';
		echo '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-upgrade' ) ) . '">Upgrade to Punchr Pro</a> for production.</p></div>';
	}


	// Show regenerated secret once
	if ( ! empty( $_GET['regen'] ) ) {
		$shown = wcpob_lite_pop_secret_once();
		if ( $shown ) {
			echo '<div class="notice notice-warning"><p><b>New secret generated:</b> <code>' . esc_html( $shown ) . '</code> (copy it now)</p>';
			echo '<p style="margin:6px 0 0;color:#b32d2e;"><b>This secret will not be shown again.</b></p></div>';
		} else {
			echo '<div class="notice notice-info"><p>Secret already displayed (or expired). Regenerate again if needed.</p></div>';
		}
	}

	$regen_url = wp_nonce_url(
		admin_url( 'admin-post.php?action=wcpob_lite_regen_secret' ),
		'wcpob_lite_regen_secret'
	);

	echo '<div class="card" style="max-width:900px;">';
	echo '<h2>Single Buyer (Lite)</h2>';
	echo '<p><b>Status:</b> ' . ( wcpob_lite_is_trial_expired() ? '<span style="color:#b32d2e;"><b>Expired</b></span>' : '<span style="color:#1d2327;"><b>Active (evaluation)</b></span>' ) . '</p>';

	echo '<form method="post">';
	wp_nonce_field( 'wcpob_lite_save' );

	echo '<p><label><b>Name</b><br>';
	echo '<input type="text" name="name" style="width:420px;" value="' . esc_attr( $buyer->name ) . '"></label></p>';

	echo '<p><label><input type="checkbox" name="is_enabled" value="1" ' . checked( (int) $buyer->is_enabled, 1, false ) . '> Enabled</label></p>';

	echo '<h3>Credentials</h3>';
	echo '<p><b>Token:</b> <code>' . esc_html( $buyer->token_id ) . '</code></p>';
	echo '<p><b>Secret:</b> <code>••••••••••••••••</code></p>';

	echo '<p><button class="button button-primary" name="wcpob_lite_save" value="1">Save</button></p>';
	echo '</form>';

	echo '<hr style="margin:18px 0;">';
	echo '<h3>How to integrate</h3>';

	// ✅ endpoints changed to /punchr/v1
	$setup  = home_url( '/wp-json/punchr/v1/setup' );
	$start  = home_url( '/wp-json/punchr/v1/start' );

	echo '<p><b>Endpoints</b></p>';
	echo '<ul style="list-style:disc;padding-left:20px;">';
	echo '<li><code>POST ' . esc_html( $setup ) . '</code> (PunchOutSetupRequest)</li>';
	echo '<li><code>GET  ' . esc_html( $start ) . '?sid=...&st=...</code> (Start)</li>';
	echo '</ul>';

	echo '<p><b>Authentication for /setup</b> (ERP-friendly)</p>';
	echo '<p>Use <b>HTTP Basic Authentication</b>:</p>';
	echo '<ul style="list-style:disc;padding-left:20px;">';
	echo '<li><b>Username</b> = <code>TOKEN</code></li>';
	echo '<li><b>Password</b> = <code>SECRET</code></li>';
	echo '</ul>';

	echo '<p><b>Example curl</b></p>';
	echo '<pre style="background:#f6f7f7;padding:12px;border:1px solid #dcdcde;max-width:900px;overflow:auto;">';
	echo esc_html(
	"curl -u \"TOKEN:SECRET\" -X POST \"" . $setup . "\" \\\n" .
	"  -H \"Content-Type: text/xml; charset=UTF-8\" \\\n" .
	"  --data-binary @punchout-setup.xml"
	);
	echo '</pre>';
	echo '<hr style="margin:18px 0;">';
	echo '<h3>Support</h3>';
	echo '<p>If you encounter a bug in Punchr Lite, email <a href="mailto:bugs@punchr.net">bugs@punchr.net</a>.</p>';
	echo '<p style="color:#646970;margin-top:6px;">For ERP integration assistance and diagnostics, Punchr Pro will include priority support.</p>';

	echo '<hr style="margin:16px 0;">';
	echo '<p style="margin:0;">';
	echo '<a class="button" style="border-color:#b32d2e;color:#b32d2e;" href="' . esc_url( $regen_url ) . '" ';
	echo 'onclick="return confirm(\'Regenerate secret? The old one will stop working immediately.\');">';
	echo 'Regenerate secret';
	echo '</a>';
	echo '</p>';

	echo '</div>';

	echo '<p style="margin-top:16px;"><a class="button" href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-logs' ) ) . '">View logs</a></p>';

	echo '<p style="margin-top:18px;">';
	echo '<a href="' . esc_url( admin_url( 'admin.php?page=wcpob-lite-upgrade' ) ) . '">Upgrade to Pro</a>';
	echo '</p>';

	echo '</div>';
}

/**
 * Logs page
 */
function wcpob_lite_admin_logs_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	global $wpdb;

	$logs_table = wcpob_lite_get_table_name( 'logs' );
	if ( ! wcpob_lite_is_allowed_table_name( $logs_table ) ) {
		echo '<div class="wrap"><h1>Punchr Lite - Logs</h1><p>Invalid table.</p></div>';
		return;
	}

	$cache_key   = 'wcpob_lite_logs_last200';
	$cache_group = 'wcpob_lite';

	$rows = wp_cache_get( $cache_key, $cache_group );
  if ( false === $rows ) {
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $rows = $wpdb->get_results(
        $wpdb->prepare(
            'SELECT id, created_at, event, severity, http_status, message
             FROM %i
             ORDER BY id DESC
             LIMIT %d',
            $logs_table,
            200
        ),
        ARRAY_A
    );

    wp_cache_set( $cache_key, $rows, $cache_group, 30 );
}

	echo '<div class="wrap">';
	echo '<h1>Punchr Lite – Logs</h1>';

	echo '<p style="margin-top:6px;">';
	echo 'Found a bug? Email <a href="mailto:bugs@punchr.net">bugs@punchr.net</a> with the log line ID and timestamp.';
	echo '</p>';

	echo '<table class="widefat striped"><thead><tr>';
	echo '<th>ID</th><th>Created</th><th>Event</th><th>Severity</th><th>HTTP</th><th>Message</th>';
	echo '</tr></thead><tbody>';

	if ( empty( $rows ) ) {
		echo '<tr><td colspan="6"><em>No logs yet.</em></td></tr>';
	} else {
		foreach ( $rows as $r ) {
			echo '<tr>';
			echo '<td>' . (int) $r['id'] . '</td>';
			echo '<td>' . esc_html( $r['created_at'] ) . '</td>';
			echo '<td><code>' . esc_html( $r['event'] ) . '</code></td>';
			echo '<td>' . esc_html( $r['severity'] ) . '</td>';
			echo '<td>' . esc_html( $r['http_status'] ?? '' ) . '</td>';
			echo '<td>' . esc_html( $r['message'] ) . '</td>';
			echo '</tr>';
		}
	}

	echo '</tbody></table></div>';
}

/**
 * Upgrade page
 */
function wcpob_lite_admin_upgrade_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	echo '<div class="wrap">';
	echo '<h1>Upgrade to Punchr Pro</h1>';

	echo '<p>Punchr Lite is designed for a single Buyer and a basic PunchOut flow.</p>';
	echo '<p><strong>Punchr Pro</strong> will add advanced features for production and enterprise environments.</p>';

	$expired = function_exists('wcpob_lite_is_trial_expired') ? wcpob_lite_is_trial_expired() : false;
	$left    = function_exists('wcpob_lite_trial_days_left') ? wcpob_lite_trial_days_left() : 0;

	if ( $expired ) {
		echo '<div class="notice notice-error"><p><strong>Evaluation expired.</strong> PunchOut setup requests are currently blocked in Lite.</p></div>';
	} else {
		echo '<div class="notice notice-info"><p><strong>Evaluation active.</strong> Days left: <strong>' . (int) $left . '</strong>.</p></div>';
	}


	echo '<h2>Lite vs Pro</h2>';
	echo '<table class="widefat striped" style="max-width:900px;">';
	echo '<thead><tr><th>Feature</th><th>Lite</th><th>Pro</th></tr></thead><tbody>';
	echo '<tr><td>Single Buyer</td><td>✅</td><td>✅</td></tr>';
	echo '<tr><td>Multiple Buyers</td><td>—</td><td>✅</td></tr>';
	echo '<tr><td>Buyer-specific credentials</td><td>—</td><td>✅</td></tr>';
	echo '<tr><td>Advanced catalog / pricing rules</td><td>—</td><td>✅</td></tr>';
	echo '<tr><td>Extended logs & diagnostics</td><td>—</td><td>✅</td></tr>';
	echo '<tr><td>Priority support</td><td>—</td><td>✅</td></tr>';
	echo '</tbody></table>';

	echo '<h2>Interested?</h2>';
	echo '<p>Punchr Pro is available. Choose an option below:</p>';

	echo '<p style="margin:10px 0 0;">';
	echo '<a class="button button-primary" style="margin-right:6px;" href="https://punchr.net" target="_blank" rel="noopener noreferrer">';
	echo 'Discover Punchr Pro';
	echo '</a>';  echo '<a class="button" href="' . esc_url( 'mailto:contact@punchr.net' ) . '">Email contact@punchr.net</a>';
	echo '</p>';


	echo '</div>';
}

/**
 * Privacy policy snippet
 */
add_action(
	'admin_init',
	function () {
		if ( ! function_exists( 'wp_add_privacy_policy_content' ) ) {
			return;
		}

		$content = wp_kses_post(
			'<p><strong>Punchr Lite – WooCommerce PunchOut Bridge</strong></p>' .
			'<p>This plugin stores limited diagnostic data to help troubleshoot PunchOut sessions.</p>' .
			'<ul>' .
				'<li>IP address and User-Agent for log entries.</li>' .
				'<li>Technical event information (timestamp, event name, HTTP status, message).</li>' .
				'<li>A SHA-256 hash of some XML payloads (payload content is not stored by default).</li>' .
			'</ul>' .
			'<p>Log data is stored locally in your WordPress database in custom tables created by the plugin. ' .
			'The plugin does not send log data to the plugin author or any third-party service.</p>'
		);

		wp_add_privacy_policy_content( 'Punchr Lite – WooCommerce PunchOut Bridge', $content );
	}
);