<?php
/**
 * Publisher Class
 *
 * @package Pubsubhubbub
 */

namespace Pubsubhubbub;

/**
 * Publisher Class
 *
 * Handles publishing feed updates to WebSub hubs.
 *
 * @package Pubsubhubbub
 */
class Publisher {

	/**
	 * Function that is called whenever a new post is published.
	 *
	 * @param int $post_id The post ID.
	 *
	 * @return void
	 */
	public static function publish_post( $post_id ) {
		// We want to notify the hub for every feed.
		$feed_urls = self::get_feed_urls_by_post_id( $post_id );

		// Publish them.
		self::publish_to_hub( $feed_urls );
	}

	/**
	 * Function that is called whenever a new comment is published.
	 *
	 * @param int $comment_id The comment ID.
	 *
	 * @return void
	 */
	public static function publish_comment( $comment_id ) {
		// Get default comment-feeds.
		$feed_urls   = array();
		$feed_urls[] = \get_bloginfo( 'comments_atom_url' );
		$feed_urls[] = \get_bloginfo( 'comments_rss2_url' );

		/**
		 * Filter the list of comment feed URLs to publish to the hub.
		 *
		 * @since 4.0.0
		 *
		 * @param array $feed_urls  List of comment feed URLs.
		 * @param int   $comment_id The comment ID that triggered the publish.
		 */
		$feed_urls = \apply_filters_deprecated( 'pubsubhubbub_comment_feed_urls', array( $feed_urls, $comment_id ), '4.0.0', 'websub_comment_feed_urls' );
		$feed_urls = \apply_filters( 'websub_comment_feed_urls', $feed_urls, $comment_id );

		// Publish them.
		self::publish_to_hub( $feed_urls );
	}

	/**
	 * Accepts either a single url or an array of urls.
	 *
	 * @param string|array $topic_urls A single topic url or an array of topic urls.
	 * @param string       $hub_url    The hub URL to publish to.
	 *
	 * @return array|\WP_Error The response or WP_Error on failure.
	 */
	public static function publish_update( $topic_urls, $hub_url ) {
		if ( ! isset( $hub_url ) ) {
			return new \WP_Error( 'missing_hub_url', \__( 'Please specify a hub url', 'pubsubhubbub' ) );
		}

		if ( ! \preg_match( '|^https?://|i', $hub_url ) ) {
			/* translators: %s is the $hub_url */
			return new \WP_Error( 'invalid_hub_url', \sprintf( \__( 'The specified hub url does not appear to be valid: %s', 'pubsubhubbub' ), \esc_url( $hub_url ) ) );
		}

		if ( ! isset( $topic_urls ) ) {
			return new \WP_Error( 'missing_topic_url', \__( 'Please specify a topic url', 'pubsubhubbub' ) );
		}

		// Check that we're working with an array.
		if ( ! \is_array( $topic_urls ) ) {
			$topic_urls = array( $topic_urls );
		}

		// Set the mode to publish.
		$post_string = 'hub.mode=publish';

		// Loop through each topic url.
		foreach ( $topic_urls as $topic_url ) {
			// Lightweight check that we're actually working w/ a valid url.
			if ( \preg_match( '|^https?://|i', $topic_url ) ) {
				// Append the topic url parameters (URL-encoded per WebSub spec).
				$post_string .= '&hub.url=' . \rawurlencode( $topic_url );
			}
		}

		$wp_version = \get_bloginfo( 'version' );
		$user_agent = \apply_filters( 'http_headers_useragent', 'WordPress/' . $wp_version . '; ' . \get_bloginfo( 'url' ) );

		$args = array(
			'timeout'             => 100,
			'limit_response_size' => 1048576,
			'redirection'         => 20,
			'user-agent'          => "$user_agent; PubSubHubbub/WebSub",
			'headers'             => array(
				'content-type' => 'application/x-www-form-urlencoded',
			),
			'body'                => $post_string,
		);

		// Make the http post request.
		return \wp_remote_post( $hub_url, $args );
	}

	/**
	 * The ability for other plugins to hook into the PuSH code.
	 *
	 * @param array $feed_urls A list of feed urls you want to publish.
	 *
	 * @return void
	 */
	public static function publish_to_hub( $feed_urls ) {
		// Remove dups (ie. they all point to feedburner).
		$feed_urls = \array_unique( $feed_urls );

		// Get the list of hubs.
		$hub_urls = self::get_hubs();

		// Loop through each hub.
		foreach ( $hub_urls as $hub_url ) {
			// Publish the update to each hub.
			$response = self::publish_update( $feed_urls, $hub_url );

			/**
			 * Fires after publishing an update to a hub.
			 *
			 * @since 4.0.0
			 *
			 * @param array|\WP_Error $response The HTTP response or WP_Error on failure.
			 */
			\do_action_deprecated( 'pubsubhubbub_publish_update_response', array( $response ), '4.0.0', 'websub_publish_update_response' );
			\do_action( 'websub_publish_update_response', $response );
		}
	}

	/**
	 * Get the endpoints from the WordPress options table.
	 *
	 * @uses apply_filters() Calls 'websub_hub_urls' filter.
	 *
	 * @return array The hub URLs.
	 */
	public static function get_hubs() {
		$endpoints = \get_option( 'pubsubhubbub_endpoints' );

		// Fail with an empty array when option retrieval fails.
		if ( ! $endpoints ) {
			$hub_urls = Pubsubhubbub::DEFAULT_HUBS;
		} else {
			$hub_urls = \explode( PHP_EOL, $endpoints );
		}

		/**
		 * Filter the list of hub URLs to publish updates to.
		 *
		 * @since 4.0.0
		 *
		 * @param array $hub_urls List of hub URLs.
		 */
		$hub_urls = \apply_filters_deprecated( 'pubsubhubbub_hub_urls', array( $hub_urls ), '4.0.0', 'websub_hub_urls' );
		$hub_urls = \apply_filters( 'websub_hub_urls', $hub_urls );

		return $hub_urls;
	}

	/**
	 * Returns a list of feed URLs for a given Post.
	 *
	 * @param int $post_id The post ID.
	 *
	 * @return array An array of feed URLs.
	 */
	public static function get_feed_urls_by_post_id( $post_id ) {
		$post      = \get_post( $post_id );
		$feed_urls = array();

		$feed_types = get_supported_feed_types();

		foreach ( $feed_types as $feed_type ) {
			$feed_urls[] = \get_feed_link( $feed_type );

			// Add tag-feeds.
			$tags = \wp_get_post_tags( $post_id );

			foreach ( $tags as $tag ) {
				$feed_urls[] = \get_term_feed_link( $tag->term_id, 'post_tag', $feed_type );
			}

			// Add category-feeds.
			$categories = \wp_get_post_categories( $post_id );

			foreach ( $categories as $category ) {
				$feed_urls[] = \get_term_feed_link( $category, 'category', $feed_type );
			}

			// Add author-feeds.
			$feed_urls[] = \get_author_feed_link( $post->post_author, $feed_type );
		}

		if ( \current_theme_supports( 'microformats2' ) ) {
			$feed_urls[] = \site_url( '/' );
		}

		/**
		 * Filter the list of feed URLs to publish to the hub for a post.
		 *
		 * @since 4.0.0
		 *
		 * @param array $feed_urls List of feed URLs.
		 * @param int   $post_id   The post ID that triggered the publish.
		 */
		$feed_urls = \apply_filters_deprecated( 'pubsubhubbub_feed_urls', array( $feed_urls, $post_id ), '4.0.0', 'websub_feed_urls' );
		$feed_urls = \apply_filters( 'websub_feed_urls', $feed_urls, $post_id );

		return $feed_urls;
	}
}
