/* global jQuery, wp */
(function ($) {
  'use strict';

  // ---- i18n fallback -------------------------------------------------------
  const { __ } = window.wp && window.wp.i18n ? window.wp.i18n : { __: (s) => s };

  // ---- helpers --------------------------------------------------------------
  // Escape text for safe insertion into HTML text nodes
  function esc(s) {
    if (s == null) return '';
    return String(s).replace(/[&<>"']/g, (c) =>
      ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[c])
    );
  }

  // Safe URL helper with stricter validation
  function safeUrl(u) {
    if (!u) return '#';
    try {
      const url = new URL(String(u), window.location.origin);
      // Only allow http/https protocols
      if (url.protocol !== 'http:' && url.protocol !== 'https:') {
        return '#';
      }
      return url.href;
    } catch (err) {
      return '#';
    }
  }

  // Scroll to bottom (single read of element)
  function scrollToBottom($el) {
    const el = $el[0];
    if (el) el.scrollTop = el.scrollHeight;
  }

  // Create external link attrs
  const EXT_LINK = ' target="_blank" rel="noopener noreferrer nofollow" ';

  // Safe integer parser
  function safeInt(val, defaultVal = 0) {
    const parsed = parseInt(val, 10);
    return isNaN(parsed) ? defaultVal : parsed;
  }

  // --------------------------------------------------------------------------
  // Conversation Memory Module (v1.1.0) - with Service Accumulation
  // --------------------------------------------------------------------------
  const ConversationMemory = {
    STORAGE_KEY: 'promptor_chat_v1',
    VERSION: '1.1.0',
    MAX_MESSAGES: 15,

    // Get conversation state from sessionStorage
    load(contextKey) {
      try {
        const key = this.STORAGE_KEY + '_' + contextKey;
        const raw = sessionStorage.getItem(key);
        if (!raw) return null;

        const data = JSON.parse(raw);

        // Version check
        if (data.version !== this.VERSION) {
          this.clear(contextKey);
          return null;
        }

        // Validate structure
        if (!data.messages || !Array.isArray(data.messages)) {
          return null;
        }

        return data;
      } catch (err) {
        return null;
      }
    },

    // Save full state to sessionStorage
    save(contextKey, messages, services) {
      try {
        const key = this.STORAGE_KEY + '_' + contextKey;

        // Keep only last N messages (FIFO)
        const trimmedMessages = messages.slice(-this.MAX_MESSAGES);

        const data = {
          version: this.VERSION,
          updated_at: Math.floor(Date.now() / 1000),
          messages: trimmedMessages,
          services: services || []
        };

        sessionStorage.setItem(key, JSON.stringify(data));
        return true;
      } catch (err) {
        // Storage might be full or disabled
        return false;
      }
    },

    // Add a new message
    addMessage(contextKey, role, content) {
      const state = this.load(contextKey) || { messages: [], services: [] };

      state.messages.push({
        role: role, // 'user' or 'assistant'
        content: content,
        ts: Math.floor(Date.now() / 1000)
      });

      this.save(contextKey, state.messages, state.services);
    },

    // Add a service (with duplicate prevention)
    addService(contextKey, service) {
      if (!service || !service.title) return false;

      const state = this.load(contextKey) || { messages: [], services: [] };

      // Generate unique ID from title
      const serviceId = service.id || service.title.toLowerCase().replace(/[^a-z0-9]+/g, '-');

      // Check if already exists
      const exists = state.services.some(s => {
        const existingId = s.id || s.title.toLowerCase().replace(/[^a-z0-9]+/g, '-');
        return existingId === serviceId;
      });

      if (!exists) {
        state.services.push({
          id: serviceId,
          title: service.title,
          description: service.description || ''
        });
        this.save(contextKey, state.messages, state.services);
        return true;
      }

      return false; // Already exists
    },

    // Get accumulated services
    getServices(contextKey) {
      const state = this.load(contextKey);
      return state && state.services ? state.services : [];
    },

    // Clear conversation
    clear(contextKey) {
      try {
        const key = this.STORAGE_KEY + '_' + contextKey;
        sessionStorage.removeItem(key);
        return true;
      } catch (err) {
        return false;
      }
    },

    // Get messages for display
    getMessages(contextKey) {
      const state = this.load(contextKey);
      return state ? state.messages : [];
    }
  };

  // --------------------------------------------------------------------------

  function initializePromptorApp($app) {
    const contextKey = $app.data('context-key');
    const localizationObjectName = 'promptor_data_' + String(contextKey || '').replace(/-/g, '_');
    const promptorData = window[localizationObjectName];

    if (typeof promptorData === 'undefined') {
      // Fail closed with a safe message
      $app.html(
        '<p style="padding:20px;text-align:center;">' +
          esc(__('Chat could not be loaded. (Context Error)', 'promptor')) +
          '</p>'
      );
      return;
    }

    // Destructure server-localized data (shape guarded)
    const {
      ajax_url,
      ai_query_nonce,
      form_nonce,
      add_to_cart_nonce,
      feedback_nonce,
      example_questions = [],
      ui = {},
      context = '',
      i18n = {},
      enable_conversation_memory = 1, // Default: enabled
    } = promptorData || {};

    // Validate required nonces
    if (!ai_query_nonce || !form_nonce || !add_to_cart_nonce || !feedback_nonce) {
      $app.html(
        '<p style="padding:20px;text-align:center;">' +
          esc(__('Security validation failed. Please refresh the page.', 'promptor')) +
          '</p>'
      );
      return;
    }

    const ajaxEndpoint =
      ajax_url ||
      (window.promptor_public && window.promptor_public.ajax_url) ||
      (typeof window.ajaxurl !== 'undefined' ? window.ajaxurl : '');

    if (!ajaxEndpoint) {
      $app.html(
        '<p style="padding:20px;text-align:center;">' +
          esc(__('Configuration error. Please contact administrator.', 'promptor')) +
          '</p>'
      );
      return;
    }

    const botAvatarUrl = ui.bot_avatar_url || '';

    const $chatLog = $app.find('.promptor-chat-log');
    const $form = $app.find('.promptor-search-form');
    const $input = $app.find('.promptor-query-input');
    const $askButton = $app.find('.promptor-ask-btn');

    // a11y: ensure chat log announces updates
    if ($chatLog.length) {
      $chatLog.attr({ role: 'log', 'aria-live': 'polite', 'aria-relevant': 'additions' });
    }

    // Mobile keyboard handling - prevent keyboard from covering input
    if ('ontouchstart' in window || navigator.maxTouchPoints > 0) {
      let initialViewportHeight = window.innerHeight;

      $input.on('focus', function() {
        // On mobile, scroll input into view when keyboard appears
        setTimeout(function() {
          if (window.innerHeight < initialViewportHeight) {
            // Keyboard is visible - scroll input into view
            $input[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
            scrollToBottom($chatLog);
          }
        }, 300);
      });

      $input.on('blur', function() {
        initialViewportHeight = window.innerHeight;
      });

      // Update viewport height on resize (keyboard show/hide)
      $(window).on('resize.promptor-keyboard', function() {
        if (document.activeElement === $input[0] && window.innerHeight < initialViewportHeight) {
          setTimeout(function() {
            scrollToBottom($chatLog);
          }, 100);
        }
      });
    }

    // Inject Service Drawer (Gemini-style sticky accumulation UI)
    const drawerHtml = `
      <div id="promptor-service-drawer" class="promptor-service-drawer" style="display:none;">
        <div class="promptor-drawer-header">
          <h4>
            📌 <span>${esc(i18n.selectedServices || __('Selected Services', 'promptor'))}</span> (<span class="drawer-count">0</span>)
          </h4>
          <button class="promptor-drawer-collapse-btn" aria-label="${esc(i18n.toggleDrawer || __('Toggle drawer', 'promptor'))}">−</button>
        </div>
        <div class="promptor-drawer-content">
          <div class="promptor-service-chips"></div>
          <button class="promptor-request-quote-btn">${esc(i18n.requestQuote || __('Request Quote', 'promptor'))}</button>
        </div>
      </div>
    `;

    // Insert drawer between chat-log and form
    $chatLog.after(drawerHtml);
    const $drawer = $app.find('#promptor-service-drawer');

    let conversationHistory = [];
    let current_query_id = 0;
    const MAX_HISTORY_LENGTH = 50; // Prevent memory issues

    function populateUI() {
      const avatarUrl = safeUrl(botAvatarUrl);
      if (avatarUrl !== '#') {
        $app.find('.promptor-header-avatar').attr('src', avatarUrl);
      }
      $app.find('.promptor-header-title').text(ui.header_title || '');
      $app.find('.promptor-header-subtitle').text(ui.header_subtitle || '');
      $input.attr('placeholder', String(ui.input_placeholder || ''));
    }

    // Restore conversation from memory on page load
    function restoreConversation() {
      const messages = ConversationMemory.getMessages(context);

      if (messages && messages.length > 0) {
        messages.forEach(msg => {
          if (msg.role === 'user') {
            addToLog(msg.content, 'user', 0, false); // false = don't save again

            // If conversation memory is enabled, restore to context array
            if (enable_conversation_memory) {
              conversationHistory.push({ role: 'user', content: msg.content });
            }
          } else if (msg.role === 'assistant') {
            addToLog(msg.content, 'ai', 0, false); // false = don't save again

            // If conversation memory is enabled, restore to context array
            if (enable_conversation_memory) {
              conversationHistory.push({ role: 'assistant', content: msg.content });
            }
          }
        });
      }

      // Restore accumulated services to UI
      const allServices = ConversationMemory.getServices(context);

      if (allServices.length > 0) {
        // UNIFIED DESIGN: Always use conversation summary style (no conditional rendering)
        // Don't render old checkbox UI - always use modern chip-based UI
        addConversationSummary(allServices);
      }

      // Update drawer (hidden by default, for future manual triggers)
      updateServiceDrawer();
    }

    // Update Service Drawer (minimal, action-triggered only)
    function updateServiceDrawer() {
      const services = ConversationMemory.getServices(context);
      const count = services.length;

      $drawer.find('.drawer-count').text(count);

      if (count === 0) {
        $drawer.hide();
        return;
      }

      // Render service chips
      const chipsHtml = services.map(s => {
        const title = esc(s.title || '');
        const id = esc(s.id || '');
        return `<div class="promptor-service-chip" data-service-id="${id}">${title}</div>`;
      }).join('');

      $drawer.find('.promptor-service-chips').html(chipsHtml);
      // Note: Drawer is NOT auto-shown. Only shown on explicit user action.
    }

    // Add Conversation Summary (frontend intelligence - ChatGPT/Gemini style)
    // Now unified for ALL service counts (1 or multiple)
    function addConversationSummary(services, selectedServiceIds = []) {
      if (!services || services.length === 0) return;

      // Remove previous summary if exists (avoid duplicates)
      $chatLog.find('.promptor-conversation-summary').remove();

      // Determine intro text based on service count
      const introText = services.length === 1
        ? esc(i18n.conversationRecommendedService || __('Based on our conversation, here is the recommended service:', 'promptor'))
        : esc(i18n.conversationRecommendedServices || __('Based on our conversation so far, you need the following services:', 'promptor'));

      const closingText = services.length > 1
        ? esc(i18n.servicesTypicallyTogether || __('These services are typically handled together.', 'promptor'))
        : '';

      // Create inline chips (clickable, ChatGPT-style with selection state)
      const chipsHtml = services.map(s => {
        const title = esc(s.title || '');
        const id = esc(s.id || '');
        const isSelected = selectedServiceIds.includes(id);
        const selectedClass = isSelected ? ' chip-selected' : '';
        return `<span class="promptor-inline-chip${selectedClass}" data-service-id="${id}" data-service-title="${title}" data-service-description="${esc(s.description || '')}">${title}</span>`;
      }).join(' ');

      // Dynamic button text based on selection count
      const selectedCount = selectedServiceIds.length;
      let buttonText;
      if (selectedCount === 0) {
        buttonText = esc(i18n.selectServiceQuote || __('Select a service to request a quote', 'promptor'));
      } else if (selectedCount === 1) {
        buttonText = esc(i18n.requestQuote || __('Request Quote', 'promptor'));
      } else {
        buttonText = esc(i18n.requestQuoteForAll || __('Request Quote for All', 'promptor'));
      }

      const summaryHtml = `
        <div class="promptor-message-wrapper ai-message promptor-conversation-summary">
          <div class="promptor-message-bubble">
            <p><strong>${introText}</strong></p>
            <p class="promptor-summary-instruction">${esc(i18n.clickServiceDetails || __('Click on a service to see details and select:', 'promptor'))}</p>
            <p class="promptor-chips-container">${chipsHtml}</p>

            <!-- Description container (inline, no scroll issues) -->
            <div class="promptor-service-description-container" style="display:none;"></div>

            ${closingText ? `<p class="promptor-summary-note">${closingText}</p>` : ''}
            <button class="promptor-cta-request-quote" ${selectedCount === 0 ? 'disabled' : ''}>${buttonText}</button>
          </div>
        </div>
      `;

      $chatLog.append(summaryHtml);
      scrollToBottom($chatLog);
    }

    // Open Quote Form for accumulated services (all at once, not per-service)
    function openQuoteForm(services) {
      // Remove any existing quote forms to prevent duplicates
      $chatLog.find('.promptor-quote-form-fullwidth').remove();

      const nameLabel = esc(i18n.yourName || __('Your Name', 'promptor'));
      const emailLabel = esc(i18n.yourEmail || __('Your Email', 'promptor'));
      const phoneLabel = esc(i18n.yourPhone || __('Your Phone', 'promptor'));
      const notesLabel = esc(i18n.anythingElse || __('Anything else you would like to add?', 'promptor'));
      const submitLabel = esc(i18n.submitInquiry || __('Submit Inquiry', 'promptor'));

      // List of services being requested
      const servicesList = services.map(s => `<li>${esc(s.title)}</li>`).join('');

      const formHtml = `
        <div class="promptor-quote-form-fullwidth">
          <div class="promptor-quote-form-container">
            <h4>${esc(i18n.requestAQuote || __('Request a Quote', 'promptor'))}</h4>
            <p>${esc(i18n.servicesInterestedIn || __('Services you are interested in:', 'promptor'))}</p>
            <ul class="promptor-services-list">${servicesList}</ul>
            <form class="promptor-contact-form promptor-drawer-form" data-type="drawer-inquiry" novalidate>
              <div class="promptor-form-field"><label>${nameLabel}*</label><input type="text" name="name" required autocomplete="name"></div>
              <div class="promptor-form-field"><label>${emailLabel}*</label><input type="email" name="email" required autocomplete="email"></div>
              <div class="promptor-form-field"><label>${phoneLabel}</label><input type="tel" name="phone" inputmode="tel" autocomplete="tel"></div>
              <div class="promptor-form-field"><label>${notesLabel}</label><textarea name="notes" rows="3"></textarea></div>
              <button type="submit" class="promptor-submit-button">${submitLabel}</button>
            </form>
          </div>
        </div>
      `;

      $chatLog.append(formHtml);
      scrollToBottom($chatLog);

      // Focus first field
      $chatLog.find('.promptor-drawer-form input[name="name"]').trigger('focus');
    }

    function addToLog(text, sender, queryId = 0, saveToMemory = true) {
      const wrapperClass = sender === 'user' ? 'user-message' : 'ai-message';
      const safeContent = esc(String(text)).replace(/\n/g, '<br>');
      
      const avatarUrl = safeUrl(botAvatarUrl);
      const botAvatar = avatarUrl !== '#'
        ? `<div class="promptor-avatar ai-avatar"><img src="${avatarUrl}" alt="AI"></div>`
        : '<div class="promptor-avatar ai-avatar"></div>';

      const messageHtml =
        `<div class="promptor-message-wrapper ${wrapperClass}">` +
        (sender === 'ai' ? botAvatar : '') +
        `<div class="promptor-message-bubble">${safeContent}</div>` +
        `</div>`;

      $chatLog.append(messageHtml);

      const qid = safeInt(queryId);
      if (sender === 'ai' && qid > 0) {
        const fbText = esc(i18n.isHelpful || __('Is this conversation helpful so far?', 'promptor'));
        const good = esc(i18n.goodResponse || __('Good response', 'promptor'));
        const bad = esc(i18n.badResponse || __('Bad response', 'promptor'));
        const thanks = esc(i18n.thankYou || __('Thank you!', 'promptor'));

        const feedbackBarHtml = `
          <div class="promptor-session-feedback-bar" data-query-id="${qid}">
            <span class="feedback-text">${fbText}</span>
            <div class="feedback-actions">
              <button class="feedback-btn" data-feedback="1" title="${good}" aria-label="${good}">👍</button>
              <button class="feedback-btn" data-feedback="-1" title="${bad}" aria-label="${bad}">👎</button>
            </div>
            <span class="feedback-thanks" style="display:none;">${thanks}</span>
          </div>`;
        $chatLog.append(feedbackBarHtml);
      }

      // Save to conversation memory
      if (saveToMemory) {
        const role = sender === 'user' ? 'user' : 'assistant';
        ConversationMemory.addMessage(context, role, text);
      }

      scrollToBottom($chatLog);
    }

    function renderCards(services, products, articles, faqs) {
      let cardsHtml = '';

      // PRODUCTS
      if (Array.isArray(products) && products.length > 0) {
        const productsTitle = esc(__('Recommended Products', 'promptor'));
        cardsHtml += `<h4 class="promptor-section-title">${productsTitle}</h4><div class="promptor-product-cards">`;
        products.forEach((p) => {
          if (!p || typeof p !== 'object') return;

          const id = safeInt(p.id);
          const link = safeUrl(p.link || '#');
          const img = p.image ? safeUrl(p.image) : '';
          const title = esc(p.title || '');
          const alt = title;
          const desc = esc(p.description || '');
          const priceHtml = p.price || ''; // Already sanitized HTML from server

          // Discount badge (if product has discount)
          const discount = p.discount_percentage || p.sale_percentage || 0;
          const discountBadge = discount > 0
            ? `<div class="promptor-product-discount-badge">%${safeInt(discount)} ${esc(__('OFF', 'promptor'))}</div>`
            : '';

          cardsHtml += `
            <div class="promptor-product-card"${id ? ` data-id="${id}"` : ''}>
              ${img && img !== '#' ? `
                <a href="${link}" class="promptor-product-image"${EXT_LINK}>
                  ${discountBadge}
                  <img src="${img}" alt="${alt}">
                </a>` : ''}
              <div class="promptor-product-details">
                <h5 class="promptor-product-title"><a href="${link}"${EXT_LINK}>${title}</a></h5>
                ${desc ? `<p class="promptor-product-description">${desc}</p>` : ''}
                <div class="promptor-product-footer">
                  ${priceHtml ? `<div class="promptor-product-price" aria-label="${esc(__('Price', 'promptor'))}">${priceHtml}</div>` : ''}
                  <button type="button" class="promptor-product-action-btn promptor-add-to-cart-btn" data-product_id="${id}">
                    ${esc(i18n.addToCart || __('Add to Cart', 'promptor'))}
                  </button>
                </div>
              </div>
            </div>`;
        });
        cardsHtml += `</div>`;
      }

      // ARTICLES & DOCUMENTS
      if (Array.isArray(articles) && articles.length > 0) {
        const articlesTitle = esc(__('Related Articles', 'promptor'));
        cardsHtml += `<h4 class="promptor-section-title">${articlesTitle}</h4><div class="promptor-article-cards">`;
        articles.forEach((a) => {
          if (!a || typeof a !== 'object') return;

          const link = safeUrl(a.link || '#');
          const img = a.image ? safeUrl(a.image) : '';
          const title = esc(a.title || '');
          const alt = title;
          const desc = esc(a.description || '');
          const fileSize = a.file_size || '';
          const isPDF = fileSize || a.type === 'pdf' || a.type === 'document';

          if (isPDF) {
            // PDF/Document card with download icon
            cardsHtml += `
              <div class="promptor-document-card">
                <div class="promptor-document-icon">
                  <svg width="32" height="32" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20M10,19L12,15H9V10H15V15L13,19H10Z"/>
                  </svg>
                </div>
                <div class="promptor-document-details">
                  <h5 class="promptor-document-title">${title}</h5>
                  ${desc ? `<p class="promptor-document-description">${desc}</p>` : ''}
                  ${fileSize ? `<span class="promptor-document-size">${esc(fileSize)}</span>` : ''}
                </div>
                <a href="${link}" class="promptor-document-download" download${EXT_LINK}>
                  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4M7 10l5 5 5-5M12 15V3"/>
                  </svg>
                  ${esc(__('Download', 'promptor'))}
                </a>
              </div>`;
          } else {
            // Regular article card
            cardsHtml += `
              <a href="${link}" class="promptor-article-card"${EXT_LINK}>
                ${img && img !== '#' ? `<div class="promptor-article-image"><img src="${img}" alt="${alt}"></div>` : ''}
                <div class="promptor-article-details">
                  <h5 class="promptor-article-title">${title}</h5>
                  ${desc ? `<p class="promptor-article-description">${desc}</p>` : ''}
                </div>
              </a>`;
          }
        });
        cardsHtml += `</div>`;
      }

      // FAQS
      if (Array.isArray(faqs) && faqs.length > 0) {
        const faqsTitle = esc(__('Frequently Asked Questions', 'promptor'));
        const readMore = esc(__('Read more…', 'promptor'));
        cardsHtml += `<h4 class="promptor-section-title">${faqsTitle}</h4><div class="promptor-faq-items">`;
        faqs.forEach((f) => {
          if (!f || typeof f !== 'object') return;
          
          const title = esc(f.title || '');
          const desc = esc(f.description || '');
          const link = safeUrl(f.link || '#');
          
          cardsHtml += `
            <details class="promptor-faq-item">
              <summary>
                <span class="promptor-faq-icon">
                  <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M11,18H13V16H11V18M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M12,20C7.59,20 4,16.41 4,12C4,7.59 7.59,4 12,4C16.41,4 20,7.59 20,12C20,16.41 16.41,20 12,20M12,6A4,4 0 0,0 8,10H10A2,2 0 0,1 12,8A2,2 0 0,1 14,10C14,12 11,11.75 11,15H13C13,12.75 16,12.5 16,10A4,4 0 0,0 12,6Z"/>
                  </svg>
                </span>
                <span class="promptor-faq-question">${title}</span>
                <span class="promptor-faq-arrow">
                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M6 9l6 6 6-6"/>
                  </svg>
                </span>
              </summary>
              <div class="promptor-faq-content">
                ${desc} ${link !== '#' ? `<a href="${link}"${EXT_LINK}>${readMore}</a>` : ''}
              </div>
            </details>`;
        });
        cardsHtml += `</div>`;
      }

      // SERVICES + form
      if (Array.isArray(services) && services.length > 0) {
        const servicesTitle = esc(__('Recommended Services', 'promptor'));
        const inquiryBtnText = esc(__('Make an Inquiry for Selected Services', 'promptor'));
        const nameLabel = esc(__('Your Name', 'promptor'));
        const emailLabel = esc(__('Your Email', 'promptor'));
        const phoneLabel = esc(__('Your Phone', 'promptor'));
        const notesLabel = esc(__('Anything else you would like to add?', 'promptor'));

        cardsHtml += `<h4 class="promptor-section-title">${servicesTitle}</h4>`;
        cardsHtml += `<div class="promptor-contact-form-container"><form class="promptor-contact-form" data-type="service-inquiry" novalidate>`;
        
        services.forEach((s, idx) => {
          if (!s || typeof s !== 'object') return;
          
          const title = esc(s.title || '');
          const desc = esc(s.description || '');
          const cid = `svc-${idx}-${Date.now()}`;
          
          cardsHtml += `
            <div class="promptor-service-card">
              <label class="promptor-service-label" for="${cid}">
                <input id="${cid}" type="checkbox" name="selected_services[]" value="${title}">
                <h4>${title}</h4>
              </label>
              ${desc ? `<p class="promptor-service-description">${desc}</p>` : ''}
            </div>`;
        });
        
        cardsHtml += `
          <div class="promptor-form-fields-wrapper" style="display:none;margin-top:15px;">
            <div class="promptor-form-field"><label>${nameLabel}*</label><input type="text" name="name" required autocomplete="name"></div>
            <div class="promptor-form-field"><label>${emailLabel}*</label><input type="email" name="email" required autocomplete="email"></div>
            <div class="promptor-form-field"><label>${phoneLabel}</label><input type="tel" name="phone" inputmode="tel" autocomplete="tel"></div>
            <div class="promptor-form-field"><label>${notesLabel}</label><textarea name="notes" rows="3"></textarea></div>
          </div>
          <button type="submit" class="promptor-submit-button">${inquiryBtnText}</button>
        </form></div>`;
      }

      return cardsHtml ? `<div class="promptor-cards-wrapper">${cardsHtml}</div>` : '';
    }

    function showTypingIndicator() {
      const avatarUrl = safeUrl(botAvatarUrl);
      const botAvatar = avatarUrl !== '#'
        ? `<div class="promptor-avatar ai-avatar"><img src="${avatarUrl}" alt="AI"></div>`
        : '<div class="promptor-avatar ai-avatar"></div>';
      
      $chatLog.append(
        `<div class="promptor-message-wrapper ai-message typing-indicator-wrapper">${botAvatar}<div class="promptor-message-bubble"><div class="typing-indicator"><span></span><span></span><span></span></div></div></div>`
      );
      scrollToBottom($chatLog);
    }

    function removeTypingIndicator() {
      $chatLog.find('.typing-indicator-wrapper').remove();
    }

    function showExampleQuestions() {
      if (Array.isArray(example_questions) && example_questions.length > 0 && conversationHistory.length === 0) {
        const buttonsHtml = example_questions
          .filter(q => q && typeof q === 'string')
          .map((q) => `<button type="button" class="promptor-example-question-btn">${esc(q)}</button>`)
          .join('');
        
        if (buttonsHtml) {
          const title = esc(__('Here are some ideas to get you started:', 'promptor'));
          $chatLog.append(
            `<div class="promptor-example-questions"><p class="promptor-example-questions-title">${title}</p><div class="button-grid">${buttonsHtml}</div></div>`
          );
        }
      }
    }

    function ajaxPost(data, { timeout = 30000 } = {}) {
      return $.ajax({
        url: ajaxEndpoint,
        type: 'POST',
        dataType: 'json',
        data,
        timeout,
      });
    }

    // Main ask handler
    $form.on('submit', function (ev) {
      ev.preventDefault();

      const query = String($input.val() || '').trim();
      if (!query) return;

      // Input length validation
      if (query.length > 5000) {
        window.alert(__('Your message is too long. Please shorten it and try again.', 'promptor'));
        return;
      }

      addToLog(query, 'user');
      conversationHistory.push({ role: 'user', content: query });
      
      // Trim history if too long
      if (conversationHistory.length > MAX_HISTORY_LENGTH) {
        conversationHistory = conversationHistory.slice(-MAX_HISTORY_LENGTH);
      }
      
      $input.val('');
      $askButton.prop('disabled', true);
      $chatLog.find('.promptor-example-questions').remove();
      showTypingIndicator();

      let historyJson = '[]';

      // Only send conversation history if memory is enabled
      if (enable_conversation_memory && conversationHistory.length > 0) {
        try {
          historyJson = JSON.stringify(conversationHistory);
        } catch (e) {
          // fallback: send only last user message
          historyJson = JSON.stringify(conversationHistory.slice(-1));
        }
      }

      ajaxPost({
        action: 'promptor_get_ai_suggestion',
        nonce: ai_query_nonce,
        query,
        history: historyJson,
        context,
      })
        .done(function (response) {
          removeTypingIndicator();
          if (response && response.success) {
            const { ai_data, query_id } = response.data || {};
            current_query_id = safeInt(query_id);

            if (ai_data && ai_data.ai_explanation) {
              addToLog(ai_data.ai_explanation, 'ai', current_query_id);
            }

            // Accumulate services to memory (frontend-only, zero backend changes)
            if (ai_data && ai_data.services && Array.isArray(ai_data.services)) {
              ai_data.services.forEach(service => {
                ConversationMemory.addService(context, service);
              });
            }

            // Get ALL accumulated services (conversation intelligence)
            const allAccumulatedServices = ConversationMemory.getServices(context);

            // UNIFIED DESIGN: Always use conversation summary for services
            // Render cards for products, articles, FAQs (but NOT services - they use chip UI)
            const cardsHtml = renderCards(
              [], // No services in old UI - always use modern chip-based UI
              (ai_data && ai_data.products) || [],
              (ai_data && ai_data.articles) || [],
              (ai_data && ai_data.faqs) || []
            );
            if (cardsHtml) {
              $chatLog.append(cardsHtml);
              scrollToBottom($chatLog);
            }

            // Show conversation summary for ALL service counts (1 or multiple)
            if (allAccumulatedServices.length > 0) {
              addConversationSummary(allAccumulatedServices);
            }

            // Keep assistant turn as JSON summary
            try {
              conversationHistory.push({ role: 'assistant', content: JSON.stringify(ai_data || {}) });
            } catch (e) {
              conversationHistory.push({ role: 'assistant', content: '' });
            }
          } else {
            const err =
              (response && response.data && response.data.message) || __('An unknown error occurred.', 'promptor');
            addToLog(String(err), 'ai');
          }
        })
        .fail(function (jqXHR) {
          removeTypingIndicator();
          const errorMsg = jqXHR.status === 0 
            ? __('Network error. Please check your connection.', 'promptor')
            : __('Sorry, an error occurred. Please try again.', 'promptor');
          addToLog(errorMsg, 'ai');
        })
        .always(function () {
          $askButton.prop('disabled', false);
          scrollToBottom($chatLog);
          // a11y: focus back to input for quicker follow-up
          $input.trigger('focus');
        });
    });

    // Update button text based on selected chips
    function updateServiceButtonText() {
      const $summary = $chatLog.find('.promptor-conversation-summary').last();
      const $button = $summary.find('.promptor-cta-request-quote');
      const $selectedChips = $summary.find('.promptor-inline-chip.chip-selected');
      const selectedCount = $selectedChips.length;

      let buttonText;
      if (selectedCount === 0) {
        buttonText = esc(i18n.selectServiceQuote || __('Select a service to request a quote', 'promptor'));
        $button.prop('disabled', true);
      } else if (selectedCount === 1) {
        buttonText = esc(i18n.requestQuote || __('Request Quote', 'promptor'));
        $button.prop('disabled', false);
      } else {
        buttonText = esc(i18n.requestQuoteForAll || __('Request Quote for All', 'promptor'));
        $button.prop('disabled', false);
      }

      $button.text(buttonText);
    }

    // Inline chip click - Show description and toggle selection (multi-description support)
    $chatLog.on('click', '.promptor-inline-chip', function () {
      const $chip = $(this);
      const $summary = $chip.closest('.promptor-conversation-summary');
      const $descContainer = $summary.find('.promptor-service-description-container');
      const chipId = $chip.data('service-id');
      const chipTitle = $chip.data('service-title');
      const chipDescription = $chip.data('service-description');

      // Toggle selection
      $chip.toggleClass('chip-selected');

      // If deselecting, remove only this chip's description
      if (!$chip.hasClass('chip-selected')) {
        const $thisDesc = $descContainer.find(`[data-desc-id="${chipId}"]`);
        $thisDesc.slideUp(200, function() {
          $(this).remove();
          // Hide container if no more descriptions
          if ($descContainer.children().length === 0) {
            $descContainer.hide();
          }
        });
      }

      // Show description inline if there's a description and chip is now selected
      if ($chip.hasClass('chip-selected') && chipDescription) {
        // Create description HTML with unique ID
        const descHtml = `
          <div class="promptor-service-description-content" data-desc-id="${esc(chipId)}" style="display:none;">
            <h4>${esc(chipTitle)}</h4>
            <p>${esc(chipDescription)}</p>
          </div>
        `;

        // Add this description to container
        $descContainer.append(descHtml);

        // Show container if hidden
        if (!$descContainer.is(':visible')) {
          $descContainer.show();
        }

        // Animate the new description
        $descContainer.find(`[data-desc-id="${chipId}"]`).slideDown(200);
      }

      // Update button text based on selection
      updateServiceButtonText();
    });

    // Close description (inline) - Not used anymore but keep for compatibility
    $chatLog.on('click', '.promptor-close-description-btn', function () {
      const $content = $(this).closest('.promptor-service-description-content');
      const $descId = $content.data('desc-id');
      const $summary = $content.closest('.promptor-conversation-summary');
      const $container = $content.parent();

      // Deselect the corresponding chip
      $summary.find(`.promptor-inline-chip[data-service-id="${$descId}"]`).removeClass('chip-selected');

      // Remove this description
      $content.slideUp(200, function() {
        $(this).remove();
        // Hide container if no more descriptions
        if ($container.children().length === 0) {
          $container.hide();
        }
      });

      // Update button text
      updateServiceButtonText();
    });

    // CTA "Request Quote" button - Opens unified form for selected services
    $chatLog.on('click', '.promptor-cta-request-quote', function () {
      const $summary = $(this).closest('.promptor-conversation-summary');
      const $selectedChips = $summary.find('.promptor-inline-chip.chip-selected');

      if ($selectedChips.length === 0) {
        window.alert(__('Please select at least one service.', 'promptor'));
        return;
      }

      // Get all services and filter to selected ones
      const allServices = ConversationMemory.getServices(context);
      const selectedServiceIds = $selectedChips.map(function() {
        return $(this).data('service-id');
      }).get();

      const selectedServices = allServices.filter(s => selectedServiceIds.includes(s.id));

      // Open unified quote form
      openQuoteForm(selectedServices);
    });

    // Drawer collapse/expand toggle with smooth animations
    $app.on('click', '.promptor-drawer-collapse-btn', function () {
      const $content = $drawer.find('.promptor-drawer-content');
      const $btn = $(this);

      if ($content.is(':visible')) {
        $content.slideUp(300, 'swing');
        $btn.text('+').attr('aria-label', __('Expand drawer', 'promptor'));
      } else {
        $content.slideDown(300, 'swing', function() {
          // Auto-scroll drawer into view on mobile after expansion
          if (window.innerWidth <= 768) {
            $content[0].scrollIntoView({ behavior: 'smooth', block: 'nearest' });
          }
        });
        $btn.text('−').attr('aria-label', __('Collapse drawer', 'promptor'));
      }
    });

    // Drawer "Request Quote" button (if drawer is manually opened)
    $app.on('click', '.promptor-request-quote-btn', function () {
      const services = ConversationMemory.getServices(context);

      if (services.length === 0) {
        window.alert(__('No services selected.', 'promptor'));
        return;
      }

      openQuoteForm(services);
    });

    // New Conversation button - Clear chat history AND accumulated services
    $app.on('click', '.promptor-new-conversation-btn', function () {
      // Confirm before clearing
      const confirmMsg = i18n.confirmNewConversation || __('Are you sure you want to start a new conversation? This will clear your chat history and selected services.', 'promptor');

      if (confirm(confirmMsg)) {
        // Clear from sessionStorage (messages + services)
        ConversationMemory.clear(context);

        // Clear UI
        $chatLog.empty();
        conversationHistory = [];

        // Hide and reset drawer
        $drawer.hide();
        $drawer.find('.promptor-service-chips').empty();
        $drawer.find('.drawer-count').text('0');

        // Show welcome message if needed
        showExampleQuestions();

        // Optional: Show a brief confirmation
        const clearedMsg = __('Chat history cleared. You can start a fresh conversation!', 'promptor');
        addToLog(clearedMsg, 'ai', 0, false);
      }
    });

    // Quick fill from example buttons
    $chatLog.on('click', '.promptor-example-question-btn', function () {
      const questionText = $(this).text();
      $input.val(questionText).trigger('focus');
      $form.trigger('submit');
    });

    // Feedback buttons
    $chatLog.on('click', '.promptor-session-feedback-bar .feedback-btn', function () {
      const $button = $(this);
      const $feedbackBar = $button.closest('.promptor-session-feedback-bar');
      const queryId = safeInt($feedbackBar.data('query-id'));
      const feedbackValue = safeInt($button.data('feedback'));

      if (!queryId) return;

      $feedbackBar.find('.feedback-btn').prop('disabled', true);
      $button.css('transform', 'scale(1.2)');

      ajaxPost({
        action: 'promptor_save_feedback',
        nonce: feedback_nonce,
        query_id: queryId,
        feedback: feedbackValue,
      })
        .done(function (response) {
          if (response && response.success) {
            $feedbackBar.find('.feedback-text, .feedback-actions').fadeOut(200, function () {
              $feedbackBar.find('.feedback-thanks').fadeIn(200);
            });
          } else {
            $feedbackBar.find('.feedback-btn').prop('disabled', false);
            window.alert(
              String((response && response.data && response.data.message) || __('Could not save feedback.', 'promptor'))
            );
          }
        })
        .fail(function () {
          $feedbackBar.find('.feedback-btn').prop('disabled', false);
          window.alert(__('An error occurred while saving feedback.', 'promptor'));
        });
    });

    // Add to cart
    $chatLog.on('click', '.promptor-add-to-cart-btn', function (ev) {
      ev.preventDefault();

      const $button = $(this);
      if ($button.hasClass('loading') || $button.hasClass('added')) return;

      const pid = safeInt($button.data('product_id'));
      if (!pid) {
        window.alert(__('Invalid product.', 'promptor'));
        return;
      }

      $button.addClass('loading').prop('disabled', true);

      ajaxPost({
        action: 'promptor_add_to_cart',
        nonce: add_to_cart_nonce,
        product_id: pid,
        query_id: current_query_id,
      })
        .done(function (response) {
          if (response && response.success) {
            $button.addClass('added').text(i18n.addedToCart || __('Added', 'promptor'));
            // WooCommerce hook if present
            if (typeof document.body !== 'undefined') {
              $(document.body).trigger('wc_fragment_refresh');
            }
          } else {
            window.alert(String((response && response.data && response.data.message) || __('An error occurred.', 'promptor')));
            $button.prop('disabled', false);
          }
        })
        .fail(function () {
          window.alert(__('A network error occurred.', 'promptor'));
          $button.removeClass('added').prop('disabled', false);
        })
        .always(function () {
          $button.removeClass('loading');
        });
    });

    // Show/hide inquiry fields on service selection
    $chatLog.on('change', 'input[name="selected_services[]"]', function () {
      const $formEl = $(this).closest('form');
      const hasChecked = $formEl.find('input[name="selected_services[]"]:checked').length > 0;
      
      if (hasChecked) {
        $formEl.find('.promptor-form-fields-wrapper').slideDown();
      } else {
        $formEl.find('.promptor-form-fields-wrapper').slideUp();
      }
    });

    // Submit inquiry form (per-message service cards)
    $chatLog.on('submit', 'form[data-type="service-inquiry"]', function (ev) {
      ev.preventDefault();

      const $contactForm = $(this);
      const $submitButton = $contactForm.find('.promptor-submit-button');
      const originalButtonText = $submitButton.text();

      // Validation
      const selectedServices = $contactForm.find('input[name="selected_services[]"]:checked');
      if (selectedServices.length === 0) {
        window.alert(__('Please select at least one service.', 'promptor'));
        return;
      }

      const name = String($contactForm.find('[name="name"]').val() || '').trim();
      const email = String($contactForm.find('[name="email"]').val() || '').trim();

      if (!name || !email) {
        window.alert(__('Please fill in your name and email.', 'promptor'));
        return;
      }

      // Basic email validation
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(email)) {
        window.alert(__('Please enter a valid email address.', 'promptor'));
        return;
      }

      $submitButton.text(__('Sending...', 'promptor')).prop('disabled', true);

      const formData = {
        action: 'promptor_submit_contact_form',
        nonce: form_nonce,
        name,
        email,
        phone: String($contactForm.find('[name="phone"]').val() || '').trim(),
        notes: String($contactForm.find('[name="notes"]').val() || '').trim(),
        services: selectedServices.map((_, el) => String($(el).val())).get(),
        query_id: current_query_id,
      };

      ajaxPost(formData)
        .done(function (response) {
          if (response && response.success) {
            const msg =
              (response.data && response.data.message) || __('Thank you! We received your inquiry.', 'promptor');
            const $success = $('<div class="promptor-success-screen"><h3></h3></div>');
            $success.find('h3').text(String(msg));
            $contactForm.closest('.promptor-contact-form-container').empty().append($success);
          } else {
            window.alert(
              String((response && response.data && response.data.message) || __('An error occurred. Please try again.', 'promptor'))
            );
            $submitButton.text(originalButtonText).prop('disabled', false);
          }
        })
        .fail(function () {
          window.alert(__('A network error occurred. Please try again.', 'promptor'));
          $submitButton.text(originalButtonText).prop('disabled', false);
        });
    });

    // Submit drawer inquiry form (accumulated services from memory)
    $chatLog.on('submit', 'form[data-type="drawer-inquiry"]', function (ev) {
      ev.preventDefault();

      const $contactForm = $(this);
      const $submitButton = $contactForm.find('.promptor-submit-button');
      const originalButtonText = $submitButton.text();

      // Get ALL accumulated services from memory (not checkboxes)
      const allServices = ConversationMemory.getServices(context);
      if (allServices.length === 0) {
        window.alert(__('No services selected.', 'promptor'));
        return;
      }

      const name = String($contactForm.find('[name="name"]').val() || '').trim();
      const email = String($contactForm.find('[name="email"]').val() || '').trim();

      if (!name || !email) {
        window.alert(__('Please fill in your name and email.', 'promptor'));
        return;
      }

      // Basic email validation
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(email)) {
        window.alert(__('Please enter a valid email address.', 'promptor'));
        return;
      }

      $submitButton.text(__('Sending...', 'promptor')).prop('disabled', true);

      const formData = {
        action: 'promptor_submit_contact_form',
        nonce: form_nonce,
        name,
        email,
        phone: String($contactForm.find('[name="phone"]').val() || '').trim(),
        notes: String($contactForm.find('[name="notes"]').val() || '').trim(),
        services: allServices.map(s => s.title), // All accumulated services
        query_id: current_query_id,
      };

      ajaxPost(formData)
        .done(function (response) {
          if (response && response.success) {
            const msg =
              (response.data && response.data.message) || __('Thank you! We received your inquiry.', 'promptor');
            const $success = $('<div class="promptor-success-screen"><h3></h3></div>');
            $success.find('h3').text(String(msg));
            $contactForm.closest('.promptor-quote-form-container').empty().append($success);
          } else {
            window.alert(
              String((response && response.data && response.data.message) || __('An error occurred. Please try again.', 'promptor'))
            );
            $submitButton.text(originalButtonText).prop('disabled', false);
          }
        })
        .fail(function () {
          window.alert(__('A network error occurred. Please try again.', 'promptor'));
          $submitButton.text(originalButtonText).prop('disabled', false);
        });
    });

    // Init
    populateUI();
    restoreConversation(); // Restore chat history from sessionStorage
    showExampleQuestions();
  }

  // DOM ready
  $(function () {
    // Initialize all instances
    $('.promptor-app').each(function () {
      try {
        initializePromptorApp($(this));
      } catch (err) {
        console.error('Promptor initialization error:', err);
        $(this).html(
          '<p style="padding:20px;text-align:center;">' +
            esc(__('Failed to initialize chat. Please refresh the page.', 'promptor')) +
            '</p>'
        );
      }
    });

    // Popup toggle (if exists)
    const $popupContainer = $('#promptor-popup-container');
    if ($popupContainer.length > 0) {
      const $toggleBtn = $('#promptor-popup-toggle');
      const $chatWindow = $popupContainer.find('.promptor-chat-window');
      const $popupInput = $chatWindow.find('.promptor-query-input');
      const $popupChatLog = $chatWindow.find('.promptor-chat-log');

      $toggleBtn.on('click', function () {
        const isActive = $chatWindow.hasClass('active');
        const isMobile = window.innerWidth <= 768;

        if (!isActive) {
          // Opening chat window
          $chatWindow.addClass('active').css('display', 'flex');
          $(this).find('.promptor-icon-chat').fadeOut(200);
          $(this).find('.promptor-icon-close').fadeIn(200);

          // Auto-focus input on desktop, delayed focus on mobile to prevent keyboard overlap
          setTimeout(function() {
            if (!isMobile) {
              $popupInput.trigger('focus');
            }
            scrollToBottom($popupChatLog);
          }, isMobile ? 400 : 100);
        } else {
          // Closing chat window
          $chatWindow.removeClass('active');
          $(this).find('.promptor-icon-chat').fadeIn(200);
          $(this).find('.promptor-icon-close').fadeOut(200);

          // Blur input to hide mobile keyboard
          $popupInput.trigger('blur');

          setTimeout(() => {
            if (!$chatWindow.hasClass('active')) {
              $chatWindow.css('display', 'none');
            }
          }, 300);
        }
      });
    }
  });
})(jQuery);