<?php
/**
 * Webhook Admin UI
 *
 * Admin settings page for webhooks
 *
 * @package Promptor
 * @subpackage Webhooks
 * @since 1.2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Promptor_Webhook_Admin_UI
 *
 * Manages webhook admin interface
 */
class Promptor_Webhook_Admin_UI {

	/**
	 * Settings instance
	 *
	 * @var Promptor_Webhook_Settings
	 */
	private $settings;

	/**
	 * Validator instance
	 *
	 * @var Promptor_Webhook_Validator
	 */
	private $validator;

	/**
	 * Constructor
	 */
	public function __construct() {
		require_once dirname( __DIR__, 2 ) . '/includes/webhooks/class-promptor-webhook-settings.php';
		require_once dirname( __DIR__, 2 ) . '/includes/webhooks/class-promptor-webhook-validator.php';

		$this->settings  = new Promptor_Webhook_Settings();
		$this->validator = new Promptor_Webhook_Validator();

		$this->init();
	}

	/**
	 * Initialize admin hooks
	 *
	 * @return void
	 */
	private function init() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ), 100 );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_post_promptor_webhook_test', array( $this, 'handle_test_webhook' ) );
	}

	/**
	 * Add webhook settings submenu
	 *
	 * @return void
	 */
	public function add_menu_page() {
		// Webhooks are Pro-only feature.
		if ( ! function_exists( 'promptor_is_pro' ) || ! promptor_is_pro() ) {
			return;
		}

		add_submenu_page(
			'promptor',
			__( 'Webhook Settings', 'promptor' ),
			__( 'Webhooks', 'promptor' ),
			'manage_options',
			'promptor-webhooks',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Register settings
	 *
	 * @return void
	 */
	public function register_settings() {
		// Only register settings if Pro is active.
		if ( ! function_exists( 'promptor_is_pro' ) || ! promptor_is_pro() ) {
			return;
		}

		register_setting(
			'promptor_webhook_settings',
			'promptor_webhook_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
			)
		);
	}

	/**
	 * Sanitize and validate settings
	 *
	 * @param array $input Input settings.
	 * @return array Sanitized settings
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Enabled.
		$sanitized['enabled'] = isset( $input['enabled'] ) && '1' === $input['enabled'];

		// Paused.
		$sanitized['paused'] = isset( $input['paused'] ) && '1' === $input['paused'];

		// URL.
		if ( isset( $input['url'] ) ) {
			$url_validation = $this->validator->validate_url( $input['url'] );

			if ( $url_validation['valid'] ) {
				$sanitized['url'] = esc_url_raw( $input['url'], array( 'http', 'https' ) );

				if ( ! empty( $url_validation['warning'] ) ) {
					add_settings_error(
						'promptor_webhook_settings',
						'webhook_url_warning',
						$url_validation['warning'],
						'warning'
					);
				}
			} else {
				add_settings_error(
					'promptor_webhook_settings',
					'webhook_url_error',
					$url_validation['error'],
					'error'
				);
			}
		}

		// Secret key (preserve existing if not changed).
		if ( isset( $input['secret_key'] ) && ! empty( $input['secret_key'] ) ) {
			$sanitized['secret_key'] = $input['secret_key'];
		} else {
			$current_settings        = $this->settings->get_settings();
			$sanitized['secret_key'] = $current_settings['secret_key'];
		}

		// Events (only lead_created for v1.2.0A).
		$sanitized['events'] = array(
			'lead_created'   => isset( $input['events']['lead_created'] ) && '1' === $input['events']['lead_created'],
			'lead_qualified' => false, // Disabled for v1.2.0A.
			'lead_scored'    => false, // Disabled for v1.2.0A.
		);

		// Advanced settings.
		if ( isset( $input['retry_attempts'] ) ) {
			$attempts                    = absint( $input['retry_attempts'] );
			$sanitized['retry_attempts'] = max( 1, min( 5, $attempts ) );
		}

		if ( isset( $input['timeout'] ) ) {
			$timeout              = absint( $input['timeout'] );
			$sanitized['timeout'] = max( 5, min( 30, $timeout ) );
		}

		$sanitized['verify_ssl'] = isset( $input['verify_ssl'] ) && '1' === $input['verify_ssl'];

		// Privacy: Store full payload (default OFF).
		$sanitized['store_full_payload'] = isset( $input['store_full_payload'] ) && '1' === $input['store_full_payload'];

		// Preserve retry_delays.
		$current_settings          = $this->settings->get_settings();
		$sanitized['retry_delays'] = $current_settings['retry_delays'];

		return $sanitized;
	}

	/**
	 * Render settings page
	 *
	 * @return void
	 */
	public function render_settings_page() {
		// Check Pro access (webhooks are Pro-only).
		if ( ! function_exists( 'promptor_is_pro' ) || ! promptor_is_pro() ) {
			wp_die( esc_html__( 'This feature is only available in Promptor Pro.', 'promptor' ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'promptor' ) );
		}

		$settings = $this->settings->get_settings();

		?>
		<div class="wrap promptor-webhooks-wrap">
			<div class="promptor-page-header">
				<div class="header-content">
					<h1>
						<span class="dashicons dashicons-update"></span>
						<?php esc_html_e( 'Webhook Settings', 'promptor' ); ?>
					</h1>
					<p class="page-subtitle"><?php esc_html_e( 'Configure webhooks to send real-time notifications to external systems like Zapier, Make, or custom endpoints.', 'promptor' ); ?></p>
				</div>
			</div>

			<?php settings_errors(); ?>

			<form method="post" action="options.php">
				<?php settings_fields( 'promptor_webhook_settings' ); ?>

				<!-- Enable Webhooks Section -->
				<div class="postbox">
					<h2 class="hndle"><span><?php esc_html_e( 'Enable Webhooks', 'promptor' ); ?></span></h2>
					<div class="inside">
						<table class="form-table" role="presentation">
							<tr>
								<th scope="row">
									<label for="webhook_enabled"><?php esc_html_e( 'Webhooks Active', 'promptor' ); ?></label>
								</th>
								<td>
									<fieldset>
										<label>
											<input type="checkbox" name="promptor_webhook_settings[enabled]" id="webhook_enabled" value="1" <?php checked( $settings['enabled'], true ); ?>>
											<?php esc_html_e( 'Enable webhook delivery', 'promptor' ); ?>
										</label>
										<p class="description">
											<?php esc_html_e( 'When enabled, webhook requests will be sent to the configured endpoint for each enabled event.', 'promptor' ); ?>
										</p>
									</fieldset>
								</td>
							</tr>

							<tr>
								<th scope="row">
									<label for="webhook_paused"><?php esc_html_e( 'Pause Deliveries', 'promptor' ); ?></label>
								</th>
								<td>
									<fieldset>
										<label>
											<input type="checkbox" name="promptor_webhook_settings[paused]" id="webhook_paused" value="1" <?php checked( $settings['paused'], true ); ?>>
											<?php esc_html_e( 'Temporarily pause deliveries', 'promptor' ); ?>
										</label>
										<p class="description">
											<?php esc_html_e( 'Events will be logged but not sent. Useful for maintenance or debugging.', 'promptor' ); ?>
										</p>
									</fieldset>
								</td>
							</tr>

							<tr>
								<th scope="row">
									<?php esc_html_e( 'Events', 'promptor' ); ?>
								</th>
								<td>
									<fieldset>
										<legend class="screen-reader-text">
											<?php esc_html_e( 'Select which events to send', 'promptor' ); ?>
										</legend>
										<label>
											<input type="checkbox" name="promptor_webhook_settings[events][lead_created]" value="1" <?php checked( $settings['events']['lead_created'] ?? false, true ); ?>>
											<strong><?php esc_html_e( 'Lead Created', 'promptor' ); ?></strong>
										</label>
										<p class="description" style="margin-left: 25px;">
											<?php esc_html_e( 'Triggered when a visitor submits a quote request with their contact details.', 'promptor' ); ?>
										</p>
									</fieldset>
								</td>
							</tr>
						</table>
					</div>
				</div>

				<!-- Delivery Settings Section -->
				<div class="postbox">
					<h2 class="hndle"><span><?php esc_html_e( 'Delivery Settings', 'promptor' ); ?></span></h2>
					<div class="inside">
						<table class="form-table" role="presentation">
							<tr>
								<th scope="row">
									<label for="webhook_url"><?php esc_html_e( 'Webhook URL', 'promptor' ); ?> <span class="description">(<?php esc_html_e( 'required', 'promptor' ); ?>)</span></label>
								</th>
								<td>
									<input type="url" name="promptor_webhook_settings[url]" id="webhook_url" value="<?php echo esc_attr( $settings['url'] ); ?>" class="regular-text code" placeholder="https://example.com/webhook" required>
									<p class="description">
										<?php esc_html_e( 'The endpoint URL where webhook payloads will be sent. HTTPS is strongly recommended.', 'promptor' ); ?>
									</p>
								</td>
							</tr>

							<tr>
								<th scope="row">
									<label for="webhook_retry_attempts"><?php esc_html_e( 'Retry Attempts', 'promptor' ); ?></label>
								</th>
								<td>
									<input type="number" name="promptor_webhook_settings[retry_attempts]" id="webhook_retry_attempts" value="<?php echo esc_attr( $settings['retry_attempts'] ); ?>" min="1" max="5" class="small-text">
									<p class="description">
										<?php esc_html_e( 'Number of automatic retry attempts for failed deliveries (1-5). Retries use exponential backoff.', 'promptor' ); ?>
									</p>
								</td>
							</tr>

							<tr>
								<th scope="row">
									<label for="webhook_timeout"><?php esc_html_e( 'Request Timeout', 'promptor' ); ?></label>
								</th>
								<td>
									<input type="number" name="promptor_webhook_settings[timeout]" id="webhook_timeout" value="<?php echo esc_attr( $settings['timeout'] ); ?>" min="5" max="30" class="small-text">
									<span class="description"><?php esc_html_e( 'seconds', 'promptor' ); ?></span>
									<p class="description">
										<?php esc_html_e( 'Maximum time to wait for the endpoint to respond (5-30 seconds).', 'promptor' ); ?>
									</p>
								</td>
							</tr>
						</table>
					</div>
				</div>

				<!-- Security Section -->
				<div class="postbox">
					<h2 class="hndle"><span><?php esc_html_e( 'Security', 'promptor' ); ?></span></h2>
					<div class="inside">
						<table class="form-table" role="presentation">
							<tr>
								<th scope="row">
									<label for="webhook_secret"><?php esc_html_e( 'Secret Key', 'promptor' ); ?></label>
								</th>
								<td>
									<input type="text" name="promptor_webhook_settings[secret_key]" id="webhook_secret" value="<?php echo esc_attr( $settings['secret_key'] ); ?>" class="regular-text code" readonly>
									<p class="description">
										<?php
										printf(
											/* translators: %s: header name */
											esc_html__( 'Used to sign webhook requests with HMAC-SHA256. The signature is sent in the %s header. Keep this secret.', 'promptor' ),
											'<code>X-Promptor-Signature</code>'
										);
										?>
									</p>
								</td>
							</tr>

							<tr>
								<th scope="row">
									<label for="webhook_verify_ssl"><?php esc_html_e( 'SSL Verification', 'promptor' ); ?></label>
								</th>
								<td>
									<fieldset>
										<label>
											<input type="checkbox" name="promptor_webhook_settings[verify_ssl]" id="webhook_verify_ssl" value="1" <?php checked( $settings['verify_ssl'], true ); ?>>
											<?php esc_html_e( 'Verify SSL/TLS certificates', 'promptor' ); ?>
										</label>
										<p class="description">
											<?php esc_html_e( 'Recommended. Disable only for local development with self-signed certificates.', 'promptor' ); ?>
										</p>
									</fieldset>
								</td>
							</tr>
						</table>
					</div>
				</div>

				<!-- Privacy & Data Retention Section -->
				<div class="postbox">
					<h2 class="hndle"><span><?php esc_html_e( 'Privacy & Data Retention', 'promptor' ); ?></span></h2>
					<div class="inside">
						<div class="notice notice-info inline" style="margin: 15px 0; padding: 12px;">
							<p style="margin: 0;">
								<span class="dashicons dashicons-info" style="color: #2271b1; vertical-align: middle;"></span>
								<strong><?php esc_html_e( 'Privacy Notice:', 'promptor' ); ?></strong>
								<?php esc_html_e( 'Webhook payloads contain customer data including names, email addresses, and message content. Delivery logs are retained for 30 days.', 'promptor' ); ?>
							</p>
						</div>
						<table class="form-table" role="presentation">
							<tr>
								<th scope="row">
									<label for="webhook_store_full_payload"><?php esc_html_e( 'Log Storage', 'promptor' ); ?></label>
								</th>
								<td>
									<fieldset>
										<label>
											<input type="checkbox" name="promptor_webhook_settings[store_full_payload]" id="webhook_store_full_payload" value="1" <?php checked( $settings['store_full_payload'] ?? false, true ); ?>>
											<?php esc_html_e( 'Store full payload data in delivery logs', 'promptor' ); ?>
										</label>
										<p class="description">
											<?php esc_html_e( 'When disabled (default), logs contain only event metadata (timestamp, status, URL). When enabled, complete customer data is logged for debugging. Only enable if required for troubleshooting.', 'promptor' ); ?>
										</p>
									</fieldset>
								</td>
							</tr>
						</table>
					</div>
				</div>

				<?php submit_button( __( 'Save Webhook Settings', 'promptor' ) ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle test webhook request
	 *
	 * @return void
	 */
	public function handle_test_webhook() {
		// Check nonce and permissions.
		check_admin_referer( 'promptor_webhook_test' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions.', 'promptor' ) );
		}

		// Send test webhook.
		// ... implementation here ...

		wp_safe_redirect( admin_url( 'admin.php?page=promptor-webhooks&test=success' ) );
		exit;
	}
}
