<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Promptor_Settings_Tab_Api {

    public function register_settings() {
        register_setting('promptor_api_options_group', 'promptor_api_settings', array($this, 'sanitize_api_settings'));
    }

public function render() {
    ?>
    <form method="post" action="options.php">
        <?php settings_fields('promptor_api_options_group'); ?>

        <div class="postbox">
            <h2 class="hndle"><?php esc_html_e('Core API Configuration', 'promptor'); ?></h2>
            <div class="inside">
                <p><?php esc_html_e("Enter your OpenAI API credentials. This is the minimum required setup.", 'promptor'); ?></p>
                <table class="form-table">
                    <?php 
                    $this->render_api_key_field();
                    $this->render_model_selection_field();
                    ?>
                </table>
            </div>
        </div>

        <div class="postbox">
            <h2 class="hndle"><?php esc_html_e('Model Behavior', 'promptor'); ?></h2>
            <div class="inside">
                <p><?php esc_html_e("Control how the AI generates responses, its personality, creativity, and length.", 'promptor'); ?></p>
                
                <div class="promptor-grid-2-col">
                    
                    <div class="promptor-col">
                        <table class="form-table">
                            <?php $this->render_slider_field_row('temperature', __('Temperature', 'promptor'), array('default' => 0.5, 'min' => 0, 'max' => 2, 'step' => 0.1, 'desc' => __('Controls randomness. Lower values make responses more focused. Higher values make them more creative.', 'promptor'))); ?>
                            <?php $this->render_text_field_row('max_tokens', __('Maximum Tokens', 'promptor'), array('default' => 1024, 'type' => 'number', 'desc' => __('The maximum number of tokens for the AI response. Helps control costs.', 'promptor'))); ?>
                        </table>
                    </div>

                    <div class="promptor-col">
                         <table class="form-table">
                            <?php $this->render_system_prompt_field_row('system_prompt', __('System Prompt', 'promptor')); ?>
                        </table>
                    </div>

                </div>
            </div>
        </div>
	<div class="promptor-grid-2-col" style="margin-bottom:20px;">
        <div class="postbox">
            <h2 class="hndle" style="color: #2271b1;"><?php esc_html_e('Semantic Search Engine', 'promptor'); ?></h2>
            <div class="inside">
                 <p><strong><?php esc_html_e("This is the core engine of your AI assistant. Configure how accurately the AI understands the user's intent.", 'promptor'); ?></strong></p>
                <table class="form-table">
                    <?php 
                    $similarity_desc = __(
                        "Determines how semantically similar a content chunk must be to the user's query to be included in the AI's context. A higher value (e.g., 0.75) is stricter and more accurate, but may find fewer results. A lower value (e.g., 0.65) is more lenient and finds more results, but they might be less relevant.",
                        'promptor'
                    );
                    $this->render_slider_field_row('similarity_threshold', __('Similarity Threshold', 'promptor'), array(
                        'default' => 0.7, 
                        'min' => 0.5, 
                        'max' => 0.9, 
                        'step' => 0.01, 
                        'desc' => $similarity_desc
                    )); 
                    ?>
                </table>
            </div>
        </div>
        <div class="postbox">
            <h2 class="hndle"><?php esc_html_e('Conversation Management', 'promptor'); ?></h2>
            <div class="inside">
                <p><?php esc_html_e("Manage long conversations to optimize performance and cost.", 'promptor'); ?></p>
                <table class="form-table">
                    <?php $this->render_checkbox_field_row('enable_summarization', __('Enable Conversation Summarization', 'promptor'), array('desc' => __('Automatically summarize long conversations to save tokens and stay within context limits.', 'promptor'))); ?>
                    <?php $this->render_text_field_row('summarization_threshold', __('Summarization Threshold', 'promptor'), array('default' => 6, 'type' => 'number', 'desc' => __('Summarize the conversation after this many messages (user + AI).', 'promptor'))); ?>
                    <?php $this->render_textarea_field_row('summarization_prompt', __('Summarization Prompt', 'promptor'), array('default' => 'Briefly summarize the key points of the preceding conversation.', 'desc' => __('The system prompt used to ask the AI to create a summary.', 'promptor'))); ?>
                </table>
            </div>
        </div>
</div>
        <?php submit_button(); ?>
    </form>
    <?php
}
    
    public function sanitize_api_settings($input) {
        $sanitized_input = array();
        $options = get_option('promptor_api_settings', array());
        $in = is_array($input) ? wp_unslash($input) : array();

        // 1) API Key (unslash + sanitize + format check)
        if ( isset( $in['api_key'] ) ) {
            $api_key = sanitize_text_field( $in['api_key'] );
            $api_key = trim( $api_key );

            if ( $api_key === '' ) {
                $sanitized_input['api_key'] = $options['api_key'] ?? '';
            } elseif ( preg_match( '/^(sk|gsk)[\-\w_]/i', $api_key ) ) {
                $sanitized_input['api_key'] = $api_key;
                if ( ( $options['api_key'] ?? '' ) !== $api_key ) {
                    add_settings_error( 'promptor_api_settings', 'api_key_valid', __( 'API Key saved.', 'promptor' ), 'success' );
                }
            } else {
                $sanitized_input['api_key'] = $options['api_key'] ?? '';
                add_settings_error( 'promptor_api_settings', 'invalid_api_key', __( 'The API key format looks unusual. Your previous key has been kept.', 'promptor' ), 'error' );
            }
        } else {
            $sanitized_input['api_key'] = $options['api_key'] ?? '';
        }

        // 2) Model (whitelist)
        $allowed_models = array('gpt-4o','gpt-4-turbo','gpt-3.5-turbo');
        $model = isset($in['model']) ? sanitize_key($in['model']) : ($options['model'] ?? 'gpt-4o');
        $sanitized_input['model'] = in_array($model, $allowed_models, true) ? $model : 'gpt-4o';

        // 3) Temperature (float clamp 0..2)
        if (isset($in['temperature'])) {
            $sanitized_input['temperature'] = max(0, min(2, (float) $in['temperature']));
        } else {
            $sanitized_input['temperature'] = $options['temperature'] ?? 0.5;
        }

        // 4) Max tokens (int clamp ≤ 8192)
        if (isset($in['max_tokens'])) {
            $sanitized_input['max_tokens'] = min(8192, absint($in['max_tokens']));
        } else {
            $sanitized_input['max_tokens'] = $options['max_tokens'] ?? 1024;
        }

        // 5) System prompt (textarea sanitize)
        $default_system_prompt = "You are a professional, helpful, and friendly sales assistant for this website. Your main goal is to understand the user's needs and, based ONLY on the context provided to you, recommend the most relevant services or products from the context.\n\nKey instructions:\n- Always be polite and professional.\n- Your answers must be based strictly on the provided context. Do not use any external knowledge.\n- If the answer is not in the context, state that you don't have information on that topic.\n- Keep your explanations concise and to the point.";
        if ( isset( $in['system_prompt'] ) ) {
            $sanitized = sanitize_textarea_field( $in['system_prompt'] );
            $sanitized_input['system_prompt'] = ! empty( $sanitized ) ? $sanitized : ( ! empty( $options['system_prompt'] ) ? $options['system_prompt'] : $default_system_prompt );
        } else {
            $sanitized_input['system_prompt'] = ! empty( $options['system_prompt'] ) ? $options['system_prompt'] : $default_system_prompt;
        }

        // 6) Similarity threshold (float clamp 0.5..0.9)
        if (isset($in['similarity_threshold'])) {
            $sanitized_input['similarity_threshold'] = max(0.5, min(0.9, (float) $in['similarity_threshold']));
        } else {
            $sanitized_input['similarity_threshold'] = $options['similarity_threshold'] ?? 0.7;
        }

        // 7) Summarization on/off
        $sanitized_input['enable_summarization'] = !empty($in['enable_summarization']) ? 1 : 0;

        // 8) Summarization threshold (int min 2)
        if (isset($in['summarization_threshold'])) {
            $sanitized_input['summarization_threshold'] = max(2, absint($in['summarization_threshold']));
        } else {
            $sanitized_input['summarization_threshold'] = $options['summarization_threshold'] ?? 6;
        }

        // 9) Summarization prompt (textarea sanitize)
        $default_summarization_prompt = 'Briefly summarize the key points of the preceding conversation.';
        if ( isset( $in['summarization_prompt'] ) ) {
            $sanitized = sanitize_textarea_field( $in['summarization_prompt'] );
            $sanitized_input['summarization_prompt'] = ! empty( $sanitized ) ? $sanitized : ( ! empty( $options['summarization_prompt'] ) ? $options['summarization_prompt'] : $default_summarization_prompt );
        } else {
            $sanitized_input['summarization_prompt'] = ! empty( $options['summarization_prompt'] ) ? $options['summarization_prompt'] : $default_summarization_prompt;
        }

        return $sanitized_input;
    }

    private function render_api_key_field() {
        $options = get_option('promptor_api_settings', array());
        $value = $options['api_key'] ?? '';
        ?>
        <tr>
            <th scope="row"><label for="api_key"><?php esc_html_e('OpenAI API Key', 'promptor'); ?></label></th>
            <td>
                <div class="api-key-wrapper">
                    <input type="password" id="api_key" name="promptor_api_settings[api_key]" value="<?php echo esc_attr( $value ); ?>" class="regular-text"
       autocomplete="off" spellcheck="false" autocapitalize="off" />
                    <button type="button" id="promptor-verify-api-key-btn" class="button button-secondary"><?php esc_html_e('Verify Key', 'promptor'); ?></button>
                    <span class="spinner"></span>
                </div>
                <p class="description" id="api-key-status"></p>
            </td>
        </tr>
        <?php
    }

    private function render_model_selection_field() {
        $options = get_option('promptor_api_settings', array());
        $selected_model = $options['model'] ?? 'gpt-4o';
        $models = array(
            'gpt-4o'        => esc_html__( 'GPT-4o (Recommended)', 'promptor' ),
            'gpt-4-turbo'   => esc_html__( 'GPT-4 Turbo', 'promptor' ),
            'gpt-3.5-turbo' => esc_html__( 'GPT-3.5 Turbo', 'promptor' ),
        );

        ?>
        <tr>
            <th scope="row"><label for="model"><?php esc_html_e('GPT Model', 'promptor'); ?></label></th>
            <td>
                <select id="model" name="promptor_api_settings[model]">
                    <?php foreach ($models as $value => $label): ?>
                        <option value="<?php echo esc_attr($value); ?>" <?php selected($selected_model, $value); ?>><?php echo esc_html($label); ?></option>
                    <?php endforeach; ?>
                </select>
                <p class="description"><?php esc_html_e('GPT-4o is recommended for its balance of cost, speed, and intelligence.','promptor');?></p>
            </td>
        </tr>
        <?php
    }

    private function render_text_field_row($id, $title, $args) {
        $options = get_option('promptor_api_settings', array());
        $value = $options[$id] ?? $args['default'] ?? '';
        $type = $args['type'] ?? 'text';
        $class = $type === 'number' ? 'small-text' : 'regular-text';
        ?>
        <tr>
            <th scope="row"><label for="<?php echo esc_attr($id); ?>"><?php echo esc_html($title); ?></label></th>
            <td>
                <input type="<?php echo esc_attr($type); ?>" id="<?php echo esc_attr($id); ?>" name="promptor_api_settings[<?php echo esc_attr($id); ?>]" value="<?php echo esc_attr($value); ?>" class="<?php echo esc_attr($class); ?>" />
                <?php if (isset($args['desc'])) { echo '<p class="description">' . esc_html($args['desc']) . '</p>'; } ?>
            </td>
        </tr>
        <?php
    }

    private function render_system_prompt_field_row($id, $title) {
        $options = get_option('promptor_api_settings', array());
        $default_prompt = "You are a professional, helpful, and friendly sales assistant for this website. Your main goal is to understand the user's needs and, based ONLY on the context provided to you, recommend the most relevant services or products from the context.

Key instructions:
- Always be polite and professional.
- Your answers must be based strictly on the provided context. Do not use any external knowledge.
- If the answer is not in the context, state that you don't have information on that topic.
- Keep your explanations concise and to the point.";
        $value = ! empty( $options[ $id ] ) ? $options[ $id ] : $default_prompt;
         ?>
        <tr>
            <th scope="row" style="vertical-align: top;"><label for="<?php echo esc_attr($id); ?>"><?php echo esc_html($title); ?></label></th>
            <td>
                <textarea id="<?php echo esc_attr($id); ?>" name="promptor_api_settings[<?php echo esc_attr($id); ?>]" rows="12" class="large-text"><?php echo esc_textarea($value); ?></textarea>
                <p class="description">
                    <?php esc_html_e('This prompt defines the AI\'s personality and core instructions. The system will automatically append instructions for language and JSON output format.', 'promptor'); ?>
                </p>
            </td>
        </tr>
        <?php
    }
    
    private function render_textarea_field_row($id, $title, $args) {
        $options = get_option('promptor_api_settings', array());
        $value = ! empty( $options[ $id ] ) ? $options[ $id ] : ( $args['default'] ?? '' );
         ?>
        <tr>
            <th scope="row"><label for="<?php echo esc_attr($id); ?>"><?php echo esc_html($title); ?></label></th>
            <td>
                <textarea id="<?php echo esc_attr($id); ?>" name="promptor_api_settings[<?php echo esc_attr($id); ?>]" rows="5" class="large-text"><?php echo esc_textarea($value); ?></textarea>
                <?php if (isset($args['desc'])) { echo '<p class="description">' . esc_html($args['desc']) . '</p>'; } ?>
            </td>
        </tr>
        <?php
    }

    private function render_slider_field_row($id, $title, $args) {
        $options = get_option('promptor_api_settings', array());
        $value = $options[$id] ?? $args['default'] ?? 0.5;
        ?>
        <tr>
            <th scope="row"><label for="<?php echo esc_attr($id); ?>"><?php echo esc_html($title); ?></label></th>
            <td>
                <div class="promptor-slider-container">
                    <input type="range" class="promptor-slider-input" id="<?php echo esc_attr($id); ?>" name="promptor_api_settings[<?php echo esc_attr($id); ?>]" 
                           value="<?php echo esc_attr($value); ?>" 
                           min="<?php echo esc_attr($args['min']); ?>" 
                           max="<?php echo esc_attr($args['max']); ?>" 
                           step="<?php echo esc_attr($args['step']); ?>">
                    <span class="promptor-slider-value"><?php echo esc_html($value); ?></span>
                </div>
                <?php if ( isset( $args['desc'] ) ) { echo '<p class="description">' . esc_html( $args['desc'] ) . '</p>'; } ?>
            </td>
        </tr>
        <?php
    }
    
    private function render_checkbox_field_row($id, $title, $args) {
        $options = get_option('promptor_api_settings', array());
        $checked = !empty($options[$id]);
        ?>
         <tr>
            <th scope="row"><?php echo esc_html($title); ?></th>
            <td>
                <label>
                    <input type="checkbox" id="<?php echo esc_attr($id); ?>" name="promptor_api_settings[<?php echo esc_attr($id); ?>]" value="1" <?php checked($checked, true); ?>>
                    <?php if (isset($args['desc'])) { echo ' ' . esc_html($args['desc']); } ?>
                </label>
            </td>
        </tr>
        <?php
    }
}