<?php
/**
 * Main plugin class
 *
 * @package WooCommerce_Product_Video_Showcase
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Woo_Product_Video_Showcase {

    public function init() {
        // Load plugin text domain
        add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

        // Add custom fields to product editor
        add_action( 'woocommerce_product_options_general_product_data', array( $this, 'add_video_url_field' ) );

        // Save custom field values
        add_action( 'woocommerce_process_product_meta', array( $this, 'save_video_url_field' ) );

        // Enqueue scripts and styles
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

        // Add video container to product thumbnail
        add_filter( 'woocommerce_product_get_image', array( $this, 'add_video_container' ), 10, 4 );

        // Remove product title on hover
        add_action( 'woocommerce_before_shop_loop', array( $this, 'remove_product_title' ) );

        // Ensure the product container has relative positioning
        add_action( 'woocommerce_before_shop_loop_item', array( $this, 'add_product_container_class' ), 5 );
        add_action( 'woocommerce_after_shop_loop_item', array( $this, 'close_product_container' ), 15 );
    }

    public function load_plugin_textdomain() {
        load_plugin_textdomain( 'product-video-showcase-for-woocommerce', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
    }

    public function add_video_url_field() {
        woocommerce_wp_text_input(
            array(
                'id' => '_product_video_url',
                'label' => __( 'Product Video URL', 'product-video-showcase-for-woocommerce' ),
                'description' => __( 'Enter the URL (YouTube or Vimeo) of the product preview video or select from media library.', 'product-video-showcase-for-woocommerce' ),
                'desc_tip' => true,
            )
        );
        echo '<p class="form-field"><button type="button" class="button wpvs-upload-video">' . esc_html__( 'Select Video', 'product-video-showcase-for-woocommerce' ) . '</button></p>';
        
        woocommerce_wp_select(
            array(
                'id' => '_product_video_fit',
                'label' => __( 'Video Fit', 'product-video-showcase-for-woocommerce' ),
                'description' => __( 'Select how the video should fit within the product image area.', 'product-video-showcase-for-woocommerce' ),
                'desc_tip' => true,
                'options' => array(
                    'cover' => __( 'Cover', 'product-video-showcase-for-woocommerce' ),
                    'contain' => __( 'Contain', 'product-video-showcase-for-woocommerce' ),
                    'fill' => __( 'Fill', 'product-video-showcase-for-woocommerce' ),
                    'scale-down' => __( 'Scale Down', 'product-video-showcase-for-woocommerce' ),
                )
            )
        );
    }

    public function save_video_url_field( $post_id ) {
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['_wpnonce'] ), 'update-post_' . $post_id ) ) {
            return;
        }

        $video_url = isset( $_POST['_product_video_url'] ) ? sanitize_text_field( wp_unslash( $_POST['_product_video_url'] ) ) : '';
        $video_fit = isset( $_POST['_product_video_fit'] ) ? sanitize_text_field( wp_unslash( $_POST['_product_video_fit'] ) ) : 'cover';
        
        if ( ! empty( $video_url ) ) {
            update_post_meta( $post_id, '_product_video_url', esc_url_raw( $video_url ) );
        }
        update_post_meta( $post_id, '_product_video_fit', $video_fit );
    }

    public function enqueue_scripts() {
        // Only load on shop/product pages and admin
        if (!is_admin() && !is_shop() && !is_product() && !is_product_category()) {
            return;
        }

        wp_enqueue_style('wpvs-style', WPVS_PLUGIN_URL . 'assets/css/style.css', array(), WPVS_VERSION);
        
        // Add the debug parameter to help with troubleshooting
        wp_enqueue_script('wpvs-script', WPVS_PLUGIN_URL . 'assets/js/script.js', array('jquery'), WPVS_VERSION . '.' . time(), true);
        
        if (is_admin()) {
            wp_enqueue_media();
            wp_enqueue_script('wpvs-admin-script', WPVS_PLUGIN_URL . 'assets/js/admin-script.js', array('jquery'), WPVS_VERSION, true);
        }
    }

    public function add_video_container($html, $product, $size, $attr) {
        // Add debugging
        error_log('Adding video container for product: ' . $product->get_id());
        
        $video_url = get_post_meta($product->get_id(), '_product_video_url', true);
        $video_fit = get_post_meta($product->get_id(), '_product_video_fit', true) ?: 'cover';
        
        if (!empty($video_url)) {
            $video_type = $this->get_video_type($video_url);
            
            // Create video container with all necessary data attributes
            $video_container = sprintf(
                '<div class="wpvp-video-container" data-video-url="%s" data-video-type="%s" data-video-fit="%s"></div>',
                esc_url($video_url),
                esc_attr($video_type),
                esc_attr($video_fit)
            );
            
            // Insert the video container before the image
            $html = $video_container . $html;
            
            // Add debugging
            error_log('Video container added with URL: ' . $video_url);
        }

        return $html;
    }

    public function remove_product_title() {
        remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );
    }

    public function add_product_container_class() {
        echo '<div class="product-inner" style="position: relative; overflow: hidden;">';
    }

    public function close_product_container() {
        echo '</div>';
    }

    private function get_video_type( $url ) {
        $youtube_regex = '/^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be)\/.+/';
        $vimeo_regex = '/^(https?:\/\/)?(www\.)?(vimeo\.com)\/.+/';
        
        if ( preg_match( $youtube_regex, $url ) ) {
            return 'youtube';
        } elseif ( preg_match( $vimeo_regex, $url ) ) {
            return 'vimeo';
        } else {
            return 'direct';
        }
    }
}