<?php
/*
Plugin Name: Product Vendor Manager for WooCommerce
Description: Easily assign and manage product vendors in WooCommerce. This plugin adds a dedicated vendor tab to each product edit screen, allowing you to track vendor details like contact person, phone, email, address, and notes — all from within your WooCommerce dashboard. Perfect for stores working with multiple vendors and seeking centralized, per-product vendor management.
Version: 1.1
Author: Tuloss Solutions
Author URI: https://x.tuloss.com
Plugin URI: https://x.tuloss.com/product-vendor-manager-for-woocommerce/
Text Domain: product-vendor-manager-for-woocommerce
Domain Path: /languages
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Requires at least: 5.0
Tested up to: 6.8
Requires PHP: 7.2
WC requires at least: 3.0
WC tested up to: 6.4
Tags: WooCommerce, supplier management, vendor management, product vendor, product vendors, product supplier, product suppliers, multi-supplier, multi-vendor, product management, WooCommerce supplier management, WooCommerce vendor management, WooCommerce suppliers, WooCommerce vendors, custom fields
Requires Plugins: woocommerce
*/

// Prevent direct file access
if (!defined('ABSPATH')) {
	exit;
}

// Add Settings link to the plugin action links 
function tulosspvmw_add_settings_link($links) {
    // Create the URL to the vendor taxonomy admin page
    $settings_link = '<a href="' . admin_url('edit-tags.php?taxonomy=tulosspvmw_vendor&post_type=product') . '" aria-label="Manage product vendors">' . __('Manage Vendors', 'product-vendor-manager-for-woocommerce') . '</a>';

    // Add the link at the beginning of the links array
    array_unshift($links, $settings_link);
    
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'tulosspvmw_add_settings_link');


// Enqueue the CSS file for the Vendor column style
add_action('admin_enqueue_scripts', 'tulosspvmw_enqueue_admin_styles');
function tulosspvmw_enqueue_admin_styles() {
    $screen = get_current_screen();
    if ($screen && 'edit-product' === $screen->id) {
        wp_enqueue_style(
            'tulosspvmw-admin-styles',
            plugin_dir_url(__FILE__) . 'tulosspvmw-admin-styles.css',
            array(),
            filemtime(plugin_dir_path(__FILE__) . 'tulosspvmw-admin-styles.css'),
            'all'
        );
    }
}

// Create a Vendor taxonomy
add_action('init', 'tulosspvmw_register_vendor_taxonomy');
function tulosspvmw_register_vendor_taxonomy() {
	$labels = array(
		'name'              => _x('Vendors', 'taxonomy general name', 'product-vendor-manager-for-woocommerce'),
		'singular_name'     => _x('Vendor', 'taxonomy singular name', 'product-vendor-manager-for-woocommerce'),
		'search_items'      => __('Search vendors', 'product-vendor-manager-for-woocommerce'),
		'all_items'         => __('All vendors', 'product-vendor-manager-for-woocommerce'),
		'edit_item'         => __('Edit vendor', 'product-vendor-manager-for-woocommerce'),
		'update_item'       => __('Update vendor', 'product-vendor-manager-for-woocommerce'),
		'add_new_item'      => __('Add new vendor', 'product-vendor-manager-for-woocommerce'),
		'new_item_name'     => __('New vendor name', 'product-vendor-manager-for-woocommerce'),
		'menu_name'         => __('Vendors', 'product-vendor-manager-for-woocommerce')
	);

    $args = array(
        'hierarchical'      => true,
        'labels'            => $labels,
        'show_ui'           => true,
        'show_admin_column' => true,
        'query_var'         => true,
        'rewrite'           => array('slug' => 'tulosspvmw_vendor'),
    );

    register_taxonomy('tulosspvmw_vendor', 'product', $args);
}

// Add custom fields in edit form
function tulosspvmw_vendor_add_custom_fields($term = null) {
	wp_nonce_field('tulosspvmw_save_vendor_meta', 'tulosspvmw_vendor_nonce');
    
	$fields = array(
        'vendor_contact' => __('Contact Person', 'product-vendor-manager-for-woocommerce'),
        'vendor_phone'   => __('Phone', 'product-vendor-manager-for-woocommerce'),
        'vendor_email'   => __('Email', 'product-vendor-manager-for-woocommerce'),
        'vendor_address' => __('Address', 'product-vendor-manager-for-woocommerce')
    );
    
    foreach ($fields as $field => $label) {
        $value = ($term && is_object($term)) ? get_term_meta($term->term_id, $field, true) : '';
        
        echo '<div class="form-field term-group">';
        echo '<label for="'.esc_attr($field).'">'.esc_html($label).'</label>';
        echo '<input type="text" id="'.esc_attr($field).'" name="'.esc_attr($field).'" value="'.esc_attr($value).'" />';
        echo '</div>';
    }
}
add_action('tulosspvmw_vendor_add_form_fields', 'tulosspvmw_vendor_add_custom_fields', 10, 1);

// Add custom fields in add form
function tulosspvmw_vendor_edit_custom_fields($term) {
    wp_nonce_field('tulosspvmw_save_vendor_meta', 'tulosspvmw_vendor_nonce');
    $fields = array(
        'vendor_contact' => __('Contact Person', 'product-vendor-manager-for-woocommerce'),
        'vendor_phone'	 => __('Phone', 'product-vendor-manager-for-woocommerce'),
        'vendor_email'	 => __('Email', 'product-vendor-manager-for-woocommerce'),
        'vendor_address' => __('Address', 'product-vendor-manager-for-woocommerce')
    );

    foreach ($fields as $field => $label) {
        $value = get_term_meta($term->term_id, $field, true);

        echo '<tr class="form-field term-group-wrap">';
        echo '<th scope="row"><label for="'.esc_attr($field).'">'.esc_html($label).'</label></th>';
        echo '<td><input type="text" id="'.esc_attr($field).'" name="'.esc_attr($field).'" value="'.esc_attr($value).'" /></td>';
        echo '</tr>';
    }
}
add_action('tulosspvmw_vendor_edit_form_fields', 'tulosspvmw_vendor_edit_custom_fields', 10, 1);


// Save the custom fields
add_action('created_tulosspvmw_vendor', 'tulosspvmw_save_vendor_custom_fields', 10, 1);
add_action('edited_tulosspvmw_vendor', 'tulosspvmw_save_vendor_custom_fields', 10, 1);
function tulosspvmw_save_vendor_custom_fields($term_id) {
    // Verify nonce properly
    if (!isset($_POST['tulosspvmw_vendor_nonce']) || !wp_verify_nonce(
            sanitize_text_field( wp_unslash( $_POST['tulosspvmw_vendor_nonce'] ) ), 
            'tulosspvmw_save_vendor_meta'
        )
    ) {
        return;
    }

    $fields = array('vendor_contact', 'vendor_phone', 'vendor_email', 'vendor_address');

    foreach ($fields as $field) {
        if (isset($_POST[$field])) {
            // Unslash and sanitize
            $value = sanitize_text_field( wp_unslash( $_POST[$field] ) );
            update_term_meta($term_id, $field, $value);
        } else {
            delete_term_meta($term_id, $field);
        }
    }
}

// Rename the "Description" field to "Notes"
add_filter('gettext', 'tulosspvmw_rename_description_to_notes', 10, 3);
function tulosspvmw_rename_description_to_notes($translated_text, $text, $domain) {
    if (is_admin() && 
    'Description' === $text && 
    isset($_GET['taxonomy']) && 
    'tulosspvmw_vendor' === wp_verify_nonce(
        sanitize_text_field( wp_unslash( $_GET['taxonomy'] ) ), 
        'tulosspvmw_save_vendor_meta'
    )
    ) {
        return __('Notes', 'product-vendor-manager-for-woocommerce');
    }
    return $translated_text;
}

// Reorganize columns
add_filter('manage_edit-tulosspvmw_vendor_columns', 'tulosspvmw_modify_vendor_columns');
function tulosspvmw_modify_vendor_columns($columns) {
    $new_columns = array(
        'cb'             => $columns['cb'],
        'name'           => __('Vendor', 'product-vendor-manager-for-woocommerce'),
        'vendor_contact' => __('Contact Person', 'product-vendor-manager-for-woocommerce'),
        'vendor_phone'   => __('Phone', 'product-vendor-manager-for-woocommerce'),
        'vendor_email'   => __('Email', 'product-vendor-manager-for-woocommerce'),
        'vendor_address' => __('Address', 'product-vendor-manager-for-woocommerce'),
        'posts'          => $columns['posts']
    );
    return $new_columns;
}

// Display column values
add_action('manage_tulosspvmw_vendor_custom_column', 'tulosspvmw_display_vendor_custom_columns', 10, 3);
function tulosspvmw_display_vendor_custom_columns($content, $column_name, $term_id) {
    $fields = array(
        'vendor_contact' => get_term_meta($term_id, 'vendor_contact', true),
        'vendor_phone'   => get_term_meta($term_id, 'vendor_phone', true),
        'vendor_email'   => get_term_meta($term_id, 'vendor_email', true),
        'vendor_address' => get_term_meta($term_id, 'vendor_address', true)
    );
    return isset($fields[$column_name]) ? esc_html($fields[$column_name]) : $content;
}
?>