<?php

namespace CodenericProductPhotoAI\Core;

use CodenericProductPhotoAI\Traits\Base;

/**
 * Class WooCommerce
 *
 * Handles WooCommerce integration for Product Photo AI.
 *
 * @package ProductPhotoAI\Core
 */
class WooCommerce
{
    use Base;

    /**
     * Initialize WooCommerce integration
     *
     * @return void
     */
    public function init()
    {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }

        // Add product meta box
        add_action('add_meta_boxes', array($this, 'add_product_meta_box'));
        
        // Add product list column
        add_filter('manage_edit-product_columns', array($this, 'add_product_column'));
        add_action('manage_product_posts_custom_column', array($this, 'populate_product_column'), 10, 2);
        
        // Add bulk actions
        add_filter('bulk_actions-edit-product', array($this, 'add_bulk_actions'));
        add_filter('handle_bulk_actions-edit-product', array($this, 'handle_bulk_actions'), 10, 3);
        
        // Add admin notices
        add_action('admin_notices', array($this, 'bulk_action_notices'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // AJAX handlers for the simplified flow
        add_action('wp_ajax_cppai_generate', array($this, 'ajax_generate_photos'));
        add_action('wp_ajax_cppai_check_status', array($this, 'ajax_check_status'));
        add_action('wp_ajax_cppai_associate_image', array($this, 'ajax_associate_image'));
        add_action('wp_ajax_cppai_get_photos', array($this, 'ajax_get_photos'));
    }

    /**
     * Display notice if WooCommerce is not active
     *
     * @return void
     */
    public function woocommerce_missing_notice()
    {
        ?>
        <div class="notice notice-error">
            			<p><?php esc_html_e('Product Photo AI requires WooCommerce to be installed and activated.', 'product-photo-ai-for-woocommerce'); ?></p>
        </div>
        <?php
    }

    /**
     * Add Product Photo AI meta box to product edit page
     *
     * @return void
     */
    public function add_product_meta_box()
    {
        		add_meta_box(
			'product-photo-ai-for-woocommerce-meta-box',
			__('Product Photo AI', 'product-photo-ai-for-woocommerce'),
            array($this, 'render_product_meta_box'),
            'product',
            'side',
            'default'
        );
    }

    /**
     * Render the Product Photo AI meta box
     *
     * @param WP_Post $post
     * @return void
     */
    public function render_product_meta_box($post)
    {
        $product = wc_get_product($post->ID);
        if (!$product) {
            return;
        }

        $image_id = $product->get_image_id();
        $job_ids = get_post_meta($post->ID, '_photoai_job_ids', true);
        $saved_images = get_post_meta($post->ID, '_photoai_generated_image_ids', true);
        $saved_count = is_array($saved_images) ? count($saved_images) : 0;
        
        wp_nonce_field('cppai_meta_box', 'cppai_nonce');
        
        // Check if API key is configured
        $settings = \CodenericProductPhotoAI\Core\Settings::get_instance();
        $api_key = $settings->get_setting('api_key', '');
        
        echo '<div id="cppai-container" data-product-id="' . esc_attr($post->ID) . '">';
        
        if (empty($api_key)) {
            echo '<div class="notice notice-error inline">';
            echo '<p>' . esc_html__('PhotoAI API key is not configured.', 'product-photo-ai-for-woocommerce') . ' ';
            echo '<a href="' . esc_url(admin_url('admin.php?page=cppai-settings')) . '" target="_blank">';
            echo esc_html__('Configure API key', 'product-photo-ai-for-woocommerce');
            echo '</a></p>';
            echo '</div>';
        } else {
            echo '<button type="button" id="open-photoai-modal" class="button button-primary button-large" ';
            echo 'data-product-id="' . esc_attr($post->ID) . '" ';
            echo 'data-seed-image-id="' . esc_attr($image_id) . '" ';
            echo 'data-seed-image-url="' . ($image_id ? esc_url(wp_get_attachment_image_url($image_id, 'medium')) : '') . '">';
            echo '<span class="dashicons dashicons-camera cppai-dashicon-camera"></span>';
            echo esc_html__('Generate AI Photos...', 'product-photo-ai-for-woocommerce');
            echo '</button>';
            
            if ($saved_count > 0) {
                echo '<p class="photoai-status-summary">';
                echo '<span class="dashicons dashicons-images-alt2 cppai-dashicon-success"></span>';
                // translators: %d is the number of AI images generated
                printf(esc_html(_n('%d AI image generated', '%d AI images generated', $saved_count, 'product-photo-ai-for-woocommerce')), esc_html($saved_count));
                echo '</p>';
            }
        }
        
        echo '</div>';
        echo '<!-- React modal will be rendered dynamically -->';

        // Enqueue inline style for the modal button
        wp_add_inline_style('wp-admin', '
            #open-photoai-modal {
                width: 100%;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 12px;
            }
        ');
    }

    /**
     * Add Product Photo AI column to products list
     *
     * @param array $columns
     * @return array
     */
    public function add_product_column($columns)
    {
        		$columns['photoai_status'] = __('AI Photos', 'product-photo-ai-for-woocommerce');
        return $columns;
    }

    /**
     * Populate Product Photo AI column in products list
     *
     * @param string $column
     * @param int $post_id
     * @return void
     */
    public function populate_product_column($column, $post_id)
    {
        if ($column === 'photoai_status') {
            $generated_images = get_post_meta($post_id, '_photoai_generated_images', true);
            $job_ids = get_post_meta($post_id, '_photoai_job_ids', true);
            
            if ($job_ids && is_array($job_ids)) {
                			echo '<span class="photoai-status processing">' . esc_html__('Processing...', 'product-photo-ai-for-woocommerce') . '</span>';
            } elseif ($generated_images && is_array($generated_images)) {
                $count = count($generated_images);
                // translators: %d is the number of photos generated
                			echo '<span class="photoai-status completed">' . esc_html(sprintf(_n('%d photo', '%d photos', $count, 'product-photo-ai-for-woocommerce'), $count)) . '</span>';
            } else {
                echo '<span class="photoai-status none">—</span>';
            }
        }
    }

    /**
     * Add bulk actions for generating AI photos
     *
     * @param array $bulk_actions
     * @return array
     */
    public function add_bulk_actions($bulk_actions)
    {
        		$bulk_actions['generate_ai_photos'] = __('Generate AI Photos', 'product-photo-ai-for-woocommerce');
        return $bulk_actions;
    }

    /**
     * Handle bulk action for generating AI photos
     *
     * @param string $redirect_to
     * @param string $doaction
     * @param array $post_ids
     * @return string
     */
    public function handle_bulk_actions($redirect_to, $doaction, $post_ids)
    {
        if ($doaction !== 'generate_ai_photos') {
            return $redirect_to;
        }

        $processed = 0;
        foreach ($post_ids as $post_id) {
            $product = wc_get_product($post_id);
            if ($product && $product->get_image_id()) {
                // Schedule AI generation (simplified - in real implementation you'd queue this)
                update_post_meta($post_id, '_photoai_bulk_queued', time());
                $processed++;
            }
        }

        $redirect_to = add_query_arg('bulk_ai_photos_generated', $processed, $redirect_to);
        return $redirect_to;
    }

    /**
     * Display bulk action notices
     *
     * @return void
     */
    public function bulk_action_notices()
    {
        if (!empty($_REQUEST['bulk_ai_photos_generated'])) {
            // Verify nonce for bulk actions
            if (!isset($_REQUEST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])), 'bulk-posts')) {
                return;
            }
            
            $count = isset($_REQUEST['bulk_ai_photos_generated']) ? intval(sanitize_text_field(wp_unslash($_REQUEST['bulk_ai_photos_generated']))) : 0;
            printf(
                '<div id="message" class="updated notice is-dismissible"><p>%s</p></div>',
                // translators: %d is the number of products that AI photo generation started for
                					esc_html(sprintf(_n('AI photo generation started for %d product.', 'AI photo generation started for %d products.', $count, 'product-photo-ai-for-woocommerce'), $count))
            );
        }
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook
     * @return void
     */
    public function enqueue_admin_assets($hook)
    {
        if (!in_array($hook, array('post.php', 'post-new.php', 'edit.php'))) {
            return;
        }

        global $post_type;
        if ($post_type !== 'product') {
            return;
        }

        // Enqueue WordPress media library
        wp_enqueue_media();

        		wp_enqueue_script(
			'cppai-admin-legacy',
            CPPAI_ASSETS_URL . '/js/admin.js',
            array('jquery', 'media-upload', 'media-views'),
            CPPAI_VERSION,
            true
        );

        		wp_localize_script('cppai-admin', 'cppai_data', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'rest_url' => rest_url(),
            'nonce' => wp_create_nonce('cppai_nonce'),
            'rest_nonce' => wp_create_nonce('wp_rest'),
            'strings' => array(
                			'generating' => __('Generating photos...', 'product-photo-ai-for-woocommerce'),
			'completed' => __('Generation completed!', 'product-photo-ai-for-woocommerce'),
			'error' => __('An error occurred. Please try again.', 'product-photo-ai-for-woocommerce'),
			'no_image' => __('Please add a product image first.', 'product-photo-ai-for-woocommerce'),
            )
        ));

        // CSS is now bundled with the JavaScript file via Vite
        // No need to enqueue separate CSS file
    }

    /**
     * AJAX handler for generating photos
     *
     * @return void
     */
    public function ajax_generate_photos()
    {
        check_ajax_referer('cppai_nonce', 'nonce');

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $num_shots = isset($_POST['num_shots']) ? intval($_POST['num_shots']) : 1;
        $style = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'professional';
        $additional_instructions = isset($_POST['additional_instructions']) ? sanitize_textarea_field(wp_unslash($_POST['additional_instructions'])) : '';
        $seed_image_id = isset($_POST['seed_image_id']) ? intval($_POST['seed_image_id']) : 0;
        
        // If no additional instructions provided, set default message
        if (empty(trim($additional_instructions))) {
            $additional_instructions = 'no additional instructions';
        }

        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        if (!$seed_image_id) {
            wp_send_json_error('Please select a seed image before generating photos');
        }

        // Call the PhotoAI controller directly
        $controller = new \CodenericProductPhotoAI\Controllers\PhotoAI\Actions();
        $result = $controller->generate_product_photos_direct($product_id, $num_shots, $style, $additional_instructions, $seed_image_id);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX handler for checking photo status
     *
     * @return void
     */
    public function ajax_check_status()
    {
        check_ajax_referer('cppai_nonce', 'nonce');

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        $controller = new \CodenericProductPhotoAI\Controllers\PhotoAI\Actions();
        $result = $controller->check_photo_status_direct($product_id);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX handler for associating an uploaded image with a product.
     *
     * @return void
     */
    public function ajax_associate_image()
    {
        check_ajax_referer('cppai_nonce', 'nonce');

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;

        if (!$product_id || !$attachment_id) {
            wp_send_json_error('Missing required parameters.');
        }

        // Verify the attachment exists
        if (!wp_attachment_is_image($attachment_id)) {
            wp_send_json_error('Invalid attachment ID.');
        }

        // Add the attachment ID to the product's generated images list
        $generated_ids = get_post_meta($product_id, '_photoai_generated_image_ids', true);
        if (!is_array($generated_ids)) {
            $generated_ids = array();
        }
        
        $generated_ids[] = $attachment_id;
        update_post_meta($product_id, '_photoai_generated_image_ids', array_unique($generated_ids));

        wp_send_json_success('Image associated with product successfully.');
    }

    /**
     * AJAX handler for getting generated photos
     *
     * @return void
     */
    public function ajax_get_photos()
    {
        check_ajax_referer('cppai_nonce', 'nonce');

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        if (!$product_id) {
            wp_send_json_error('Invalid product ID');
        }

        $controller = new \CodenericProductPhotoAI\Controllers\PhotoAI\Actions();
        $result = $controller->get_generated_photos_direct($product_id);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }
} 