<?php

namespace CodenericProductPhotoAI\Core;

use CodenericProductPhotoAI\Traits\Base;

/**
 * Class Settings
 *
 * Handles plugin settings and configuration.
 *
 * @package ProductPhotoAI\Core
 */
class Settings
{
    use Base;

    /**
     * Settings option name
     */
    const OPTION_NAME = 'cppai_settings';

    /**
     * Default settings
     */
    private $default_settings = array(
        'api_key' => '',
        'default_style' => 'professional'
    );

    /**
     * Initialize settings
     *
     * @return void
     */
    public function init()
    {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_settings_scripts'));
    }

    /**
     * Register plugin settings
     *
     * @return void
     */
    public function register_settings()
    {
        register_setting(
            'cppai_settings_group',
            self::OPTION_NAME,
            array($this, 'sanitize_settings')
        );

        add_settings_section(
            'cppai_general_section',
            __('General Settings', 'product-photo-ai-for-woocommerce'),
            array($this, 'general_section_callback'),
            'cppai-settings'
        );

        add_settings_field(
            'api_key',
            			__('API Key', 'product-photo-ai-for-woocommerce'),
			array($this, 'api_key_callback'),
			'cppai-settings',
            'cppai_general_section'
        );

        add_settings_field(
            'default_style',
            			__('Default Style', 'product-photo-ai-for-woocommerce'),
			array($this, 'default_style_callback'),
			'cppai-settings',
            'cppai_general_section'
        );
    }

    /**
     * Add settings page to admin menu
     *
     * @return void
     */
    public function add_settings_page()
    {
        		add_submenu_page(
			'cppai-menu',
			__('Settings', 'product-photo-ai-for-woocommerce'),
			__('Settings', 'product-photo-ai-for-woocommerce'),
			'manage_options',
			'cppai-settings',
            array($this, 'settings_page_callback')
        );
    }

    /**
     * Enqueue scripts for settings page
     *
     * @param string $hook_suffix
     * @return void
     */
    public function enqueue_settings_scripts($hook_suffix)
    {
        // Check if we're on the settings page
        // The hook suffix contains our settings page slug
        if (strpos($hook_suffix, 'cppai-settings') === false) {
            return;
        }

        // Enqueue jQuery (in case it's not already enqueued)
        wp_enqueue_script('jquery');

        // Add inline script for API key functionality
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                var toggleButton = $("#toggle-api-key");
                var apiKeyInput = $("#api_key");
                
                if (toggleButton.length && apiKeyInput.length) {
                    toggleButton.on("click", function(e) {
                        e.preventDefault();
                        
                        var input = apiKeyInput;
                        var button = $(this);
                        var currentType = input.attr("type");
                        
                        if (currentType === "password") {
                            input.attr("type", "text");
                            button.text("Hide");
                        } else {
                            input.attr("type", "password");
                            button.text("Show");
                        }
                    });
                    
                    // Add real-time validation feedback
                    apiKeyInput.on("input", function() {
                        var input = $(this);
                        var value = input.val().trim();
                        var container = input.closest(".api-key-container");
                        
                        // Remove existing validation messages
                        container.nextAll(".validation-message").remove();
                        
                        if (value.length > 0 && !value.startsWith("ak_")) {
                            var errorMsg = $("<p>")
                                .addClass("description validation-message")
                                .css("color", "#dc3232")
                                .html("<strong>⚠ API Key must start with \\"ak_\\"</strong>");
                            container.after(errorMsg);
                        } else if (value.startsWith("ak_") && value.length > 3) {
                            var successMsg = $("<p>")
                                .addClass("description validation-message")
                                .css("color", "#46b450")
                                .html("<strong>✓ Valid API Key format</strong>");
                            container.after(successMsg);
                        }
                    });
                }
            });
        ');
    }

    /**
     * Settings page callback
     *
     * @return void
     */
    public function settings_page_callback()
    {
        ?>
        <div class="wrap">
            			<h1><?php esc_html_e('Product Photo AI Settings', 'product-photo-ai-for-woocommerce'); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('cppai_settings_group');
                do_settings_sections('cppai-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * General section callback
     *
     * @return void
     */
    public function general_section_callback()
    {
        echo '<p>' . esc_html__('Configure the general settings for Product Photo AI.', 'product-photo-ai-for-woocommerce') . '</p>';
    }

    /**
     * API key field callback
     *
     * @return void
     */
    public function api_key_callback()
    {
        $settings = $this->get_settings();
        $api_key = $settings['api_key'];
        $is_valid_format = $this->is_valid_api_key_format($api_key);
        
        echo '<div class="api-key-container">';
        echo '<input type="password" id="api_key" name="' . esc_attr(self::OPTION_NAME) . '[api_key]" value="' . esc_attr($api_key) . '" class="regular-text" placeholder="ak_your_api_key_here" />';
        echo '<button type="button" id="toggle-api-key" class="button cppai-toggle-api-key">' . esc_html__('Show', 'product-photo-ai-for-woocommerce') . '</button>';
        echo '</div>';
        echo '<p class="description">' . esc_html__('Your PhotoAI API authentication key. Must start with "ak_". Keep this secure and do not share it.', 'product-photo-ai-for-woocommerce') . '</p>';
        
        if (!empty($api_key)) {
            if ($is_valid_format) {
                echo '<p class="description cppai-status-success"><strong>' . esc_html__('✓ API Key configured and valid format', 'product-photo-ai-for-woocommerce') . '</strong></p>';
            } else {
                echo '<p class="description cppai-status-error"><strong>' . esc_html__('⚠ API Key format invalid. Must start with "ak_"', 'product-photo-ai-for-woocommerce') . '</strong></p>';
            }
        } else {
            echo '<p class="description cppai-status-error"><strong>' . esc_html__('⚠ API Key required for PhotoAI to function. Get your free API key at ', 'product-photo-ai-for-woocommerce') . '<a href="https://product-photo-ai.com" target="_blank">product-photo-ai.com</a></strong></p>';
        }
    }

    /**
     * Default style field callback
     *
     * @return void
     */
    public function default_style_callback()
    {
        $settings = $this->get_settings();
        $styles = array(
            'professional' => __('Professional', 'product-photo-ai-for-woocommerce'),
            'lifestyle' => __('Lifestyle', 'product-photo-ai-for-woocommerce'),
            'minimal' => __('Minimal', 'product-photo-ai-for-woocommerce'),
            'luxury' => __('Luxury', 'product-photo-ai-for-woocommerce')
        );

        echo '<select id="default_style" name="' . esc_attr(self::OPTION_NAME) . '[default_style]">';
        foreach ($styles as $value => $label) {
            echo '<option value="' . esc_attr($value) . '"' . selected($settings['default_style'], $value, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Default background style for generated photos.', 'product-photo-ai-for-woocommerce') . '</p>';
    }

    /**
     * Validate API key format
     *
     * @param string $api_key
     * @return bool
     */
    private function is_valid_api_key_format($api_key)
    {
        return !empty($api_key) && strpos($api_key, 'ak_') === 0 && strlen($api_key) > 3;
    }

    /**
     * Sanitize settings
     *
     * @param array $input
     * @return array
     */
    public function sanitize_settings($input)
    {
        $sanitized = array();

        if (isset($input['api_key'])) {
            $api_key = sanitize_text_field($input['api_key']);
            
            // Validate API key format
            if (!empty($api_key) && !$this->is_valid_api_key_format($api_key)) {
                add_settings_error(
                    self::OPTION_NAME,
                    'invalid_api_key_format',
                    __('API Key must start with "ak_" and cannot be empty.', 'product-photo-ai-for-woocommerce'),
                    'error'
                );
                // Keep the invalid key so user can see and fix it
                $sanitized['api_key'] = $api_key;
            } else {
                $sanitized['api_key'] = $api_key;
            }
        }

        if (isset($input['default_style'])) {
            $allowed_styles = array('professional', 'lifestyle', 'minimal', 'luxury');
            $sanitized['default_style'] = in_array($input['default_style'], $allowed_styles) ? $input['default_style'] : 'professional';
        }

        return $sanitized;
    }

    /**
     * Get settings with defaults
     *
     * @return array
     */
    public function get_settings()
    {
        $settings = get_option(self::OPTION_NAME, array());
        return wp_parse_args($settings, $this->default_settings);
    }

    /**
     * Get a specific setting value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function get_setting($key, $default = null)
    {
        $settings = $this->get_settings();
        return isset($settings[$key]) ? $settings[$key] : $default;
    }

    /**
     * Update a specific setting
     *
     * @param string $key
     * @param mixed $value
     * @return bool
     */
    public function update_setting($key, $value)
    {
        $settings = $this->get_settings();
        $settings[$key] = $value;
        return update_option(self::OPTION_NAME, $settings);
    }
} 