<?php

namespace CodenericProductPhotoAI\Controllers\PhotoAI;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Class Actions
 *
 * Handles PhotoAI related actions for WooCommerce integration.
 * 
 * Error Handling:
 * - All API errors are properly parsed from JSON responses
 * - HTTP status codes are mapped to appropriate error types
 * - 401 errors (like "Invalid API key format") are handled as 'unauthorized'
 * - 404 errors are handled as 'not_found'
 * - Rate limiting (429) and server errors (5xx) are properly categorized
 * - Status checking gracefully handles errors by marking jobs as failed
 *
 * @package ProductPhotoAI\Controllers\PhotoAI
 */
class Actions
{
    /**
     * Get API endpoint base URL from settings
     *
     * @return string
     */
    private function get_api_base_url()
    {
        $base_url = defined('CPPAI_API_URL') ? CPPAI_API_URL : 'https://product-photo-ai.com/api/photoai/v1';
        return $base_url;
    }

    /**
     * Get API key from settings
     *
     * @return string
     */
    private function get_api_key()
    {
        $settings = \CodenericProductPhotoAI\Core\Settings::get_instance();
        return $settings->get_setting('api_key', '');
    }

    /**
     * Validate API key format
     *
     * @param string $api_key
     * @return bool
     */
    private function is_valid_api_key_format($api_key)
    {
        return !empty($api_key) && strpos($api_key, 'ak_') === 0 && strlen($api_key) > 3;
    }

    /**
     * Parse API error response and return appropriate WP_Error
     *
     * @param string $response_body
     * @param int $response_code
     * @return WP_Error
     */
    private function handle_api_error($response_body, $response_code)
    {
        $error_data = json_decode($response_body, true);
        
        // Default error message
        $error_message = 'PhotoAI service returned an error';
        
        // Try to extract error message from JSON response
        if ($error_data && isset($error_data['error'])) {
            $error_message = $error_data['error'];
        } elseif ($error_data && isset($error_data['message'])) {
            $error_message = $error_data['message'];
        } else {
            // Fallback to raw response body if JSON parsing fails
            $error_message = !empty($response_body) ? $response_body : $error_message;
        }

        // Determine error code based on HTTP status
        $error_code = 'api_error';
        switch ($response_code) {
            case 401:
                $error_code = 'unauthorized';
                break;
            case 403:
                $error_code = 'forbidden';
                break;
            case 404:
                $error_code = 'not_found';
                break;
            case 429:
                $error_code = 'rate_limited';
                break;
            case 500:
            case 502:
            case 503:
            case 504:
                $error_code = 'server_error';
                break;
        }

        return new WP_Error($error_code, $error_message, array('status' => $response_code));
    }

    /**
     * Get headers for API requests
     *
     * @param array $additional_headers
     * @return array
     */
    private function get_api_headers($additional_headers = array())
    {
        $headers = array();

        $api_key = $this->get_api_key();
        if (!empty($api_key)) {
            $headers['Authorization'] = 'Bearer ' . $api_key;
        }

        return array_merge($headers, $additional_headers);
    }

    /**
     * Generate product photos for a WooCommerce product (REST API endpoint)
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function generate_product_photos(WP_REST_Request $request)
    {
        $product_id = $request->get_param('product_id');
        $num_shots = $request->get_param('num_shots') ?: 1;
        $background_style = $request->get_param('background_style') ?: 'professional';
        $additional_instructions = $request->get_param('additional_instructions') ?: '';
        $seed_image_id = $request->get_param('seed_image_id');

        $result = $this->generate_product_photos_direct($product_id, $num_shots, $background_style, $additional_instructions, $seed_image_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Generate product photos for a WooCommerce product (Direct method)
     *
     * @param int $product_id
     * @param int $num_shots
     * @param string $background_style
     * @param string $additional_instructions
     * @param int $seed_image_id
     * @return array|WP_Error
     */
    public function generate_product_photos_direct($product_id, $num_shots = 1, $background_style = 'professional', $additional_instructions = '', $seed_image_id = null)
    {
        try {
            if (!$product_id) {
                return new WP_Error('missing_product_id', 'Product ID is required', array('status' => 400));
            }

            // Check if API key is configured and valid format
            $api_key = $this->get_api_key();
            if (empty($api_key)) {
                return new WP_Error('missing_api_key', 'PhotoAI API key is not configured. Please go to Product Photo AI → Settings to configure your API key.', array('status' => 401));
            }
            
            if (!$this->is_valid_api_key_format($api_key)) {
                return new WP_Error('invalid_api_key_format', 'PhotoAI API key format is invalid. API key must start with "ak_". Please check your API key in settings.', array('status' => 401));
            }

            // Get WooCommerce product
            $product = wc_get_product($product_id);
            if (!$product) {
                return new WP_Error('product_not_found', 'Product not found', array('status' => 404));
            }

            // Use seed image if provided, otherwise fall back to product image
            $image_id = $seed_image_id ?: $product->get_image_id();
            if (!$image_id) {
                return new WP_Error('no_seed_image', 'No seed image provided', array('status' => 400));
            }

            $image_path = get_attached_file($image_id);

            if (!$image_path || !file_exists($image_path)) {
                return new WP_Error('image_not_found', 'Seed image file not found', array('status' => 400));
            }

            // Prepare form data for API call
            $boundary = wp_generate_uuid4();
            $delimiter = '-------------' . $boundary;

            // Build form data for API call
            $form_data = array(
                'product_image' => array(
                    'name' => basename($image_path),
                    'type' => mime_content_type($image_path),
                    'content' => file_get_contents($image_path)
                ),
                'num_shots' => $num_shots,
                'product_description' => $product->get_name() . ' - ' . $product->get_short_description()
            );

            // Add additional instructions if provided
            if (!empty($additional_instructions)) {
                $form_data['additional_instructions'] = $additional_instructions;
            }

            $post_data = $this->build_multipart_form_data($delimiter, $form_data);

            // Make API call to generate photos
            $response = wp_remote_post($this->get_api_base_url() . '/generate', array(
                'headers' => $this->get_api_headers(array(
                    'Content-Type' => 'multipart/form-data; boundary=' . $delimiter,
                    'Content-Length' => strlen($post_data)
                )),
                'body' => $post_data,
                'timeout' => 60
            ));

            if (is_wp_error($response)) {
                return new WP_Error('api_error', 'Failed to connect to PhotoAI service: ' . $response->get_error_message(), array('status' => 500));
            }

            $response_body = wp_remote_retrieve_body($response);
            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code !== 200) {
                return $this->handle_api_error($response_body, $response_code);
            }

            $api_response = json_decode($response_body, true);

            if (!$api_response || !isset($api_response['photo_ids'])) {
                return new WP_Error('invalid_response', 'Invalid response from PhotoAI service', array('status' => 500));
            }

            // Store job IDs in product meta to track the polling
            update_post_meta($product_id, '_photoai_job_ids', $api_response['photo_ids']);

            // Clean up any old meta fields that are no longer used
            delete_post_meta($product_id, '_photoai_generated_images');
            delete_post_meta($product_id, '_photoai_processed_jobs');
            delete_post_meta($product_id, '_photoai_job_attachments');

            return array(
                'success' => true,
                'message' => 'Photo generation started',
                'job_ids' => $api_response['photo_ids'],
                'product_id' => $product_id
            );
        } catch (\Exception $e) {
            return new WP_Error('server_error', $e->getMessage(), array('status' => 500));
        }
    }

    /**
     * Check the status of photo generation job and images (REST API endpoint)
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function check_photo_status(WP_REST_Request $request)
    {
        $product_id = $request->get_param('product_id');
        $result = $this->check_photo_status_direct($product_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Check the status of photo generation job and images (Direct method)
     *
     * @param int $product_id
     * @return array|WP_Error
     */
    public function check_photo_status_direct($product_id)
    {
        try {
            if (!$product_id) {
                return new WP_Error('missing_product_id', 'Product ID is required', array('status' => 400));
            }

            $image_ids = get_post_meta($product_id, '_photoai_job_ids', true);

            if (!$image_ids || !is_array($image_ids)) {
                return array(
                    'success' => true,
                    'jobs' => [],
                    'all_completed' => true
                );
            }

            $individual_jobs = array();
            $completed_count = 0;
            $all_completed = true;

            // Check each image individually
            foreach ($image_ids as $image_id) {
                $response = wp_remote_get($this->get_api_base_url() . '/photo/' . $image_id, array(
                    'headers' => $this->get_api_headers(),
                    'timeout' => 30
                ));

                if (is_wp_error($response)) {
                    // If API call fails, assume still processing
                    $individual_jobs[] = array(
                        'id' => $image_id,
                        'status' => 'processing',
                        'created_at' => null
                    );
                    $all_completed = false;
                    continue;
                }

                $response_body = wp_remote_retrieve_body($response);
                $response_code = wp_remote_retrieve_response_code($response);

                // Handle non-200 status codes
                if ($response_code !== 200) {
                    // Parse the error response to get a better error message
                    $error_data = json_decode($response_body, true);
                    $error_message = 'API error: ' . $response_code;
                    
                    if ($error_data && isset($error_data['error'])) {
                        $error_message = $error_data['error'];
                    } elseif ($response_code === 404) {
                        $error_message = 'Image not found';
                    }
                    
                    // For status checking, we'll treat API errors as failed jobs
                    $individual_jobs[] = array(
                        'id' => $image_id,
                        'status' => 'failed',
                        'created_at' => null,
                        'error' => $error_message
                    );
                    $completed_count++; // Count as completed (failed)
                    continue;
                }

                $image_data = json_decode($response_body, true);

                if (!$image_data) {
                    // If response is invalid, assume still processing
                    $individual_jobs[] = array(
                        'id' => $image_id,
                        'status' => 'processing',
                        'created_at' => null
                    );
                    $all_completed = false;
                    continue;
                }

                // Transform image data into "job" format expected by frontend
                $job_status = array(
                    'id' => $image_data['id'],
                    'status' => $image_data['status'],
                    'created_at' => $image_data['created_at'],
                    'sequence_number' => $image_data['sequence_number'] ?? null,
                    'shot_description' => $image_data['shot_description'] ?? null
                );

                // If image is completed, add the result URL
                if ($image_data['status'] === 'completed' && !empty($image_data['result_image_url'])) {
                    $job_status['result'] = $image_data['result_image_url'];
                    $completed_count++;
                } else if ($image_data['status'] === 'failed') {
                    $job_status['error'] = $image_data['error_message'] ?? 'Generation failed';
                    $completed_count++; // Count failed as "finished"
                } else {
                    // Still pending or processing
                    $all_completed = false;
                }

                $individual_jobs[] = $job_status;
            }

            // If all images are completed, clean up the job IDs from post meta
            if ($all_completed) {
                delete_post_meta($product_id, '_photoai_job_ids');
            }

            return array(
                'success' => true,
                'product_id' => $product_id,
                'jobs' => $individual_jobs,
                'completed_count' => $completed_count,
                'all_completed' => $all_completed
            );
        } catch (\Exception $e) {
            return new WP_Error('server_error', $e->getMessage(), array('status' => 500));
        }
    }

    /**
     * Build multipart form data for API requests
     *
     * @param string $delimiter
     * @param array $fields
     * @return string
     */
    private function build_multipart_form_data($delimiter, $fields)
    {
        $data = '';

        foreach ($fields as $name => $field) {
            $data .= "--" . $delimiter . "\r\n";

            if (is_array($field) && isset($field['content'])) {
                // File field
                $data .= 'Content-Disposition: form-data; name="' . $name . '"; filename="' . $field['name'] . '"' . "\r\n";
                $data .= 'Content-Type: ' . $field['type'] . "\r\n\r\n";
                $data .= $field['content'] . "\r\n";
            } else {
                // Regular field
                $data .= 'Content-Disposition: form-data; name="' . $name . '"' . "\r\n\r\n";
                $data .= $field . "\r\n";
            }
        }

        $data .= "--" . $delimiter . "--\r\n";

        return $data;
    }

    /**
     * Gets the already-generated images for a product (REST API endpoint)
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_generated_photos(WP_REST_Request $request)
    {
        $product_id = $request->get_param('product_id');
        $result = $this->get_generated_photos_direct($product_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Gets the already-generated images for a product (Direct method)
     *
     * @param int $product_id
     * @return array|WP_Error
     */
    public function get_generated_photos_direct($product_id)
    {
        try {
            if (!$product_id) {
                return new WP_Error('missing_product_id', 'Product ID is required', array('status' => 400));
            }

            $generated_images = get_post_meta($product_id, '_photoai_generated_image_ids', true);
            if (!$generated_images || !is_array($generated_images)) {
                $generated_images = array();
            }

            $images_data = array();
            foreach ($generated_images as $attachment_id) {
                $image_url = wp_get_attachment_url($attachment_id);
                if ($image_url) {
                    $images_data[] = array(
                        'id' => $attachment_id,
                        'url' => $image_url,
                        'thumbnail' => wp_get_attachment_image_url($attachment_id, 'thumbnail'),
                    );
                }
            }
            return array('success' => true, 'images' => $images_data);
        } catch (\Exception $e) {
            return new WP_Error('server_error', $e->getMessage(), array('status' => 500));
        }
    }
}
