<?php

declare(strict_types=1);

namespace CodenericProductPhotoAI\Assets;

use CodenericProductPhotoAI\Traits\Base;
use CodenericProductPhotoAI\Libs\Assets;

/**
 * Class Admin
 *
 * Handles admin functionalities for the ProductPhotoAI.
 *
 * @package ProductPhotoAI\Admin
 */
class Admin
{

	use Base;

	/**
	 * Script handle for ProductPhotoAI.
	 */
	const HANDLE = 'cppai-admin';

	/**
	 * JS Object name for ProductPhotoAI.
	 */
	const OBJ_NAME = 'CPPAI';

	/**
	 * Development script path for ProductPhotoAI.
	 */
	const DEV_SCRIPT = 'src/admin/main.tsx';

	/**
	 * List of allowed screens for script enqueue.
	 *
	 * @var array
	 */
	private $allowed_screens = array(
		'toplevel_page_cppai-menu', // Main admin dashboard page
		'product-photo-ai_page_cppai-settings', // Settings submenu page
		'post.php',     // Product edit page
		'post-new.php', // New product page
		'edit.php',     // Products list page
	);

	/**
	 * Frontend bootstrapper.
	 *
	 * @return void
	 */
	public function bootstrap()
	{
		add_action('admin_enqueue_scripts', array($this, 'enqueue_script'));
	}

	/**
	 * Enqueue script based on the current screen.
	 *
	 * @param string $screen The current screen.
	 */
	public function enqueue_script($screen)
	{
		$current_screen = $screen;

		if (in_array($current_screen, $this->allowed_screens, true)) {
			// For post-related screens, make sure we're dealing with products
			if (in_array($current_screen, array('post.php', 'post-new.php', 'edit.php'))) {
				global $post_type;
				if ($post_type !== 'product') {
					return;
				}
			}

			Assets\enqueue_asset(
				CPPAI_DIR . '/assets/admin/dist',
				self::DEV_SCRIPT,
				$this->get_config()
			);
			wp_localize_script(self::HANDLE, self::OBJ_NAME, $this->get_data());
		}
	}

	/**
	 * Get the script configuration.
	 *
	 * @return array The script configuration.
	 */
	public function get_config()
	{
		return array(
			'dependencies' => array('react', 'react-dom'),
			'css-dependencies' => array('wp-components'),
			'handle'       => self::HANDLE,
			'in-footer'    => true,
		);
	}

	/**
	 * Get data for script localization.
	 *
	 * @return array The localized script data.
	 */
	public function get_data()
	{

		return array(
			'developer' => 'prappo',
			'isAdmin'   => is_admin(),
			'apiUrl'    => rest_url(),
			'nonce'     => wp_create_nonce('wp_rest'),
			'userInfo'  => $this->get_user_data(),
		);
	}

	/**
	 * Get user data for script localization.
	 *
	 * @return array The user data.
	 */
	private function get_user_data()
	{
		$username   = '';
		$avatar_url = '';

		if (is_user_logged_in()) {
			// Get current user's data .
			$current_user = wp_get_current_user();

			// Get username.
			$username = $current_user->user_login; // or use user_nicename, display_name, etc.

			// Get avatar URL.
			$avatar_url = get_avatar_url($current_user->ID);
		}

		return array(
			'username' => $username,
			'avatar'   => $avatar_url,
		);
	}
}
