<?php
/**
 * Runtime requirements checks for Product Fees Toolkit for WooCommerce.
 *
 * Provides utility methods to determine if WordPress and WooCommerce meet
 * the minimum version requirements and whether WooCommerce is active.
 *
 * @package ProductFeesToolkitForWooCommerce
 */

declare(strict_types=1);

namespace PFTFW\Infrastructure;

/**
 * Provides static checks for plugin runtime requirements.
 */
final class Requirements {

	/** Minimum supported WordPress version. */
	public const MIN_WP = '6.0';
	/** Minimum supported WooCommerce version. */
	public const MIN_WC = '9.0';

	/**
	 * Determine whether WooCommerce is installed and active.
	 *
	 * @return bool True when WooCommerce is active, false otherwise.
	 */
	public static function wc_active(): bool {
		return class_exists( 'WooCommerce' ) && defined( 'WC_VERSION' );
	}

	/**
	 * Check whether the current WordPress version meets the minimum requirement.
	 *
	 * @return bool True when the site WordPress version is compatible, false otherwise.
	 */
	public static function wp_version_ok(): bool {
		global $wp_version;
		return version_compare( (string) $wp_version, self::MIN_WP, '>=' );
	}

	/**
	 * Check whether the current WooCommerce version meets the minimum requirement.
	 *
	 * Returns false if WooCommerce is not active.
	 *
	 * @return bool True when WooCommerce is active and the version is compatible.
	 */
	public static function wc_version_ok(): bool {
		if ( ! self::wc_active() ) {
			return false;
		}
		return version_compare( (string) WC_VERSION, self::MIN_WC, '>=' );
	}
}
