<?php
/**
 * Plugin autoloader.
 *
 * Converts PFTFW namespaced classes to WP-style class files first, then PSR-4 fallback.
 *
 * @package ProductFeesToolkitForWooCommerce
 */

declare(strict_types=1);

namespace PFTFW\Infrastructure;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin class autoloader.
 *
 * Registers a namespaced autoloader that first attempts WordPress-style
 * class file resolution (class-{hyphenated}.php) with a condensed variant
 * fallback, and then falls back to PSR-4 class file resolution.
 *
 * @since 1.0.0
 */
final class Autoloader {
	/**
	 * Register the autoloader for this plugin only.
	 *
	 * @return void
	 */
	public static function register(): void {
		spl_autoload_register(
			static function ( $class_name ) {
				if ( 0 !== strpos( (string) $class_name, 'PFTFW\\' ) ) {
					return;
				}

				$relative = substr( (string) $class_name, strlen( 'PFTFW\\' ) );
				$parts    = explode( '\\', (string) $relative );
				$class    = (string) array_pop( $parts );

				$subdir = '';
				if ( ! empty( $parts ) ) {
					$subdir = implode( '/', $parts ) . '/';
				}

				$dirs = array( PFTFW_PLUGIN_DIR . 'src/' . $subdir );
				if ( '' !== $subdir ) {
					$dirs[] = PFTFW_PLUGIN_DIR . 'src/' . strtolower( $subdir );
				}

				// Normalize underscores as word boundaries before hyphenation so
				// e.g. Product_Settings => product-settings (matches class-product-settings.php).
				$base       = str_replace( '_', '', $class );
				$hyphenated = strtolower( preg_replace( '/(?<!^)[A-Z]/', '-$0', (string) $base ) );
				$condensed  = strtolower( preg_replace( '/[^A-Za-z0-9]+/', '', (string) $class ) );

				foreach ( $dirs as $dir ) {
					// 1) WordPress-style: class-{hyphenated-lowercase}.php.
					$wp_file = $dir . 'class-' . $hyphenated . '.php';
					if ( is_readable( $wp_file ) ) {
						require_once $wp_file;
						return;
					}

					// 1b) Alternate WP-style for CamelCase classes: class-{lowercasewithoutseparators}.php.
					$alt_wp = $dir . 'class-' . $condensed . '.php';
					if ( is_readable( $alt_wp ) ) {
						require_once $alt_wp;
						return;
					}

					// 2) PSR-4 fallback: {ClassName}.php.
					$psr4_file = $dir . $class . '.php';
					if ( is_readable( $psr4_file ) ) {
						require_once $psr4_file;
						return;
					}
				}
			}
		);
	}
}
